/** 
* @file OpenGLMethods.h
* Declaration of class OpenGLMethods.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/

#ifndef __OpenGLMethods_h__
#define __OpenGLMethods_h__

#include "Representations/Perception/Image.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/PlayersPercept.h"
#include "Representations/Perception/SensorDataBuffer.h"
#include "Representations/Motion/JointData.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Tools/Math/Points3D.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorTableCuboids.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorTableReferenceColor.h"
#include "Tools/RingBuffer.h"

#include "Representations/Perception/PSDPercept.h"

/**
* @class OpenGLMethods
*
* Defines static methods to initialize and paint with openGL
*/
class OpenGLMethods 
{
public:
  /** IDs for different OpenGL lists */
  enum {colorSpaceCubeList = 1, 
    colorSpaceSubCubeList,
    colorSpaceColorTableList,
    colorSpaceColorTableCuboidsList,
    colorSpaceColorTableReferenceColorList,
    colorSpacePixelList, 
    radarViewer3DAxesList,
    radarViewer3DPoints3DList,
    radarViewer3DBallPerceptList,
    radarViewer3DLandmarksPerceptList,
    radarViewer3DPlayersPerceptList,
    radarViewer3DLinesPerceptList,
    radarViewer3DObstaclesPerceptList,
    radarViewer3DObstaclesModelList,
    radarViewer3DSensorDataList,
    radarViewer3DJointDataList,
    radarViewer3DAngleOfViewList,
    radarViewer3DImageList,
    radarViewer3DImageOnGroundList,
	radarViewer3DPSDPerceptList
  };
  
  /** returns (value+offset)/scale */
  static double transformCoordinates(
    int scale,
    int offset,
    int value
    );
  
  /** sets the window pixel format for the specified HDC*/
  static BOOL setWindowPixelFormat(HDC hDC);
  
  /** creates an openGL view context*/
  static BOOL createViewGLContext(HDC hDC, HGLRC& hGLContext);
  
  /** writes a cube to an openGL list*/
  static void paintCubeToOpenGLList(
    int xSize, int ySize, int zSize, int listID, bool paintFullCube,
    int scale,
    int offsetX,int offsetY,int offsetZ,
    int r, int g, int b
    );
  
  /** writes a cube to an openGL list*/
  static void paintCubeToOpenGLList(
    int xLow, int yLow, int zLow, 
    int xHigh, int yHigh, int zHigh, 
    int listID, bool paintFullCube,
    int scale,
    int offsetX,int offsetY,int offsetZ,
    int r, int g, int b,
    bool beginList = true, bool endList = true
    );
  
  /** writes a sphere and the necessary texture coordinates*/
  static void paintSphere(float* c,float r,int n);

  /** writes a xyz coordinate system to an openGL list*/
  static void paintCoordinateSystem(int listID);
  
  /** writes a Points3D structure to an openGL list */
  static void paintPoints3DToOpenGLList(const Points3D& points, int listID);
  
  /** writes a LinesPercept to an openGL list */
  static void paintLinesPerceptToOpenGLList(const LinesPercept& linesPercept, int listID);

  /** writes an ObstaclesPercept to an openGL list */
  static void paintObstaclesPerceptToOpenGLList(const ObstaclesPercept& obstaclesPercept, int listID);

  /** writes an ObstaclesModel to an openGL list */
  static void paintObstaclesModelToOpenGLList(const ObstaclesModel& obstaclesModel, int listID);

  /** writes SensorData to an openGL list */
  static void paintSensorDataToOpenGLList(const SensorData& sensorData, int listID);

  /** writes JointData to an openGL list */
  static void paintJointDataToOpenGLList(const RingBuffer<JointData, 200>& jointDataRingBuffer, int listID);

  /** writes SensorData to an openGL list */
  static void paintSensorDataToOpenGLList(const RingBuffer<SensorData, 200>& sensorDataRingBuffer, int listID);

  /** writes a LandmarksPercept to an openGL list */
  static void paintLandmarksPerceptToOpenGLList(const CameraMatrix& cameraMatrix, const LandmarksPercept& landmarksPercept, int listID);

  /** writes a BallPercept to an openGL list */
  static void paintBallPerceptToOpenGLList(const BallPercept& ballPercept, int listID);

  /** writes a PlayersPercept to an openGL list */
  static void paintPlayersPerceptToOpenGLList(const PlayersPercept& playersPercept, int listID);

  /** writes the angle of view to an openGL list */
  static void paintAngleOfViewToOpenGLList(const CameraMatrix& cameraMatrix, const CameraInfo& cameraInfo, int listID);

  /** writes the PSDPercept to an openGL list */
  static void paintPSDPerceptToOpenGLList(const CameraMatrix& cameraMatrix, const PSDPercept& psdPercept, int listID);

  /** writes an image to an openGL list */
  static void paintRotatedImageToOpenGLList(
    const CameraMatrix& cameraMatrix,
    const Image& image,
    bool clipPointsBelowGround,
    int listID,
    bool beginList,
    bool endList);

  static void paintColorTable(
    const ColorTable& colorTable,
    int listID);

  static void paintColorTableCuboids(
    const ColorTableCuboids& colorTableCuboids,
    int listID);

  static void paintColorTableReferenceColor(
    const ColorTableReferenceColor& colorTableReferenceColor,
    int listID);

  /** writes the pixels of the image to an openGL list*/
  static void paintImagePixelsToOpenGLList(
    const Image& image,
    int colorModel,
    int zComponent,
    bool polygon,
    int listID
    );
  
  /** writes the pixels of the image between y1 and y2 to an openGL list*/
  static void paintImagePixelsToOpenGLList(
    const Image& image,
    int colorModel,
    int zComponent,
    bool polygon,
    int listID,
    int x1,
    int x2,
    int y1,
    int y2
    );

private:
  /** inserts an invisible quad into the display list to overcome an error of the ATI Mobility FireGL 9000. */
  static void fireGLHack();
};
#endif // __OpenGLMethods_h__


/*
* Change log :
*
* $Log: OpenGLMethods.h,v $
* Revision 1.1.1.1  2004/05/22 17:30:12  cvsadm
* created new repository GT2004_WM
*
* Revision 1.5  2004/02/14 13:05:57  roefer
* SensorData drawn for feet and knees
*
* Revision 1.4  2004/01/14 19:19:02  mellmann
* Visualization of PSDPercept.
*
* Revision 1.3  2003/12/15 11:49:07  juengel
* Introduced CameraInfo
*
* Revision 1.2  2003/10/23 07:26:57  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.1  2003/10/07 10:11:08  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.6  2003/09/26 15:28:24  juengel
* Renamed DataTypes to representations.
*
* Revision 1.5  2003/09/11 15:56:47  juengel
* Added paintSensorDataToOpenGLList and paintJointDataToOpenGLList.
*
* Revision 1.4  2003/08/23 14:28:33  roefer
* fireGLHack added
*
* Revision 1.3  2003/08/18 11:53:19  juengel
* Added paintColorTableReferenceColorThresholds.
*
* Revision 1.2  2003/07/21 13:41:53  juengel
* Moved ColorTableReferenceColor and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.1.1.1  2003/07/02 09:40:27  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.17  2003/06/26 12:35:45  juengel
* Visualization of ObstaclesModel.
*
* Revision 1.16  2003/04/01 14:00:59  juengel
* Added visualization for color tables.
*
* Revision 1.15  2003/03/28 11:37:31  timrie
* Added changelog
*
* Revision 1.1  2002/09/10 15:49:12  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
*/
