/** 
 * @file ImageMethods.h
 * Declaration of class ImageMethods.
 *
 * @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
 */

#ifndef __ImageMethods_h__
#define __ImageMethods_h__

#include "Representations/Perception/Image.h"
#include "Representations/Perception/ColorClassImage.h"

/**
* @class ImageMethods
*
* Defines static methods to paint and manipulate images.
*/
class ImageMethods 
{
public:

  /** 
  * Paints an image to a CDC. The three color channels of the
  * image are interpreted as y, u and v.
  * @param dcMem The CDC the image is painted to.
  * @param image the image to paint.
  * @param destinationRect Specifies where the image is painted in the CDC.
  */
	static void ImageMethods::paintImage2CDCAsYUV
	(
	CDC& dc,
	const Image& image,
	CRect destinationRect
	);
  /** 
  * Paints an image to a CDC. The three color channels of the
  * image are interpreted as y, u and v.
  * @param dcMem The CDC the image is painted to.
  * @param image the image to paint.
  * @param destinationRect Specifies where the image is painted in the CDC.
	* @param swImage Specifies if a color image or a black and white Image is drawn.
  */
  static void paintImage2CDCAsYUV
	(
  CDC& dcMem,
  const Image& image,
  CRect destinationRect,
	bool bwImage
  );

  /** 
  * Paints a color class image to a CDC.
  * @param dc The CDC the image is painted to.
  * @param colorClassImage the image to paint.
  * @param destinationRect Specifies where the color class image is painted in the CDC.
  */
  static void paintColorClassImage2CDC(
    CDC& dc, 
    const ColorClassImage& colorClassImage,
    CRect destinationRect
    );

  /** 
  * Paints the color space usage of an image to a CDC.
  * The three color channels of the image are interpreted as y, u and v.
  * The view to the color space is along the y axis.
  * @param image the image to paint.
  * @param dcMem The CDC the color space usage is painted to.
  */
  static void paint_U_V_ColorSpaceUsage2CDC(
    CDC& dcMem,
    const Image& image
    );

  /** 
  * Paints the color space usage of an image to a bitmap.
  * The three color channels of the image are interpreted as y, u and v.
  * The view to the color space is along the y axis.
  * The pixels are added to the bitmap so the old bitmap is not overwritten
  * everywehre.
  * @param image the image to paint.
  * @param pBuffer Points to the bitmap the color space usage is painted to.
  */
  static void paint_U_V_ColorSpaceUsage2CDC(
    BITMAPINFOHEADER* pBuffer,
    const Image& image
    );

  /** 
  * Paints the color space usage of an image to a CDC.
  * The three color channels of the image are interpreted as y, u and v.
  * The view to the color space is along the v axis.
  * @param image The image.
  * @param dcMem The CDC the color space usage is painted to.
  */
  static void paint_U_Y_ColorSpaceUsage2CDC(
    CDC& dcMem,
    const Image& image
    );

  /** 
  * Paints the color space usage of an image to a bitmap.
  * The three color channels of the image are interpreted as y, u and v.
  * The view to the color space is along the v axis.
  * The pixels are added to the bitmap so the old bitmap is not overwritten
  * everywehre.
  * @param image the image to paint.
  * @param pBuffer Points to the bitmap the color space usage is painted to.
  */
  static void paint_U_Y_ColorSpaceUsage2CDC(
    BITMAPINFOHEADER* pBuffer,
    const Image& image
    );

  /** 
  * Paints the color space usage of an image to a CDC.
  * The three color channels of the image are interpreted as y, u and v.
  * The view to the color space is along the u axis.
  * @param image The image.
  * @param dcMem The CDC the color space usage is painted to.
  */
  static void paint_V_Y_ColorSpaceUsage2CDC(
    CDC& dcMem,
    const Image& image
    );

  /** 
  * Paints the color space usage of an image to a bitmap.
  * The three color channels of the image are interpreted as y, u and v.
  * The view to the color space is along the u axis.
  * The pixels are added to the bitmap so the old bitmap is not overwritten
  * everywehre.
  * @param image the image to paint.
  * @param pBuffer Points to the bitmap the color space usage is painted to.
  */
  static void paint_V_Y_ColorSpaceUsage2CDC(
    BITMAPINFOHEADER* pBuffer,
    const Image& image
    );

  /** 
  * Writes the RobotControl logo to the image and initializes the rest of the image.
  * @param image The image to paint.
  */
  static void writeRobotControlLogoToImage(Image& image);
};
#endif // __ImageMethods_h__


/*
 * Change log :
 * 
 * $Log: ImageMethods.h,v $
 * Revision 1.1.1.1  2004/05/22 17:30:10  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2004/04/07 13:00:44  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.2  2004/03/29 15:19:03  Marc
 * Intruduced the Black and White Image
 * Normal Images (not Jpeg) images were now send as Color Image with BW
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:28:24  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2002/12/18 16:22:56  dueffert
 * doxygen docu corrected
 *
 * Revision 1.3  2002/12/15 23:33:02  dueffert
 * rgb-yuv-convertion moved to Image
 *
 * Revision 1.2  2002/09/20 23:28:34  juengel
 * Moved instance of DebugDrawingManager to RobotControlMainFrame.
 *
 * Revision 1.1  2002/09/10 15:49:12  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 */
