/** 
* @file DrawingMethods.cpp
* Implementation of class DrawingMethods.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/
#include "DrawingMethods.h"


#include "Tools/FieldDimensions.h"
#include "Tools/Math/Geometry.h"
#include "Platform/SystemCall.h"

void DrawingMethods::paintFieldPolygons
(
 DebugDrawing& fieldPolygonsDrawing,
 Player::teamColor ownColor
 )
{
  if(ownColor == Player::blue)
  {
    //yellow goal
    fieldPolygonsDrawing.rectangle(
      xPosOwnGoalpost, xPosOwnGoal,
      yPosLeftGoal, yPosRightGoal,
      1,
      Drawings::ps_solid,
      DebugDrawing::Color(63,127,200),
      Drawings::bs_solid,
      DebugDrawing::Color(63,127,200));
    
    //skyblue goal
    fieldPolygonsDrawing.rectangle(
      xPosOpponentGoalpost, xPosOpponentGoal,
      yPosLeftGoal, yPosRightGoal,
      1,
      Drawings::ps_solid,
      DebugDrawing::Color(200,200,63),
      Drawings::bs_solid,
      DebugDrawing::Color(200,200,63));
  }
  else
  {
    //skyblue goal
    fieldPolygonsDrawing.rectangle(
      xPosOwnGroundline, xPosOwnGoal,
      yPosLeftGoal, yPosRightGoal,
      1,
      Drawings::ps_solid,
      DebugDrawing::Color(200,200,63),
      Drawings::bs_solid,
      DebugDrawing::Color(200,200,63));
    
    //yellow goal
    fieldPolygonsDrawing.rectangle(
      xPosOpponentGroundline, xPosOpponentGoal,
      yPosLeftGoal, yPosRightGoal,
      1,
      Drawings::ps_solid,
      DebugDrawing::Color(63,127,200),
      Drawings::bs_solid,
      DebugDrawing::Color(63,127,200));
  }
  Vector2<int> points[16];
  points[0].x = xPosOwnGroundline;
  points[0].y = yPosLeftGroundline;
  points[1].x = xPosOwnSideCorner;
  points[1].y = yPosLeftSideline;
  points[2].x = xPosOpponentSideCorner;
  points[2].y = yPosLeftSideline;
  points[3].x = xPosOpponentGroundline;
  points[3].y = yPosLeftGroundline;
  points[4].x = xPosOpponentGroundline;
  points[4].y = yPosLeftGoal;
  points[5].x = xPosOpponentGoalpost;
  points[5].y = yPosLeftGoal;
  points[6].x = xPosOpponentGoalpost;
  points[6].y = yPosRightGoal;
  points[7].x = xPosOpponentGroundline;
  points[7].y = yPosRightGoal;
  points[8].x = xPosOpponentGroundline;
  points[8].y = yPosRightGroundline;
  points[9].x = xPosOpponentSideCorner;
  points[9].y = yPosRightSideline;
  points[10].x = xPosOwnSideCorner;
  points[10].y = yPosRightSideline;
  points[11].x = xPosOwnGroundline;
  points[11].y = yPosRightGroundline;
  points[12].x = xPosOwnGroundline;
  points[12].y = yPosRightGoal;
  points[13].x = xPosOwnGoalpost;
  points[13].y = yPosRightGoal;
  points[14].x = xPosOwnGoalpost;
  points[14].y = yPosLeftGoal;
  points[15].x = xPosOwnGroundline;
  points[15].y = yPosLeftGoal;
  
  fieldPolygonsDrawing.polygon(
    &points[0],
    16,
    1,
    Drawings::ps_solid,
    DebugDrawing::Color(63,127,63),
    Drawings::bs_solid,
    DebugDrawing::Color(63,127,63));
}


void DrawingMethods::paintFieldLines
(
 DebugDrawing& fieldLinesDrawing
 )
{
  //Center Circle
  fieldLinesDrawing.ellipse(
    xPosHalfWayLine - centerCircleRadius, yPosCenterGoal - centerCircleRadius,
    xPosHalfWayLine + centerCircleRadius, yPosCenterGoal + centerCircleRadius,
    1, Drawings::ps_solid, 
    DebugDrawing::Color(0,0,0), 
    Drawings::bs_null,
    DebugDrawing::Color(63,127,63));
  
  //Lines parallel to the side lines
  fieldLinesDrawing.line(
    xPosBackFlags,yPosLeftFlags,
    xPosFrontFlags,yPosLeftFlags);  
  
  fieldLinesDrawing.line(
    xPosOwnSideCorner, yPosLeftSideline,
    xPosOpponentSideCorner,yPosLeftSideline);  
  
  fieldLinesDrawing.line(
    xPosOwnGoal, yPosLeftGoal,
    xPosOwnGroundline,yPosLeftGoal);  
  
  fieldLinesDrawing.line(
    xPosOwnGoal, yPosRightGoal,
    xPosOwnGroundline,yPosRightGoal);
  
  fieldLinesDrawing.line(
    xPosOpponentGoal, yPosLeftGoal, 
    xPosOpponentGroundline,yPosLeftGoal);
  
  fieldLinesDrawing.line(
    xPosOpponentGoal, yPosRightGoal, 
    xPosOpponentGroundline,yPosRightGoal);
  
  fieldLinesDrawing.line(
    xPosOwnSideCorner,yPosRightSideline, 
    xPosOpponentSideCorner,yPosRightSideline);
  
  fieldLinesDrawing.line(
    xPosBackFlags,yPosRightFlags,
    xPosFrontFlags,yPosRightFlags);  
  
  //Lines parallel to the ground lines
  fieldLinesDrawing.line(
    xPosOwnGoal,yPosLeftGoal, 
    xPosOwnGoal,yPosRightGoal);
  
  fieldLinesDrawing.line(
    xPosBackFlags,yPosLeftFlags, 
    xPosBackFlags,yPosLeftGoal);
  
  fieldLinesDrawing.line(
    xPosBackFlags,yPosRightFlags, 
    xPosBackFlags,yPosRightGoal);
  
  
  fieldLinesDrawing.line(
    xPosOwnGroundline,yPosRightGroundline, 
    xPosOwnGroundline,yPosLeftGroundline);
  
  
  fieldLinesDrawing.line(
    xPosOwnPenaltyArea,yPosLeftPenaltyArea, 
    xPosOwnPenaltyArea,yPosRightPenaltyArea);
  
  fieldLinesDrawing.line(
    xPosOpponentPenaltyArea,yPosLeftPenaltyArea, 
    xPosOpponentPenaltyArea,yPosRightPenaltyArea);
  
#ifdef LARGE_FIELD
  fieldLinesDrawing.line(
    xPosOwnGroundline,yPosLeftPenaltyArea, 
    xPosOwnPenaltyArea,yPosLeftPenaltyArea);
  
  fieldLinesDrawing.line(
    xPosOwnPenaltyArea,yPosRightPenaltyArea, 
    xPosOwnGroundline,yPosRightPenaltyArea);
  
  fieldLinesDrawing.line(
    xPosOpponentGroundline,yPosLeftPenaltyArea, 
    xPosOpponentPenaltyArea,yPosLeftPenaltyArea);
  
  fieldLinesDrawing.line(
    xPosOpponentPenaltyArea,yPosRightPenaltyArea, 
    xPosOpponentGroundline,yPosRightPenaltyArea);
#endif
  
  fieldLinesDrawing.line(
    xPosHalfWayLine,yPosRightSideline, 
    xPosHalfWayLine,yPosLeftSideline);
  
  fieldLinesDrawing.line(
    xPosOpponentGroundline,yPosRightGroundline, 
    xPosOpponentGroundline,yPosLeftGroundline);
  
  
  fieldLinesDrawing.line(
    xPosFrontFlags,yPosLeftFlags, 
    xPosFrontFlags,yPosLeftGoal);
  
  fieldLinesDrawing.line(
    xPosFrontFlags,yPosRightFlags, 
    xPosFrontFlags,yPosRightGoal);
  
  fieldLinesDrawing.line(
    xPosOpponentGoal,yPosLeftGoal, 
    xPosOpponentGoal,yPosRightGoal);
  
  ////////
  fieldLinesDrawing.line(
    xPosOpponentGroundline,yPosRightGroundline, 
    xPosOpponentSideCorner,yPosRightSideline);
  
  fieldLinesDrawing.line(
    xPosOpponentGroundline,yPosLeftGroundline, 
    xPosOpponentSideCorner,yPosLeftSideline);
  
  fieldLinesDrawing.line(
    xPosOwnGroundline,yPosRightGroundline, 
    xPosOwnSideCorner,yPosRightSideline);
  
  fieldLinesDrawing.line(
    xPosOwnGroundline,yPosLeftGroundline, 
    xPosOwnSideCorner,yPosLeftSideline);
  
  //Diagonal lines
  fieldLinesDrawing.line(
    xPosFrontFlags,yPosRightFlags, 
    xPosOpponentGroundline,yPosRightGroundline);
  fieldLinesDrawing.line(
    xPosFrontFlags,yPosRightFlags, 
    xPosOpponentSideCorner,yPosRightSideline);
  
  fieldLinesDrawing.line(
    xPosFrontFlags,yPosLeftFlags, 
    xPosOpponentGroundline,yPosLeftGroundline);
  fieldLinesDrawing.line(
    xPosFrontFlags,yPosLeftFlags, 
    xPosOpponentSideCorner,yPosLeftSideline);
  
  fieldLinesDrawing.line(
    xPosBackFlags,yPosRightFlags, 
    xPosOwnGroundline,yPosRightGroundline);
  fieldLinesDrawing.line(
    xPosBackFlags,yPosRightFlags, 
    xPosOwnSideCorner,yPosRightSideline);
  
  fieldLinesDrawing.line(
    xPosBackFlags,yPosLeftFlags, 
    xPosOwnGroundline,yPosLeftGroundline);
  fieldLinesDrawing.line(
    xPosBackFlags,yPosLeftFlags, 
    xPosOwnSideCorner,yPosLeftSideline);
  
}

void DrawingMethods::paintPerceptCollectionForRadarView
(   
 DebugDrawing& perceptsDrawing, 
 const LandmarksPercept& landmarksPercept,
 const BallPercept& ballPercept,
 const PlayersPercept& playersPercept,
 const ObstaclesPercept& obstaclesPercept,
 const LinesPercept& linesPercept,
 const EdgesPercept& edgesPercept
 )
{
  Pose2D origin(0,0,0);
  //(pi_2,Vector2<double>(0,yPosRightSideline * 0.3));
  /*  
  perceptsDrawing.line(0,yPosLeftSideline,
  0,yPosRightSideline,
  Drawings::ps_null,0,DebugDrawing::Color(0,0,0));
  perceptsDrawing.line(int(1.3 * yPosRightSideline),0,
  int(1.3 * yPosLeftSideline),0,
  Drawings::ps_null,0,DebugDrawing::Color(0,0,0));
  */
  perceptsDrawing.circle((int) origin.translation.x,(int) origin.translation.y,100,1,
    Drawings::ps_solid,DebugDrawing::Color(0,0,0),
    Drawings::bs_solid,DebugDrawing::Color(0,0,0));
  
  paintPerceptCollectionForFieldView(perceptsDrawing,landmarksPercept,
    ballPercept, playersPercept, obstaclesPercept, linesPercept, edgesPercept, origin, Player::blue);
}

void DrawingMethods::paintPerceptCollectionForFieldView
(
 DebugDrawing& perceptsDrawing, 
 const LandmarksPercept& landmarksPercept,
 const BallPercept& ballPercept,
 const PlayersPercept& playersPercept,
 const ObstaclesPercept& obstaclesPercept,
 const LinesPercept& linesPercept,
 const EdgesPercept& edgesPercept,
 const RobotPose& origin,
 const Player::teamColor teamColor
 )
{
  RobotPose newOrigin = origin;
  /*if(teamColor == Player::blue)
  {
  newOrigin.translation.x *= -1;
  newOrigin.translation.y *= -1;
  newOrigin.rotation += pi;
  }
  */
  paintLandmarksPerceptForFieldView(perceptsDrawing, landmarksPercept, newOrigin);
  paintPlayersPerceptForFieldView(perceptsDrawing, playersPercept, newOrigin);
  paintObstaclesPerceptForFieldView(perceptsDrawing, obstaclesPercept, newOrigin);
  paintBallPerceptForFieldView(perceptsDrawing, ballPercept, newOrigin);
  paintLinesPerceptForFieldView(perceptsDrawing, linesPercept, newOrigin);
  paintEdgesPerceptForFieldView(perceptsDrawing, edgesPercept, newOrigin);
}

void DrawingMethods::paintBallPerceptForFieldView(DebugDrawing& ballDrawing,
                                                  const BallPercept& ballPercept,
                                                  const Pose2D& origin)
{
  if(ballPercept.ballWasSeen)
  {
    Vector2<double> ballOffset;
    
    // bearing based
    ballPercept.getOffsetBearingBased(ballOffset);
    
    Vector2<double> bp = (origin + Pose2D(ballOffset)).translation;
    ballDrawing.line((int) origin.translation.x,(int) origin.translation.y,
      (int) bp.x,(int) bp.y,
      Drawings::ps_solid,2,DebugDrawing::Color(255,0,0)
      );
    ballDrawing.circle((int) bp.x,(int) bp.y,(int)ballRadius,2,Drawings::ps_solid,
      DebugDrawing::Color(255,0,0),Drawings::bs_null,DebugDrawing::Color(0,0,0));
    
    // size based
    ballPercept.getOffsetSizeBased(ballOffset);
    
    bp = (origin + Pose2D(ballOffset)).translation;
    ballDrawing.line((int) origin.translation.x,(int) origin.translation.y,
      (int) bp.x,(int) bp.y,
      Drawings::ps_solid,2,DebugDrawing::Color(255,128,64)
      );
    ballDrawing.circle((int) bp.x,(int) bp.y,(int)ballRadius,2,Drawings::ps_solid,
      DebugDrawing::Color(255,128,64),Drawings::bs_null,DebugDrawing::Color(0,0,0));
  }
}

void DrawingMethods::paintLandmarksPerceptForFieldView(DebugDrawing& landmarksDrawing,
                                                       const LandmarksPercept &landmarksPercept,
                                                       const Pose2D& origin)
{
  for (int i=0;i<landmarksPercept.numberOfFlags;i++)
  {
    const Flag& flag = landmarksPercept.flags[i];
    
    if(!flag.isOnBorder(flag.x.min))
    {
      Vector2<double> lpright;
      lpright.x=cos(flag.x.min)*flag.distance*1.2;
      lpright.y=sin(flag.x.min)*flag.distance*1.2;
      lpright = (origin + Pose2D(lpright)).translation;
      landmarksDrawing.line((int) origin.translation.x,(int) origin.translation.y,
        (int) lpright.x,(int) lpright.y,
        //        Drawings::ps_solid,1,DebugDrawing::Color(200,220,220));
        Drawings::ps_solid,1,DebugDrawing::Color(0,255,0));
    }
    
    if(!flag.isOnBorder(flag.x.max))
    {
      Vector2<double> lpleft;
      lpleft.x=cos(flag.x.max)*flag.distance*1.2;
      lpleft.y=sin(flag.x.max)*flag.distance*1.2;
      lpleft = (origin + Pose2D(lpleft)).translation;
      landmarksDrawing.line((int) origin.translation.x,(int) origin.translation.y,
        (int) lpleft.x,(int) lpleft.y,
        Drawings::ps_solid,1,DebugDrawing::Color(255,0,0));
      //        Drawings::ps_solid,1,DebugDrawing::Color(220,220,220));
    }
    
    Vector2<double> lp;
    lp.x=cos(flag.angle)*flag.distance;
    lp.y=sin(flag.angle)*flag.distance;
    lp = (origin + Pose2D(lp)).translation;
    
    //   Vector2<double> lp=origin.translation;
    //    lp.x+=cos(flag.angle)*flag.distance;
    //    lp.y+=sin(flag.angle)*flag.distance;
    
    landmarksDrawing.line((int) origin.translation.x,(int) origin.translation.y,
      (int) lp.x,(int) lp.y,
      Drawings::ps_dash,1,DebugDrawing::Color(150,150,150));
    
    DebugDrawing::Color lowerColor(0,0,0);
    lowerColor = ColorClasses::colorClassToRGB(flag.getLowerColor());
    DebugDrawing::Color upperColor(0,0,0);
    upperColor = ColorClasses::colorClassToRGB(flag.getUpperColor());
    
    landmarksDrawing.circle((int)lp.x,(int)lp.y,int(flagRadius),1,Drawings::ps_solid,
      lowerColor,Drawings::bs_solid,lowerColor);
    landmarksDrawing.circle((int)lp.x,(int)lp.y,(int)(flagRadius*0.5),1,Drawings::ps_solid,
      upperColor,Drawings::bs_solid,upperColor);
    landmarksDrawing.circle((int)lp.x,(int)lp.y,int(flagRadius),1,Drawings::ps_solid,
      DebugDrawing::Color(0,0,0),Drawings::bs_null,DebugDrawing::Color(140,140,140));
  }
  for (int j=0;j<landmarksPercept.numberOfGoals;j++)
  {
    const Goal& goal = landmarksPercept.goals[j];
    
    if(!goal.isOnBorder(goal.x.max))
    {
      Vector2<double> gpleft;
      
      gpleft.x=cos(goal.x.max)*goal.distance*1.2;
      gpleft.y=sin(goal.x.max)*goal.distance*1.2;
      gpleft = (origin + Pose2D(gpleft)).translation;
      landmarksDrawing.line((int)origin.translation.x,(int)origin.translation.y,
        (int)gpleft.x,(int)gpleft.y,
        Drawings::ps_solid,1,DebugDrawing::Color(255,0,0));
    }
    if(!goal.isOnBorder(goal.x.min))
    {
      Vector2<double> gpright;
      gpright.x=cos(goal.x.min)*goal.distance*1.2;
      gpright.y=sin(goal.x.min)*goal.distance*1.2;
      gpright = (origin + Pose2D(gpright)).translation;
      landmarksDrawing.line((int)origin.translation.x,(int)origin.translation.y,
        (int)gpright.x,(int)gpright.y,
        Drawings::ps_solid,1,DebugDrawing::Color(0,255,0));
    }
    
    Pose2D gp;
    gp.translation.x = cos(goal.angle)*goal.distance;
    gp.translation.y = sin(goal.angle)*goal.distance;
    gp = origin + gp;
    
    landmarksDrawing.line((int) origin.translation.x,(int) origin.translation.y,
      (int) gp.translation.x,(int) gp.translation.y,
      Drawings::ps_dash,1,DebugDrawing::Color(150,150,150));
    
    Pose2D leftCorner(gp);
    leftCorner.rotation = origin.rotation + goal.angle + goal.rotation;
    leftCorner = leftCorner + Pose2D(0,yPosLeftGoal);
    
    Pose2D rightCorner(gp);
    rightCorner.rotation = origin.rotation + goal.angle + goal.rotation;
    rightCorner = rightCorner + Pose2D(0,yPosRightGoal);
    
    landmarksDrawing.line(
      (int)leftCorner.translation.x, (int)leftCorner.translation.y, 
      (int)rightCorner.translation.x, (int)rightCorner.translation.y,
      Drawings::ps_solid,3,ColorClasses::colorClassToRGB(goal.color));
  }
}

void DrawingMethods::paintPlayersPerceptForFieldView(DebugDrawing& playersDrawing,
                                                     const PlayersPercept& playersPercept,
                                                     const Pose2D& origin)
{
  for (int i=0;i<playersPercept.numberOfBluePlayers;i++)
  {
    Pose2D pp = origin + Pose2D(playersPercept.bluePlayers[i].offset);
    
    playersDrawing.line((int) origin.translation.x,(int) origin.translation.y,
      (int)(pp.translation.x), (int)(pp.translation.y),
      Drawings::ps_solid,1,DebugDrawing::Color(0,0,255));
    
    playersDrawing.line((int) pp.translation.x - 70,(int) pp.translation.y - 70,
      (int) pp.translation.x - 70,(int) pp.translation.y + 70,
      Drawings::ps_solid,1,DebugDrawing::Color(0,0,255));
    playersDrawing.line((int) pp.translation.x - 70,(int) pp.translation.y + 70,
      (int) pp.translation.x + 70,(int) pp.translation.y + 70,
      Drawings::ps_solid,1,DebugDrawing::Color(0,0,255));
    playersDrawing.line((int) pp.translation.x + 70,(int) pp.translation.y + 70,
      (int) pp.translation.x + 70,(int) pp.translation.y - 70,
      Drawings::ps_solid,1,DebugDrawing::Color(0,0,255));
    playersDrawing.line((int) pp.translation.x + 70,(int) pp.translation.y - 70,
      (int) pp.translation.x - 70,(int) pp.translation.y - 70,
      Drawings::ps_solid,1,DebugDrawing::Color(0,0,255));
  }
  
  for (int j=0;j<playersPercept.numberOfRedPlayers;j++)
  {
    Pose2D pp = origin + Pose2D(playersPercept.redPlayers[j].offset);
    
    playersDrawing.line((int) origin.translation.x,(int) origin.translation.y,
      (int)(pp.translation.x), (int)(pp.translation.y),
      Drawings::ps_solid,1,DebugDrawing::Color(255,0,0));
    
    playersDrawing.line((int) pp.translation.x - 70,(int) pp.translation.y - 70,
      (int) pp.translation.x - 70,(int) pp.translation.y + 70,
      Drawings::ps_solid,1,DebugDrawing::Color(255,0,0));
    playersDrawing.line((int) pp.translation.x - 70,(int) pp.translation.y + 70,
      (int) pp.translation.x + 70,(int) pp.translation.y + 70,
      Drawings::ps_solid,1,DebugDrawing::Color(255,0,0));
    playersDrawing.line((int) pp.translation.x + 70,(int) pp.translation.y + 70,
      (int) pp.translation.x + 70,(int) pp.translation.y - 70,
      Drawings::ps_solid,1,DebugDrawing::Color(255,0,0));
    playersDrawing.line((int) pp.translation.x + 70,(int) pp.translation.y - 70,
      (int) pp.translation.x - 70,(int) pp.translation.y - 70,
      Drawings::ps_solid,1,DebugDrawing::Color(255,0,0));
  }
  
}

void DrawingMethods::paintWorldState(DebugDrawing& debugDrawing,
                                     const RobotPose& robotPose,
                                     const BallModel& ballModel,
                                     const PlayerPoseCollection& playerPoseCollection,
                                     Player::teamColor ownColor,
                                     unsigned long timestamp,
                                     bool oracled)
{
  // ball
  DebugDrawing::Color seenBallColor  = oracled ? DebugDrawing::Color(255,200,100) : DebugDrawing::Color(255,150,50);
  DebugDrawing::Color communicatedBallColor = DebugDrawing::Color(255,100,100);

  // communicated ball
  if (!oracled && timestamp - ballModel.communicated.timeWhenLastObserved < 3000) 
  {
    debugDrawing.line(int(ballModel.communicated.x),
      int(ballModel.communicated.y), 
      (int)robotPose.translation.x,(int)robotPose.translation.y,
      Drawings::ps_dot,1,communicatedBallColor);
    debugDrawing.circle(int(ballModel.communicated.x), int(ballModel.communicated.y), 
      (int)ballRadius,1,Drawings::ps_solid,communicatedBallColor,Drawings::bs_solid,communicatedBallColor);
  }

  // seen ball
  debugDrawing.line(int(ballModel.seen.x),int(ballModel.seen.y), 
    (int)robotPose.translation.x,(int)robotPose.translation.y,
    Drawings::ps_dash,1,seenBallColor);
  debugDrawing.circle(int(ballModel.seen.x),int(ballModel.seen.y), 
    (int)ballRadius,1,Drawings::ps_solid,seenBallColor,Drawings::bs_solid,
    timestamp - ballModel.seen.timeWhenLastSeen < 2000 ? seenBallColor : DebugDrawing::Color(255,255,255));
//  debugDrawing.arrow(int(ballModel.seen.x),int(ballModel.seen.y),
//    int(ballModel.seen.x+ballModel.seen.speed.x),int(ballModel.seen.y+ballModel.seen.speed.y), 
//    Drawings::ps_solid, 1, seenBallColor);

  DebugDrawing::Color ownCol = ownColor == Player::red ? 
    (oracled ? DebugDrawing::Color(255,125,125) : DebugDrawing::Color(255,0,0))
    : (oracled ? DebugDrawing::Color(125,125,255) : DebugDrawing::Color(0,0,255));

  DebugDrawing::Color opponentCol = ownColor == Player::blue ? 
    (oracled ? DebugDrawing::Color(255,125,125) : DebugDrawing::Color(255,0,0))
    : (oracled ? DebugDrawing::Color(125,125,255) : DebugDrawing::Color(0,0,255));

  // robot pose
  drawRobot(debugDrawing,robotPose.getPose(),
    ownCol,DebugDrawing::Color(0,0,0));

  /* Vielleicht das in ein eigenes Drawing, welches nur fr den ausgewhlten Spieler angezeigt wird?
  // player pose collection
  int i;
  for(i = 0; i < playerPoseCollection.numberOfOwnPlayers; ++i)
  {
    drawRobot(debugDrawing,
      playerPoseCollection.getOwnPlayerPose(i).getPose(),
      ownCol,DebugDrawing::Color(128,128,128));
    debugDrawing.circle(
      int(playerPoseCollection.getOwnPlayerPose(i).getPose().translation.x),
      int(playerPoseCollection.getOwnPlayerPose(i).getPose().translation.y),
      int(200.0 - (playerPoseCollection.getOwnPlayerPose(i).getValidity() * 100.0)),
      1,
      Drawings::ps_solid, ownCol,
      Drawings::bs_null, DebugDrawing::Color(0,0,0)
      );
  }
  for(i = 0; i < playerPoseCollection.numberOfOpponentPlayers; ++i)
  {
    drawRobot(debugDrawing,
      playerPoseCollection.getOpponentPlayerPose(i).getPose(),
      opponentCol,DebugDrawing::Color(128,128,128));
    debugDrawing.circle(
      int(playerPoseCollection.getOpponentPlayerPose(i).getPose().translation.x),
      int(playerPoseCollection.getOpponentPlayerPose(i).getPose().translation.y),
      int(200.0 - (playerPoseCollection.getOpponentPlayerPose(i).getValidity() * 100.0)),
      1,
      Drawings::ps_solid, opponentCol,
      Drawings::bs_null, DebugDrawing::Color(0,0,0)
      );
  }*/
}

void DrawingMethods::drawRobot
(
 DebugDrawing& debugDrawing,
 Pose2D pose,
 const DebugDrawing::Color& color,
 const DebugDrawing::Color& headColor
 )
{
  int x[] = { 38,38,-162,-162,-30,  0, 66,66, 0},
    y[] = {-37,37,  37, -37,  0,-37,-25,25,37};
  Vector2<int> points[9];
  
  for(int i = 0; i < 9; ++i)
  {
    Pose2D p = pose + Pose2D(Vector2<double>(x[i],y[i]));
    points[i] = Vector2<int>(int(p.translation.x + 0.5),
      int(p.translation.y + 0.5));
  }
  debugDrawing.polygon(points,4,1,Drawings::ps_solid,color,
    Drawings::bs_solid,color);
  debugDrawing.polygon(points + 4,5,1,Drawings::ps_solid,headColor,
    Drawings::bs_solid,headColor);
  
}

void DrawingMethods::paintLinesPerceptForFieldView
(
 DebugDrawing& linesDrawing,
 const LinesPercept& linesPercept,
 const Pose2D& origin)
{
  for(int i = 0; i < LinesPercept::numberOfTypes; ++i)
  {
    static DebugDrawing::Color colors[] =
    {
      DebugDrawing::Color(Drawings::gray),
        DebugDrawing::Color(Drawings::green),
        DebugDrawing::Color(Drawings::yellow),
        DebugDrawing::Color(Drawings::skyblue),
        DebugDrawing::Color(Drawings::red),
        DebugDrawing::Color(Drawings::blue),
        DebugDrawing::Color(Drawings::orange)
    };
    const DebugDrawing::Color& color = colors[i];
    for(int j = 0; j < linesPercept.numberOfPoints[i]; j++)
    {
      Pose2D linePoint = 
        origin + Pose2D(linesPercept.points[i][j]);
      double angle = linesPercept.points[i][j].angle;
      if(angle != LinesPercept::UNDEF)
      {
        Vector2<double> linePoint2 = (linePoint + Pose2D(angle) + 
          Pose2D(Vector2<double>(100,0))).translation;
        linesDrawing.arrow(        
          (int)linePoint.translation.x,
          (int)linePoint.translation.y,
          (int)linePoint2.x,
          (int)linePoint2.y,
          Drawings::ps_solid, 1, color);
      }
      linesDrawing.circle(
        (int)linePoint.translation.x,
        (int)linePoint.translation.y,
        10,
        1,
        Drawings::ps_solid, color,
        Drawings::bs_solid, color);
    }
  }
}

void DrawingMethods::paintEdgesPerceptForFieldView
(
 DebugDrawing& linesDrawing,
 const EdgesPercept& edgesPercept,
 const Pose2D& origin)
{
  static DebugDrawing::Color colors[] =
  {
    DebugDrawing::Color(Drawings::gray),
    DebugDrawing::Color(Drawings::green),
    DebugDrawing::Color(Drawings::yellow),
    DebugDrawing::Color(Drawings::skyblue),
    DebugDrawing::Color(Drawings::red),
    DebugDrawing::Color(Drawings::blue),
    DebugDrawing::Color(Drawings::orange)
  };
  for(int i = 0; i < edgesPercept.numberOfEdges; i++)
  {
    const DebugDrawing::Color& color = colors[i%7];
    Vector2<double> point1 = (origin + Pose2D(edgesPercept.edges[i].point1)).translation;
    Vector2<double> point2 = (origin + Pose2D(edgesPercept.edges[i].point1)).translation;
    linesDrawing.line(
      (int)point1.x,
      (int)point1.y,
      (int)point2.x,
      (int)point2.y,
      Drawings::ps_solid, 1, color);
  }
}

void DrawingMethods::paintObstaclesPerceptForFieldView
(
 DebugDrawing& obstaclesDrawing,
 const ObstaclesPercept& obstaclesPercept,
 const Pose2D& origin)
{
  DebugDrawing::Color color(127,255,127);
  for(int j = 0; j < obstaclesPercept.numberOfPoints; j++)
  {
    Vector2<double> nearObstaclePoint = 
      (origin + Pose2D(obstaclesPercept.nearPoints[j])).translation;
    
    Vector2<double> farObstaclePoint = 
      (origin + Pose2D(obstaclesPercept.farPoints[j])).translation;
    
    obstaclesDrawing.line(
      (int)nearObstaclePoint.x, (int)nearObstaclePoint.y, 
      (int)farObstaclePoint.x, (int)farObstaclePoint.y,
      Drawings::ps_solid, 1, color);
    
    DebugDrawing::Color color(0,0,0);
    switch(obstaclesPercept.obstacleType[j])
    {
      case ObstaclesPercept::unknown: color = Drawings::black; break;
      case ObstaclesPercept::opponent: color = Drawings::red; break;
      case ObstaclesPercept::teammate: color = Drawings::green; break;
      case ObstaclesPercept::goal: color = Drawings::blue; break;
      case ObstaclesPercept::border: color = Drawings::light_gray; break;
    }
    obstaclesDrawing.circle(
      (int)farObstaclePoint.x, (int)farObstaclePoint.y,
      40, 1, Drawings::ps_solid, color, Drawings::bs_solid, color);
    
    if(obstaclesPercept.farPointIsOnImageBorder[j])
      obstaclesDrawing.circle( (int)farObstaclePoint.x, (int)farObstaclePoint.y,
      20, 1, Drawings::ps_solid, color, Drawings::bs_solid, color);
  }
  
  
  DebugDrawing::Color freePartColor[2];
  freePartColor[0] = Drawings::red;
  freePartColor[1] = Drawings::green;
  
  for(int i = 0; i < 2; i++)
  {
    if(obstaclesPercept.angleToFreePartOfGoalWasDetermined[i])
    {
      Vector2<double> vLeft = (
        origin + 
        Pose2D(obstaclesPercept.angleToFreePartOfGoal[i] + obstaclesPercept.widthOfFreePartOfGoal[i] / 2.0) + 
        Pose2D(Vector2<double>(obstaclesPercept.distanceToFreePartOfGoal[i], 0))
        ).translation;
      
      Vector2<double> vRight = (
        origin + 
        Pose2D(obstaclesPercept.angleToFreePartOfGoal[i] - obstaclesPercept.widthOfFreePartOfGoal[i] / 2.0) + 
        Pose2D(Vector2<double>(obstaclesPercept.distanceToFreePartOfGoal[i] ? obstaclesPercept.distanceToFreePartOfGoal[i] : 500, 0))
        ).translation;
      
      obstaclesDrawing.line(
        int(origin.translation.x), int(origin.translation.y),
        int(vLeft.x), int(vLeft.y),
        Drawings::ps_solid, 5, freePartColor[i]);
      
      obstaclesDrawing.line(
        int(origin.translation.x), int(origin.translation.y),
        int(vRight.x), int(vRight.y),
        Drawings::ps_solid, 5, freePartColor[i]);
    }
  }
}

void DrawingMethods::paintObstaclesModelForFieldView
(
 DebugDrawing& obstaclesDrawing,
 const ObstaclesModel& obstaclesModel,
 const Pose2D& origin)
{
  Vector2<double> point, point2, lastPoint;
  double angle, openingAngle;
  angle = ObstaclesModel::getAngleOfSector(ObstaclesModel::numOfSectors-1);
  openingAngle = tan(pi / ObstaclesModel::numOfSectors);
  
  lastPoint.x = cos(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]
    - sin(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]*openingAngle;
  lastPoint.y = sin(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]
    + cos(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]*openingAngle;
  lastPoint = (origin + Pose2D(lastPoint)).translation;
  
  for(int i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    angle = ObstaclesModel::getAngleOfSector(i);
    
    point.x = cos(angle) * obstaclesModel.distance[i]
      - sin(angle) * obstaclesModel.distance[i]*openingAngle;
    point.y = sin(angle) * obstaclesModel.distance[i]
      + cos(angle) * obstaclesModel.distance[i]*openingAngle;
    
    point2.x = cos(angle) * obstaclesModel.distance[i]
      + sin(angle) * obstaclesModel.distance[i]*openingAngle;
    point2.y = sin(angle) * obstaclesModel.distance[i]
      - cos(angle) * obstaclesModel.distance[i]*openingAngle;
    
    point = (origin + Pose2D(point)).translation;
    point2 = (origin + Pose2D(point2)).translation;
    
    //obstaclesDrawing.line(
    //  (int) origin.translation.x,(int) origin.translation.y,
    //  (int) point2.x,(int) point2.y,
    //  Drawings::ps_solid,1,DebugDrawing::Color(128,200,128));
    //    if (obstaclesModel.distance[i] < ObstaclesModel::maxDistance)

    DebugDrawing::Color color(0,0,0);
  
    switch(obstaclesModel.obstacleType[i])
    {
      case ObstaclesPercept::unknown: color = Drawings::black; break;
      case ObstaclesPercept::opponent: color = Drawings::red; break;
      case ObstaclesPercept::teammate: color = Drawings::green; break;
      case ObstaclesPercept::goal: color = Drawings::blue; break;
      case ObstaclesPercept::border: color = Drawings::light_gray; break;
    }

    obstaclesDrawing.line(
      (int) point2.x,(int) point2.y,
      (int) point.x,(int) point.y,
      Drawings::ps_solid,4, color);
    
    obstaclesDrawing.line(
      (int) point2.x,(int) point2.y,
      (int) lastPoint.x,(int) lastPoint.y,
      Drawings::ps_solid,1,DebugDrawing::Color(127,127,127));
    lastPoint = point;
  }
  
  /*
  point.x = 0;
  point.y = 90;
  point2.x = obstaclesModel.corridorInFront;
  point2.y = 90;
  
    point = (origin + Pose2D(point)).translation;
    point2 = (origin + Pose2D(point2)).translation;
    
      obstaclesDrawing.line(
      (int) point.x,(int) point.y,
      (int) point2.x,(int) point2.y,
      Drawings::ps_solid,2,DebugDrawing::Color(100,100,255));
      
        point.x = 0;
        point.y = -90;
        point2.x = obstaclesModel.corridorInFront;
        point2.y = -90;
        
          point = (origin + Pose2D(point)).translation;
          point2 = (origin + Pose2D(point2)).translation;
          
            obstaclesDrawing.line(
            (int) point.x,(int) point.y,
            (int) point2.x,(int) point2.y,
            Drawings::ps_solid,2,DebugDrawing::Color(100,100,255));
            
              point.x = obstaclesModel.corridorInFront;
              point.y = 90;
              point2.x = obstaclesModel.corridorInFront;
              point2.y = -90;
              
                point = (origin + Pose2D(point)).translation;
                point2 = (origin + Pose2D(point2)).translation;
                
                  obstaclesDrawing.line(
                  (int) point.x,(int) point.y,
                  (int) point2.x,(int) point2.y,
                  Drawings::ps_solid,2,DebugDrawing::Color(100,100,255));
  */
  
  // Free parts of goal:
  DebugDrawing::Color freePartColor[2];
  freePartColor[0] = DebugDrawing::Color(127, 0 , 0);
  freePartColor[1] = DebugDrawing::Color(0, 127 , 0);
  
  for(i = 0; i < 2; i++)
  {
    if(obstaclesModel.angleToFreePartOfGoalWasDetermined[i])
    {
      Vector2<double> vLeft = (
        origin + 
        Pose2D(obstaclesModel.angleToFreePartOfGoal[i] + obstaclesModel.widthOfFreePartOfGoal[i] / 2.0) + 
        Pose2D(Vector2<double>(obstaclesModel.distanceToFreePartOfGoal[i], 0))
        ).translation;
      
      Vector2<double> vRight = (
        origin + 
        Pose2D(obstaclesModel.angleToFreePartOfGoal[i] - obstaclesModel.widthOfFreePartOfGoal[i] / 2.0) + 
        Pose2D(Vector2<double>(obstaclesModel.distanceToFreePartOfGoal[i], 0))
        ).translation;
      
      obstaclesDrawing.line(
        int(origin.translation.x), int(origin.translation.y),
        int(vLeft.x), int(vLeft.y),
        Drawings::ps_solid, 3, freePartColor[i]);
      
      obstaclesDrawing.line(
        int(origin.translation.x), int(origin.translation.y),
        int(vRight.x), int(vRight.y),
        Drawings::ps_solid, 3, freePartColor[i]);
    }
  }
  
  // Next free teammate
  if(obstaclesModel.angleToNextFreeTeammateWasDetermined)
  {
    Vector2<double> v = (
      origin + 
      Pose2D(obstaclesModel.angleToNextFreeTeammate) + 
      Pose2D(Vector2<double>(obstaclesModel.distanceToNextFreeTeammate, 0))
      ).translation;
    
    obstaclesDrawing.line(
      int(origin.translation.x), int(origin.translation.y),
      int(v.x), int(v.y),
      Drawings::ps_solid, 3, Drawings::orange);
  }
}

void DrawingMethods::paintCollisionPerceptForFieldView
(
 DebugDrawing& collisionPerceptDrawing,
 const CollisionPercept& collisionPercept,
 const Pose2D& origin)
{
  Vector2<double> point;
  if(collisionPercept.getCollisionHindLeft())
  {
    point.x = -150;
    point.y = 80;
    point = (origin + Pose2D(point)).translation;
    collisionPerceptDrawing.largeDot((int)point.x, (int)point.y, Drawings::red, Drawings::gray);
  }
  if(collisionPercept.getCollisionHindRight())
  {
    point.x = -150;
    point.y = -80;
    point = (origin + Pose2D(point)).translation;
    collisionPerceptDrawing.largeDot((int)point.x, (int)point.y, Drawings::red, Drawings::gray);
  }
  
  if(collisionPercept.getCollisionFrontLeft())
  {
    point.x = 50;
    point.y = 80;
    point = (origin + Pose2D(point)).translation;
    collisionPerceptDrawing.largeDot((int)point.x, (int)point.y, Drawings::red, Drawings::gray);
  }
  
  if(collisionPercept.getCollisionFrontRight())
  {
    point.x = 50;
    point.y = -80;
    point = (origin + Pose2D(point)).translation;
    collisionPerceptDrawing.largeDot((int)point.x, (int)point.y, Drawings::red, Drawings::gray);
  }
  
  if(collisionPercept.getCollisionHead())
  {
    point.x = 120;
    point.y = 0;
    point = (origin + Pose2D(point)).translation;
    collisionPerceptDrawing.largeDot((int)point.x, (int)point.y, Drawings::red, Drawings::gray);
  }
}

void DrawingMethods::paintObstaclesModelForRadarView
(
 DebugDrawing& obstaclesDrawing,
 const ObstaclesModel& obstaclesModel)
{
  paintObstaclesModelForFieldView(obstaclesDrawing, obstaclesModel, Pose2D(0,0,0));
}

void DrawingMethods::paintAngleBoundaryToImage
(
 DebugDrawing& perceptsDrawing, 
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo,
 const ConditionalBoundary boundary 
 )
{
  Vector2<int> corner[4];
  
  Geometry::calculatePointByAngles(
    Vector2<double>(boundary.x.max,boundary.y.max),
    cameraMatrix, cameraInfo, 
    corner[0]
    );
  
  Geometry::calculatePointByAngles(
    Vector2<double>(boundary.x.min,boundary.y.max),
    cameraMatrix, cameraInfo, 
    corner[1]
    );
  
  Geometry::calculatePointByAngles(
    Vector2<double>(boundary.x.min,boundary.y.min),
    cameraMatrix, cameraInfo, 
    corner[2]
    );
  
  Geometry::calculatePointByAngles(
    Vector2<double>(boundary.x.max,boundary.y.min),
    cameraMatrix, cameraInfo, 
    corner[3]
    );
  
  perceptsDrawing.line(
    (int)corner[0].x, (int)corner[0].y, 
    (int)corner[1].x, (int)corner[1].y,
    Drawings::ps_solid, 1, DebugDrawing::Color(200,200,200));
  perceptsDrawing.line(
    (int)corner[1].x, (int)corner[1].y, 
    (int)corner[2].x, (int)corner[2].y,
    Drawings::ps_solid, 1, DebugDrawing::Color(200,200,200));
  perceptsDrawing.line(
    (int)corner[2].x, (int)corner[2].y, 
    (int)corner[3].x, (int)corner[3].y,
    Drawings::ps_solid, 1, DebugDrawing::Color(200,200,200));
  perceptsDrawing.line(
    (int)corner[3].x, (int)corner[3].y, 
    (int)corner[0].x, (int)corner[0].y,
    Drawings::ps_solid, 1, DebugDrawing::Color(200,200,200));
  
}

void DrawingMethods::paintSpecialPerceptForImageView
(
 DebugDrawing& perceptsDrawing, 
 const SpecialPercept& specialPercept,
 const CameraMatrix& cameraMatrix
 )
{
/*
Add drawing of special percept here
switch(specialPercept.type)
{
case SpecialPercept::xxx:
}
  */
}


void DrawingMethods::paintBallPerceptForImageView
(
 DebugDrawing& perceptsDrawing, 
 const BallPercept& ballPercept,
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo
 )
{
  if(ballPercept.ballWasSeen)
  {
    Geometry::Circle circle;
    Vector2<int> center;
    
    Geometry::calculatePointByAngles(ballPercept.anglesToCenter, cameraMatrix, cameraInfo, center);
    circle.center.x = center.x;
    circle.center.y = center.y;
    circle.radius = Geometry::angleSizeToPixelSize(ballPercept.ballRadiusAsAngle, cameraInfo);
    
    perceptsDrawing.circle(
      int(circle.center.x),int(circle.center.y),int(circle.radius),2,
      Drawings::ps_solid,DebugDrawing::Color(255,100,50),
      Drawings::bs_null,Drawings::white);
    
    perceptsDrawing.dot(
      int(circle.center.x),int(circle.center.y), 
      DebugDrawing::Color(255,100,50),Drawings::black);  
  }
}  
  void DrawingMethods::paintLandmarksPerceptForImageView
    (
    DebugDrawing& perceptsDrawing, 
    const LandmarksPercept& landmarksPercept,
    const CameraMatrix& cameraMatrix,
    const CameraInfo& cameraInfo
    )
  {
    Vector2<int> corner[4];
    int penWidth[4];
    DebugDrawing::Color topColor, bottomColor, color;
    
    int i;
    for (i = 0; i < landmarksPercept.numberOfFlags; i++)
    {
      const Flag& flag = landmarksPercept.flags[i];
      
      (flag.isOnBorder(flag.y.max))? penWidth[0] = 0 : penWidth[0] = 1;
      (flag.isOnBorder(flag.x.min))? penWidth[1] = 0 : penWidth[1] = 1;
      (flag.isOnBorder(flag.y.min))? penWidth[2] = 0 : penWidth[2] = 1;
      (flag.isOnBorder(flag.x.max))? penWidth[3] = 0 : penWidth[3] = 1;
      
      switch(flag.type)
      {
      case Flag::pinkAboveYellow:
        topColor = DebugDrawing::Color(255,0,255);
        bottomColor = DebugDrawing::Color(255,255,0);
        break;
      case Flag::pinkAboveSkyblue:
        topColor = DebugDrawing::Color(255,0,255);
        bottomColor = DebugDrawing::Color(64,127,255);
        break;
      case Flag::yellowAbovePink:
        topColor = DebugDrawing::Color(255,255,0);
        bottomColor = DebugDrawing::Color(255,0,255);
        break;
      case Flag::skyblueAbovePink:
        topColor = DebugDrawing::Color(64,127,255);
        bottomColor = DebugDrawing::Color(255,0,255);
        break;
      }
      
      Geometry::calculatePointByAngles(
        Vector2<double>(flag.x.max,flag.y.max),
        cameraMatrix, cameraInfo, 
        corner[0]
        );
      
      Geometry::calculatePointByAngles(
        Vector2<double>(flag.x.min,flag.y.max),
        cameraMatrix, cameraInfo, 
        corner[1]
        );
      
      Geometry::calculatePointByAngles(
        Vector2<double>(flag.x.min,flag.y.min),
        cameraMatrix, cameraInfo, 
        corner[2]
        );
      
      Geometry::calculatePointByAngles(
        Vector2<double>(flag.x.max,flag.y.min),
        cameraMatrix, cameraInfo, 
        corner[3]
        );
      
      perceptsDrawing.line(
        (int)corner[0].x, (int)corner[0].y, 
        (int)corner[1].x, (int)corner[1].y,
        Drawings::ps_solid, penWidth[0], topColor);
      perceptsDrawing.line(
        (int)corner[1].x, (int)corner[1].y, 
        (int)corner[2].x, (int)corner[2].y,
        Drawings::ps_solid, penWidth[1], DebugDrawing::Color(200,200,200));
      perceptsDrawing.line(
        (int)corner[2].x, (int)corner[2].y, 
        (int)corner[3].x, (int)corner[3].y,
        Drawings::ps_solid, penWidth[2], bottomColor);
      perceptsDrawing.line(
        (int)corner[3].x, (int)corner[3].y, 
        (int)corner[0].x, (int)corner[0].y,
        Drawings::ps_solid, penWidth[3], DebugDrawing::Color(200,200,200));
      
    } // for (int i=0;i<landmarksPercept.numberOfFlags;i++)
    
    for (i = 0; i < landmarksPercept.numberOfGoals; i++)
    {
      const Goal& goal = landmarksPercept.goals[i];
      
      (goal.isOnBorder(goal.y.max))? penWidth[0] = 1 : penWidth[0] = 3;
      (goal.isOnBorder(goal.x.min))? penWidth[1] = 1 : penWidth[1] = 3;
      (goal.isOnBorder(goal.y.min))? penWidth[2] = 1 : penWidth[2] = 3;
      (goal.isOnBorder(goal.x.max))? penWidth[3] = 1 : penWidth[3] = 3;
      
      switch(goal.color)
      {
      case yellow:
        color = DebugDrawing::Color(255,255,127);
        break;
      case skyblue:
        color = DebugDrawing::Color(127,255,255);
        break;
      }
      
      Geometry::calculatePointByAngles(
        Vector2<double>(goal.x.max,goal.y.max),
        cameraMatrix, cameraInfo, 
        corner[0]
        );
      
      Geometry::calculatePointByAngles(
        Vector2<double>(goal.x.min,goal.y.max),
        cameraMatrix, cameraInfo, 
        corner[1]
        );
      
      Geometry::calculatePointByAngles(
        Vector2<double>(goal.x.min,goal.y.min),
        cameraMatrix, cameraInfo, 
        corner[2]
        );
      
      Geometry::calculatePointByAngles(
        Vector2<double>(goal.x.max,goal.y.min),
        cameraMatrix, cameraInfo, 
        corner[3]
        );
      
      perceptsDrawing.line(
        (int)corner[0].x, (int)corner[0].y, 
        (int)corner[1].x, (int)corner[1].y,
        Drawings::ps_solid, penWidth[0], color);
      perceptsDrawing.line(
        (int)corner[1].x, (int)corner[1].y, 
        (int)corner[2].x, (int)corner[2].y,
        Drawings::ps_solid, penWidth[1], color);
      perceptsDrawing.line(
        (int)corner[2].x, (int)corner[2].y, 
        (int)corner[3].x, (int)corner[3].y,
        Drawings::ps_solid, penWidth[2], color);
      perceptsDrawing.line(
        (int)corner[3].x, (int)corner[3].y, 
        (int)corner[0].x, (int)corner[0].y,
        Drawings::ps_solid, penWidth[3], color);
      
    } // for (int i=0;i<landmarksPercept.numberOfGoals;i++)
}

void DrawingMethods::paintLinesPerceptForImageView
(
 DebugDrawing& perceptsDrawing, 
 const LinesPercept& linesPercept,
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo
 )
{
  static DebugDrawing::Color colors[] =
  {
    DebugDrawing::Color(Drawings::gray),
      DebugDrawing::Color(Drawings::green),
      DebugDrawing::Color(Drawings::yellow),
      DebugDrawing::Color(Drawings::skyblue),
      DebugDrawing::Color(Drawings::red),
      DebugDrawing::Color(Drawings::blue),
      DebugDrawing::Color(Drawings::orange)
  };
  DebugDrawing::Color white(Drawings::white);
  
  for(int i = 0; i < LinesPercept::numberOfTypes; ++i)
  {
    const DebugDrawing::Color& color = colors[i];
    for(int j = 0; j < linesPercept.numberOfPoints[i]; ++j)
    {
      Vector2<int> point;
      Geometry::calculatePointInImage(
        linesPercept.points[i][j],
        cameraMatrix, cameraInfo,
        point);
      perceptsDrawing.circle(point.x,point.y, 3, 1, Drawings::ps_solid, Drawings::black, Drawings::bs_solid, color);
      double angle = linesPercept.points[i][j].angle;
      if(angle != LinesPercept::UNDEF)
      {
        Vector2<double> v = (Pose2D(angle - pi_2, Vector2<double>(
          linesPercept.points[i][j].x,
          linesPercept.points[i][j].y)) +
          Pose2D(Vector2<double>(linesPercept.points[i][j].abs() / 20,0))).translation;
        Vector2<int> point2;
        Geometry::calculatePointInImage(
          Vector2<int>(int(v.x), int(v.y)),
          cameraMatrix, cameraInfo,
          point2);
        point2 -= point;
        point2 = point + Vector2<int>(point2.y, -point2.x) * 10 / point2.abs();
        perceptsDrawing.arrow(        
          (int)point.x,
          (int)point.y,
          (int)point2.x,
          (int)point2.y,
          Drawings::ps_solid, 1, color);
      }
    }
  }
}

void DrawingMethods::paintEdgesPerceptForImageView
(
 DebugDrawing& perceptsDrawing, 
 const EdgesPercept& edgesPercept,
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo
 )
{
  static DebugDrawing::Color colors[] =
  {
    DebugDrawing::Color(Drawings::gray),
    DebugDrawing::Color(Drawings::green),
    DebugDrawing::Color(Drawings::yellow),
    DebugDrawing::Color(Drawings::skyblue),
    DebugDrawing::Color(Drawings::red),
    DebugDrawing::Color(Drawings::blue),
    DebugDrawing::Color(Drawings::orange)
  };
  DebugDrawing::Color white(Drawings::white);
  
  for(int i = 0; i < edgesPercept.numberOfEdges; ++i)
  {
    const DebugDrawing::Color& color = colors[i%7];
    Vector2<int> point1, point2;
    Geometry::calculatePointInImage(
      edgesPercept.edges[i].point1,
      cameraMatrix, cameraInfo,
      point1);
//    point1 = edgesPercept.edges[i].point1;
    Geometry::calculatePointInImage(
      edgesPercept.edges[i].point2,
      cameraMatrix, cameraInfo,
      point2);
//    point2 = edgesPercept.edges[i].point2;
//    perceptsDrawing.dot(point.x,point.y,black,color);
//    perceptsDrawing.circle(point.x,point.y, 3, 1, Drawings::ps_solid, Drawings::black, Drawings::bs_solid, color);
    perceptsDrawing.line(point1.x,point1.y, point2.x,point2.y, Drawings::ps_solid, 1, color);
  }
}

void DrawingMethods::paintObstaclesPerceptForImageView
(
 DebugDrawing& perceptsDrawing, 
 const ObstaclesPercept& obstaclesPercept,
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo
 )
{
  DebugDrawing::Color color(0,0,0);
  for(int j = 0; j < obstaclesPercept.numberOfPoints; ++j)
  {
    Vector2<int> nearPoint,farPoint;
    Geometry::calculatePointInImage(obstaclesPercept.nearPoints[j], cameraMatrix, cameraInfo, nearPoint);
    Geometry::calculatePointInImage(obstaclesPercept.farPoints[j], cameraMatrix, cameraInfo, farPoint);
    switch(obstaclesPercept.obstacleType[j])
    {
      case ObstaclesPercept::unknown: color = Drawings::black; break;
      case ObstaclesPercept::opponent: color = Drawings::red; break;
      case ObstaclesPercept::teammate: color = Drawings::green; break;
      case ObstaclesPercept::goal: color = Drawings::blue; break;
      case ObstaclesPercept::border: color = Drawings::gray; break;
    }
    perceptsDrawing.line(nearPoint.x, nearPoint.y, farPoint.x, farPoint.y,
      Drawings::ps_solid, 1, DebugDrawing::Color(127, 255, 127) );
    if(obstaclesPercept.farPointIsOnImageBorder[j])
      perceptsDrawing.dot(farPoint.x, farPoint.y, DebugDrawing::Color(255, 255, 255), DebugDrawing::Color(127, 255, 127));
    perceptsDrawing.circle(farPoint.x, farPoint.y, 2, 1, Drawings::ps_solid, color, Drawings::bs_solid, color);
  }
}

void DrawingMethods::paintFreePartOfGoalPerceptForImageView
(
 DebugDrawing& perceptsDrawing, 
 const ObstaclesPercept& obstaclesPercept,
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo
 )
{
  DebugDrawing::Color freePartColor[2];
  freePartColor[0] = Drawings::red;
  freePartColor[1] = Drawings::green;
  
  for(int i = 0; i < 2; i++)
  {
    if(obstaclesPercept.angleToFreePartOfGoalWasDetermined[i])
    {
      Vector2<int> leftBottomPoint, rightBottomPoint;
      
      double leftAngle = obstaclesPercept.angleToFreePartOfGoal[i] + obstaclesPercept.widthOfFreePartOfGoal[i] / 2.0;
      double rightAngle = obstaclesPercept.angleToFreePartOfGoal[i] - obstaclesPercept.widthOfFreePartOfGoal[i] / 2.0;
      
      Vector2<int> leftPointOnField(int(cos(leftAngle) * obstaclesPercept.distanceToFreePartOfGoal[i]),
        int(sin(leftAngle) * obstaclesPercept.distanceToFreePartOfGoal[i]) );
      
      Vector2<int> rightPointOnField(int(cos(rightAngle) * obstaclesPercept.distanceToFreePartOfGoal[i]),
        int(sin(rightAngle) * obstaclesPercept.distanceToFreePartOfGoal[i]) );
      
      Geometry::calculatePointInImage(leftPointOnField ,cameraMatrix, cameraInfo, leftBottomPoint);
      Geometry::calculatePointInImage(rightPointOnField ,cameraMatrix, cameraInfo, rightBottomPoint);
      
      perceptsDrawing.line(leftBottomPoint.x, leftBottomPoint.y, rightBottomPoint.x, rightBottomPoint.y,
        Drawings::ps_solid, 5, freePartColor[i]);
    }
  }
}

void DrawingMethods::paintObstaclesModelForImageView
(
 DebugDrawing& obstaclesDrawing, 
 const ObstaclesModel& obstaclesModel,
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo
 )
{
  Vector2<int> pointOnField;
  Vector2<int> pointInImage, lastPointInImage;
  double angle;
  DebugDrawing::Color lastColor(0,0,0);
  DebugDrawing::Color color(0,0,0);

  
  Vector2<double> lookingDirectionAngles;
  Geometry::calculateAnglesForPoint(Vector2<int>(cameraInfo.resolutionWidth / 2, cameraInfo.resolutionHeight / 2), cameraMatrix, cameraInfo, lookingDirectionAngles);

  angle = ObstaclesModel::getAngleOfSector(0);
  pointOnField.x = (int)(cos(angle) * obstaclesModel.distance[0]);
  pointOnField.y = (int)(sin(angle) * obstaclesModel.distance[0]);
  Geometry::calculatePointInImage(pointOnField, cameraMatrix, cameraInfo, lastPointInImage);

  switch(obstaclesModel.obstacleType[0])
  {
    case ObstaclesPercept::unknown: lastColor = Drawings::black; break;
    case ObstaclesPercept::opponent: lastColor = Drawings::red; break;
    case ObstaclesPercept::teammate: lastColor = Drawings::green; break;
    case ObstaclesPercept::goal: lastColor = Drawings::blue; break;
    case ObstaclesPercept::border: lastColor = Drawings::gray; break;
  }


  int i;
  for(i = 1; i < ObstaclesModel::numOfSectors; i++)
  {
    angle = ObstaclesModel::getAngleOfSector(i);
    pointOnField.x = (int)(cos(angle) * obstaclesModel.distance[i]);
    pointOnField.y = (int)(sin(angle) * obstaclesModel.distance[i]);

    Geometry::calculatePointInImage(pointOnField, cameraMatrix, cameraInfo, pointInImage);
    switch(obstaclesModel.obstacleType[i])
    {
      case ObstaclesPercept::unknown: color = Drawings::black; break;
      case ObstaclesPercept::opponent: color = Drawings::red; break;
      case ObstaclesPercept::teammate: color = Drawings::green; break;
      case ObstaclesPercept::goal: color = Drawings::blue; break;
      case ObstaclesPercept::border: color = Drawings::gray; break;
    }

    // Do not paint the parts of the obstacles model that are outside the image.
    if(fabs((double)normalize(lookingDirectionAngles.x - angle) ) < cameraInfo.openingAngleWidth)
    {
      Vector2<int> pointBetween = pointInImage / 2 + lastPointInImage / 2;
      obstaclesDrawing.line(
        (int) pointInImage.x,(int) pointInImage.y, 
        (int) pointBetween.x,(int) pointBetween.y, 
        Drawings::ps_solid, 3, color);
      obstaclesDrawing.line(
        (int) pointBetween.x,(int) pointBetween.y, 
        (int) lastPointInImage.x,(int) lastPointInImage.y, 
        Drawings::ps_solid, 3, lastColor);
    }
    lastPointInImage = pointInImage;
    lastColor = color;
  }
}

void DrawingMethods::paintFreePartOfGoalModelForImageView
(
 DebugDrawing& freePartOfGoalDrawing, 
 const ObstaclesModel& obstaclesModel,
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo
 )
{
  Vector2<double> lookingDirectionAngles;
  Geometry::calculateAnglesForPoint(Vector2<int>(cameraInfo.resolutionWidth / 2, cameraInfo.resolutionHeight / 2), cameraMatrix, cameraInfo, lookingDirectionAngles);

  // Free part of goal
  DebugDrawing::Color freePartColor[2];
  freePartColor[0] = DebugDrawing::Color(127, 0 , 0);
  freePartColor[1] = DebugDrawing::Color(0, 127 , 0);
  int i;
  for(i = 0; i < 2; i++)
  {
    if(obstaclesModel.angleToFreePartOfGoalWasDetermined[i])
    {
      Vector2<int> leftBottomPoint, rightBottomPoint;
      
      double leftAngle = obstaclesModel.angleToFreePartOfGoal[i] + obstaclesModel.widthOfFreePartOfGoal[i] / 2.0;
      double rightAngle = obstaclesModel.angleToFreePartOfGoal[i] - obstaclesModel.widthOfFreePartOfGoal[i] / 2.0;
      
      // Do not paint the angles if they are outside the image.
      if(
        fabs((double)normalize(lookingDirectionAngles.x - leftAngle) ) < cameraInfo.openingAngleWidth &&
        fabs((double)normalize(lookingDirectionAngles.x - rightAngle) ) < cameraInfo.openingAngleWidth
        )
      {
        Vector2<int> leftPointOnField(int(cos(leftAngle) * obstaclesModel.distanceToFreePartOfGoal[i]),
          int(sin(leftAngle) * obstaclesModel.distanceToFreePartOfGoal[i]) );
        
        Vector2<int> rightPointOnField(int(cos(rightAngle) * obstaclesModel.distanceToFreePartOfGoal[i]),
          int(sin(rightAngle) * obstaclesModel.distanceToFreePartOfGoal[i]) );
        
        Geometry::calculatePointInImage(leftPointOnField ,cameraMatrix, cameraInfo, leftBottomPoint);
        Geometry::calculatePointInImage(rightPointOnField ,cameraMatrix, cameraInfo, rightBottomPoint);
        
        freePartOfGoalDrawing.line(leftBottomPoint.x, leftBottomPoint.y, rightBottomPoint.x, rightBottomPoint.y,
          Drawings::ps_solid, 3, freePartColor[i]);
      }
    }
  }
  
  // Next free teammate
  if(obstaclesModel.angleToNextFreeTeammateWasDetermined)
  {
    Vector2<int> farPointOnField(int(cos(obstaclesModel.angleToNextFreeTeammate) * obstaclesModel.distanceToNextFreeTeammate),
      int(sin(obstaclesModel.angleToNextFreeTeammate) * obstaclesModel.distanceToNextFreeTeammate) );
    
    Vector2<int> nearPointOnField(int(cos(obstaclesModel.angleToNextFreeTeammate) * 100),
      int(sin(obstaclesModel.angleToNextFreeTeammate) * 100) );
    
    Vector2<int> nearPoint, farPoint;
    
    Geometry::calculatePointInImage(nearPointOnField ,cameraMatrix, cameraInfo, nearPoint);
    Geometry::calculatePointInImage(farPointOnField ,cameraMatrix, cameraInfo, farPoint);
    
    freePartOfGoalDrawing.line(
      nearPoint.x, nearPoint.y,
      farPoint.x, farPoint.y,
      Drawings::ps_solid, 3, Drawings::blue);
  }
 }

 
 void DrawingMethods::paintPSDPerceptForImageView
   (
   DebugDrawing& perceptsDrawing, 
   const PSDPercept& psdPercept,
   const CameraMatrix& cameraMatrix,
   const CameraInfo& cameraInfo
   )
 {
   DebugDrawing::Color gray(Drawings::white);
   DebugDrawing::Color white(Drawings::white);
   
   RotationMatrix rotationMatrix = cameraMatrix.rotation;
   
   for (int i=0;i<psdPercept.numOfPercepts;i++)
   {
     /* The psd point*/ 
     Vector3<double> vectorToPointWorld, vectorToGroundWorld, vectorToPointImage, vectorToGroundImage;
     vectorToPointWorld.x = psdPercept[i].x - cameraMatrix.translation.x;
     vectorToPointWorld.y = psdPercept[i].y - cameraMatrix.translation.y;
     vectorToPointWorld.z = psdPercept[i].z - cameraMatrix.translation.z;
     
     vectorToGroundWorld.x = psdPercept[i].x - cameraMatrix.translation.x;
     vectorToGroundWorld.y = psdPercept[i].y - cameraMatrix.translation.y;
     vectorToGroundWorld.z = - cameraMatrix.translation.z;
     
     
     vectorToPointImage = rotationMatrix.invert() * vectorToPointWorld;
     vectorToGroundImage = rotationMatrix.invert() * vectorToGroundWorld;
     
     double scale = (cameraInfo.resolutionWidth - 1) / 2.0 / tan(cameraInfo.openingAngleWidth / 2) / vectorToPointImage.x;
     int xPSD = (int)((cameraInfo.resolutionWidth - 1) / 2 - vectorToPointImage.y * scale);
     int yPSD = (int)((cameraInfo.resolutionHeight - 1) / 2 - vectorToPointImage.z * scale);
     
     scale = (cameraInfo.resolutionWidth - 1) / 2.0 / tan(cameraInfo.openingAngleWidth / 2) / vectorToGroundImage.x;
     int xGround = (int)((cameraInfo.resolutionWidth - 1) / 2 - vectorToGroundImage.y * scale);
     int yGround = (int)((cameraInfo.resolutionHeight - 1) / 2 - vectorToGroundImage.z * scale);
     
     if(psdPercept[i].tooFarAway)
     {
       if(psdPercept[i].z < -90)
         perceptsDrawing.circle(xPSD, yPSD, 4, 1, Drawings::ps_solid, Drawings::green, Drawings::bs_solid, Drawings::blue);
       else
         perceptsDrawing.circle(xPSD, yPSD, 4, 1, Drawings::ps_solid, Drawings::green, Drawings::bs_solid, Drawings::gray);
     }
     else
     {
       if(psdPercept[i].z > 60)
       {
         perceptsDrawing.line(xPSD, yPSD, xGround, yGround, Drawings::ps_solid, 2, Drawings::gray);
         perceptsDrawing.circle(xPSD, yPSD, 4, 1, Drawings::ps_solid, Drawings::red, Drawings::bs_solid, Drawings::gray);
       }
       else if(psdPercept[i].z > -90)
       {
         perceptsDrawing.circle(xPSD, yPSD, 4, 1, Drawings::ps_solid, Drawings::red, Drawings::bs_solid, Drawings::black);
       }
       else
       {
         perceptsDrawing.line(xPSD, yPSD, xGround, yGround, Drawings::ps_solid, 2, Drawings::blue);
         perceptsDrawing.circle(xPSD, yPSD, 4, 1, Drawings::ps_solid, Drawings::red, Drawings::bs_solid, Drawings::blue);
       }
     }
   }
 }
 
 
 /**
 * Paints a special percept to a debug drawing for the field view
 */
 void DrawingMethods::paintSpecialPerceptForFieldView
   (
   DebugDrawing& specialPerceptDrawing, 
   const SpecialPercept& specialPercept,
   const RobotPose& origin,
   Player::teamColor teamColor)
 {
 /*
 Add drawing of special percept here
 switch(specialPercept.type)
 {
 case SpecialPercept::xxx:
 }
   */
 }
 
 
 /*
 * Change log :
 * 
 * $Log: DrawingMethods.cpp,v $
 * Revision 1.8  2004/06/17 14:34:44  dassler
 * GT2004HeadControl updated
 * Now looks after propergated ball, followed up withe the communicated ball
 * GT2004HeadPathPlanner work now with time optimized moves
 * Middle Beacons removed of the Landmarkspercept
 *
 * Revision 1.7  2004/06/17 10:50:22  roefer
 * Visualization of line orientation fixed
 *
 * Revision 1.6  2004/06/17 09:24:43  roefer
 * Lines percept includes line orientation
 *
 * Revision 1.4  2004/06/15 10:58:28  thomas
 * added edge-specialist, edges-percept, debug-drawings etc. (not yet called from image-processor)
 *
 * Revision 1.3  2004/06/14 23:20:07  spranger
 * -changed some functions in Geometry from int to double including ballradius(fieldimensions)
 * -maybe all Geometric-functions in Geometry should be as precise as possible, avoiding int to double conversion errors
 *
 * Revision 1.2  2004/05/22 22:52:03  juengel
 * Renamed ballP_osition to ballModel.
 *
 * Revision 1.1.1.1  2004/05/22 17:30:06  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.12  2004/03/28 14:09:51  jhoffman
 * - added error member variables to ball model
 * - minor improvement in handling of updateRP in motiontesterdialog
 *
 * Revision 1.11  2004/03/01 11:44:21  juengel
 * visualization of obstacle types
 *
 * Revision 1.10  2004/02/26 12:10:07  juengel
 * changed visualization of lines percept
 *
 * Revision 1.9  2004/02/12 14:28:53  juengel
 * Changed color of ball percept.
 *
 * Revision 1.8  2004/02/06 10:16:23  juengel
 * percepts for image view are painted in different debug drawings now
 *
 * Revision 1.7  2004/02/03 13:20:47  spranger
 * renamed all references to  class BallP_osition to BallModel (possibly changed include files)
 *
 * Revision 1.6  2003/12/15 11:46:57  juengel
 * Introduced CameraInfo
 *
 * Revision 1.5  2003/12/09 18:11:45  loetzsch
 * Added parameter timestamp to
 * Drawingmethods::paintWorldStateToDebugDrawing
 *
 * Revision 1.4  2003/12/07 18:59:27  loetzsch
 * changed methods for field view
 *
 * Revision 1.3  2003/11/15 17:09:02  juengel
 * changed BallPercept
 *
 * Revision 1.2  2003/10/21 15:04:51  roefer
 * Improved world state drawing
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.4  2003/09/26 21:23:20  loetzsch
 * renamed class JointState to CollisionPercept
 *
 * Revision 1.3  2003/09/25 11:21:58  juengel
 * Removed BlobCollection.
 *
 * Revision 1.2  2003/07/03 23:09:38  juengel
 * removed localization validity circles
 * removed corridors
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.37  2003/06/27 13:17:56  jhoffman
 * work on obstacle avoider challenge,
 * added headcontrolmode,
 * added method needed to determine empty space,
 * updatet drawing method to reflect actual corridor size
 *
 * Revision 1.36  2003/06/25 18:45:33  juengel
 * Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
 *
 * Revision 1.35  2003/06/19 18:34:00  juengel
 * Reduced line with for obstaclesModel.
 *
 * Revision 1.34  2003/06/19 12:08:56  juengel
 * Changed drawing of obstaclesModel.
 *
 * Revision 1.33  2003/06/13 14:52:37  juengel
 * Added paintJointStateForFieldView.
 *
 * Revision 1.32  2003/06/05 11:33:27  mkunz
 * validity circles for the other players added
 *
 * Revision 1.31  2003/05/26 08:22:44  juengel
 * Added paintObstaclesModelForImageView
 *
 * Revision 1.30  2003/05/16 14:07:48  risler
 * added radar view obstacles drawing
 *
 * Revision 1.29  2003/05/15 00:56:47  risler
 * added functions for sector to angle mapping to ObstaclesModel
 * sector to angle mapping unified
 * added rule AvoidObstacles
 *
 * Revision 1.28  2003/05/14 19:55:18  risler
 * ObstaclesModel Drawing displays all sectors
 * PSDPercept contains all points from one frame
 *
 * Revision 1.27  2003/05/14 13:08:38  risler
 * removed DefaultObstaclesLocator
 * renamed MicroSectorsObstaclesLocator to DefaultObstaclesLocator
 * ObstaclesModel contains increased number of sectors
 * DefaultObstaclesLocator clean up
 *
 * Revision 1.26  2003/05/05 12:25:53  juengel
 * Visualization of angleToFreePartOfOpponentGoal changed.
 *
 * Revision 1.25  2003/05/01 18:10:03  loetzsch
 * renamed CommunicatedBallPosition::timeWhenLastReceived to CommunicatedBallPosition::timeWhenLastObserved
 *
 * Revision 1.24  2003/05/01 17:09:10  loetzsch
 * Redesign of ball modeling:
 * - Modularized class BallPosition
 * - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
 *   and "TeamBallLocator" for modelling communicated positions
 * - Removed solution JumpingBallLocator
 * - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
 * - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
 *
 * Revision 1.23  2003/05/01 10:21:40  roefer
 * freePartOfO*Goal in field drawing
 *
 * Revision 1.22  2003/04/23 21:07:21  roefer
 * Ball center is black now
 *
 * Revision 1.21  2003/04/23 16:18:38  juengel
 * Changed visualization of percepts.
 *
 * Revision 1.20  2003/04/16 07:00:16  roefer
 * Bremen GO checkin
 *
 * Revision 1.19  2003/04/14 16:19:34  loetzsch
 * ATH after GermanOpen CVS merge
 * improvements in the color table 64 dialog bar
 *
 * Revision 1.2  2003/04/10 14:14:44  Jan Hoffmann
 * added corridor to obstacle model
 *
 * Revision 1.1.1.1  2003/04/09 14:23:00  loetzsch
 * started Aibo Team Humboldt's GermanOpen CVS
 *
 * Revision 1.18  2003/04/05 16:49:03  juengel
 * Added specialPercept to DrawingMethods.
 *
 * Revision 1.17  2003/04/02 13:20:10  juengel
 * percepts of blue flags are more blue now.
 *
 * Revision 1.16  2003/03/31 12:32:38  juengel
 * Changed color for ball percept
 *
 * Revision 1.15  2003/03/29 21:08:03  juengel
 * Added paintObstaclesAndPSDPerceptForImageView.
 *
 * Revision 1.14  2003/03/29 13:44:37  roefer
 * Ball percept visualization in image
 *
 * Revision 1.13  2003/03/28 14:32:44  juengel
 * Added visualization of psdPercept.
 *
 * Revision 1.12  2003/03/25 10:53:30  juengel
 * Changed colors for goals and obstacles percept.
 *
 * Revision 1.11  2003/03/22 12:06:44  jhoffman
 * changed obstacle visualization
 *
 * Revision 1.10  2003/03/22 11:06:25  roefer
 * Progress in LinesImageProcessor2
 *
 * Revision 1.9  2003/03/11 11:27:49  juengel
 * Added obstaclesPercept to drawingMethods.
 *
 * Revision 1.8  2003/03/10 14:07:13  juengel
 * Prepared new type sideOfPenaltyArea
 *
 * Revision 1.7  2003/03/04 17:43:54  roefer
 * New penalty area
 *
 * Revision 1.6  2003/01/30 13:48:20  loetzsch
 * Redesign of class BallPosition
 *
 * Revision 1.5  2003/01/30 13:16:49  loetzsch
 * Redesign of class BallPosition
 *
 * Revision 1.4  2002/12/14 19:45:54  roefer
 * paintLinesPerceptToImageView added
 *
 * Revision 1.3  2002/12/14 17:56:08  roefer
 * Names of drawing functions changed
 *
 * Revision 1.2  2002/11/20 15:46:27  juengel
 * PathPlanner added.
 *
 * Revision 1.1  2002/11/12 10:52:52  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.7  2002/10/13 15:51:05  juengel
 * second debug drawing clean up
 *
 * Revision 1.6  2002/10/10 11:39:56  juengel
 * Changed width of line for goal in PaintLandmarksPerceptForFieldView(...)
 *
 * Revision 1.5  2002/09/25 12:05:43  loetzsch
 * removed BarPercept and PatternPercept
 *
 * Revision 1.4  2002/09/22 18:40:52  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.3  2002/09/22 09:15:33  risler
 * Geometry.h moved to directory Math
 *
 * Revision 1.2  2002/09/17 23:55:23  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.10  2002/08/30 17:20:33  juengel
 * Bug fixed again.
 *
 * Revision 1.9  2002/07/05 15:51:33  dueffert
 * bug fixed - how did that ever compile?
 *
 * Revision 1.8  2002/06/18 07:06:56  mkunz
 * no middle calculation when whiteSeen
 *
 * Revision 1.7  2002/06/17 19:24:14  mkunz
 * BarPerceptor searches for own end only now
 *
 * Revision 1.6  2002/06/13 11:52:39  mkunz
 * Bar end poses are now in world coordinates
 *
 * Revision 1.5  2002/06/08 16:14:27  mkunz
 * bar paintings
 *
 * Revision 1.4  2002/06/08 11:43:59  mkunz
 * Special Percept on Field
 *
 * Revision 1.3  2002/06/06 10:52:46  kspiess
 * distribution for multiple balls;
 *
 * Revision 1.2  2002/06/02 23:21:10  roefer
 * Single color table and progress in LinesSelfLocator
 *
 * Revision 1.1.1.1  2002/05/10 12:40:33  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.29  2002/04/25 20:29:58  roefer
 * New BallPercept and BallPosition, GTMath errors in SimGT2002 fixed
 *
 * Revision 1.28  2002/04/23 10:25:07  risler
 * circles for selfLoc validity added
 *
 * Revision 1.27  2002/04/23 00:01:38  roefer
 * Redesign of Bremen process layout
 *
 * Revision 1.26  2002/04/20 19:37:20  roefer
 * #ifdef _WIN32 removed
 *
 * Revision 1.25  2002/04/16 16:13:14  dueffert
 * no message
 *
 * Revision 1.2  2002/04/12 07:10:38  juengel
 * Player Specialist added.
 *
 * Revision 1.24  2002/04/08 19:53:14  juengel
 * Drawing of percept collections in images added.
 *
 * Revision 1.23  2002/04/07 13:02:13  roefer
 * Improved goal drawing
 *
 * Revision 1.22  2002/04/05 13:42:03  juengel
 * Step by step ...
 *
 * Revision 1.21  2002/03/29 17:34:38  roefer
 * Field now follows specification from Open-R website
 *
 * Revision 1.20  2002/03/24 17:47:02  juengel
 * LinesPercept defined and LinesPerceptor added.
 *
 * Revision 1.19  2002/03/23 22:25:44  roefer
 * LandmarksPercepts now in player coordinates
 *
 * Revision 1.18  2002/03/06 10:20:56  juengel
 * Drawing of field polygons depends on team color.
 *
 * Revision 1.17  2002/02/22 16:10:04  tim
 * Anpassungen an das genderte PlayersPercept
 *
 * Revision 1.16  2002/02/19 14:26:34  roefer
 * drawArc removed
 *
 * Revision 1.15  2002/02/18 15:41:43  loetzsch
 * continued percept visualization
 *
 * Revision 1.14  2002/02/18 12:05:18  loetzsch
 * Percept visualization continued
 *
 * Revision 1.13  2002/02/13 22:43:02  roefer
 * First working versions of DefaultLandmarksPerceptor and MonteCarloSelfLocator
 *
 * Revision 1.12  2002/02/11 11:13:20  roefer
 * BallPerceptor and BallLocator inserted
 *
 * Revision 1.11  2002/02/11 00:55:38  loetzsch
 * ::Paint Methoden in ::paint umbenannt,
 * BallPerzept- Visualisierung fertig
 *
 * Revision 1.10  2002/02/10 12:45:15  loetzsch
 * radar viewer and percept visualization continued
 *
 * Revision 1.9  2002/02/08 13:28:08  loetzsch
 * Visualization of PerceptCollection and RadarViewer started
 * (not implemented yet)
 *
 * Revision 1.8  2002/02/06 18:57:58  roefer
 * Simplifications for oracle and debug drawings
 *
 * Revision 1.7  2002/02/05 03:37:57  loetzsch
 * added a method for painting percept collections.
 *
 * replaced direct member access in WorldState
 * and PerceptCollection members by
 * inline const VALUE& get...() const   and
 * inline void set...(const Value&) methods.
 *
 * Revision 1.6  2002/02/03 16:11:57  juengel
 * wordStateOracle drawing added
 *
 * Revision 1.5  2002/02/03 14:37:58  juengel
 * Drawing of the world state removed from Berlin2001BehaviorControl.
 * Drawing method for world states added to DrawingMethods.
 * Drawing of the world state added to the Processes with BehaviorControl.
 *
 * Revision 1.4  2002/01/19 08:09:08  juengel
 * Umrisse von filledBlobs sind nun grau.
 *
 * Revision 1.3  2002/01/08 15:52:07  juengel
 * no message
 *
 * Revision 1.2  2002/01/04 16:16:57  juengel
 * no message
 *
 * Revision 1.1  2002/01/04 14:36:00  juengel
 * no message
 *
 *
*/
