/** 
* @file ImageDrawingManager.h
* Declaration of class ImageDrawingManager.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/

#ifndef __ImageDrawingManager_h_
#define __ImageDrawingManager_h_

#include "StdAfx.h"

#include "Visualization/DebugDrawing.h"
#include "Tools/Debugging/DebugImages.h"
#include "DebugDrawingManagerEnums.h"
#include "Representations/Perception/Image.h"

using DrawingPapers::ImageDrawingPaper;
using DrawingPapers::numberOfImageDrawingPapers;
using Images::ImageID;
using Images::numberOfImageIDs;
/**
* A ImageDrawingManager object handles DebugDrawing objects. Incoming debug drawings and Images
* are distributed to several drawing papers. These papers can be displyed
* in a CDC or be saved to a MetaFile.
*/
class ImageDrawingManager
{
public:
  /** Constructor */
  ImageDrawingManager();
  /** Destructor */
  ~ImageDrawingManager();
  
  /**
  * Gives the Image Drawing Manager a reference to a debug drawing.
  * This debug drawing is copied to an internal representation.
  * @param pDebugDrawing Pointer to the debug drawing.
  */
  void setDebugDrawing(DebugDrawing* pDebugDrawing);
  
  /**
  * Specifies which image is shown on the specified image drawing paper.
  * Only one image can be shown on one image drawing paper.
  * @param imageDrawingPaper Specifies the drawing paper where the image is shown.
  * @param imageID Specifies the image that is shown.
  */
  void setImageID(
    ImageDrawingPaper imageDrawingPaper,
    ImageID imageID
    );
  
    /**
    * Returns the id of the image which is shown on the specified drawing paper.
    * @param imageDrawingPaper Specifies the image drawing paper.
    * @return imageID Specifies the image that is shown on the specified drawing paper.
  */
  ImageID getImageID(ImageDrawingPaper imageDrawingPaper);
  
  void notifyImage(ImageID imageID);
  
  /**
  * Specifies if the specified drawing is shown on the specified drawing paper.
  * @param imageDrawingPaper The image drawing paper.
  * @param imageDrawing The debug drawing.
  * @param show Specifies if the drawing is shown or not.
  */
  void showDrawingOnDrawingPaper(
    ImageDrawingPaper imageDrawingPaper,
    Drawings::ImageDrawing imageDrawing,
    bool show
    );
  
  
    /**
    * @param imageDrawingPaper The drawing paper.
    * @param imageDrawing The drawing.
    * @return Retruns if the specified drawing is shown on the specified drawing paper.
  */
  bool isDrawingShownOnDrawingPaper(
    ImageDrawingPaper imageDrawingPaper,
    Drawings::ImageDrawing imageDrawing
    );
  
    /**
    * Paints the content of the specified drawing paper. 
    * Call this member function from the OnPaint function of the CWnd displaying the device context.
    *
    * @param rect Specifies the rect in the device context to paint to.
    * @param imageDrawingPaper Specifies the drawing paper to paint.
    *
  */
  void paint(
    CRect* rect, 
    ImageDrawingPaper imageDrawingPaper
    );
  
    /**
    * The ImageDrawingManager calls the RedrawWindow method of the
    * CWnd object each time something changes at the specified drawing paper.
    * @param imageDrawingPaper Specifies the drawing paper.
    * @param pWnd A pointer to a CWnd object.
  */
  void setWnd(
    ImageDrawingPaper imageDrawingPaper,
    CWnd* pWnd
    );
  
    /**
    * Copies the specified drawing paper to the clipboard.
    * @param drawingPaper Specifies the drawing paper.
  */
  void copyDrawingPaperToClipBoard(ImageDrawingPaper drawingPaper);

    /**
    * Copies the image selected in the specified drawing paper to the clipboard.
    * @param drawingPaper Specifies the drawing paper.
  */
  void copyImageToClipBoard(ImageDrawingPaper drawingPaper);

  /**
  * Saves which drawings and which image are shown on which drawing paper to the reistry.
  */
  void saveProperties();
  
  /**
  * Loads which drawings and which image are shown on which drawing paper from the registry.
  */
  void loadProperties();
  
  /**
  * Calls the RedrawWindow function of all CWnd objects 
  * displaying the specified image.
  * @param imageID The image to draw.
  */
  void repaintDrawingPapers(ImageID imageID);

  void setSendDebugImagesFromRobot(bool sendDebugImagesFromRobot);
  bool getSendDebugImagesFromRobot();
  
  /**
  * Returns whether images are drawn in high resolution
  */
  bool getHighRes(ImageDrawingPaper drawingPaper);

  /**
  * Sets whether images are drawn in high resolution
  */
  void setHighRes(ImageDrawingPaper drawingPaper, bool highRes);

private:
/**
* Generates a Metafile containing all the drawings of the specified
* @param drawingPaper Specifies the drawing paper.
* drawing paper. The handle to that Metafile is stored in the List of meta file
* handles.
  */
  void generateMetaFile(ImageDrawingPaper drawingPaper);
  
  /**
  * Calls the RedrawWindow function of the CWnd object displaying the
  * specified drawing paper.
  * @param drawingPaper Specifies the drawing paper.
  */
  void repaintDrawingPaper(ImageDrawingPaper drawingPaper);
  
  /**
  * Calls the RedrawWindow function of all CWnd objects 
  * displaying a drawing paper containing
  * the specified drawing.
  * @param drawing Specifies a drawing.
  */
  void repaintDrawingPapers(Drawings::ImageDrawing drawing);
  

  void createNewBitmap(ImageDrawingPaper drawingPaper);

  void drawDrawingPaperToOffScreenDC(ImageDrawingPaper drawingPaper);

  void drawImageToOffScreenDC(
    ImageDrawingPaper drawingPaper
    );

  void drawDrawingToOffScreenDC(
    Drawings::ImageDrawing drawing, 
    ImageDrawingPaper drawingPaper
    );

  void sendDebugKeysForSimulatedRobots();
  
  /** Pointers to the debug drawings*/
  DebugDrawing* imageDrawing[Drawings::numberOfImageDrawings];
  
  bool isDrawingValid[Drawings::numberOfImageDrawings];

  CDC dcOffScreen[numberOfImageDrawingPapers];
  CBitmap* bmpOffScreen[numberOfImageDrawingPapers];
  HBITMAP oldBitmap[numberOfImageDrawingPapers];

  int currentWidth[numberOfImageDrawingPapers];
  int currentHeight[numberOfImageDrawingPapers];


  /** Stores if a special drawing is painted on a special paper. */
  bool showImageDrawing[numberOfImageDrawingPapers][Drawings::numberOfImageDrawings];

  /** Stores if a special drawing is painted on at least one paper. */
  bool imageDrawingIsShownOnAtLeastOnePaper[Drawings::numberOfImageDrawings];

  /** Stores if a special debug image is painted on at least one paper. */
  bool debugImageIsShownAtLeastOnce[Images::numberOfImageIDs];

  /** Stores, which of the images is shown on a special paper. */
  ImageID imageID[numberOfImageDrawingPapers];

  /** Pointer to the CWnds of where the drawing papers are shown. */
  CWnd* pWnd[numberOfImageDrawingPapers];

  bool sendDebugImagesFromRobot;

  bool highRes[numberOfImageDrawingPapers];

public:
  Image imageYUV[numberOfImageIDs];
//  ColorClassImage imageColorClass;
};

#endif //ImageDrawingManager_h





/*
* Change log :
* 
* $Log: ImageDrawingManager.h,v $
* Revision 1.1.1.1  2004/05/22 17:30:20  cvsadm
* created new repository GT2004_WM
*
* Revision 1.6  2004/04/07 13:00:46  risler
* ddd checkin after go04 - second part
*
* Revision 1.2  2004/04/06 13:19:37  risler
* cleaned up and improved high resolution image support
*
* Revision 1.1.1.1  2004/03/29 08:28:43  Administrator
* initial transfer from tamara
*
* Revision 1.5  2003/12/15 11:47:07  juengel
* Introduced CameraInfo
*
* Revision 1.4  2003/12/07 19:00:38  loetzsch
* removed some includes
*
* Revision 1.3  2003/11/03 20:16:09  juengel
* color class images can be sent from robot now
*
* Revision 1.2  2003/10/12 20:23:59  juengel
* DebugImages are only requested if selected.
*
* Revision 1.1  2003/10/07 10:11:08  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:28:24  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:27  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.7  2003/03/25 10:56:40  juengel
* DebugDrawings have to be requested in configuration "WIN_32" now.
*
* Revision 1.6  2002/12/18 16:22:56  dueffert
* doxygen docu corrected
*
* Revision 1.5  2002/11/12 10:49:02  juengel
* New debug drawing macros - second step
* -moved /Tools/Debugging/PaintMethods.h and . cpp
*  to /Visualization/DrawingMethods.h and .cpp
* -moved DebugDrawing.h and .cpp from /Tools/Debugging/
*  to /Visualization
*
* Revision 1.4  2002/09/21 23:17:05  juengel
* Removed different debug image types.
*
* Revision 1.3  2002/09/20 23:28:34  juengel
* Moved instance of DebugDrawingManager to RobotControlMainFrame.
*
* Revision 1.2  2002/09/19 23:38:54  juengel
* Changed debug image mechanisms.
*
* Revision 1.1  2002/09/10 15:49:12  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.2  2002/06/11 16:52:30  roefer
* Replaced type of oldBitmap, temporary MFC objects only survive one message handling
*/
