/** 
 * @file ImageDrawingManager.cpp
 * Implementation of class ImageDrawingManager.
 *
 * @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
 */

#include "ImageDrawingManager.h"
#include "Visualization/ImageMethods.h"
#include "Visualization/PaintMethodsWin32.h"

#include "RobotControlDebugKeyTables.h"


ImageDrawingManager::ImageDrawingManager()
{
  int i;
  for(i = 0; i < Drawings::numberOfImageDrawings; i++)
  {
    imageDrawing[i] = NULL;
    isDrawingValid[i] = false;
  }

  for(i = 0; i < numberOfImageDrawingPapers; i++)
  {
    ImageDrawingPaper currentDrawingPaper = (ImageDrawingPaper)i;
    this->pWnd[currentDrawingPaper] = NULL;
    imageID[currentDrawingPaper] = Images::noImage;
    bmpOffScreen[currentDrawingPaper] = 0;
    highRes[currentDrawingPaper] = (1 == AfxGetApp()->GetProfileInt("DebugDrawingManager\\ImageDrawingPaper\\HighRes", DrawingPapers::getDrawingPaperName(currentDrawingPaper), (currentDrawingPaper==DrawingPapers::largeImageViewer) ) );
  }

  sendDebugImagesFromRobot = false;
}
  

ImageDrawingManager::~ImageDrawingManager()
{
  int i;
  for(i = 0; i < Drawings::numberOfImageDrawings; i++)
  {
    if(imageDrawing[i] != NULL)
    {
      delete imageDrawing[i];
    }
  }
  for(i = 0; i < numberOfImageDrawingPapers; i++)
  {
    ImageDrawingPaper currentDrawingPaper = (ImageDrawingPaper)i;
    
    if(bmpOffScreen[currentDrawingPaper])
    {
      dcOffScreen[currentDrawingPaper].SelectObject(oldBitmap[currentDrawingPaper]);
      bmpOffScreen[currentDrawingPaper]->DeleteObject();
      delete bmpOffScreen[currentDrawingPaper];
      bmpOffScreen[currentDrawingPaper] = 0;
    }
  }
}

void ImageDrawingManager::setDebugDrawing(DebugDrawing* pDebugDrawing)
{
  if(imageDrawing[pDebugDrawing->imageDrawingID] != NULL)
  {
    delete imageDrawing[pDebugDrawing->imageDrawingID];
  }
  imageDrawing[pDebugDrawing->imageDrawingID] = new DebugDrawing(pDebugDrawing);
  isDrawingValid[pDebugDrawing->imageDrawingID] = true;
  //    addDrawingToDrawingPapers(pDebugDrawing->imageDrawingID);
  repaintDrawingPapers(pDebugDrawing->imageDrawingID);
}

void ImageDrawingManager::showDrawingOnDrawingPaper
(
 ImageDrawingPaper drawingPaper,
 Drawings::ImageDrawing drawing,
 bool show
 )
{
  this->showImageDrawing[drawingPaper][drawing] = show;

  imageDrawingIsShownOnAtLeastOnePaper[drawing] = false;
  for(int i = 0; i < numberOfImageDrawingPapers; i++)
  {
    if(showImageDrawing[i][drawing]) imageDrawingIsShownOnAtLeastOnePaper[drawing] = true;
  }
  sendDebugKeysForSimulatedRobots();
  repaintDrawingPaper(drawingPaper);
}

void ImageDrawingManager::sendDebugKeysForSimulatedRobots()
{
  DebugKeyTable::debugKeyID debugKeyID;
  for(int i = 0; i < Drawings::numberOfImageDrawings; i++)
  {
    if(Drawings::getDebugKeyID((Drawings::ImageDrawing)i, debugKeyID))
    {
      if(imageDrawingIsShownOnAtLeastOnePaper[i])
        getDebugKeyTables().forSimulatedRobots.set(debugKeyID, DebugKey::always);
      else
        getDebugKeyTables().forSimulatedRobots.set(debugKeyID, DebugKey::disabled);
    }
  }
  for(i = 0; i < Images::numberOfImageIDs; i++)
  {
    if(Images::getDebugKeyID((Images::ImageID)i, debugKeyID))
    {
      //local
      if(debugImageIsShownAtLeastOnce[i] && !sendDebugImagesFromRobot)
        getDebugKeyTables().forSimulatedRobots.set(debugKeyID, DebugKey::always);
      else
        getDebugKeyTables().forSimulatedRobots.set(debugKeyID, DebugKey::disabled);

      //robot
      if(debugImageIsShownAtLeastOnce[i] && sendDebugImagesFromRobot)
        getDebugKeyTables().forPhysicalRobots.set(debugKeyID, DebugKey::always);
      else
        getDebugKeyTables().forPhysicalRobots.set(debugKeyID, DebugKey::disabled);
    }
  }
  getDebugKeyTables().sendForSimulatedRobots();
  getDebugKeyTables().sendForPhysicalRobots();
}

bool ImageDrawingManager::isDrawingShownOnDrawingPaper
(
 ImageDrawingPaper drawingPaper,
 Drawings::ImageDrawing drawing
 )
{
  return showImageDrawing[drawingPaper][drawing];
}

void ImageDrawingManager::setWnd
(
 ImageDrawingPaper drawingPaper, 
 CWnd* pWnd
 )
{
  this->pWnd[drawingPaper] = pWnd;
  dcOffScreen[drawingPaper].CreateCompatibleDC(pWnd->GetDC());
  createNewBitmap(drawingPaper);
}

void ImageDrawingManager::repaintDrawingPaper
(
 ImageDrawingPaper drawingPaper
 )
{
  if(pWnd[drawingPaper] != NULL)
  {
    pWnd[drawingPaper]->RedrawWindow(NULL,NULL,RDW_INVALIDATE);
  }
}

void ImageDrawingManager::repaintDrawingPapers
(
 Drawings::ImageDrawing drawing
 )
{
  for(int i = 0; i < numberOfImageDrawingPapers; i++)
  {
    ImageDrawingPaper currentDrawingPaper = (ImageDrawingPaper)i;
    if(showImageDrawing[currentDrawingPaper][drawing])
    {
      repaintDrawingPaper(currentDrawingPaper);
    } // if(showImageDrawing[currentDrawingPaper][drawing])
  } // for(int i = 0; i < Drawings::numberOfImageDrawingPapers; i++)
}

void ImageDrawingManager::repaintDrawingPapers
(
 ImageID imageID
 )
{
  for(int i = 0; i < numberOfImageDrawingPapers; i++)
  {
    ImageDrawingPaper currentDrawingPaper = (ImageDrawingPaper)i;
    if(this->imageID[currentDrawingPaper] == imageID)
    {
      repaintDrawingPaper(currentDrawingPaper);
    } // if(showImageDrawing[currentDrawingPaper][drawing])
  } // for(int i = 0; i < Drawings::numberOfImageDrawingPapers; i++)
}

void ImageDrawingManager::paint
(
 CRect* rect, 
 ImageDrawingPaper drawingPaper
 )
{
  if(pWnd[drawingPaper] == NULL) return;
  int x, y;
  x = (rect->right - rect->left);
  y = (rect->bottom - rect->top);

  double imageWidth, imageHeight;
  imageWidth = imageYUV[imageID[drawingPaper]].cameraInfo.resolutionWidth;
  imageHeight = imageYUV[imageID[drawingPaper]].cameraInfo.resolutionHeight;

  if( x / imageWidth > y / imageHeight )
  {
    x = (int)(imageWidth * y / imageHeight);
  }
  else
  {
    y = (int)(imageHeight * x / imageWidth);
  }

  if(x != currentWidth[drawingPaper] || y != currentHeight[drawingPaper])
  {
    currentWidth[drawingPaper] = x;
    currentHeight[drawingPaper] = y;
    createNewBitmap(drawingPaper);
  }

  drawDrawingPaperToOffScreenDC(drawingPaper);

  //copy the off screen dc to the dc for painting
  CDC* pDC = pWnd[drawingPaper]->GetDC();
  pDC->BitBlt(rect->left, rect->top, currentWidth[drawingPaper], currentHeight[drawingPaper],
    &dcOffScreen[drawingPaper], 0, 0, SRCCOPY);
  pWnd[drawingPaper]->ReleaseDC(pDC);
}

void ImageDrawingManager::copyDrawingPaperToClipBoard
(
 ImageDrawingPaper drawingPaper
 )
{
  HENHMETAFILE metaFileHandle;
  CMetaFileDC* metaFileDC;
  metaFileDC = new CMetaFileDC();

  metaFileDC->CreateEnhanced(NULL, NULL, NULL, NULL);

  for(int j = 0; j < Drawings::numberOfImageDrawings; j++)
  {  
    Drawings::ImageDrawing currentDrawing = (Drawings::ImageDrawing)j;
    if(
      showImageDrawing[drawingPaper][currentDrawing] &&
      this->imageDrawing[currentDrawing] != NULL
      )
    {
      PaintMethodsWin32::paintDebugDrawingToCDC(imageDrawing[currentDrawing], *metaFileDC);
    } // if(showDrawing[drawingPaper][currentDrawing])
  } // for(int j = 0; j < Drawings::numberOfDrawings; j++)

  metaFileHandle = metaFileDC->CloseEnhanced();
  
  OpenClipboard(NULL);
	EmptyClipboard();
	::SetClipboardData(CF_ENHMETAFILE,metaFileHandle);
	CloseClipboard();

  DeleteEnhMetaFile(metaFileHandle);
  metaFileDC->DeleteDC();
  delete metaFileDC;
}

void ImageDrawingManager::copyImageToClipBoard
(
 ImageDrawingPaper drawingPaper
 )
{
  OpenClipboard(NULL);
	EmptyClipboard();
	::SetClipboardData(CF_BITMAP,bmpOffScreen[drawingPaper]->m_hObject);
	CloseClipboard();
}
 
void ImageDrawingManager::setImageID
(
 ImageDrawingPaper drawingPaper,
 ImageID imageID
 )
{
  ImageID oldImageID = this->imageID[drawingPaper];
  this->imageID[drawingPaper] = imageID;
  
  repaintDrawingPaper(drawingPaper);

  debugImageIsShownAtLeastOnce[imageID] = true;
  debugImageIsShownAtLeastOnce[oldImageID] = false;
  for(int i = 0; i < numberOfImageDrawingPapers; i++)
  {
    if(this->imageID[i] == oldImageID) debugImageIsShownAtLeastOnce[oldImageID] = true;
  }
  sendDebugKeysForSimulatedRobots();
}

ImageID ImageDrawingManager::getImageID
(
 ImageDrawingPaper drawingPaper
 )
{
  return imageID[drawingPaper];
}

void ImageDrawingManager::notifyImage
(
 ImageID imageID
 )
{
  int i;
  if(imageID == Images::rawImage)
  {
    for(i = 0; i < Drawings::numberOfImageDrawings; i++)
    {
      Drawings::ImageDrawing currentDrawing = (Drawings::ImageDrawing)i;
      isDrawingValid[currentDrawing] = false;
    }
  }
  //  drawImageToOffScreenDCs(imageID);
  repaintDrawingPapers(imageID);
}

void ImageDrawingManager::loadProperties()
{
  int i;
  int j;
  for(j = 0; j < Drawings::numberOfImageDrawings; j++)
  {
    imageDrawingIsShownOnAtLeastOnePaper[j] = false;
  }
  for(i = 0; i < numberOfImageDrawingPapers; i++)
  {
    for(j = 0; j < Drawings::numberOfImageDrawings; j++)
    {
      showImageDrawing[i][j] = false;
    }
    CString paperString;
    paperString = DrawingPapers::getDrawingPaperName(ImageDrawingPaper(i)); 
    

    this->imageID[(ImageDrawingPaper)i] = Images::noImage;
    CString defaultString;
    switch ((ImageDrawingPaper)i)
    {
      case DrawingPapers::imageViewer1: defaultString = Images::getImageIDName(Images::rawImage); break;
      case DrawingPapers::imageViewer2: defaultString = Images::getImageIDName(Images::segmentedImage1); break;
      default: defaultString = Images::getImageIDName(Images::noImage);
    }

    CString imageString = AfxGetApp()->GetProfileString("DebugDrawingManager\\ImageDrawingPaper\\Images", paperString, defaultString); 
    for(int k=0; k < numberOfImageIDs; k++)
    {
      if (Images::getImageIDName(ImageID(k)) == imageString)
      {
        imageID[(ImageDrawingPaper)i] = (ImageID)k;
      }
    }

    CString drawingsString = AfxGetApp()->GetProfileString("DebugDrawingManager\\ImageDrawingPaper\\Drawings", paperString);
    int currentReturnPosition = 0;
    int currentLength = drawingsString.GetLength();
    while(currentLength != 0)
    {
      currentReturnPosition = drawingsString.Find(",");
      CString extractedString = drawingsString.Left(currentReturnPosition);

      for(int j=0; j < Drawings::numberOfImageDrawings; j++)
      {
        if (Drawings::getDrawingName(Drawings::ImageDrawing(j)) == extractedString)
        {
          showImageDrawing[i][j] = true;
          imageDrawingIsShownOnAtLeastOnePaper[j] = true;
        }
      }
      drawingsString = drawingsString.Right(currentLength - currentReturnPosition - 1);
      currentLength = drawingsString.GetLength();
    }
  }
  sendDebugKeysForSimulatedRobots();
}

void ImageDrawingManager::saveProperties()
{
  int i,j;
  for(i = 0; i < numberOfImageDrawingPapers; i++)
  {
    CString paperString;
    paperString = DrawingPapers::getDrawingPaperName(ImageDrawingPaper(i)); 

    CString drawingsString = "";
    for(j = 0; j < Drawings::numberOfImageDrawings; j++)
    {
      if(showImageDrawing[i][j])
      {
        drawingsString.Insert(9999, Drawings::getDrawingName(Drawings::ImageDrawing(j)));
        drawingsString.Insert(9999, ",");
      }
    }
    AfxGetApp()->WriteProfileString("DebugDrawingManager\\ImageDrawingPaper\\Drawings", paperString, drawingsString);
    
    CString imageString;
    imageString = Images::getImageIDName(imageID[ImageDrawingPaper(i)]); 
    AfxGetApp()->WriteProfileString("DebugDrawingManager\\ImageDrawingPaper\\Images", paperString, imageString);
  }
}

void ImageDrawingManager::createNewBitmap(ImageDrawingPaper drawingPaper)
{
  if(bmpOffScreen[drawingPaper])
  {
    dcOffScreen[drawingPaper].SelectObject(oldBitmap[drawingPaper]);
    bmpOffScreen[drawingPaper]->DeleteObject();
    delete bmpOffScreen[drawingPaper];
  }
  bmpOffScreen[drawingPaper] = new CBitmap;
  bmpOffScreen[drawingPaper]->CreateCompatibleBitmap(
    pWnd[drawingPaper]->GetDC(), 
    currentWidth[drawingPaper],
    currentHeight[drawingPaper]);
  oldBitmap[drawingPaper] = *dcOffScreen[drawingPaper].SelectObject(bmpOffScreen[drawingPaper]);
}

void ImageDrawingManager::drawDrawingPaperToOffScreenDC(ImageDrawingPaper drawingPaper)
{
  drawImageToOffScreenDC(drawingPaper);
  int i;
  for(i = 0; i < Drawings::numberOfImageDrawings; i++)
  {
    Drawings::ImageDrawing currentDrawing = (Drawings::ImageDrawing)i;
    if(showImageDrawing[drawingPaper][currentDrawing] &&
       isDrawingValid[currentDrawing])
    {
      drawDrawingToOffScreenDC(currentDrawing, drawingPaper);
    }
  }
}

void ImageDrawingManager::drawImageToOffScreenDC
(
 ImageDrawingPaper drawingPaper
 )
{
  dcOffScreen[drawingPaper].SaveDC();
  dcOffScreen[drawingPaper].SetMapMode(MM_ANISOTROPIC);
  dcOffScreen[drawingPaper].SetViewportExt(currentWidth[drawingPaper], currentHeight[drawingPaper]);
  dcOffScreen[drawingPaper].SetViewportOrg(0,0);
  
  switch(imageID[drawingPaper])
  {
  case Images::noImage: 
    {
      CBrush br(GetSysColor(COLOR_APPWORKSPACE));
      CRect rect;
      rect.left = 0;
      rect.right = currentWidth[drawingPaper];
      rect.top = 0;
      rect.bottom = currentHeight[drawingPaper];
      dcOffScreen[drawingPaper].FillRect(&rect, &br);
      break;
    }
/*  case Images::segmentedImage:
    {
      ImageMethods::paintColorClassImage2CDC(
        dcOffScreen[drawingPaper],
        imageColorClass,
        CRect(0,0,currentWidth[drawingPaper], currentHeight[drawingPaper]));
      break;
    }
*/
  default:
    {
      dcOffScreen[drawingPaper].SetWindowExt(currentWidth[drawingPaper],currentHeight[drawingPaper]);
      ImageMethods::paintImage2CDCAsYUV(
        dcOffScreen[drawingPaper],
        imageYUV[imageID[drawingPaper]],
        CRect(0,0,currentWidth[drawingPaper], currentHeight[drawingPaper]), highRes[drawingPaper]);
      break;
    }
  }
  dcOffScreen[drawingPaper].RestoreDC(-1);
}

void ImageDrawingManager::drawDrawingToOffScreenDC
(
 Drawings::ImageDrawing drawingID, 
 ImageDrawingPaper drawingPaper
 )
{
  int imageWidth, imageHeight;
  imageWidth = imageYUV[imageID[drawingPaper]].cameraInfo.resolutionWidth;
  imageHeight = imageYUV[imageID[drawingPaper]].cameraInfo.resolutionHeight;

  dcOffScreen[drawingPaper].SaveDC();
  dcOffScreen[drawingPaper].SetMapMode(MM_ANISOTROPIC);
  dcOffScreen[drawingPaper].SetViewportExt(currentWidth[drawingPaper], currentHeight[drawingPaper]);
  dcOffScreen[drawingPaper].SetViewportOrg(0,0);
  if(imageDrawing[drawingID])
  {
    dcOffScreen[drawingPaper].SetWindowExt(imageWidth,imageHeight);
    PaintMethodsWin32::paintDebugDrawingToCDC(imageDrawing[drawingID], dcOffScreen[drawingPaper]);
  }
  dcOffScreen[drawingPaper].RestoreDC(-1);
}

void ImageDrawingManager::setSendDebugImagesFromRobot(bool sendDebugImagesFromRobot)
{
  this->sendDebugImagesFromRobot = sendDebugImagesFromRobot;
  sendDebugKeysForSimulatedRobots();
}

bool ImageDrawingManager::getSendDebugImagesFromRobot()
{
  return sendDebugImagesFromRobot;
}

void ImageDrawingManager::setHighRes(ImageDrawingPaper drawingPaper, bool highRes)
{
  this->highRes[drawingPaper] = highRes;
  repaintDrawingPaper(drawingPaper);
  
  CString paperString = DrawingPapers::getDrawingPaperName(drawingPaper);
  AfxGetApp()->WriteProfileInt("DebugDrawingManager\\ImageDrawingPaper\\HighRes", paperString, (getHighRes(drawingPaper) ? 1 : 0) );
}

bool ImageDrawingManager::getHighRes(ImageDrawingPaper drawingPaper)
{
  return highRes[drawingPaper];
}

/*
 * Change log :
 * 
 * $Log: ImageDrawingManager.cpp,v $
 * Revision 1.3  2004/06/09 13:31:29  thomas
 * added storing of highres-settings in (large-)imageviewer
 *
 * Revision 1.2  2004/06/09 13:12:31  thomas
 * modified default-value for highres in (large-)imageviewer
 *
 * Revision 1.1.1.1  2004/05/22 17:30:20  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.5  2004/04/07 13:00:46  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.3  2004/04/06 13:19:37  risler
 * cleaned up and improved high resolution image support
 *
 * Revision 1.2  2004/03/29 15:19:02  Marc
 * Intruduced the Black and White Image
 * Normal Images (not Jpeg) images were now send as Color Image with BW
 *
 * Revision 1.4  2004/01/04 12:08:28  juengel
 * adaptation for larger images
 *
 * Revision 1.3  2003/11/03 20:16:09  juengel
 * color class images can be sent from robot now
 *
 * Revision 1.2  2003/10/12 20:23:59  juengel
 * DebugImages are only requested if selected.
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/01 10:23:14  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.1.1.1  2003/07/02 09:40:27  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.8  2003/03/25 10:56:40  juengel
 * DebugDrawings have to be requested in configuration "WIN_32" now.
 *
 * Revision 1.7  2003/02/13 09:38:54  juengel
 * segmentedImage is not a color class image any more.
 *
 * Revision 1.6  2002/11/12 10:49:02  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.5  2002/09/22 18:40:52  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.4  2002/09/21 23:17:05  juengel
 * Removed different debug image types.
 *
 * Revision 1.3  2002/09/20 23:28:34  juengel
 * Moved instance of DebugDrawingManager to RobotControlMainFrame.
 *
 * Revision 1.2  2002/09/19 23:38:54  juengel
 * Changed debug image mechanisms.
 *
 * Revision 1.1  2002/09/10 15:49:12  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.3  2002/06/11 16:52:30  roefer
 * Replaced type of oldBitmap, temporary MFC objects only survive one message handling
 *
 * Revision 1.2  2002/06/10 11:27:12  roefer
 * Maybe reason for crash during termination of RobotControl removed
 *
 * Revision 1.1.1.1  2002/05/10 12:40:27  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.6  2002/04/25 14:50:36  kallnik
 * changed double/float to double
 * added several #include GTMath
 *
 * PLEASE use double
 *
 * Revision 1.5  2002/02/11 00:53:33  loetzsch
 * ::Paint Methoden in ::paint umbenannt
 *
 * Revision 1.4  2002/02/07 23:25:27  loetzsch
 * Changed registry access
 *
 * Revision 1.3  2002/02/02 23:46:22  juengel
 * Copy image and drawings to clipboard added.
 *
 * Revision 1.2  2002/01/26 20:27:02  juengel
 * Memory Leaks ausgebaut.
 *
 * Revision 1.1  2002/01/26 18:07:33  juengel
 * no message
 *
 */











