/**
* @file DebugDrawingManagerMessageHandler.cpp
*
* Implementation of class DebugDrawingManagerMessageHandler.
*
* @author Martin Ltzsch
*/

#include "StdAfx.h"

#include "DebugDrawingManagerMessageHandler.h"
#include "RobotControlQueues.h"
#include "DebugDrawingManager.h"
#include "Platform/SystemCall.h"

#include "Representations/Cognition/RobotState.h"

bool DebugDrawingManagerMessageHandler::handleMessage(InMessage& message)
{
  switch(message.getMessageID())
  {
  case idDebugDrawing2:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      char shapeType;
      char id;
      char typeOfDrawing;
      message.bin >> shapeType;
      message.bin >> id;
      message.bin >> typeOfDrawing;
      
      if((Drawings::TypeOfDrawing)typeOfDrawing == Drawings::drawingOnImage)
        id += Drawings::numberOfFieldDrawings;
      
      debugDrawingManager.incompleteDrawing[id]->addShapeFromQueue(message, (Drawings::ShapeType)shapeType);
    }
    return true;

  case idDebugDrawing:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      DebugDrawing debugDrawing;
      message.bin >> debugDrawing;
      debugDrawingManager.setDebugDrawing(&debugDrawing);
    }
    return true;

  case idDebugDrawingFinished:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      char id, typeOfDrawing;
      message.bin >> id;
      message.bin >> typeOfDrawing;
      if((Drawings::TypeOfDrawing)typeOfDrawing == Drawings::drawingOnField)
        debugDrawingManager.drawingFinished((Drawings::FieldDrawing)id);
      else
        debugDrawingManager.drawingFinished((Drawings::ImageDrawing)id);
    }
    return true;

  case idDebugImage:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      int id;
      message.bin >> id;
      message.bin >> debugDrawingManager.imageDrawingManager.imageYUV[id];
      debugDrawingManager.notifyImage((Images::ImageID)id);
    }
    return true;

  case idDebugColorClassImage:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      int id;
      message.bin >> id;
      ColorClassImage colorClassImage;
      message.bin >> colorClassImage;
      colorClassImage.convertToImage(debugDrawingManager.imageDrawingManager.imageYUV[id]);
      debugDrawingManager.notifyImage((Images::ImageID)id);
    }
    return true;

  case idImage:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      message.bin >> debugDrawingManager.imageDrawingManager.imageYUV[Images::rawImage];
			debugDrawingManager.notifyImage(Images::rawImage);
    }
    return true;

  case idWorldState:
    {
      CameraMatrix cameraMatrix;
      CameraInfo cameraInfo;
      RobotState robotState;
      Player player;
      BallModel ballModel;
      PlayerPoseCollection playerPoseCollection;
      ObstaclesModel obstaclesModel;
      RobotPose robotPose;

      message.bin >> RECEIVE_WORLDSTATE(robotPose,
        ballModel,playerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo);
      
      int robotNumber = message.getRobotNumber();

      if (robotNumber<8) // player number is not undefined
      {
        if (getQueues().isFromSelectedOrUndefinedRobot(message))
        {
          teamColorOfSelectedRobot = message.getTeamColor();
          repaintField();
        }        

        timeWhenLastWorldStateReceived[robotNumber] = SystemCall::getCurrentSystemTime();  

        if (message.getTeamColor()!=teamColorOfSelectedRobot)
        {
          // flip the robot pose 
          robotPose.translation *= -1;
          robotPose.rotation += fromDegrees(180);

          // flip the ball position
          ballModel.communicated *=-1;
          ballModel.seen *=-1;
        }

        robotPoses[robotNumber] = robotPose; // store the robotpose for drawing of percepts

        // draw the world state to the corresponding drawing in the array
        worldStateFieldDrawing[robotNumber] = DebugDrawing(Drawings::worldState);
        DrawingMethods::paintWorldState(worldStateFieldDrawing[robotNumber], robotPose,
          ballModel, playerPoseCollection, message.getTeamColor(),
          message.getTimeStamp(), false);
        
        // draw the obstacles model to the corresponding drawing in the array
        obstaclesModelFieldDrawing[robotNumber] = DebugDrawing(Drawings::models_obstaclesField);
        DrawingMethods::paintObstaclesModelForFieldView(
          obstaclesModelFieldDrawing[robotNumber],
              obstaclesModel, robotPose);

        // concat all the stored drawings into one drawing
        DebugDrawing worldStateDrawing(Drawings::worldState);
        DebugDrawing obstaclesModelDrawing(Drawings::models_obstaclesField);

        int i;
        for (i=0;i<8;i++)
        {
          if (SystemCall::getTimeSince(timeWhenLastWorldStateReceived[i]) < 5000)
            // paint only recently received world states
          {
            worldStateDrawing += worldStateFieldDrawing[i];
            obstaclesModelDrawing += obstaclesModelFieldDrawing[i];
          }
        }
        debugDrawingManager.setDebugDrawing(&worldStateDrawing);
        debugDrawingManager.setDebugDrawing(&obstaclesModelDrawing);
      }  
      
      if (getQueues().isFromSelectedOrUndefinedRobot(message))
      {
        message.resetReadPosition();
        
        DebugDrawing models_obstacles_drawing(Drawings::models_obstacles);
        DrawingMethods::paintObstaclesModelForImageView(models_obstacles_drawing,
          obstaclesModel, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&models_obstacles_drawing);

        DebugDrawing models_freePartOfGoal_drawing(Drawings::models_freePartOfGoal);
        DrawingMethods::paintFreePartOfGoalModelForImageView(models_freePartOfGoal_drawing,
          obstaclesModel, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&models_freePartOfGoal_drawing);

        DebugDrawing drawingRadar(Drawings::models_obstaclesRadar);
        DrawingMethods::paintObstaclesModelForRadarView(drawingRadar,
          obstaclesModel);
        debugDrawingManager.setDebugDrawing(&drawingRadar);
      }
    }
    return true;

  case idOracledWorldState:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      // paint the oracled world state
      Player player;
      RobotPose oracledRobotPose;
      BallModel oracledBallModel;
      PlayerPoseCollection oracledPlayerPoseCollection;
      ObstaclesModel obstaclesModel;
      RobotState robotState;
      CameraMatrix cameraMatrix;
      CameraInfo cameraInfo;
      
      message.bin >> RECEIVE_WORLDSTATE(oracledRobotPose,oracledBallModel,oracledPlayerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo);
      DebugDrawing drawing(Drawings::worldStateOracle);
      DrawingMethods::paintWorldState(drawing, oracledRobotPose,
        oracledBallModel, oracledPlayerPoseCollection, player.getTeamColor(), 
        message.getTimeStamp(), true);
      debugDrawingManager.setDebugDrawing(&drawing);
      
    }
    return true;

  case idPercepts:
    {
      Player player;
      CameraMatrix cameraMatrix;
      CameraInfo cameraInfo;
      LandmarksPercept landmarksPercept;
      BallPercept ballPercept;
      PlayersPercept playersPercept;
      ObstaclesPercept obstaclesPercept;
      PSDPercept psdPercept;
      CollisionPercept collisionPercept;
      LinesPercept linesPercept;
      EdgesPercept edgesPercept;

      message.bin >> RECEIVE_PERCEPTS(cameraMatrix, cameraInfo, ballPercept,
        landmarksPercept,linesPercept,edgesPercept,playersPercept, obstaclesPercept, psdPercept, collisionPercept);

      int robotNumber = message.getRobotNumber();

      if (robotNumber<8) // player number is not undefined
      {
        if (getQueues().isFromSelectedOrUndefinedRobot(message))
        {
          teamColorOfSelectedRobot = message.getTeamColor();
          repaintField();
        }        

        timeWhenLastPerceptsReceived[robotNumber] = SystemCall::getCurrentSystemTime();  

        // draw the percepts to the corresponding drawing in the array
        perceptsFieldDrawing[robotNumber] = DebugDrawing(Drawings::percepts_ballFlagsGoalsField);
        DrawingMethods::paintPerceptCollectionForFieldView(perceptsFieldDrawing[robotNumber],
          landmarksPercept, ballPercept, playersPercept, obstaclesPercept, linesPercept, edgesPercept,
          robotPoses[robotNumber],message.getTeamColor());

        // concat all the stored drawings into one drawing
        int i;
        DebugDrawing drawingField(Drawings::percepts_ballFlagsGoalsField);
        for (i=0;i<8;i++)
        {
          if (SystemCall::getTimeSince(timeWhenLastPerceptsReceived[i]) < 5000)
            // paint only recently received percepts
          {
            drawingField += perceptsFieldDrawing[i];
          }
        }
        debugDrawingManager.setDebugDrawing(&drawingField);
      }

      if (getQueues().isFromSelectedOrUndefinedRobot(message))
      {
        
        DebugDrawing drawingRadar(Drawings::percepts_ballFlagsGoalsRadar);
        DrawingMethods::paintPerceptCollectionForRadarView(drawingRadar,
          landmarksPercept, ballPercept, playersPercept, obstaclesPercept, linesPercept, edgesPercept);
        debugDrawingManager.setDebugDrawing(&drawingRadar);
/*        
        DebugDrawing drawingImage(Drawings::percepts_ballFlagsGoals);
        DrawingMethods::paintPerceptCollectionForImageView(drawingImage,
          landmarksPercept, ballPercept, playersPercept, linesPercept, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&drawingImage);
*/
        DebugDrawing drawingImage(Drawings::percepts_ball);
        DrawingMethods::paintBallPerceptForImageView(drawingImage, ballPercept, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&drawingImage);

        drawingImage.reset();
        drawingImage.imageDrawingID = Drawings::percepts_flagsGoals;
        DrawingMethods::paintLandmarksPerceptForImageView(drawingImage, landmarksPercept, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&drawingImage);

        drawingImage.reset();
        drawingImage.imageDrawingID = Drawings::percepts_freePartOfGoal;
        DrawingMethods::paintFreePartOfGoalPerceptForImageView(drawingImage, obstaclesPercept, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&drawingImage);

        drawingImage.reset();
        drawingImage.imageDrawingID = Drawings::percepts_lines;
        DrawingMethods::paintLinesPerceptForImageView(drawingImage, linesPercept, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&drawingImage);

        drawingImage.reset();
        drawingImage.imageDrawingID = Drawings::percepts_edges;
        DrawingMethods::paintEdgesPerceptForImageView(drawingImage, edgesPercept, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&drawingImage);

        drawingImage.reset();
        drawingImage.imageDrawingID = Drawings::percepts_obstacles;
        DrawingMethods::paintObstaclesPerceptForImageView(drawingImage, obstaclesPercept, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&drawingImage);

        drawingImage.reset();
        drawingImage.imageDrawingID = Drawings::percepts_psd;
        DrawingMethods::paintPSDPerceptForImageView(drawingImage, psdPercept, cameraMatrix, cameraInfo);
        debugDrawingManager.setDebugDrawing(&drawingImage);
      }
    }
    return true;

  case idSpecialPercept:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      Player player;
      SpecialPercept specialPercept;
      CameraMatrix cameraMatrix;
      message.bin >> player >> specialPercept >> cameraMatrix;
      
      DebugDrawing drawingField(Drawings::percepts_specialField);
      
      DrawingMethods::paintSpecialPerceptForFieldView(drawingField,
        specialPercept,robotPoses[message.getRobotNumber()],player.getTeamColor());
      debugDrawingManager.setDebugDrawing(&drawingField);
      
      DebugDrawing drawingImage(Drawings::percepts_special);
      DrawingMethods::paintSpecialPerceptForImageView(drawingImage, specialPercept, cameraMatrix);
      debugDrawingManager.setDebugDrawing(&drawingImage);
      
    }
    return true;

  default:
    return false;
  }
}

DebugDrawingManagerMessageHandler::DebugDrawingManagerMessageHandler(DebugDrawingManager& debugDrawingManager)
:debugDrawingManager(debugDrawingManager),
teamColorOfSelectedRobot(Player::red)
{
  int i;
  for (i=0;i<8;i++)
  {
    timeWhenLastPerceptsReceived[i] = 0;
    timeWhenLastWorldStateReceived[i] = 0;
  }
}

void DebugDrawingManagerMessageHandler::repaintField()
{
  debugDrawingManager.setDebugDrawing(teamColorOfSelectedRobot==Player::red ? 
    &debugDrawingManager.fieldPolygonsDrawingRed 
    : &debugDrawingManager.fieldPolygonsDrawingBlue);

  debugDrawingManager.setDebugDrawing(&debugDrawingManager.fieldLinesDrawing);
}


/*
* Change Log:
*
* $Log: DebugDrawingManagerMessageHandler.cpp,v $
* Revision 1.3  2004/06/15 10:58:28  thomas
* added edge-specialist, edges-percept, debug-drawings etc. (not yet called from image-processor)
*
* Revision 1.2  2004/05/22 22:52:02  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:30:19  cvsadm
* created new repository GT2004_WM
*
* Revision 1.7  2004/04/07 13:00:46  risler
* ddd checkin after go04 - second part
*
* Revision 1.3  2004/04/06 13:19:37  risler
* cleaned up and improved high resolution image support
*
* Revision 1.2  2004/03/29 15:19:02  Marc
* Intruduced the Black and White Image
* Normal Images (not Jpeg) images were now send as Color Image with BW
*
* Revision 1.6  2004/02/06 10:16:23  juengel
* percepts for image view are painted in different debug drawings now
*
* Revision 1.5  2004/02/03 13:20:47  spranger
* renamed all references to  class BallP_osition to BallModel (possibly changed include files)
*
* Revision 1.4  2004/01/10 10:09:14  juengel
* Added CameraInfo to and removed Player from (SEND|RECEIVE)_(PERCEPTS|WORLDSTATE).
*
* Revision 1.3  2003/12/15 11:46:14  juengel
* Introduced CameraInfo
*
* Revision 1.2  2003/12/09 18:13:31  loetzsch
* Added parameter timestamp to
* Drawingmethods::paintWorldStateToDebugDrawing
*
* Moved prepainted instances for fieldlines and field polygons from
* DebugDrawingManagerMessageHAndler to DebugDrawingManager
*
* Revision 1.1  2003/12/07 19:04:11  loetzsch
* The message handling for the debug drawing manager is now done in a
* separate class.
*
* In the field view, world states and percepts from up to 8 robots are displayed.
*
*/
