/** 
* @file DebugDrawingManager.h
* Declaration of class DebugDrawingManager.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/

#ifndef __DebugDrawingManager_h_
#define __DebugDrawingManager_h_

#include "StdAfx.h"

#include "ImageDrawingManager.h"
#include "DebugDrawingManagerMessageHandler.h"

using DrawingPapers::FieldDrawingPaper;
using DrawingPapers::numberOfFieldDrawingPapers;

/**
 * A DebugDrawingManager object handles DebugDrawing objects. Incoming debug drawings
 * and Images are distributed to several drawing papers. These papers can be displayed
 * in a CDC or be saved to a MetaFile.
 */
class DebugDrawingManager
{
public:
  /** Constructor */
  DebugDrawingManager();
  /** Destructor */
  ~DebugDrawingManager();

  /** Handles incoming Messages */
  DebugDrawingManagerMessageHandler messageHandler;

  void drawingFinished(Drawings::FieldDrawing drawingID);

  void drawingFinished(Drawings::ImageDrawing drawingID);

  /**
  * Specifies which image is shown on the specified image drawing paper.
  * Only one image can be shown on one image drawing paper.
  * @param imageDrawingPaper Specifies the drawing paper where the image is shown.
  * @param imageID Specifies the image that is shown.
  */
  void setImageID(
    ImageDrawingPaper imageDrawingPaper,
    ImageID imageID
    );
  
    /**
    * Returns the id of the image which is shown on the specified drawing paper.
    * @param imageDrawingPaper Specifies the image drawing paper.
    * @return imageID Specifies the image that is shown on the specified drawing paper.
  */
  ImageID getImageID(ImageDrawingPaper imageDrawingPaper);
  
  /**
  * Gives the Debug Drawing Manager a reference to a debug drawing.
  * This debug drawing is copied to an internal representation.
  * @param pDebugDrawing Pointer to the debug drawing.
  */
  void setDebugDrawing(DebugDrawing* pDebugDrawing);
  
  void notifyImage(ImageID imageID);
  
  /**
  * Specifies if the specified drawing is shown on the specified drawing paper.
  * @param fieldDrawingPaper The drawing paper.
  * @param fieldDrawing The debug drawing.
  * @param show Specifies if the drawing is shown or not.
  */
  void showDrawingOnDrawingPaper(
    FieldDrawingPaper fieldDrawingPaper,
    Drawings::FieldDrawing fieldDrawing,
    bool show
    );
  
    /**
    * Specifies if the specified drawing is shown on the specified drawing paper.
    * @param imageDrawingPaper The drawing paper.
    * @param imageDrawing The debug drawing.
    * @param show Specifies if the drawing is shown or not.
  */
  void showDrawingOnDrawingPaper(
    ImageDrawingPaper imageDrawingPaper,
    Drawings::ImageDrawing imageDrawing,
    bool show
    );
  
    /**
    * Rotates the view of the specified drawing paper.
    * @param fieldDrawingPaper The drawing paper.
  */
  void rotate(FieldDrawingPaper fieldDrawingPaper);
  
  /**
  * Enables or disables the auto rotate property of the specified drawing paper.
  * If the auto rotate property is enabled, the paint method rotates the drawing paper
  * such that it is shown with maximal size.
  * @param fieldDrawingPaper The drawing paper.
  * @param enable Whether or not to enable the auto rotate property.
  */
  void enableAutoRotate(FieldDrawingPaper fieldDrawingPaper, bool enable);
  
  /**
  * @param fieldDrawingPaper The drawing paper.
  * @return Retruns if the specified drawing paper has the auto rotate property.
  */
  bool isAutoRotateEnabled(FieldDrawingPaper fieldDrawingPaper);

  /**
  * @param fieldDrawingPaper The drawing paper.
  * @param fieldDrawing The field drawing.
  * @return Retruns if the specified drawing is shown on the specified drawing paper.
  */
  bool isDrawingShownOnDrawingPaper(
    FieldDrawingPaper fieldDrawingPaper,
    Drawings::FieldDrawing fieldDrawing
    );
  

    /**
    * @param imageDrawingPaper The drawing paper.
    * @param imageDrawing The image drawing.
    * @return Retruns if the specified drawing is shown on the specified drawing paper.
  */
  bool isDrawingShownOnDrawingPaper(
    ImageDrawingPaper imageDrawingPaper,
    Drawings::ImageDrawing imageDrawing
    );
  
  DebugDrawing fieldPolygonsDrawingRed; /*!< field polygons for the red robot */
  DebugDrawing fieldPolygonsDrawingBlue; /*!< field polygons for the blue robot */
  DebugDrawing fieldLinesDrawing; /*!< field lines */
  

  XFORM  calcPaintXForm(
    CRect *dcRect, FieldDrawingPaper fieldDrawingPaper
  );

    /**
    * Paints the content of the specified drawing paper. 
    * Call this member function from the OnPaint function of the CWnd displaying the device context.
    *
    * @param pDC A pointer to a device context.
    * @param rect Specifies the rect in the device context to paint to.
    * @param fieldDrawingPaper Specifies the drawing paper to paint.
    *
  */
  void paint(
    CDC* pDC, CRect* dcRect, 
    FieldDrawingPaper fieldDrawingPaper
    );

    /**
    * Paints the content of the specified drawing paper. 
    * Call this member function from the OnPaint function of the CWnd displaying the device context.
    *
    * @param rect Specifies the rect in the device context to paint to.
    * @param imageDrawingPaper Specifies the drawing paper to paint.
    *
  */
  void paint(
    CRect *dcRect, 
    ImageDrawingPaper imageDrawingPaper
    );

  
    /**
    * The DebugDrawingManager calls the RedrawWindow method of the
    * CWnd object each time something changes at the specified drawing paper.
    * @param fieldDrawingPaper Specifies the drawing paper.
    * @param pWnd A pointer to a CWnd object.
  */
  void setCallBackMethod(
    FieldDrawingPaper fieldDrawingPaper,
    CWnd* pWnd
    );
    
    /**
    * The DebugDrawingManager calls the RedrawWindow method of the
    * CWnd object each time something changes at the specified drawing paper.
    * @param imageDrawingPaper Specifies the drawing paper.
    * @param pWnd A pointer to a CWnd object.
  */
  void setCallBackMethod(
    ImageDrawingPaper imageDrawingPaper,
    CWnd* pWnd
    );
  
  
    /**
    * Copies the specified drawing paper to the clipboard.
    * @param drawingPaper Specifies the drawing paper.
  */
  void copyDrawingPaperToClipBoard(FieldDrawingPaper drawingPaper);
    
  /**
  * Copies the specified drawing paper to the clipboard.
  * @param drawingPaper Specifies the drawing paper.
  */
  void copyDrawingPaperToClipBoard(ImageDrawingPaper drawingPaper);
  
  /**
  * Copies the image selected in the specified drawing paper to the clipboard.
  * @param drawingPaper Specifies the drawing paper.
  */
  void copyImageToClipBoard(ImageDrawingPaper drawingPaper);
  
  /**
  * Saves which drawings and which image are shown on which drawing paper and the rotation angle of the drawing papers to the reistry.
  */
  void saveProperties();
  
  /**
  * Loads which drawings and which image are shown on which drawing paper and the rotation angle of the drawing papers from the registry.
  */
  void loadProperties();

  /**
  * Increases the magnification for the specified drawing paper
  */
  void zoomIn(FieldDrawingPaper drawingPaper);

  /**
  * Decreases the magnification for the specified drawing paper
  */
  void zoomOut(FieldDrawingPaper drawingPaper);

  /**
  * Changes the magnification for the specified drawing paper according to factor
  */
  void zoomDrawingPaper(FieldDrawingPaper drawingPaper, double factor);

  void setSendDebugImagesFromRobot(bool sendDebugImagesFromRobot);
  bool getSendDebugImagesFromRobot();

  /**
  * Returns whether images are drawn in high resolution
  */
  bool getHighRes(ImageDrawingPaper drawingPaper);

  /**
  * Sets whether images are drawn in high resolution
  */
  void setHighRes(ImageDrawingPaper drawingPaper, bool highRes);

protected:
  friend class DebugDrawingManagerMessageHandler;

/**
* Generates a Metafile containing all the drawings of the specified
* @param drawingPaper Specifies the drawing paper.
* drawing paper. The handle to that Metafile is stored in the List of meta file
* handles.
  */
  void generateMetaFile(FieldDrawingPaper drawingPaper);
  
  /**
  * Calls the RedrawWindow function of the CWnd object displaying the
  * specified drawing paper.
  * @param drawingPaper Specifies the drawing paper.
  */
  void repaintDrawingPaper(FieldDrawingPaper drawingPaper);
    
  /**
  * Calls the RedrawWindow function of all CWnd objects 
  * displaying a drawing paper containing
  * the specified drawing.
  * @param drawing Specifies a drawing.
  */
  void repaintDrawingPapers(Drawings::FieldDrawing drawing);
  
  /** Calculates how the image has to be scaled to fit in the rect*/
  double calculateScale(CRect* dcRect, double rotationAngle);

  void sendDebugKeysForSimulatedRobots();


   /** Pointers to the incomplete incoming drawings*/
  DebugDrawing* incompleteDrawing[Drawings::numberOfFieldDrawings + Drawings::numberOfImageDrawings];

 
 /** Pointers to the debug drawings*/
  DebugDrawing* fieldDrawing[Drawings::numberOfFieldDrawings];
  
  /** Stores if a special drawing is painted on a special paper. */
  bool showFieldDrawing[numberOfFieldDrawingPapers][Drawings::numberOfFieldDrawings];

  /** Stores if a special drawing is painted on at least one paper. */
  bool fieldDrawingIsShownOnAtLeastOnePaper[Drawings::numberOfFieldDrawings];
  
  /** Rotation angle in degrees of each drawing paper if auto rotate displays
  * that paper vertical*/
  int verticalRotationAngle[numberOfFieldDrawingPapers];
  
  /** Rotation angle in degrees of each drawing paper if auto rotate displays
  * that paper horizontal. Rotation angle of the drawing paper, if it has not the auto rotate property*/
  int horizontalRotationAngle[numberOfFieldDrawingPapers];
  
  /** Specifies if a special drawing paper has the auto rotate property*/
  bool autoRotate[numberOfFieldDrawingPapers];

  /** The meta file handles of the field drawing papers */
  HENHMETAFILE metaFileHandleFieldDrawingPaper[numberOfFieldDrawingPapers];
  
  /** Pointer to the CWnds of where the drawing papers are shown. */
  CWnd* pWndField[numberOfFieldDrawingPapers];

  double zoom[numberOfFieldDrawingPapers];

public:  
  ImageDrawingManager imageDrawingManager;
};

DebugDrawingManager& getDebugDrawingManager();

#endif //DebugDrawingManager_h

/*
* Change log :
* 
* $Log: DebugDrawingManager.h,v $
* Revision 1.2  2004/06/20 14:27:24  kindler
* - Removed arkward scale-parameter in paint functions, all drawings are now done in world coordinates.
* - It's easy now to calculate the inverse transformation for interactive DebugDrawings (like the CeilingCamView)
*
* Revision 1.1.1.1  2004/05/22 17:30:19  cvsadm
* created new repository GT2004_WM
*
* Revision 1.6  2004/04/07 13:00:46  risler
* ddd checkin after go04 - second part
*
* Revision 1.2  2004/04/06 13:19:37  risler
* cleaned up and improved high resolution image support
*
* Revision 1.1.1.1  2004/03/29 08:28:43  Administrator
* initial transfer from tamara
*
* Revision 1.5  2003/12/09 18:14:06  loetzsch
* Moved prepainted instances for fieldlines and field polygons from
* DebugDrawingManagerMessageHAndler to DebugDrawingManager
*
* Revision 1.4  2003/12/07 19:03:29  loetzsch
* The message handling for the debug drawing manager is now done in a
* separate class
*
* Revision 1.3  2003/12/06 06:31:20  loetzsch
* no message
*
* Revision 1.2  2003/11/03 20:16:09  juengel
* color class images can be sent from robot now
*
* Revision 1.1  2003/10/07 10:11:08  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:28:24  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:27  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.10  2003/03/25 10:56:40  juengel
* DebugDrawings have to be requested in configuration "WIN_32" now.
*
* Revision 1.9  2002/12/18 16:22:56  dueffert
* doxygen docu corrected
*
* Revision 1.8  2002/11/12 10:49:02  juengel
* New debug drawing macros - second step
* -moved /Tools/Debugging/PaintMethods.h and . cpp
*  to /Visualization/DrawingMethods.h and .cpp
* -moved DebugDrawing.h and .cpp from /Tools/Debugging/
*  to /Visualization
*
* Revision 1.7  2002/11/11 11:27:11  juengel
* First Step: New debug drawing macros.
*
* Revision 1.6  2002/10/13 15:51:05  juengel
* second debug drawing clean up
*
* Revision 1.5  2002/10/10 11:41:03  juengel
* First clean up in DebugDrawingManager
*
* Revision 1.4  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.3  2002/09/20 23:28:34  juengel
* Moved instance of DebugDrawingManager to RobotControlMainFrame.
*
* Revision 1.2  2002/09/19 23:38:54  juengel
* Changed debug image mechanisms.
*
*/
