/** 
 * @file DebugDrawingManager.cpp
 * Implementation of class DebugDrawingManager.
 *
 * @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
 */

#include "DebugDrawingManager.h"
#include "Visualization/ImageMethods.h"
#include "Visualization/PaintMethodsWin32.h"
#include "Visualization/DrawingMethods.h"
#include "RobotControl.h"

#pragma warning(disable:4355) // VC++: "this" in initializer list is ok

DebugDrawingManager::DebugDrawingManager()
: messageHandler(*this),
fieldPolygonsDrawingBlue(Drawings::fieldPolygons),
fieldPolygonsDrawingRed(Drawings::fieldPolygons),
fieldLinesDrawing(Drawings::fieldLines)
{
  int i;
  for(i = 0; i < Drawings::numberOfFieldDrawings; i++)
  {
    this->fieldDrawing[i] = NULL;
  }

  for(i = 0; i < Drawings::numberOfFieldDrawings + Drawings::numberOfImageDrawings; i++)
  {
    this->incompleteDrawing[i] = new DebugDrawing();
  }

  for(i = 0; i < numberOfFieldDrawingPapers; i++)
  {
    FieldDrawingPaper currentDrawingPaper = (FieldDrawingPaper)i;
    this->pWndField[currentDrawingPaper] = NULL;
    verticalRotationAngle[currentDrawingPaper] = 90;
    horizontalRotationAngle[currentDrawingPaper] = 0;
    autoRotate[currentDrawingPaper] = false;
    zoom[currentDrawingPaper] = 7;
  }
  zoom[DrawingPapers::mainWindow] = 0.86;

  showFieldDrawing[DrawingPapers::mainWindow][Drawings::fieldPolygons] = true;
  showFieldDrawing[DrawingPapers::mainWindow][Drawings::fieldLines] = true;
  autoRotate[DrawingPapers::mainWindow] = true;

  DrawingMethods::paintFieldPolygons(fieldPolygonsDrawingRed, Player::red);
  DrawingMethods::paintFieldPolygons(fieldPolygonsDrawingBlue, Player::blue);
  DrawingMethods::paintFieldLines(fieldLinesDrawing);

  setDebugDrawing(&fieldLinesDrawing);
  setDebugDrawing(&fieldPolygonsDrawingBlue);
}

DebugDrawingManager::~DebugDrawingManager()
{
  int i;
  for(i = 0; i < Drawings::numberOfFieldDrawings; i++)
  {
    if(fieldDrawing[i] != NULL)
    {
      delete fieldDrawing[i];
    }
  }
   for(i = 0; i < Drawings::numberOfFieldDrawings + Drawings::numberOfImageDrawings; i++)
  {
    delete incompleteDrawing[i];
  }

}

void DebugDrawingManager::drawingFinished(Drawings::FieldDrawing id)
{
  incompleteDrawing[id]->fieldDrawingID = id;
  incompleteDrawing[id]->typeOfDrawing = Drawings::drawingOnField;
  setDebugDrawing(incompleteDrawing[id]);
  delete incompleteDrawing[id];
  incompleteDrawing[id] = new DebugDrawing();
}

void DebugDrawingManager::drawingFinished(Drawings::ImageDrawing id)
{
  incompleteDrawing[id + Drawings::numberOfFieldDrawings]->imageDrawingID = id;
  incompleteDrawing[id + Drawings::numberOfFieldDrawings]->typeOfDrawing = Drawings::drawingOnImage;
  setDebugDrawing(incompleteDrawing[id + Drawings::numberOfFieldDrawings]);
  delete incompleteDrawing[id + Drawings::numberOfFieldDrawings];
  incompleteDrawing[id + Drawings::numberOfFieldDrawings] = new DebugDrawing();
}


void DebugDrawingManager::setDebugDrawing(DebugDrawing* pDebugDrawing)
{
  if(pDebugDrawing->typeOfDrawing == Drawings::drawingOnField)
  {
    if(fieldDrawing[pDebugDrawing->fieldDrawingID] != NULL)
    {
      delete fieldDrawing[pDebugDrawing->fieldDrawingID];
    }
    fieldDrawing[pDebugDrawing->fieldDrawingID] = new DebugDrawing(pDebugDrawing);
    repaintDrawingPapers(pDebugDrawing->fieldDrawingID);
  }
  
  if(pDebugDrawing->typeOfDrawing == Drawings::drawingOnImage)
  {
    imageDrawingManager.setDebugDrawing(pDebugDrawing);
  }
}


void DebugDrawingManager::showDrawingOnDrawingPaper
(
 FieldDrawingPaper drawingPaper,
 Drawings::FieldDrawing drawing,
 bool show
 )
{
  this->showFieldDrawing[drawingPaper][drawing] = show;

  fieldDrawingIsShownOnAtLeastOnePaper[drawing] = false;
  for(int i = 0; i < numberOfFieldDrawingPapers; i++)
  {
    if(showFieldDrawing[i][drawing]) fieldDrawingIsShownOnAtLeastOnePaper[drawing] = true;
  }
  sendDebugKeysForSimulatedRobots();
  
  repaintDrawingPaper(drawingPaper);
}

void DebugDrawingManager::sendDebugKeysForSimulatedRobots()
{
  for(int i = 0; i < Drawings::numberOfFieldDrawings; i++)
  {
    if(fieldDrawingIsShownOnAtLeastOnePaper[i])
    {
      DebugKeyTable::debugKeyID debugKeyID;
      if(Drawings::getDebugKeyID((Drawings::FieldDrawing)i, debugKeyID))
        getDebugKeyTables().forSimulatedRobots.set(debugKeyID, DebugKey::always);
    }
    else
    {
      DebugKeyTable::debugKeyID debugKeyID;
      if(Drawings::getDebugKeyID((Drawings::FieldDrawing)i, debugKeyID))
        getDebugKeyTables().forSimulatedRobots.set(debugKeyID, DebugKey::disabled);
    }
  }
  getDebugKeyTables().sendForSimulatedRobots();
}


void DebugDrawingManager::showDrawingOnDrawingPaper
(
 ImageDrawingPaper drawingPaper,
 Drawings::ImageDrawing drawing,
 bool show
 )
{
  imageDrawingManager.showDrawingOnDrawingPaper(drawingPaper, drawing, show);
}

bool DebugDrawingManager::isDrawingShownOnDrawingPaper
(
 FieldDrawingPaper drawingPaper,
 Drawings::FieldDrawing drawing
 )
{
  return showFieldDrawing[drawingPaper][drawing];
}

bool DebugDrawingManager::isDrawingShownOnDrawingPaper
(
 ImageDrawingPaper drawingPaper,
 Drawings::ImageDrawing drawing
 )
{
  return imageDrawingManager.isDrawingShownOnDrawingPaper(drawingPaper, drawing);
}

void DebugDrawingManager::setCallBackMethod
(
 FieldDrawingPaper drawingPaper, 
 CWnd* pWnd
 )
{
  this->pWndField[drawingPaper] = pWnd;
}

void DebugDrawingManager::setCallBackMethod
(
 ImageDrawingPaper drawingPaper, 
 CWnd* pWnd
 )
{
  imageDrawingManager.setWnd(drawingPaper, pWnd);
}

void DebugDrawingManager::repaintDrawingPaper
(
 FieldDrawingPaper drawingPaper
 )
{
  if(pWndField[drawingPaper] != NULL)
  {
    pWndField[drawingPaper]->RedrawWindow(NULL,NULL,RDW_INVALIDATE);
  }
}

void DebugDrawingManager::repaintDrawingPapers
(
 Drawings::FieldDrawing drawing
 )
{
  for(int i = 0; i < numberOfFieldDrawingPapers; i++)
  {
    FieldDrawingPaper currentDrawingPaper = (FieldDrawingPaper)i;
    if(showFieldDrawing[currentDrawingPaper][drawing])
    {
      repaintDrawingPaper(currentDrawingPaper);
    }
  }
}

double DebugDrawingManager::calculateScale(
  CRect *dcRect, double rotationAngle
)
{
  // Calculate aspect ratio of the drawing
  //
  double paperWidth  = xPosOpponentGoal - xPosOwnGoal;
  double paperHeight = yPosLeftFlags - yPosRightFlags;
  double paperRatio  = paperWidth / paperHeight; 
  
  // Calculate bounding rectangle of turned drawing
  //
  double diagonalLength = 
    sqrt(paperWidth*paperWidth + paperHeight*paperHeight);
  
  double alpha = atan(paperHeight / paperWidth);

  int x1 = (int)(diagonalLength/2.0 * cos( alpha + rotationAngle));
  int y1 = (int)(diagonalLength/2.0 * sin( alpha + rotationAngle));

  int x2 = (int)(diagonalLength/2.0 * cos(-alpha + rotationAngle));
  int y2 = (int)(diagonalLength/2.0 * sin(-alpha + rotationAngle));
  
  // Calculate aspect ratio of the turned drawing
  //
  double turnedWidth  = 2 * max(abs(x1), abs(x2));
  double turnedHeight = 2 * max(abs(y1), abs(y2));
  double turnedRatio  = turnedWidth / turnedHeight;
  
  // Calculate aspect ratio of the dc
  //
  double dcWidth  = dcRect->Width();
  double dcHeight = dcRect->Height();
  double dcRatio  = dcWidth / dcHeight; 
  
  if (dcRatio < turnedRatio) {
    // Fit horizontal
    //
    return dcWidth / turnedWidth;
  } 
  else {
    // Fit vertical
    //
    return dcHeight / turnedHeight;
  }
}


XFORM DebugDrawingManager::calcPaintXForm( 
  CRect *rect, FieldDrawingPaper drawingPaper
)
{
  double angle;
  double paperWidth  = xPosOpponentGoal - xPosOwnGoal;
  double paperHeight = yPosLeftFlags - yPosRightFlags;

  if (((rect->Width() < rect->Height() && paperWidth > paperHeight) ||
       (rect->Width() > rect->Height() && paperWidth < paperHeight))
        && autoRotate[drawingPaper])
    angle = fromDegrees(verticalRotationAngle[drawingPaper]);
  else
    angle = fromDegrees(horizontalRotationAngle[drawingPaper]);

  double scale = calculateScale(rect, angle) * zoom[drawingPaper];
  
  XFORM xform;
  xform.eM11 = (float)(cos(angle)  * scale);
  xform.eM12 = (float)(sin(angle)  * scale);
  xform.eM21 = (float)(sin(angle)  * scale);
  xform.eM22 = (float)(-cos(angle) * scale);
  xform.eDx  = (float)(rect->Width()  / 2.0);
  xform.eDy  = (float)(rect->Height() / 2.0);

  return xform;
}


void DebugDrawingManager::paint(
  CDC *pDC, CRect *rect,
  FieldDrawingPaper drawingPaper
)
{
  // Offscreen device context to prevent flickering
  // 
  CDC      dcOffScreen; 
  CBitmap  bmpOffScreen;
  CBitmap *oldBitmap;

  int currentWidth  = rect->Width();
  int currentHeight = rect->Height();

  dcOffScreen.CreateCompatibleDC(pDC);
  bmpOffScreen.CreateCompatibleBitmap(pDC, currentWidth, currentHeight);
  
  oldBitmap = dcOffScreen.SelectObject(&bmpOffScreen);

  // Generate white background
  //
  CPen  noPen;
  CPen *oldPen;
  
  noPen.CreatePen(PS_NULL, 0, RGB(0,0,0));
  oldPen = dcOffScreen.SelectObject(&noPen);
    
  dcOffScreen.Rectangle(0, 0, currentWidth+1, currentHeight+1);
  dcOffScreen.SelectObject(oldPen);

  SetGraphicsMode(dcOffScreen.m_hDC, GM_ADVANCED);

  // Set up world coordinate transformation
  //
  XFORM xform = calcPaintXForm(rect, drawingPaper);
  SetWorldTransform(dcOffScreen.m_hDC, &xform); 

  for (int i=0; i<Drawings::numberOfFieldDrawings; i++)
  {  
    Drawings::FieldDrawing currentDrawing = (Drawings::FieldDrawing)i;
    
    if (fieldDrawing[currentDrawing] == NULL)
      continue;

    if (showFieldDrawing[drawingPaper][currentDrawing])
      PaintMethodsWin32::paintDebugDrawingToCDC(fieldDrawing[currentDrawing], dcOffScreen);
  }

  ModifyWorldTransform(dcOffScreen, NULL, MWT_IDENTITY);

  // Copy the off-screen DC to the paint DC
  //
  pDC->BitBlt(
    rect->left,    rect->top, 
    rect->Width(), rect->Height(),
    &dcOffScreen, 0, 0, SRCCOPY
  );

  dcOffScreen.SelectObject(oldBitmap);
  dcOffScreen.DeleteDC();
}

void DebugDrawingManager::paint(
  CRect *dcRect, ImageDrawingPaper drawingPaper
)
{
  imageDrawingManager.paint(dcRect, drawingPaper);
}


void DebugDrawingManager::copyDrawingPaperToClipBoard
(
 FieldDrawingPaper drawingPaper
 )
{
  HENHMETAFILE metaFileHandle;
  CMetaFileDC* metaFileDC;
  metaFileDC = new CMetaFileDC();

  metaFileDC->CreateEnhanced(NULL, NULL, NULL, NULL);

  for(int j = 0; j < Drawings::numberOfFieldDrawings; j++)
  {  
    Drawings::FieldDrawing currentDrawing = (Drawings::FieldDrawing)j;
    if(
      showFieldDrawing[drawingPaper][currentDrawing] &&
      this->fieldDrawing[currentDrawing] != NULL
      )
    {
      PaintMethodsWin32::paintDebugDrawingToCDC(fieldDrawing[currentDrawing], *metaFileDC);
    } // if(showDrawing[drawingPaper][currentDrawing])
  } // for(int j = 0; j < Drawings::numberOfDrawings; j++)

  metaFileHandle = metaFileDC->CloseEnhanced();
  
  OpenClipboard(NULL);
	EmptyClipboard();
	::SetClipboardData(CF_ENHMETAFILE,metaFileHandle);
	CloseClipboard();

  DeleteEnhMetaFile(metaFileHandle);
  metaFileDC->DeleteDC();
  delete metaFileDC;
}

void DebugDrawingManager::copyDrawingPaperToClipBoard
(
 ImageDrawingPaper drawingPaper
 )
{
  imageDrawingManager.copyDrawingPaperToClipBoard(drawingPaper);
}

void DebugDrawingManager::copyImageToClipBoard
(
 ImageDrawingPaper drawingPaper
 )
{
  imageDrawingManager.copyImageToClipBoard(drawingPaper);
}

void DebugDrawingManager::loadProperties()
{
  int i,j;
  for(j = 0; j < Drawings::numberOfFieldDrawings; j++)
  {
    fieldDrawingIsShownOnAtLeastOnePaper[j] = false;
  }
  for(i = 0; i < numberOfFieldDrawingPapers; i++)
  {
    for(j = 0; j < Drawings::numberOfFieldDrawings; j++)
    {
      showFieldDrawing[i][j] = false;
    }
    CString paperString;
    paperString = DrawingPapers::getDrawingPaperName(FieldDrawingPaper(i)); 
    

    CString drawingsString = AfxGetApp()->GetProfileString("DebugDrawingManager\\FieldDrawingPaper\\Drawings", paperString);
    int currentReturnPosition = 0;
    int currentLength = drawingsString.GetLength();
    if (currentLength == 0)
    {
      showFieldDrawing[(DrawingPapers::FieldDrawingPaper)i][Drawings::fieldLines] = true;
    }
    while(currentLength != 0)
    {
      currentReturnPosition = drawingsString.Find(",");
      CString extractedString = drawingsString.Left(currentReturnPosition);

      for(int j=0; j < Drawings::numberOfFieldDrawings; j++)
      {
        if (Drawings::getDrawingName(Drawings::FieldDrawing(j)) == extractedString)
        {
          showFieldDrawing[i][j] = true;
          fieldDrawingIsShownOnAtLeastOnePaper[j] = true;
        }
      }

      drawingsString = drawingsString.Right(currentLength - currentReturnPosition - 1);
      currentLength = drawingsString .GetLength();
    }
    int horizontalRotationAngleInt =
      AfxGetApp()->GetProfileInt("DebugDrawingManager\\FieldDrawingPaper\\horizontalRotation", paperString, 0);
    
    int verticalRotationAngleInt =
      AfxGetApp()->GetProfileInt("DebugDrawingManager\\FieldDrawingPaper\\verticalRotation", paperString, 90);
    
    horizontalRotationAngle[FieldDrawingPaper(i)] = horizontalRotationAngleInt;

    verticalRotationAngle[FieldDrawingPaper(i)] = verticalRotationAngleInt;

    int defaultInt = 0;
    if(FieldDrawingPaper(i) == DrawingPapers::mainWindow)
      defaultInt = 1;

    if(AfxGetApp()->GetProfileInt("DebugDrawingManager\\FieldDrawingPaper\\autoRotate", paperString, defaultInt) == 1)
    {
      this->autoRotate[FieldDrawingPaper(i)] = true;
    }
    else
    {
      this->autoRotate[FieldDrawingPaper(i)] = false;
    }
  }
  sendDebugKeysForSimulatedRobots();
  imageDrawingManager.loadProperties();
}

void DebugDrawingManager::saveProperties()
{
  int i,j;
  for(i = 0; i < numberOfFieldDrawingPapers; i++)
  {
    CString paperString;
    paperString = DrawingPapers::getDrawingPaperName(FieldDrawingPaper(i)); 

    CString drawingsString = "";
    for(j = 0; j < Drawings::numberOfFieldDrawings; j++)
    {
      if(showFieldDrawing[i][j])
      {
        drawingsString.Insert(9999, Drawings::getDrawingName(Drawings::FieldDrawing(j)));
        drawingsString.Insert(9999, ",");
      }
    }
    AfxGetApp()->WriteProfileString("DebugDrawingManager\\FieldDrawingPaper\\Drawings", paperString, drawingsString);
    
    int horizontalRotationAngleInt
      = horizontalRotationAngle[FieldDrawingPaper(i)];
    
    AfxGetApp()->WriteProfileInt(
      "DebugDrawingManager\\FieldDrawingPaper\\horizontalRotation", paperString, 
      horizontalRotationAngleInt
      );

    int verticalRotationAngleInt
      = verticalRotationAngle[FieldDrawingPaper(i)];
    
    AfxGetApp()->WriteProfileInt(
      "DebugDrawingManager\\FieldDrawingPaper\\verticalRotation", paperString, 
      verticalRotationAngleInt
      );

    AfxGetApp()->WriteProfileInt(
      "DebugDrawingManager\\FieldDrawingPaper\\autoRotate", paperString,
      (int)autoRotate[FieldDrawingPaper(i)]
      );
  }

  imageDrawingManager.saveProperties();
}

void DebugDrawingManager::setImageID
(
 ImageDrawingPaper drawingPaper,
 ImageID imageID
 )
{
  imageDrawingManager.setImageID(drawingPaper, imageID);
}

ImageID DebugDrawingManager::getImageID
(
 ImageDrawingPaper drawingPaper
 )
{
  return imageDrawingManager.getImageID(drawingPaper);
}

void DebugDrawingManager::enableAutoRotate
(
 FieldDrawingPaper drawingPaper,
 bool enable
 )
{
  autoRotate[drawingPaper] = enable;
  horizontalRotationAngle[drawingPaper] = 0;
  verticalRotationAngle[drawingPaper] = 90;
  pWndField[drawingPaper]->RedrawWindow(NULL,NULL,RDW_INVALIDATE);
}

bool DebugDrawingManager::isAutoRotateEnabled
(
 FieldDrawingPaper drawingPaper
 )
{
  return autoRotate[drawingPaper];
}

void DebugDrawingManager::rotate
(
 FieldDrawingPaper drawingPaper
 )
{
  if(autoRotate[drawingPaper])
  {
    verticalRotationAngle[drawingPaper] += 180;
    horizontalRotationAngle[drawingPaper] += 180;
  }
  else
  {
    horizontalRotationAngle[drawingPaper] += 90;
  }
  pWndField[drawingPaper]->RedrawWindow(NULL,NULL,RDW_INVALIDATE);
}

void DebugDrawingManager::notifyImage
(
 ImageID imageID
 )
{
  imageDrawingManager.notifyImage(imageID);
}

void DebugDrawingManager::zoomIn(FieldDrawingPaper fieldDrawingPaper)
{
  zoom[fieldDrawingPaper] -= 3;
  if(zoom[fieldDrawingPaper] <= 0) zoom[fieldDrawingPaper] +=3;
}

void DebugDrawingManager::zoomOut(FieldDrawingPaper fieldDrawingPaper)
{
  zoom[fieldDrawingPaper] += 3;
}

void DebugDrawingManager::zoomDrawingPaper(FieldDrawingPaper fieldDrawingPaper, double factor)
{
  zoom[fieldDrawingPaper] *= factor;
}

void DebugDrawingManager::setSendDebugImagesFromRobot(bool sendDebugImagesFromRobot)
{
  imageDrawingManager.setSendDebugImagesFromRobot(sendDebugImagesFromRobot);
}

bool DebugDrawingManager::getSendDebugImagesFromRobot()
{
  return imageDrawingManager.getSendDebugImagesFromRobot();
}

DebugDrawingManager& getDebugDrawingManager()
{
  return getRobotControlApp().pMainFrame->debugDrawingManager;
}

bool DebugDrawingManager::getHighRes(ImageDrawingPaper drawingPaper)
{
  return imageDrawingManager.getHighRes(drawingPaper);
}

void DebugDrawingManager::setHighRes(ImageDrawingPaper drawingPaper, bool highRes)
{
  imageDrawingManager.setHighRes(drawingPaper, highRes);
}

/*
 * Change log :
 * 
 * $Log: DebugDrawingManager.cpp,v $
 * Revision 1.2  2004/06/20 14:27:24  kindler
 * - Removed arkward scale-parameter in paint functions, all drawings are now done in world coordinates.
 * - It's easy now to calculate the inverse transformation for interactive DebugDrawings (like the CeilingCamView)
 *
 * Revision 1.1.1.1  2004/05/22 17:30:19  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.9  2004/04/07 13:00:46  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.2  2004/04/06 13:19:36  risler
 * cleaned up and improved high resolution image support
 *
 * Revision 1.1.1.1  2004/03/29 08:28:43  Administrator
 * initial transfer from tamara
 *
 * Revision 1.8  2004/02/07 15:45:48  juengel
 * Removed RadarViewer stuff.
 *
 * Revision 1.7  2003/12/09 18:14:06  loetzsch
 * Moved prepainted instances for fieldlines and field polygons from
 * DebugDrawingManagerMessageHAndler to DebugDrawingManager
 *
 * Revision 1.6  2003/12/07 19:03:29  loetzsch
 * The message handling for the debug drawing manager is now done in a
 * separate class
 *
 * Revision 1.5  2003/12/06 16:44:45  loetzsch
 * improved message handling
 *
 * Revision 1.4  2003/12/06 06:31:20  loetzsch
 * no message
 *
 * Revision 1.3  2003/11/30 01:53:21  loetzsch
 * prepared RobotControl port to Visual C++ .Net
 *
 * Revision 1.2  2003/11/03 20:16:09  juengel
 * color class images can be sent from robot now
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/01 10:23:13  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.1.1.1  2003/07/02 09:40:27  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.13  2003/03/25 10:56:40  juengel
 * DebugDrawings have to be requested in configuration "WIN_32" now.
 *
 * Revision 1.12  2003/02/08 18:38:55  juengel
 * iproved copyDrawingPaperToClipBoard()
 *
 * Revision 1.11  2002/11/13 10:43:06  juengel
 * clipboard functions restored.
 *
 * Revision 1.10  2002/11/12 16:08:11  loetzsch
 * Added (char) and (int) casts to DebugDrawing Macros
 *
 * Revision 1.9  2002/11/12 10:49:02  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.8  2002/11/11 11:27:11  juengel
 * First Step: New debug drawing macros.
 *
 * Revision 1.7  2002/10/13 15:51:05  juengel
 * second debug drawing clean up
 *
 * Revision 1.6  2002/10/10 12:56:03  juengel
 * First clean up in DebugDrawingManager
 *
 * Revision 1.5  2002/10/10 11:45:00  juengel
 * First clean up in DebugDrawingManager
 *
 * Revision 1.4  2002/09/22 18:40:52  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.3  2002/09/20 23:28:34  juengel
 * Moved instance of DebugDrawingManager to RobotControlMainFrame.
 *
 * Revision 1.2  2002/09/19 23:38:53  juengel
 * Changed debug image mechanisms.
 *
 */
