/** 
* @file RobotControl/Visualization/DebugDrawing.h
* Declaration of class DebugDrawing.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#ifndef __DebugDrawing_h_
#define __DebugDrawing_h_

#include "Tools/Math/Pose2D.h"
#include "Tools/Debugging/DebugDrawings.h"

#include "Tools/MessageQueue/InMessage.h"
#include "Tools/List.h"

/**
* The DebugDrawing class defines a class of drawing objects for debug purposes. 
*
* The DebugDrawing object provides member functions for painting simple graphic structures like lines, ellipses, polygons and text.
* To use a DebugDrawing object, construct it, and then call its member functions.
*
* @author Matthias Jngel
*/
class DebugDrawing
{
public:
  /** Default constructor. */
  DebugDrawing() {}

  /** Constructs a DebugDrawing object. */
  DebugDrawing(Drawings::FieldDrawing fieldDrawing);
  
  /** Constructs a DebugDrawing object. */
  DebugDrawing(Drawings::ImageDrawing imageDrawing);
  
  /** Copy constructor for a DebugDrawing object. */
  DebugDrawing(const DebugDrawing& other);
  
  /** Copy constructor for a DebugDrawing object. */
  DebugDrawing(const DebugDrawing* pDebugDrawing);
  
  /** Destructor. */
  ~DebugDrawing() {reset();}

  /** The function empties the drawing. */
  void reset();
  
  /** Assignment operator.*/
  const DebugDrawing& operator=(const DebugDrawing& other);
  
  /** Adds the contents of another debug drawing to this one. */
  const DebugDrawing& operator+=(const DebugDrawing& other);

  /**
  * A struct representing a color.
  */
  struct Color
  {
    Color(){};
    Color(unsigned char red, unsigned char green, unsigned char blue)
    {
      this->red = red;
      this->green = green;
      this->blue = blue;
    }
    Color(unsigned long rgbColor)
    {
      this->red = (unsigned char)(rgbColor % 0x100);
      rgbColor /= 0x100;
      this->green = (unsigned char)(rgbColor % 0x100);
      rgbColor /= 0x100;
      this->blue = (unsigned char)(rgbColor % 0x100);
    }
    Color(Drawings::Color color)
    {
      switch(color)
      {
      case Drawings::red:
        red = 255; green = 0; blue = 0;
        break;
      case Drawings::green:
        red = 0; green = 255; blue = 0;
        break;
      case Drawings::blue:
        red = 0; green = 0; blue = 255;
        break;
      case Drawings::yellow:
        red = 255; green = 255; blue = 0;
        break;
      case Drawings::orange:
        red = 255; green = 128; blue = 64;
        break;
      case Drawings::pink:
        red = 255; green = 0; blue = 255;
        break;
      case Drawings::skyblue:
        red = 0; green = 255; blue = 255;
        break;


      case Drawings::white:
        red = 255; green = 255; blue = 255;
        break;
      case Drawings::light_gray:
        red = 192; green = 192; blue = 192;
        break;
      case Drawings::gray:
        red = 128; green = 128; blue = 128;
        break;
      case Drawings::dark_gray:
        red = 64; green = 64; blue = 64;
        break;
      case Drawings::black:
        red = 0; green = 0; blue = 0;
        break;
      case Drawings::yellowOrange:
        red = 255; green = 192; blue = 32;
        break;
      }
    }

    unsigned char red;
    unsigned char blue;
    unsigned char green;
  };
  
  /** the maximum number of points in a polygon */
  enum { MAX_NUMBER_OF_POINTS=16 };

  /** base class for all drawing elements */
  class Element
  {
  public:
    enum {LINE, POLYGON, ELLIPSE} type; 
    Drawings::PenStyle penStyle;
    Color penColor;
    int width;
  };

  /** Stores a polygon */
  class Polygon : public Element
  {
  public: 
    Vector2<int> points[MAX_NUMBER_OF_POINTS];
    int nCount;
    Drawings::FillStyle fillStyle;
    Color fillColor;
    Polygon() {type = POLYGON;}
  };
      
  /** Stores a line */
  class Line : public Element
  {
  public:
    int xStart, yStart, xEnd, yEnd;
    Line() {type = LINE;}
  };

  /** Stores an ellipse */
  class Ellipse : public Element
  {
  public:
    int top, bottom, left, right;
    Drawings::FillStyle fillStyle;
    Color fillColor;
    Ellipse() {type = ELLIPSE;}
  };

  /** Contains all elements of this debug drawing */
  List<Element*> elements;

  /**
  * Adds an arrow to the debug drawing
  * @param offset vector pointing to the tip of the arrow 
  * @param direction vector pointing in the direction of the arrow
  * @param color
  */
  void arrow( 
		Vector2<double> offset,
    Vector2<double> direction,
    Color color
    );

  /**
  * Adds an arrow to the debug drawing
  * @param xStart Specifies the x-coordinate of the startpoint for the line. 
  * @param yStart Specifies the y-coordinate of the startpoint for the line.
  * @param xEnd Specifies the x-coordinate of the endpoint for the line.
  * @param yEnd Specifies the y-coordinate of the endpoint for the line.
  * @param penStyle not used!
  * @param width not used!
  * @param color Specifies the color of the line.
  */
  void arrow(
    int xStart, 
    int yStart, 
    int xEnd,
    int yEnd,
    Drawings::PenStyle penStyle,
    int width,
    Color color
    );
  
  
  /**
  * Adds a line to the debug drawing.
  * @param xStart Specifies the x-coordinate of the startpoint for the line. 
  * @param yStart Specifies the y-coordinate of the startpoint for the line.
  * @param xEnd Specifies the x-coordinate of the endpoint for the line.
  * @param yEnd Specifies the y-coordinate of the endpoint for the line.
  * @param penStyle Specifies the penStyle of the Line.
  * @param width Specifies the width of the line.
  * @param color Specifies the color of the line.
  */
  void line(
    int xStart, 
    int yStart, 
    int xEnd,
    int yEnd,
    Drawings::PenStyle penStyle,
    int width,
    Color color
    );

  /**
  * Adds a line to the debug drawing. The line is a solid black line with width 1.
  * @param xStart Specifies the x-coordinate of the startpoint for the line. 
  * @param yStart Specifies the y-coordinate of the startpoint for the line.
  * @param xEnd Specifies the x-coordinate of the endpoint for the line.
  * @param yEnd Specifies the y-coordinate of the endpoint for the line.
	*/
  void line(int xStart, int yStart, int xEnd, int yEnd);
  
  /**
  * Adds a polygon to the debug drawing.
  * @param points Points to an array of points that specifies the vertices of the polygon. Each point in the array is a Point.
  * @param nCount Specifies the number of vertices in the array.
  * @param width Specifies the width of the border.
  * @param penStyle Specifies the penStyle of the border.
  * @param penColor Specifies the color of the border.
  * @param fillStyle Specifies the fillStyle of the polygon.
  * @param fillColor Specifies the color of the polygon.
  */
  void polygon(
    const Vector2<int>* points,
    int nCount,
    int width,
    Drawings::PenStyle penStyle,
    Color penColor,
    Drawings::FillStyle fillStyle,
    Color fillColor
    );
  
  /**
   * Adds a filled rectangle to the debug drawing.
   * The rectangle extends up to, but does not include, the right and bottom coordinates.
   * @param left Specifies the x-coordinate of the upper-left corner of the rectangle.
   * @param right Specifies the x-coordinate of the lower-right corner of the rectangle.
   * @param top Specifies the y-coordinate of the upper-left corner of the rectangle.
   * @param bottom Specifies the y-coordinate of the lower-right corner of the rectangle.
   * @param width Specifies the width of the border.
   * @param penStyle Specifies the penStyle of the border.
   * @param penColor Specifies the color of the border.
   * @param fillStyle Specifies the fillStyle of the ellipse.
   * @param fillColor Specifies the color of the ellipse.
   */  
  void rectangle(
    int left, int right, int top, int bottom, 
    int width,
    Drawings::PenStyle penStyle,
    Color penColor,
    Drawings::FillStyle fillStyle,
    Color fillColor
    );
  
  /**
   * Adds a filled rectangle to the debug drawing. The border of the rectangle is a solid black line with width 1.
   * The rectangle extends up to, but does not include, the right and bottom coordinates.
   * @param left Specifies the x-coordinate of the upper-left corner of the rectangle.
   * @param right Specifies the x-coordinate of the lower-right corner of the rectangle.
   * @param top Specifies the y-coordinate of the upper-left corner of the rectangle.
   * @param bottom Specifies the y-coordinate of the lower-right corner of the rectangle.
   * @param color Specifies the color of the rectangle.
   */  
  void rectangle(int left, int right, int top, int bottom, Color color)
    {rectangle(left,right,top,bottom,1,Drawings::ps_solid,Color(0,0,0),Drawings::bs_solid,color);}
  
    /**
    * Adds a filled square to the debug drawing. The border of the square is a solid line with width 0.
    * The square is a 3x3 square.
    * @param x Specifies the center of the dot.
    * @param y Specifies the center of the dot.
    * @param penColor Specifies the penColor of the dot.
    * @param fillColor Specifies the fillColor of the dot.
  */  
  void dot(
    int x, int y, Color penColor, Color fillColor
    );

    /**
    * Adds a filled square to the debug drawing. The border of the square is a solid line with width 5.
    * The square is a 10x10 square.
    * @param x Specifies the center of the dot.
    * @param y Specifies the center of the dot.
    * @param penColor Specifies the penColor of the dot.
    * @param fillColor Specifies the fillColor of the dot.
  */  
  void largeDot(
    int x, int y, Color penColor, Color fillColor
    );
  
    /**
    * Adds a filled ellipse to the debug drawing. The border of the ellipse is a solid black line with width 1.
    * The ellipse extends up to, but does not include, the right and bottom coordinates.
    * The figure drawn by this function extends up to, but does not include, the right and bottom coordinates.
    * @param left Specifies the x-coordinate of the upper-left corner of the ellipses bounding rectangle.
    * @param right Specifies the x-coordinate of the lower-right corner of the ellipses bounding rectangle.
    * @param top Specifies the y-coordinate of the upper-left corner of the ellipses bounding rectangle.
    * @param bottom Specifies the y-coordinate of the lower-right corner of the ellipses bounding rectangle.
    * @param color Specifies the color of the ellipse.
  */
  void ellipse(
    int left, int right, int top, int bottom, Color color
    );
  
    /**
    * Adds a filled ellipse to the debug drawing.
    * The ellipse extends up to, but does not include, the right and bottom coordinates.
    * The figure drawn by this function extends up to, but does not include, the right and bottom coordinates.
    * @param left Specifies the x-coordinate of the upper-left corner of the ellipses bounding rectangle.
    * @param right Specifies the x-coordinate of the lower-right corner of the ellipses bounding rectangle.
    * @param top Specifies the y-coordinate of the upper-left corner of the ellipses bounding rectangle.
    * @param bottom Specifies the y-coordinate of the lower-right corner of the ellipses bounding rectangle.
    * @param width Specifies the width of the border.
    * @param penStyle Specifies the penStyle of the border.
    * @param penColor Specifies the color of the border.
    * @param fillStyle Specifies the fillStyle of the ellipse.
    * @param fillColor Specifies the color of the ellipse.
  */  
  void ellipse(
    int left, int right, int top, int bottom, 
    int width,
    Drawings::PenStyle penStyle,
    Color penColor,
    Drawings::FillStyle fillStyle,
    Color fillColor
    );
  
    /**
    * Adds a filled circle to the debug drawing.
    * The figure drawn by this function extends up to, but does not include, the right and bottom coordinates.
    * @param xCenter Specifies the x-coordinate of the center of the circle.
    * @param yCenter Specifies the y-coordinate of the center of the circle.
    * @param radius Specifies the radius of the circle.
    * @param penWidth Specifies the width of the pen to draw with.
    * @param penStyle Specifies the penStyle of the border.
    * @param penColor Specifies the color of the border.
    * @param fillStyle Specifies the fillStyle of the ellipse.
    * @param fillColor Specifies the color of the ellipse.
  */  
  void circle(
    int xCenter, int yCenter, int radius, 
    int penWidth,
    Drawings::PenStyle penStyle,
    Color penColor,
    Drawings::FillStyle fillStyle,
    Color fillColor
    );
  
  
  /** draws a Pose2D as an Arrow on the field */
  void pose2DSample(Pose2D pose);
  
  /** draws a Pose2D as an Arrow on the field */
  void pose2DSample(Pose2D pose, Color color);

  bool addShapeFromQueue(InMessage& message, Drawings::ShapeType shapeType);

  //private:
  /** the kind of the drawing */
  Drawings::TypeOfDrawing typeOfDrawing;
  Drawings::FieldDrawing fieldDrawingID;
  Drawings::ImageDrawing imageDrawingID;
};

/**
* Streaming operator that reads a DebugDrawing from a stream.
* @param stream The stream from which is read.
* @param debugDrawing The DebugDrawing object.
* @return The stream.
*/ 
In& operator>>(In& stream, DebugDrawing& debugDrawing);

/**
* Streaming operator that writes a DebugDrawing to a stream.
* @param stream The stream to write on.
* @param debugDrawing The DebugDrawing object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const DebugDrawing& debugDrawing);


#endif //DebugDrawing_h

/*
* Change log :
* 
* $Log: DebugDrawing.h,v $
* Revision 1.2  2004/05/24 13:45:10  tim
* added potential field viewer again
*
* Revision 1.1.1.1  2004/05/22 17:30:01  cvsadm
* created new repository GT2004_WM
*
* Revision 1.4  2003/12/07 18:58:50  loetzsch
* added += operator
*
* Revision 1.3  2003/11/03 20:19:22  juengel
* Added color class yellowOrange
*
* Revision 1.2  2003/10/30 22:46:28  roefer
* DebugDrawings now preserve sequence of drawing instructions
*
* Revision 1.1  2003/10/07 10:11:08  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.9  2003/06/24 10:10:05  jhoffman
* added arrow so it can be sent
*
* Revision 1.8  2003/06/10 18:01:53  juengel
* Added LARGE_DOT.
*
* Revision 1.7  2003/03/29 13:44:01  roefer
* Parameter of polygon changed
*
* Revision 1.6  2003/03/23 19:12:10  loetzsch
* OUTPUT not allowed in the RobotControl thread anymore.
* Use getQueues().toGUI.out instead.
*
* Revision 1.5  2003/02/08 18:37:49  juengel
* changed rgb values for skyblue
*
* Revision 1.4  2002/12/18 16:22:56  dueffert
* doxygen docu corrected
*
* Revision 1.3  2002/11/25 12:22:24  dueffert
* doxygen comments added
*
* Revision 1.2  2002/11/12 16:08:11  loetzsch
* Added (char) and (int) casts to DebugDrawing Macros
*
* Revision 1.1  2002/11/12 10:52:52  juengel
* New debug drawing macros - second step
* -moved /Tools/Debugging/PaintMethods.h and . cpp
*  to /Visualization/DrawingMethods.h and .cpp
* -moved DebugDrawing.h and .cpp from /Tools/Debugging/
*  to /Visualization
*
* Revision 1.7  2002/11/11 11:27:10  juengel
* First Step: New debug drawing macros.
*
* Revision 1.6  2002/10/14 15:17:20  dueffert
* doxygen warnings removed
*
* Revision 1.5  2002/10/14 13:14:24  dueffert
* doxygen comments corrected
*
* Revision 1.4  2002/10/10 13:30:14  jhoffman
* added ISL related stuff
*
* Revision 1.3  2002/10/01 15:31:02  loetzsch
* fixed gcc errors
*
* Revision 1.2  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:53:58  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.12  2002/09/03 16:02:24  juengel
* Debug Drawing ball removed, renamed ball2 to ball.
*
* Revision 1.11  2002/07/23 13:48:28  loetzsch
* - new streaming classes
* - removed many #include statements
* - exchanged StaticQueue by MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.10  2002/07/08 14:04:46  roefer
* gcc compatibility improved
*
* Revision 1.9  2002/06/17 18:34:43  kspiess
* new drawing method for pose2d with color
*
* Revision 1.8  2002/06/12 11:35:36  roefer
* Deleting lists is now less stack-intensive
*
* Revision 1.7  2002/06/08 11:44:00  mkunz
* Special Percept on Field
*
* Revision 1.6  2002/06/03 13:12:57  kspiess
* debug drawing for potentialfield added
*
* Revision 1.5  2002/06/01 10:12:20  juengel
* DebugKey sendCrosshairsDrawing added.
*
* Revision 1.4  2002/05/25 17:37:03  juengel
* 3D and offset for lookAtPoint
*
* Revision 1.3  2002/05/17 09:57:50  brunn
* added debugDrawing bar
*
* Revision 1.2  2002/05/16 08:33:59  juengel
* Added DebugDrawing "pattern"
*
* Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.29  2002/04/23 14:16:47  jhoffman
* added gradient image processor
*
* Revision 1.1.1.1  2002/04/10 11:03:22  juengel
* no message
*
* Revision 1.28  2002/04/08 19:53:14  juengel
* Drawing of percept collections in images added.
*
* Revision 1.27  2002/04/07 13:02:13  roefer
* Improved goal drawing
*
* Revision 1.26  2002/04/04 18:43:59  juengel
* FlagSpecialist improved.
*
* Revision 1.25  2002/04/02 10:30:34  juengel
* GridImageProcessor enhanced.
*
* Revision 1.24  2002/03/24 17:47:02  juengel
* LinesPercept defined and LinesPerceptor added.
*
* Revision 1.23  2002/03/16 13:44:49  juengel
* GridImageProcessor created.
*
* Revision 1.22  2002/02/18 15:41:17  loetzsch
* changed sequence of debug drawings
*
* Revision 1.21  2002/02/11 00:55:38  loetzsch
* ::Paint Methoden in ::paint umbenannt,
* BallPerzept- Visualisierung fertig
*
* Revision 1.20  2002/02/10 12:45:15  loetzsch
* radar viewer and percept visualization continued
*
* Revision 1.19  2002/02/10 12:19:02  juengel
* oracledWorldState and worldState drawing exchangeg.
*
* Revision 1.18  2002/02/08 13:28:08  loetzsch
* Visualization of PerceptCollection and RadarViewer started
* (not implemented yet)
*
* Revision 1.17  2002/02/06 15:34:44  Denny
* added method for drawing of Pose2D as Arrows on the field
*
* Revision 1.16  2002/02/05 03:41:37  loetzsch
* renamed DebugDrawing::blobs to DebugDrawing::octangles
* and DebugDrawing::filledBlobs to DebugDrawing::filledOctangles.
*
* added ids for blob collections and percept collections
*
* Revision 1.15  2002/02/04 13:47:09  kspiess
* BremenBerlin2001PlayersLocator in GT2001PlayersLocator umbenannt
* alte Aufrufe in neue gendert
* DebugDrawing fr GT2001PlayersLocator eingebaut
*
* Revision 1.14  2002/02/03 14:37:58  juengel
* Drawing of the world state removed from Berlin2001BehaviorControl.
* Drawing method for world states added to PaintMethods.
* Drawing of the world state added to the Processes with BehaviorControl.
*
* Revision 1.13  2002/01/30 15:50:47  risler
* removed bug: if(..) OUTPUT(..)
*
* Revision 1.12  2002/01/30 15:34:13  risler
* enabled debug drawings from robot
*
* Revision 1.11  2002/01/30 15:24:24  risler
* enabled debug drawings from robot
*
* Revision 1.10  2002/01/28 11:41:35  roefer
* DebugDrawing for world state inserted
*
* Revision 1.9  2002/01/23 12:36:16  juengel
* DebugDrawing Makros eingefgt.
*
* Revision 1.8  2002/01/22 14:55:15  juengel
* ImageToPerceptCollection eingefhrt
*
* Revision 1.7  2002/01/22 11:01:47  kspiess
* Eintrge fr BremenBerlin2001PlayersLocator eingefgt
*
* Revision 1.6  2002/01/17 15:13:17  roefer
* Views and debug output in SimGT2002
*
* Revision 1.5  2002/01/06 12:15:01  juengel
* Streaming Operatoren iplementiert
*
* Revision 1.4  2002/01/04 14:35:19  juengel
* no message
*
* Revision 1.3  2001/12/14 17:35:49  juengel
* struct Color, enum penStyle und enum fillStyle in die Klasse DebugDrawing hereingenommen.
*
* Revision 1.2  2001/12/12 17:34:26  risler
* changed Point to Vector2
*
* Revision 1.1  2001/12/12 15:21:09  juengel
* CDebugKeyToolBar in eigenes .cpp und .h File verlagert.
*
* Revision 1.8  2001/12/10 17:47:10  risler
* change log added
*
*/
