/**
* @file RobotControlQueues.h
*
* Declaration of the main queues in RobotControl.
*
* @author Martin Ltzsch
*/

#ifndef __RobotControlQueues_h_
#define __RobotControlQueues_h_

#include "Tools/MessageQueue/MessageQueue.h"
#include "Tools/MessageQueue/LogPlayer.h"

/** 
* @class CRobotControlQueuesToRobots
*
* The common interface to both physical and simulated robots.
*
* @author Martin Ltzsch
*/
class CRobotControlQueuesToRobots
{
public:
  /** A queue for messages to the selected robot */
  MessageQueue selectedRobot;

  /** A queue for messages to all robots */
  MessageQueue allRobots;

  /** Queues to specific robots specified by their robot number */
  MessageQueue robot[8];
};

/**
* @class CRobotControlQueues
* 
* The main message queues in RobotControl.
* 
* @author Martin Ltzsch
*/
class CRobotControlQueues
{
public:
  /** Constructor */
  CRobotControlQueues();

  /** The queues to the physical robots */
  CRobotControlQueuesToRobots toPhysical;

  /** The queues to the simulated robots */
  CRobotControlQueuesToRobots toSimulated;

  /** A queue for messages that are handled by the GUI */
  MessageQueue toGUI;

  /** A queue for recording messages from the queueFromPhysicalRobots */
  LogPlayer logPlayer;

protected:
  /** A queue for messages from the physical robots */
  MessageQueue fromPhysicalRobots;

  /** A queue for messages from the simulated robots */
  MessageQueue fromSimulatedRobots;

  // Dialogs and Toolbars are not allowed to access the protected 
  // queues as this would lead to synchronisation problems.
  // All classes that can access these queues are listed below.
  friend class CMessageHandlerForQueueFromPhysicalRobots;
  friend class CRobotControlApp;
  friend class OVirtualRobotComm;
  friend class CLogPlayerToolBar;
  friend class CRobotControlPhysicalRobots;
public:
  /** 
  * Returns whether a certain message is from a 
  * selected (simulated or physical) robot.
  * If the robot number is undefined, true is returned too.
  */
  static bool isFromSelectedOrUndefinedRobot(const InMessage& message);
};

/**
* Returns a reference to the RobotControl queues interface.
* Use that function only inside the thread of the RobotControl GUI !!!
*/
CRobotControlQueues& getQueues();


#endif //__RobotControlQueues_h_

/*
* Change Log:
*
* $Log: RobotControlQueues.h,v $
* Revision 1.2  2004/06/08 11:55:07  wachter
* Robot control now uses asyncrhonous communication with CAsyncSocket instead of wlan-threds now.
*
* Revision 1.1.1.1  2004/05/22 17:26:51  cvsadm
* created new repository GT2004_WM
*
* Revision 1.4  2004/02/16 12:26:41  nistico
* Added noise reduction functionality for jpeg images in log file player
*
* Revision 1.3  2003/12/09 19:49:24  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.2  2003/12/07 18:58:03  loetzsch
* added isFromSelectedOrUndefinedRobot()
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/03/24 14:05:42  loetzsch
* moved the logPlayer instance from CRobotControlApp to CRobotControlQueues
*
* Revision 1.2  2003/03/05 17:09:27  loetzsch
* redesign of the queues and debug key tables in RobotControl
*
* Revision 1.1  2002/09/10 15:49:04  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.1  2002/07/23 13:45:14  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - new debug message handling
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
*/
