/**
* @file RobotControlPhysicalRobots.h
*
* Definition of class CRobotControlWLanThread and class CRobotControlPhysicalRobots
*
* @author Uwe Dffert
* @author Martin Ltzsch
*/

#ifndef __RobotControlPhysicalRobots_h_
#define __RobotControlPhysicalRobots_h_

#include "Tools/MessageQueue/MessageQueue.h"
#include "Platform/Win32/Thread.h"
#include "RobotControlQueues.h"
#include "RobotControlDebugConnection.h"

/** 
* @class CRobotControlWLANConfiguration
*
* Stores all information of a (team) wlan configuration 
*
* @author Uwe Dffert
* @author Max Risler
*/
class CRobotControlWLANConfiguration
{
public:
  /** constructor */
  CRobotControlWLANConfiguration();

  /** 
  * Fills this configuration with data read from the Registry
  * @param index number of the WLAN configuration in the Registry to read from
  */
  void readFromProfile(int index);

  /** 
  * Writes the data in this configuration to the Registry
  * @param index number of the WLAN configuration in the Registry to write to
  */
  void writeToProfile(int index);

  /** 
  * deletes a WLAN configuration in the Registry
  * @param index number of the WLAN configuration in the Registry to delete
  */
  void deleteProfile(int index);

  char name[65]; /*!< The name of the configuration */
  unsigned long ipRedRouter; /*!< The ip of the router for the red robots */
	unsigned long ipBlueRouter; /*!< The ip of the router for the blue robots */
	int portRedRouter; /*!< The port of the router for the red robots */
	int portBlueRouter; /*!< The port of the router for the blue robots */
  bool autostartRed; /*!< ??? */
  bool autostartBlue; /*!< ??? */
  unsigned long ipRed[4]; /*!< The ips for the red team */
  unsigned long ipBlue[4]; /*!< The ips for the blue team */
	bool useRed[4]; /*!< Which of the red robots shall be connected */
	bool useBlue[4]; /*!< Which of the red robots shall be connected */
	char essidRed[65]; /*!< The essid of the red team */
	char essidBlue[65]; /*!< The essid of the blue team */
	unsigned long netmaskRed; /*!< The netmask of the red team */
	unsigned long netmaskBlue; /*!< The netmask of the blue team */
	int apmodeRed; /*!< The apmode of the red team */
	int apmodeBlue; /*!< The apmode of the blue team */
	int channelRed; /*!< The WLAN channel of the red team */
	int channelBlue; /*!< The WLAN channel of the blue team */
  char wepkeyRed[17]; /*!< The wepkey of the red team */
  char wepkeyBlue[17]; /*!< The wepkey of the blue team */
  char teamIdentifierBlue[17];
  char teamIdentifierRed[17];
};



/** 
* @class CRobotControlPhysicalRobots
*
* Encapsulates the WLAN interface to the physical robots.
* 
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class CRobotControlPhysicalRobots 
	//: public Thread<CRobotControlPhysicalRobots>
{
public:
  DECLARE_SYNC;

  /** Default constructor */
  CRobotControlPhysicalRobots();
  
  /** Destructor */
  ~CRobotControlPhysicalRobots();
  
  /** Should be called when the application is idle, used for communication */
  void onIdle();
  
  /** Connects all robots contained in config */
  void connect(CRobotControlWLANConfiguration& config);
  
  /** Disconnects all connected robots */
  void disconnect();
  
  /** Selects a physical robot. If -1, then no robot is selected. */
  void setSelectedRobot(int robot);
  
  /** Returns the selected physical robot. If -1, then no robot is selected. */
  int getSelectedRobot() const;
  
  /** return whether a certain robot is connected via WLAN */
  bool isConnected(int robot) const;
  
  /** return whether at least one robot is connected and select that if no one else is selected */
  bool isSomeoneConnected();
  
protected:
  CRobotControlDebugConnection *debugConnection[8];
  MessageQueue queueToRobot[8];
  MessageQueue queueFromRobot[8];
  
   /** The robot that is connected to the gui */
  int selectedRobot;
};

/** Returns a reference to the physical robots instance inside RobotControl */
CRobotControlPhysicalRobots& getPhysicalRobots();

#endif //__RobotControlPhysicalRobots_h_

/*
* Change Log:
*
* $Log: RobotControlPhysicalRobots.h,v $
* Revision 1.2  2004/06/08 11:55:07  wachter
* Robot control now uses asyncrhonous communication with CAsyncSocket instead of wlan-threds now.
*
* Revision 1.1.1.1  2004/05/22 17:26:51  cvsadm
* created new repository GT2004_WM
*
* Revision 1.5  2004/01/27 23:20:23  wachter
* Added  teamIdentifier to WLANConfigurationDialog
*
* Revision 1.4  2004/01/04 19:06:12  wachter
* Moved connect(..) into WLan-thread.
*
* Revision 1.3  2003/12/13 11:32:06  loetzsch
* renamed WLANConfiguration to CRobotControlWLANConfiguration
* and moved to RobotControlPhysicalRobots.h
*
* Revision 1.2  2003/12/04 00:52:05  loetzsch
* small beautification
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/08/17 18:35:37  roefer
* Communication with router standardized and synchronized
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/03/28 14:04:19  dueffert
* team wlan support improved
*
* Revision 1.2  2003/03/25 11:46:53  dueffert
* team wlan support improved
*
* Revision 1.1  2003/03/24 14:06:51  loetzsch
* added class CRobotControlPhysicalRobot
*
*/
