/**
* @file RobotControlDebugConnection.cpp
*
* Implementation of class RobotControlDebugConnection.
*
* @author Michael Wachter
*/

#include "RobotControlDebugConnection.h"

#define ERRORMESSAGE(t1) error.out.text << t1; error.out.finishMessage(idText);

#include "Tools/Debugging/QueueFillRequest.h"
#include "Platform/SystemCall.h"
#include <stdlib.h>

CRobotControlDebugConnection::CRobotControlDebugConnection(MessageQueue &from, MessageQueue &to, MessageQueue &error): 
queueFromRobot(from),
queueToRobot(to),
error(error)
{
   connected = false;
   sendBuffer = NULL;
   receivingData = false;
   resetBuffers();
}

void CRobotControlDebugConnection::resetBuffers()
{
	if (!sendBuffer == NULL){
		free(sendBuffer);
		sendBuffer = NULL;
	}
	sendBufferSize=0;
	sendPosition=0;
	if (receivingData) 
	{ 
		free(receiveBuffer);
		
	}
	receiveBuffer = (char*)&receiveBufferForSize;
	receiveBufferSize = sizeof(int);
	receivingData = false;
	receivePosition   = 0;
	
}

CRobotControlDebugConnection::~CRobotControlDebugConnection()
{
   Close();
   resetBuffers();
}


void CRobotControlDebugConnection::wLanConnect(const unsigned long remoteIP, int remotePort)
{
   
		if (!Create(0,SOCK_STREAM, FD_READ + FD_WRITE + FD_CONNECT + FD_CLOSE))
		{
			ERRORMESSAGE("CRobotControlDebugConnection : Could not create socket ");
			return;
		}
	
  
   sprintf(ipAddr, "%i.%i.%i.%i", remoteIP >> 24,  remoteIP >> 16 & 255,  remoteIP >> 8 & 255, remoteIP & 255);
   error.out.text << "CRobotControlDebugConnection: trying to connect to " << ipAddr << ":" << remotePort;
   error.out.finishMessage(idText);

   Connect(ipAddr,remotePort);
}


void CRobotControlDebugConnection::OnConnect(int nErrorCode)   
{
   CString errorMessage;
   if (0 != nErrorCode)
   {
      switch( nErrorCode )
      {
         case WSAEADDRINUSE: 
            errorMessage = "Address in use.";
            break;
         case WSAEADDRNOTAVAIL: 
            errorMessage = "Address not available from local machine.";
            break;
         case WSAEAFNOSUPPORT: 
            errorMessage = "Addresses in the specified family cannot be used with this socket.";
            break;
         case WSAECONNREFUSED: 
            errorMessage = "Connection attempt forcefully rejected.";
            break;
         case WSAEDESTADDRREQ: 
            errorMessage = "Destination address required.";
            break;
         case WSAEFAULT: 
            errorMessage = "The lpSockAddrLen argument is incorrect.";
            break;
         case WSAEINVAL: 
            errorMessage = "The socket is already bound to an address.";
            break;
         case WSAEISCONN: 
            errorMessage = "The socket is already connected.";
            break;
         case WSAEMFILE: 
            errorMessage = "No more file descriptors are available.";
            break;
         case WSAENETUNREACH: 
            errorMessage = "The network cannot be reached from this host at this time.";
            break;
         case WSAENOBUFS: 
            errorMessage = "No buffer space is available. The socket cannot be connected.";
            break;
         case WSAENOTCONN: 
            errorMessage = "The socket is not connected.";
            break;
         case WSAETIMEDOUT: 
            errorMessage = "The attempt to connect timed out without establishing a connection.";
            break;
         default:
            errorMessage = "Unknown Error";
            break;
      }
     
	  ERRORMESSAGE ("CRobotControlDebugConnection: Can't connect to " << ipAddr << ": " << errorMessage);
      connected=false;
	  Close();
	  resetBuffers();
	  return;
	  
   }
   
   CAsyncSocket::OnConnect(nErrorCode);
   ERRORMESSAGE("CRobotControlDebugConnection: Connection to " << ipAddr << " established");
   connected=true;
}

void CRobotControlDebugConnection::OnReceive(int nErrorCode)
{
	int bytesRead;
	bytesRead = Receive(receiveBuffer + receivePosition, receiveBufferSize - receivePosition);
	if (bytesRead == SOCKET_ERROR)
	{
		ERRORMESSAGE ("CRobotControlDebugConnection: Socket-Error");
		return;
	}
	receivePosition+=bytesRead;
	
	// Fehler abfragen
    
    if (receivePosition == receiveBufferSize)
	{
	    // all data in buffer. 

		if (receivingData)
		{
			// Streamen
            InBinaryMemory stream(receiveBuffer,receiveBufferSize);
            stream >> queueFromRobot;

            // Speicher freigeben
			free(receiveBuffer);
			receiveBuffer = NULL;

			// nun wieder size lesen
			receiveBuffer = (char*)&receiveBufferForSize;
			receiveBufferSize = sizeof(int);
            receivingData = false;
			receivePosition = 0;
		
		}
		else
		{
            receiveBufferSize = *(int*)receiveBuffer;
		    receiveBuffer = (char*)malloc(receiveBufferSize);
			receivingData = true;
			receivePosition = 0;
		}
	}
}

void CRobotControlDebugConnection::OnSend(int nErrorCode)
{
   int bytesSend;
   // copypasted from MFC-documentation CAsyncSocket::onSend()
   while (sendPosition < sendBufferSize)
   {
     if ((bytesSend = Send(sendBuffer + sendPosition, sendBufferSize - sendPosition)) == SOCKET_ERROR)
      {
         if (GetLastError() == WSAEWOULDBLOCK) break;
         else
         {
			 ERRORMESSAGE("CRobotControlDebugConnection: Error Sending");
         }
      }
      else
      {
         sendPosition += bytesSend;
      }
   }

   // ERRORMESSAGE("Send : " << sendPosition << " : " << sendBufferSize);

   if (sendPosition == sendBufferSize)
      {
		 free (sendBuffer);
		 sendBuffer = NULL;
         sendBufferSize = 0;
		 sendPosition = 0;
	//	 ERRORMESSAGE("CRobotControlDebugConnection: Sending done");
      }
 

   CAsyncSocket::OnSend(nErrorCode);
}


void CRobotControlDebugConnection::sendMessageQueue()
{
    if ((!queueToRobot.isEmpty()) && (sendBufferSize==0))
	{
       // Size bestimmen
       sendBufferSize = queueToRobot.getStreamedSize();

	   // Allocate memeory
       sendBuffer = (char*)malloc(sendBufferSize + sizeof(int));
       
	   *(int*)sendBuffer = sendBufferSize;

	   // Streamen
	   OutBinaryMemory memoryStream((char*)sendBuffer+sizeof(int));
       memoryStream << queueToRobot;

	   sendBufferSize+= sizeof(int);
	   queueToRobot.clear();
	   OnSend(0);
	}
}


void CRobotControlDebugConnection::OnClose(int nErrorCode)
{
	 CString errorMessage = "";
     switch( nErrorCode )
      {
         case WSAENETDOWN:
		     errorMessage = "Network subsystem failed.";
		     break;
	     case WSAECONNRESET:
		     errorMessage = "Connection reset by beer.";
		     break;
	     case WSAECONNABORTED:
		     errorMessage = "Connection timeout." ;
			 break;
         default:
             errorMessage = "Unknown Error";
	  }

	  CAsyncSocket::OnClose(nErrorCode);
	  ERRORMESSAGE("CRobotControlDebugConnection: Connection closed : " << errorMessage);
	  resetBuffers();
	  connected = false;
}

bool CRobotControlDebugConnection::wLanIsConnected()
{
    return(connected);
}

void CRobotControlDebugConnection::wLanDisconnect()
{   
	if (connected)
	{
		Close();
		ERRORMESSAGE("CRobotContolDebugConnection: Connection closed");
		resetBuffers();
		connected = false;
	}
}

/*
 * Changelog
 *
 * $Log: RobotControlDebugConnection.cpp,v $
 * Revision 1.1  2004/06/08 11:55:07  wachter
 * Robot control now uses asyncrhonous communication with CAsyncSocket instead of wlan-threds now.
 *
 *
 */