/**
 *  @file HSIColorTableToolCore.h
 *  Declaration of class HSIColorTableTool.
 *  @author<A href=mailto:kspiess@informatik.uni-bremen.de>Kai Spiess</A>
 */
#ifndef __HSIColorTableToolCore_h
#define __HSIColorTableToolCore_h


#include "Representations/Perception/ColorTable64.h"	
#include "HSIDataTypes.h"
#include "Representations/Perception/ColorClassImage.h"
#include "Representations/Perception/Image.h"


/** 
 * This class represents the core of the HSI colortable tool including colortables, images 
 * and methods for image segmentation, converting images from one color space to another
 * and getting color classes from the colortable.
 */
class HSIColorTableToolCore
{
public:
	void growHSIColorRange(int h, int s, int i, int colorClass);
	void shrinkHSIColorRange(int h, int i, int s, int colorClass);
  /** The maximum number of undos. */
  enum{MAXIMUM_NUMBER_OF_UNDOS = 6};
  /** The undos of HSI color range edit. */
  HSIColorRangeUndo hsiColorRangeEditUndo[MAXIMUM_NUMBER_OF_UNDOS];
  /** The number of stored undos. */
  int numberOfUndos;
  /** The actual undo number. */
	int actualUndo;
  /** This stores an HSI color range for the specified color class.
   *  @param colorClass The color class.
   */
	void setUndo(int colorClass);
  /** This restores the last edited HSI color range. */
	void undoEditHSIColorRange();
  /** This updates color classification for the segmented images with the converted 
   *  YUV colortable.
   */
	void updateSegmentedImagesWithColorTable64();

  /** The number of images handled by the tool. */
  enum{NUMBER_OF_IMAGES = 4};

  /** This determines the color class of a color in HSI color space using the HSI colortable. 
   *  @param H The H value of the color.
   *  @param S The S value of the color.
   *  @param I The I value of the color.
   *  @return The color class of the given color.
   */
	colorClass getColorClassByHSI(int H, int S, int I);
  /** The creates an YUV color table from the HSI color table of the tool. 
   *  @param colorTable64 The YUV colortable.
   */
  void createColorTable64FromHSIColorTable(ColorTable64& colorTable64);
  /** This initialized the HSI colortable toolcore. */
  void init();
  /** This clears an rgbImage. All colors are set to black (0,0,0).
   *  @param image The image to clear.
   */
  void doImageSegmentation();
  /** This is loading an HSI colortable from a file.
   *  @param filename The name of the file.
   */
  void loadHSIColorTableFromFile(const char* filename);
  /** This is saving an HSI colortable to a file.
   *  @param filename The name of the file.
   */
  void saveHSIColorTableToFile(const char* filename);
  /** Stores an image at the specified place in the tool
   *  @param image The image to store
   *  @param index The place where the image should be stored
   */
  void setImageToPlace(Image &image, int index);

  /** The original YUV images. */
	Image originalYUVImages[NUMBER_OF_IMAGES];
  /** The HSI image representation, converted from the YUV image. */
  Image hsiImages[NUMBER_OF_IMAGES];
  /** The color classified images. */
  ColorClassImage colorClassImages[NUMBER_OF_IMAGES];
  /** The HSI colortable of the tool. */
  HSIColorTable hsiColorTable;
  /** The colortable for segmentation in YUV color space . */
  ColorTable64 colorTable64;
  /** The flags whether an image is stored in a place or not*/
  bool isImageAtPlace[NUMBER_OF_IMAGES];
};


#endif //__HSIColorTableToolCore_h


/*
 * Change log :
 * 
 * $Log: HSIColorTableToolCore.h,v $
 * Revision 1.1.1.1  2004/05/22 17:28:26  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/05/01 17:09:33  roefer
 * Streamlined HSI stuff
 *
 * Revision 1.2  2004/01/17 17:07:41  roefer
 * HSI-Toolbar does not crash anymore, but image size is still 176x144
 *
 * Revision 1.1  2003/10/07 10:10:07  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:28:24  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.1  2002/09/10 15:49:09  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.2  2002/08/29 13:51:12  dueffert
 * includes in correct case, system includes in <>
 *
 * Revision 1.1.1.1  2002/05/10 12:40:25  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.3  2002/04/05 08:17:00  kspiess
 * gleichzeitiges Speichern von HSI und C64 Farbtabelle eingebaut;
 * Fehler beim Anklicken einzelner Pixel im Zoom-Dialog zur Verschiebung
 * der HSI-Grenzen behoben;
 * Fehler beim mehrstufigen Undo behoben
 *
 * Revision 1.2  2002/02/08 14:46:26  kspiess
 * Change log added
 *
 *
 */