/**
 *  @file HSIColorTableToolCore.cpp
 *  Implementation of class HSIColorTableToolCore.
 *  @author<A href=mailto:kspiess@informatik.uni-bremen.de>Kai Spiess</A>
 */
#include "StdAfx.h"

#include "HSIColorTableToolCore.h"

#include "Tools/Streams/OutStreams.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/Math/Common.h"
#include "Representations/Perception/ColorTable64.h"


void HSIColorTableToolCore::doImageSegmentation()
{
  for(int i=0; i < NUMBER_OF_IMAGES; i++)
    if(this->isImageAtPlace[i])
    {
      colorClassImages[i].width = hsiImages[i].cameraInfo.resolutionWidth;
      colorClassImages[i].height = hsiImages[i].cameraInfo.resolutionHeight;
      for(int y=0; y < this->hsiImages[i].cameraInfo.resolutionHeight; y++)
        for(int x=0; x < this->hsiImages[i].cameraInfo.resolutionWidth; x++)
          colorClassImages[i].image[y][x] = this->getColorClassByHSI(hsiImages[i].image[y][0][x],
                                                                     hsiImages[i].image[y][1][x],
                                                                     hsiImages[i].image[y][2][x]);
    }
} 


void HSIColorTableToolCore::loadHSIColorTableFromFile(const char* filename)
{
	InTextFile stream(filename);
	stream >> hsiColorTable;
}


void HSIColorTableToolCore::saveHSIColorTableToFile(const char* filename)
{
  CString saveFilename(filename);
	OutTextFile streamHSI(saveFilename+".hsi");
	streamHSI << hsiColorTable;

  createColorTable64FromHSIColorTable(colorTable64);
  OutBinaryFile streamC64(saveFilename+".c64");
  streamC64 << colorTable64;
}


void HSIColorTableToolCore::setImageToPlace(Image &image, int index)
{
  this->originalYUVImages[index] = image;
  this->hsiImages[index].convertFromYCbCrToHSI(this->originalYUVImages[index]);
  this->isImageAtPlace[index] = TRUE;
  this->doImageSegmentation();
}


void HSIColorTableToolCore::init()
{
  for(int i = 0; i < NUMBER_OF_IMAGES; i++)
    this->isImageAtPlace[i] = false;
  this->actualUndo = -1;
  this->numberOfUndos = 0;
  this->hsiColorTable.init();

}


void HSIColorTableToolCore::createColorTable64FromHSIColorTable(ColorTable64 &colorTable64)
{
  for(int y = 0; y < 256; y += 4)
    for (int cb = 0; cb < 256; cb += 4)
      for (int cr = 0; cr < 256; cr += 4)
      {
        unsigned char h,
                      s,
                      i;
        Image::convertFromYCbCrToHSI((unsigned char) y,
                                     (unsigned char) cb,
                                     (unsigned char) cr,
                                     h, s, i);
        colorTable64.addColorClass(getColorClassByHSI(h, s, i), y, cb, cr);
      }
}


colorClass HSIColorTableToolCore::getColorClassByHSI(int H, int S, int I)
{
  int cc = noColor;
  while(cc < NUMBER_OF_COLOR_CLASSES)
  {
    if(hsiColorTable.colorClasses[cc].H_max < hsiColorTable.colorClasses[cc].H_min)
    {
      if((H <= hsiColorTable.colorClasses[cc].H_max ||
          H >= hsiColorTable.colorClasses[cc].H_min) &&
         S <= hsiColorTable.colorClasses[cc].S_max &&
         S >= hsiColorTable.colorClasses[cc].S_min &&
         I <= hsiColorTable.colorClasses[cc].I_max &&
         I >= hsiColorTable.colorClasses[cc].I_min)
          return colorClass(cc);
    }
    else
    {
      if(  H <= hsiColorTable.colorClasses[cc].H_max
        && H >= hsiColorTable.colorClasses[cc].H_min
        && S <= hsiColorTable.colorClasses[cc].S_max
        && S >= hsiColorTable.colorClasses[cc].S_min
        && I <= hsiColorTable.colorClasses[cc].I_max
        && I >= hsiColorTable.colorClasses[cc].I_min)
          return colorClass(cc); 
    }
    ++cc;
  }
  return noColor;
}


void HSIColorTableToolCore::updateSegmentedImagesWithColorTable64()
{
  this->createColorTable64FromHSIColorTable(this->colorTable64);
  for(int i=0; i < NUMBER_OF_IMAGES; i++)
  {
    if(this->isImageAtPlace[i])
      this->colorTable64.generateColorClassImage(this->originalYUVImages[i], this->colorClassImages[i]);
  }
}

void HSIColorTableToolCore::undoEditHSIColorRange()
{
  if(numberOfUndos != 0)
  {
     HSIColorRangeUndo actual = this->hsiColorRangeEditUndo[actualUndo];
//     actual.color = this->m_hsiColorRangeEditUndo[actualUndo].color;

     this->hsiColorTable.colorClasses[actual.colorClass].H_max = actual.hsiColorRange.H_max;
     this->hsiColorTable.colorClasses[actual.colorClass].H_min = actual.hsiColorRange.H_min;
     this->hsiColorTable.colorClasses[actual.colorClass].S_max = actual.hsiColorRange.S_max;
     this->hsiColorTable.colorClasses[actual.colorClass].S_min = actual.hsiColorRange.S_min;
     this->hsiColorTable.colorClasses[actual.colorClass].I_max = actual.hsiColorRange.I_max;
     this->hsiColorTable.colorClasses[actual.colorClass].I_min = actual.hsiColorRange.I_min;

     this->hsiColorRangeEditUndo[actualUndo].set = false;

     if(actualUndo == 0)
     {
       if(this->hsiColorRangeEditUndo[actualUndo].set = true)
         actualUndo = MAXIMUM_NUMBER_OF_UNDOS - 1;
     }
     else
       actualUndo--;
     
     numberOfUndos--;
   }
  this->doImageSegmentation();
}

void HSIColorTableToolCore::setUndo(int colorClass)
{
  HSIColorRange hsiColorRange;
  hsiColorRange.H_max = this->hsiColorTable.colorClasses[colorClass].H_max;
  hsiColorRange.H_min = this->hsiColorTable.colorClasses[colorClass].H_min;
  hsiColorRange.S_max = this->hsiColorTable.colorClasses[colorClass].S_max;
  hsiColorRange.S_min = this->hsiColorTable.colorClasses[colorClass].S_min;
  hsiColorRange.I_max = this->hsiColorTable.colorClasses[colorClass].I_max;
  hsiColorRange.I_min = this->hsiColorTable.colorClasses[colorClass].I_min;

  if(actualUndo == MAXIMUM_NUMBER_OF_UNDOS - 1)
    actualUndo = 0;
  else
    actualUndo++;

  this->hsiColorRangeEditUndo[actualUndo].setHSIColorRangeUndo(hsiColorRange, colorClass);
  if(numberOfUndos != MAXIMUM_NUMBER_OF_UNDOS)
    numberOfUndos++;

}

void HSIColorTableToolCore::shrinkHSIColorRange(int h, int s, int i, int colorClass)
{
 if(this->getColorClassByHSI(h,s,i) == colorClass)
 {
  int hMax = this->hsiColorTable.colorClasses[colorClass].H_max;
  int hMin = this->hsiColorTable.colorClasses[colorClass].H_min;
  int sMax = this->hsiColorTable.colorClasses[colorClass].S_max;
  int sMin = this->hsiColorTable.colorClasses[colorClass].S_min;
  int iMax = this->hsiColorTable.colorClasses[colorClass].I_max;
  int iMin = this->hsiColorTable.colorClasses[colorClass].I_min;

  this->setUndo(colorClass);


  int hMinDist, hMaxDist, sMinDist, sMaxDist, iMinDist, iMaxDist;
  hMinDist = hMaxDist = sMinDist = sMaxDist = iMinDist = iMaxDist = 999;

  int dist = 999;
  int flag = 0;

  //hMin <= hMax
  if(hMin <= hMax)
  {
    if(h > hMin && h < hMax)
    {
      hMaxDist = hMax-h;
      hMinDist = h-hMin;
      if(hMinDist < dist)
      {
        dist = hMinDist;
        flag = 1;
      }
      if(hMaxDist < dist)
      {
        dist = hMaxDist;
        flag = 2;
      }
    }
  }
  //hMax < hMin
  else
  {
    if(h < hMax)
    {
      hMaxDist = hMax-h;
      hMinDist = h + 255 - hMin;
      if(hMinDist < dist)
      {
        dist = hMinDist;
        flag = 1;
      }
      if(hMaxDist < dist)
      {
        dist = hMaxDist;
        flag = 2;
      }
    }
    else if(h > hMin)
    {
      hMaxDist = 255 + hMax - h;
      hMinDist = h-hMin;
      if(hMinDist < dist)
      {
        dist = hMinDist;
        flag = 1;
      }
      if(hMaxDist < dist)
      {
        dist = hMaxDist;
        flag = 2;
      }
    }
  }

  if(s > sMin && s < sMax)
  {
    sMinDist = s - sMin;
    sMaxDist = sMax - s;
    if(sMinDist < dist)
    {
      dist = sMinDist;
      flag = 3;
    }
    if(sMaxDist < dist)
    {
      dist = sMaxDist;
      flag = 4;
    }
  }

  if(i > iMin && i < iMax)
  {
    iMinDist = i - iMin;
    iMaxDist = iMax -i;
    if(iMinDist < dist)
    {
      dist = iMinDist;
      flag = 5;
    }
    if(iMaxDist < dist)
    {
      dist = iMaxDist;
      flag = 6;
    }
  }

  if(dist < 999)
  {
    switch(flag)
    {
      case 1: this->hsiColorTable.colorClasses[colorClass].H_min = h; break;
      case 2: this->hsiColorTable.colorClasses[colorClass].H_max = h; break;
      case 3: this->hsiColorTable.colorClasses[colorClass].S_min = s; break;
      case 4: this->hsiColorTable.colorClasses[colorClass].S_max = s; break;
      case 5: this->hsiColorTable.colorClasses[colorClass].I_min = i; break;
      case 6: this->hsiColorTable.colorClasses[colorClass].I_max = i; break;
    }
  }
 }
}


void HSIColorTableToolCore::growHSIColorRange(int h, int s, int i, int colorClass)
{
  int hMax = this->hsiColorTable.colorClasses[colorClass].H_max;
  int hMin = this->hsiColorTable.colorClasses[colorClass].H_min;
  int sMax = this->hsiColorTable.colorClasses[colorClass].S_max;
  int sMin = this->hsiColorTable.colorClasses[colorClass].S_min;
  int iMax = this->hsiColorTable.colorClasses[colorClass].I_max;
  int iMin = this->hsiColorTable.colorClasses[colorClass].I_min;

  this->setUndo(colorClass);

  //hMin <= hMax
  if(hMin <= hMax)
  {
    if(h < hMin || h > hMax)
    {
      if(h < hMin)
      {
        if((hMin - h) < (255 - hMax + h))
        {
          this->hsiColorTable.colorClasses[colorClass].H_min = h;
        }
        else
        {
          this->hsiColorTable.colorClasses[colorClass].H_max = h;
        }
      }
      else if(h > hMax)
      {
        if((h - hMax) < (255 + hMin - h))
        {
          this->hsiColorTable.colorClasses[colorClass].H_max = h;
        }
        else
        {
          this->hsiColorTable.colorClasses[colorClass].H_min = h;
        }
      }
    }
  }
  //hMax < hMin
  else
  {
    if(h > hMax && h < hMin)
    {
      if((h - hMax) < (hMin -h))
      {
        this->hsiColorTable.colorClasses[colorClass].H_max = h;
      }
      else if((hMin - h) < (h-hMax))
      {
        this->hsiColorTable.colorClasses[colorClass].H_min = h;
      }
    }
  }


  if(s < sMin)
  {
    this->hsiColorTable.colorClasses[colorClass].S_min = s;
  }
  else if(s > sMax)
  {
      this->hsiColorTable.colorClasses[colorClass].S_max = s;
  }

  if(i < iMin)
  {
    this->hsiColorTable.colorClasses[colorClass].I_min = i;
  }
  else if(i > iMax)
  {
      this->hsiColorTable.colorClasses[colorClass].I_max = i;
  }
}


/*
 * Change log :
 * 
 * $Log: HSIColorTableToolCore.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:28:26  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.5  2004/05/01 17:09:33  roefer
 * Streamlined HSI stuff
 *
 * Revision 1.4  2004/03/16 13:20:12  tim
 * Fixed bug in colortable conversion
 *
 * Revision 1.3  2004/01/17 17:07:41  roefer
 * HSI-Toolbar does not crash anymore, but image size is still 176x144
 *
 * Revision 1.2  2003/12/15 11:47:06  juengel
 * Introduced CameraInfo
 *
 * Revision 1.1  2003/10/07 10:10:07  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:28:24  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.7  2003/05/11 23:37:55  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.6  2003/03/23 19:11:50  loetzsch
 * OUTPUT not allowed in the RobotControl thread anymore.
 * Use getQueues().toGUI.out instead.
 *
 * Revision 1.5  2002/10/01 09:03:17  kspiess
 * fixed click-pixel-routine in zoomed dialog
 *
 * Revision 1.4  2002/09/22 18:40:55  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.3  2002/09/19 12:05:49  loetzsch
 * MathLib -> Mathlib
 *
 * Revision 1.2  2002/09/17 23:55:23  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:49:09  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.3  2002/08/29 13:51:12  dueffert
 * includes in correct case, system includes in <>
 *
 * Revision 1.2  2002/07/23 13:45:44  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - 5 instead of 3 debug queues in RobotControl
 * - new debug message handling
 * - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
 * - general clean up
 *
 * Revision 1.1.1.1  2002/05/10 12:40:25  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.5  2002/04/25 14:50:36  kallnik
 * changed double/float to double
 * added several #include GTMath
 *
 * PLEASE use double
 *
 * Revision 1.4  2002/04/08 18:06:10  kspiess
 * automatische generierung von testC64.c64 entfernt
 *
 * Revision 1.3  2002/04/05 08:17:00  kspiess
 * gleichzeitiges Speichern von HSI und C64 Farbtabelle eingebaut;
 * Fehler beim Anklicken einzelner Pixel im Zoom-Dialog zur Verschiebung
 * der HSI-Grenzen behoben;
 * Fehler beim mehrstufigen Undo behoben
 *
 * Revision 1.2  2002/02/08 14:46:26  kspiess
 * Change log added
 *
 *
 */
