/**
* @file Xabsl2DlgBar.cpp
*
* Implementation of class CXabsl2DlgBar
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
* @author <a href="mailto:loetzsch@informatik.hu-berlin.de">Martin Ltzsch</a>
*/

#include "StdAfx.h"
#include "Xabsl2DlgBar.h"

#include "RobotControlQueues.h"
#include "RobotControlDebugKeyTables.h"

#include "Tools/Module/SolutionRequest.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

CXabsl2DlgBar::CXabsl2DlgBar()
: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CXabsl2DlgBar)
  //}}AFX_DATA_INIT
  
  bmpOffScreen = NULL;
  oldBitmap = NULL;
  lineHeight = 13;
  leftOfColumnPercent[0] = 0;
  leftOfColumnPercent[1] = 0.25;
  leftOfColumnPercent[2] = 0.5;
  leftOfColumnPercent[3] = 0.7;
  
  leftOfColumnPercent[4] = 0.05;
  leftOfColumnPercent[5] = 0.2;
  leftOfColumnPercent[6] = 0.35;
  leftOfColumnPercent[7] = 0.6;
  leftOfColumnPercent[8] = 0.8;
  leftOfColumnPercent[9] = 0.85;
  
  horizontalSpacing = 5;
  
}

CXabsl2DlgBar::~CXabsl2DlgBar()
{
  if(bmpOffScreen)
  {
    dcOffScreen.SelectObject(oldBitmap);
    bmpOffScreen->DeleteObject();
    delete(bmpOffScreen);
  }
  int i;
  for(i = 0; i < numberOfOutputSymbols; i++)
  {
    delete outputSymbolAlternatives[i];
  }
  for(i = 0; i < numberOfOptions; i++)
  {
    delete optionParameterNames[i];
  }
  for(i = 0; i < numberOfBasicBehaviors; i++)
  {
    delete basicBehaviorParameterNames[i];
  }
  for(i = 0; i < 20; i++)
  {
    activeOptionParameterName[i].RemoveAll();
    activeOptionParameterValue[i].RemoveAll();
  }
  for(i = 0; i < 20; i++)
  {
    activeBasicBehaviorParameterName[i].Empty();
    activeBasicBehaviorParameterValue[i].Empty();
  }
}

void CXabsl2DlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CXabsl2DlgBar)
	DDX_Control(pDX, IDC_XABSL2_TEST_HEADCONTROL_CHECK, m_testHeadControlCheck);
  DDX_Control(pDX, IDC_XABSL2_TEST_ON_ROBOT_CHECK, m_TestOnRobotCheck);
  DDX_Control(pDX, IDC_XABSL2_SEND_BUTTON, m_SendButton);
  DDX_Control(pDX, IDC_XABSL2_PARAMETER_STATIC, m_ParameterStatic);
  DDX_Control(pDX, IDC_XABSL2_PARAM_EDIT5, m_Param5Edit);
  DDX_Control(pDX, IDC_XABSL2_PARAM_EDIT4, m_Param4Edit);
  DDX_Control(pDX, IDC_XABSL2_PARAM_EDIT3, m_Param3Edit);
  DDX_Control(pDX, IDC_XABSL2_PARAM_EDIT2, m_Param2Edit);
  DDX_Control(pDX, IDC_XABSL2_PARAM_EDIT1, m_Param1Edit);
  DDX_Control(pDX, IDC_XABSL2_OUTPUT_SYMBOL_VALUE_COMBO, m_OutputSymbolValueCombo);
  DDX_Control(pDX, IDC_XABSL2_OUTPUT_SYMBOL_COMBO, m_OutputSymbolCombo);
  DDX_Control(pDX, IDC_XABSL2_OPTIONS_COMBO, m_OptionsCombo);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CXabsl2DlgBar, CDynamicBarDlg)
//{{AFX_MSG_MAP(CXabsl2DlgBar)
ON_WM_PAINT()
ON_WM_SIZE()
ON_WM_CONTEXTMENU()
ON_BN_CLICKED(IDC_XABSL2_TEST_ON_ROBOT_CHECK, OnXabsl2TestOnRobotCheck)
ON_BN_CLICKED(IDC_XABSL2_SEND_BUTTON, OnXabsl2SendButton)
ON_CBN_SELCHANGE(IDC_XABSL2_OPTIONS_COMBO, OnSelchangeXabsl2OptionsCombo)
ON_CBN_SELCHANGE(IDC_XABSL2_OUTPUT_SYMBOL_COMBO, OnSelchangeXabsl2OutputSymbolCombo)
ON_CBN_SELCHANGE(IDC_XABSL2_OUTPUT_SYMBOL_VALUE_COMBO, OnSelchangeXabsl2OutputSymbolValueCombo)
	ON_BN_CLICKED(IDC_XABSL2_TEST_HEADCONTROL_CHECK, OnXabsl2TestHeadcontrolCheck)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


BOOL CXabsl2DlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();
  AddSzControl(m_ParameterStatic, mdResize, mdNone);
  AddSzControl(m_OptionsCombo, mdResize, mdNone);
  AddSzControl(m_SendButton, mdRepos, mdNone);
  AddSzControl(m_TestOnRobotCheck, mdRepos, mdNone);
  AddSzControl(m_testHeadControlCheck, mdRepos, mdNone);
  AddSzControl(m_OutputSymbolValueCombo, mdResize, mdNone);
  AddSzControl(m_Param3Edit, mdResize, mdNone);
  AddSzControl(m_Param4Edit, mdRepos, mdNone);
  AddSzControl(m_Param5Edit, mdRepos, mdNone);
  
  clear();
  sendDebugRequest();

  return TRUE;
}

void CXabsl2DlgBar::OnPaint() 
{
  CPaintDC dc(this); // device context for painting
  CBrush backGroundBrush(RGB(255,255,255));
  dcOffScreen.FillRect(&currentRect, &backGroundBrush);
  createFonts();
  horizontalLineList.RemoveAll();
  verticalLineList.RemoveAll();
  
  /* begin of drawing section */
  int i;
  int line = 0;
  CString text;
  text.Format("%s - %s",selectedBehaviorControlName,selectedAgentName);
  drawText(bold, 0, 0, "Agent: ");
  drawText(normal, 0, 1, text);
  
  // option activation path...
  horizontalLineList.Add(0);
  drawText(bold, 1, 0, "Option Activation Path:");
  line = 2;
  
  for(i = 0; i < numberOfActiveOptions; i++)
  {
    drawText(normal, line, 0, activeOptionName.ElementAt(i)); 
    drawText(normal, line, 6, activeOptionDurationOfActivation.ElementAt(i) + " s"); 
    drawText(normal, line, 2, activeOptionActiveState.ElementAt(i)); 
    drawText(normal, line, 9, activeOptionDurationOfStateActivation.ElementAt(i) + " s"); 
    line++;
    //    verticalLineList.Add(VerticalLine(2, line - 2, line - 1));
    horizontalLineList.Add(line - 2);
    //    horizontalLineList.Add(line - 1);
    if(activeOptionNumberOfParameters.ElementAt(i) != 0)
    {
      for(int j = 0; j < activeOptionNumberOfParameters.ElementAt(i); j++)
      {
        drawText(normal, line, 4, activeOptionParameterName[j].ElementAt(i) );
        drawText(normal, line, 3, activeOptionParameterValue[j].ElementAt(i) );
        line++;
      }
    }
  }
  horizontalLineList.Add(line - 1);
  
  drawText(bold, line, 0, "Active Basic Behavior:");
  drawText(normal, line, 2, activeBasicBehaviorName);
  line++;
  for(int j = 0; j < activeBasicBehaviorNumberOfParameters; j++)
  {
    drawText(normal, line, 4, activeBasicBehaviorParameterName[j] );
    drawText(normal, line, 3, activeBasicBehaviorParameterValue[j] );
    line++;
  }
  // executed motion request
  horizontalLineList.Add(line - 1);
  drawText(bold, line, 0, "Generated Action:");
  drawText(normal, line, 2, generatedAction);
  line++;
  
  if (currentEngineID == SolutionRequest::undefined)
  {
    // display a message if no engine is connected
    line++;
    drawText(bold, line, 0, "No engine connected.");
    line++;
  }
  
  // output symbols
  int firstSymbolLine = numberOfLines - numberOfShownInputAndOutputSymbols - 2;
  if (firstSymbolLine < 4) firstSymbolLine = 4;
  if (line < firstSymbolLine) line = firstSymbolLine;
  
  horizontalLineList.Add(line - 1);
  drawText(bold, line++, 0, "output symbols:");
  for(i = 0; i < numberOfOutputSymbols; i++)
  {
    if(showOutputSymbol[i])
    {
      drawTextFixedX(normal, line, 10, outputSymbolName[i]);
      drawText(normal, line, 3, outputSymbolValue[i]);
      line++;
    }
  }
  
  // input symbols
  horizontalLineList.Add(line - 1);
  drawText(bold, line++, 0, "input symbols:");
  for(i = 0; i < numberOfInputSymbols; i++)
  {
    if(showInputSymbol[i])
    {
      drawTextFixedX(normal, line, 10, inputSymbolName[i]);
      drawText(normal, line, 3, inputSymbolValue[i]);
      line++;
    }
  }
  
  drawHorizontalLines();
  drawVerticalLines();
  /* end of drawing section*/
  
  deleteFonts();
  dc.BitBlt(paintRect.left, paintRect.top, paintRect.right, paintRect.bottom,
    &dcOffScreen, 0, 0, SRCCOPY);
}

void CXabsl2DlgBar::OnSize(UINT nType, int cx, int cy) 
{
  paintRect.top = 82;
  paintRect.bottom = cy;
  paintRect.left = 0;
  paintRect.right = cx;
  
  currentRect.top = 0;
  currentRect.left = 0;
  currentRect.bottom = paintRect.bottom - paintRect.top;
  currentRect.right = paintRect.right;
  
  numberOfLines = currentRect.bottom / lineHeight;
  for(int i = 0; i < numberOfColumns; i++)
  {
    leftOfColumnPosition[i] = (int)(horizontalSpacing + 
      (currentRect.right - currentRect.left - 2 * horizontalSpacing) * leftOfColumnPercent[i]);
  }
  
  CPaintDC dc(this); // device context for painting
  
  if(!dcOffScreen)
    dcOffScreen.CreateCompatibleDC(&dc);
  
  if(bmpOffScreen)
  {
    dcOffScreen.SelectObject(oldBitmap);
    bmpOffScreen->DeleteObject();
    delete(bmpOffScreen);
  }
  bmpOffScreen = new CBitmap;
  bmpOffScreen->CreateCompatibleBitmap(
    &dc, paintRect.right, paintRect.bottom - paintRect.top);
  oldBitmap = dcOffScreen.SelectObject(bmpOffScreen);
  
  
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  
  CDynamicBarDlg::OnSize(nType,cx,cy);
}

void CXabsl2DlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu popup;
  
  popup.LoadMenu(IDP_XABSL2);
  popup.GetSubMenu(0)->AppendMenu(0,3000,"reload files");
  
  CMenu* inputSymbolMenu = popup.GetSubMenu(0)->GetSubMenu(0);
  CMenu* outputSymbolMenu = popup.GetSubMenu(0)->GetSubMenu(1);
  
  CMenu* currentSubMenu = inputSymbolMenu;
  char currentSubMenuTitle[100] = "";
  
  inputSymbolMenu->AppendMenu(MF_SEPARATOR,0,"");
  inputSymbolMenu->AppendMenu(0,3001,"clear all");
  inputSymbolMenu->AppendMenu(MF_SEPARATOR,0,"");
  int i;
  int m = 3;
  UINT nFlags;
  for(i = 0; i < numberOfInputSymbols; i++)
  {
    char* point; 
    nFlags = 0;
    if(m%20 == 19) nFlags |= MF_MENUBARBREAK;
    if (point=strchr(inputSymbolName[i],'.'))
    {
      int l = point - inputSymbolName[i];
      if (strncmp(currentSubMenuTitle,inputSymbolName[i], l)!=0)
      {
        currentSubMenu = new CMenu();
        currentSubMenu->CreateMenu();
        strncpy(currentSubMenuTitle,inputSymbolName[i],l);
        currentSubMenuTitle[l]=0;
        inputSymbolMenu->AppendMenu(MF_POPUP | nFlags,(int)currentSubMenu->m_hMenu, currentSubMenuTitle);
        m++;
      }
      nFlags = 0;
      if(showInputSymbol[i]) nFlags |= MF_CHECKED;
      currentSubMenu->AppendMenu(
        nFlags,
        IDC_XABSL_DIALOG_CONTEXT_MENU_INPUT + i,
        point+1);
    } else {
      currentSubMenuTitle[0]=0;
      m++;
      if(showInputSymbol[i]) nFlags |= MF_CHECKED;
      inputSymbolMenu->AppendMenu(
        nFlags,
        IDC_XABSL_DIALOG_CONTEXT_MENU_INPUT + i,
        inputSymbolName[i]);
    }
  }
  
  outputSymbolMenu->AppendMenu(MF_SEPARATOR,0,"");
  outputSymbolMenu->AppendMenu(0,3002,"clear all");
  outputSymbolMenu->AppendMenu(MF_SEPARATOR,0,"");
  m=3;
  for(i = 0; i < numberOfOutputSymbols; i++)
  {
    nFlags = 0;
    if(showOutputSymbol[i]) nFlags |= MF_CHECKED;
    if(m%20 == 19) nFlags |= MF_MENUBARBREAK;
    outputSymbolMenu->AppendMenu(
      nFlags,
      IDC_XABSL_DIALOG_CONTEXT_MENU_OUTPUT + i,
      outputSymbolName[i]);
    m++;
  }
  
  UINT nID = popup.GetSubMenu(0)->TrackPopupMenu(
    TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RIGHTBUTTON | TPM_RETURNCMD,
    point.x,point.y,AfxGetMainWnd());
  
  switch(nID)
  {
  case 3000: 
    reload();
    sendIntermediateCode();
    sendDebugRequest();
    break;
  case 3001:
    for (i = 0; i < numberOfInputSymbols; i++)
    {
      showInputSymbol[i] = false;
      AfxGetApp()->WriteProfileInt(
        CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\InputSymbols\\"),
        inputSymbolName[i],false);
    }
    countNumberOfShownInputAndOutputSymbols();
    sendDebugRequest();
    break;
  case 3002:
    for (i = 0; i < numberOfOutputSymbols; i++)
    {
      showOutputSymbol[i] = false;
      AfxGetApp()->WriteProfileInt(
        CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\OutputSymbols\\"),
        outputSymbolName[i],false);
    }
    countNumberOfShownInputAndOutputSymbols();
    sendDebugRequest();
    break;
  default:
    if (nID >= IDC_XABSL_DIALOG_CONTEXT_MENU_INPUT)
    {
      showInputSymbol[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_INPUT] = !showInputSymbol[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_INPUT];
      AfxGetApp()->WriteProfileInt(
        CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\InputSymbols\\"),
        inputSymbolName[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_INPUT],
        showInputSymbol[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_INPUT]
        );
      if(showInputSymbol[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_INPUT]) numberOfShownInputAndOutputSymbols++;
      else numberOfShownInputAndOutputSymbols--;
      sendDebugRequest();
    }
    else if (nID >= IDC_XABSL_DIALOG_CONTEXT_MENU_OUTPUT)
    {
      showOutputSymbol[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_OUTPUT] = !showOutputSymbol[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_OUTPUT];
      AfxGetApp()->WriteProfileInt(
        CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\OutputSymbols\\"),
        outputSymbolName[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_OUTPUT],
        showOutputSymbol[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_OUTPUT]
        );
      if(showOutputSymbol[nID - IDC_XABSL_DIALOG_CONTEXT_MENU_OUTPUT]) numberOfShownInputAndOutputSymbols++;
      else numberOfShownInputAndOutputSymbols--;
      sendDebugRequest();
    }
  }
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  
  for (i=0; i < (int)inputSymbolMenu->GetMenuItemCount(); i++)
  {
    currentSubMenu = inputSymbolMenu->GetSubMenu(i);
    if (currentSubMenu) delete currentSubMenu;
  }
}

bool CXabsl2DlgBar::handleMessage(InMessage& message)
{
  int i;
  if (message.getMessageID() == idXabsl2DebugMessage)
  {
    //clear option activation path
    activeOptionName.RemoveAll();
    activeOptionDurationOfActivation.RemoveAll();
    activeOptionActiveState.RemoveAll();
    activeOptionDurationOfStateActivation.RemoveAll();
    activeOptionNumberOfParameters.RemoveAll();
    for(i = 0; i < 20; i++)
    {
      activeOptionParameterName[i].RemoveAll();
      activeOptionParameterValue[i].RemoveAll();
    }
    
    for(i = 0; i < 20; i++)
    {
      activeBasicBehaviorParameterName[i].Empty();
      activeBasicBehaviorParameterValue[i].Empty();
    }
    
    //
    char behaviorControlID;
    SolutionRequest::xabsl2EngineID newEngineID;
    message.bin >> behaviorControlID;
    newEngineID= (SolutionRequest::xabsl2EngineID)behaviorControlID;
    if(newEngineID != currentEngineID) 
    {
      clear();
      currentEngineID = newEngineID;
      selectedBehaviorControlName = 
        SolutionRequest::getXabsl2EngineIDName(currentEngineID);
      reload();
    }
    else
    {
      currentEngineID = newEngineID;
    }
    message.bin >> selectedAgentName.GetBuffer(100);
    
    int z; 
    char name[300], valueString[300];
    double valueDouble;
    char valueChar;
    unsigned long valueUnsignedLong;
    // watched decimal input symbols:
    message.bin >> z; 
    for (i = 0; i < z; i++)
    {
      message.bin >> name;
      message.bin >> valueDouble;
      inputSymbolValue[inputSymbolIndex[name]].Format("%.2f", valueDouble);
    }
    
    // watched boolean input symbols:
    message.bin >> z; 
    for (i = 0; i < z; i++)
    {
      message.bin >> name;
      message.bin >> valueChar;
      if(valueChar == 0) inputSymbolValue[inputSymbolIndex[name]] = "false";
      else inputSymbolValue[inputSymbolIndex[name]] = "true";
    }
    
    // watched enumerated input symbols:
    message.bin >> z; 
    for (i = 0; i < z; i++)
    {
      message.bin >> name;
      message.bin >> valueString;
      
      CString shortValueString = valueString;
      int endOfPrefix = shortValueString.Find('.');  
      shortValueString.Delete(0,endOfPrefix + 1);
      
      inputSymbolValue[inputSymbolIndex[name]] = shortValueString;
    }
    
    // watched enumerated output symbols:
    message.bin >> z; 
    for (i = 0; i < z; i++)
    {
      message.bin >> name;
      message.bin >> valueString;
      
      CString shortValueString = valueString;
      int endOfPrefix = shortValueString.Find('.');  
      shortValueString.Delete(0,endOfPrefix + 1);
      
      outputSymbolValue[outputSymbolIndex[name]] = shortValueString;
    }
    
    //option activation path:
    CString string;
    message.bin >> numberOfActiveOptions;
    for(i = 0; i < numberOfActiveOptions; i++)
    { 
      message.bin >> name;
      message.bin >> valueUnsignedLong;
      string.Format("%.1f", valueUnsignedLong / 1000.0);
      activeOptionName.Add(name);
      activeOptionDurationOfActivation.Add(string);
      
      message.bin >> name;
      message.bin >> valueUnsignedLong;
      string.Format("%.1f", valueUnsignedLong / 1000.0);
      activeOptionActiveState.Add(name);
      activeOptionDurationOfStateActivation.Add(string);
      
      message.bin >> valueChar;
      activeOptionNumberOfParameters.Add(valueChar);
      
      for(int j = 0; j < 20; j++)
      {
        if(j < valueChar)
        {
          message.bin >> name;
          activeOptionParameterName[j].Add(name);
          message.bin >> valueDouble;
          string.Format("%.2f", valueDouble);
          activeOptionParameterValue[j].Add(string);
        }
        else
        {
          activeOptionParameterName[j].Add("empty");
          activeOptionParameterValue[j].Add("empty");
        }
      }
    }
    
    //active basic behavior:
    message.bin >> name;
    activeBasicBehaviorName = name;
    message.bin >> activeBasicBehaviorNumberOfParameters;
    for(int j = 0; j < activeBasicBehaviorNumberOfParameters; j++)
    {
      message.bin >> name;
      activeBasicBehaviorParameterName[j] = name;
      message.bin >> valueDouble;
      string.Format("%.2f", valueDouble);
      activeBasicBehaviorParameterValue[j] = string;
    }
    
    //generated action;
    message.bin >> name;
    generatedAction = name;
    
    
    if (this->IsWindowVisible())
    {
      RedrawWindow(NULL, NULL, RDW_INVALIDATE);
    }
    return true;
  }
  return 0;
}

void CXabsl2DlgBar::drawText
(
 FontType fontType,
 int line, 
 int column, 
 CString string)
{
  dcOffScreen.SelectObject(&boldFont);
  switch(fontType)
  {
  case normal:
    dcOffScreen.SelectObject(&normalFont);
    break;
  case italic:
    dcOffScreen.SelectObject(&italicFont);
    break;
  case bold:
    dcOffScreen.SelectObject(&boldFont);
    break;
  default:
    dcOffScreen.SelectObject(&normalFont);
    break;
  }
  CBrush backGroundBrush(RGB(255,255,255));
  dcOffScreen.FillRect(
    CRect(leftOfColumnPosition[column] - 5,
    currentRect.top + line * lineHeight,
    currentRect.right,
    currentRect.top + line * lineHeight + lineHeight), 
    &backGroundBrush);
  dcOffScreen.TextOut(leftOfColumnPosition[column], currentRect.top + line * lineHeight, string);
}

void CXabsl2DlgBar::drawHorizontalLines()
{
  for(int i = 0; i < horizontalLineList.GetSize(); i++)
  {
    dcOffScreen.MoveTo(horizontalSpacing, (horizontalLineList[i] + 1) * lineHeight);
    dcOffScreen.LineTo(currentRect.right - horizontalSpacing, (horizontalLineList[i] + 1) * lineHeight);
  }
}

void CXabsl2DlgBar::drawVerticalLines()
{
  for(int i = 0; i < verticalLineList.GetSize(); i++)
  {
    dcOffScreen.MoveTo(
      leftOfColumnPosition[verticalLineList[i].column] - 5, 
      (verticalLineList[i].begin + 1) * lineHeight);
    
    dcOffScreen.LineTo(
      leftOfColumnPosition[verticalLineList[i].column] - 5, 
      (verticalLineList[i].end + 1) * lineHeight);
  }
}

void CXabsl2DlgBar::drawTextFixedX
(
 FontType fontType,
 int line, 
 int x, 
 CString string)
{
  dcOffScreen.SelectObject(&boldFont);
  switch(fontType)
  {
  case normal:
    dcOffScreen.SelectObject(&normalFont);
    break;
  case italic:
    dcOffScreen.SelectObject(&italicFont);
    break;
  case bold:
    dcOffScreen.SelectObject(&boldFont);
    break;
  default:
    dcOffScreen.SelectObject(&normalFont);
    break;
  }
  CBrush backGroundBrush(RGB(255,255,255));
  dcOffScreen.FillRect(
    CRect(x,
    currentRect.top + line * lineHeight,
    currentRect.right,
    currentRect.top + line * lineHeight + lineHeight), 
    &backGroundBrush);
  dcOffScreen.TextOut(horizontalSpacing + x, currentRect.top + line * lineHeight, string);
}

void CXabsl2DlgBar::createFonts()
{
  normalFont.CreateFont(14,0,0,0,FW_NORMAL,FALSE,FALSE,0,ANSI_CHARSET,
    OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,
    DEFAULT_PITCH | FF_SWISS,"Arial");
  
  italicFont.CreateFont(14,0,0,0,FW_NORMAL,TRUE,FALSE,0,ANSI_CHARSET,
    OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,
    DEFAULT_PITCH | FF_SWISS,"Arial");
  
  boldFont.CreateFont(14,0,0,0,FW_BOLD,FALSE,FALSE,0,ANSI_CHARSET,
    OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,
    DEFAULT_PITCH | FF_SWISS,"Arial");
  
  oldFont = dcOffScreen.SelectObject(&boldFont);
}

void CXabsl2DlgBar::deleteFonts()
{
  dcOffScreen.SelectObject(oldFont);
  boldFont.DeleteObject();
  normalFont.DeleteObject();
  italicFont.DeleteObject();
}


void CXabsl2DlgBar::countNumberOfShownInputAndOutputSymbols()
{
  numberOfShownInputSymbols = 0;
  numberOfShownOutputSymbols = 0;
  
  int i;
  for(i = 0; i < numberOfInputSymbols; i++)
    if(showInputSymbol[i]) numberOfShownInputSymbols++;
    
    for(i = 0; i < numberOfOutputSymbols; i++)
      if(showOutputSymbol[i]) numberOfShownOutputSymbols++;

  numberOfShownInputAndOutputSymbols = numberOfShownInputSymbols + numberOfShownOutputSymbols; 
}

void CXabsl2DlgBar::clear()
{
	m_OutputSymbolValueCombo.ResetContent();
	m_OutputSymbolCombo.ResetContent();
	m_OptionsCombo.ResetContent();
  currentSelection = nothing;

  int i;
  for(i = 0; i < numberOfOutputSymbols; i++)
  {
    delete outputSymbolAlternatives[i];
  }
  for(i = 0; i < numberOfOptions; i++)
  {
    delete optionParameterNames[i];
  }
  for(i = 0; i < numberOfBasicBehaviors; i++)
  {
    delete basicBehaviorParameterNames[i];
  }
  for(i = 0; i < 20; i++)
  {
    activeOptionParameterName[i].RemoveAll();
    activeOptionParameterValue[i].RemoveAll();
  }
  for(i = 0; i < 20; i++)
  {
    activeBasicBehaviorParameterName[i].Empty();
    activeBasicBehaviorParameterValue[i].Empty();
  }
    
  inputSymbolIndex.RemoveAll();
  outputSymbolIndex.RemoveAll();

  numberOfInputSymbols = 0;
  numberOfOutputSymbols = 0;
  numberOfOptions = 0;
  numberOfBasicBehaviors = 0;

  numberOfShownInputAndOutputSymbols = 0;
  numberOfShownInputSymbols = 0;
  numberOfShownOutputSymbols = 0;

  inputSymbolName.RemoveAll();
  inputSymbolValue.RemoveAll();

  typeOfInputSymbol.RemoveAll();
  showInputSymbol.RemoveAll();

  outputSymbolName.RemoveAll();
  outputSymbolValue.RemoveAll();
  showOutputSymbol.RemoveAll();

  numberOfOutputSymbolAlternatives.RemoveAll();
  outputSymbolAlternatives.RemoveAll();

  selectedOutputSymbolAlternative.RemoveAll();
 
  optionName.RemoveAll();
  numberOfOptionParameters.RemoveAll();
  optionParameterNames.RemoveAll();

  basicBehaviorName.RemoveAll();
  numberOfBasicBehaviorParameters.RemoveAll();
  basicBehaviorParameterNames.RemoveAll();


  currentEngineID = SolutionRequest::undefined;

  selectedBehaviorControlName.Empty();
  selectedAgentName.Empty();

  numberOfActiveOptions = 0;
  activeOptionName.RemoveAll();
  activeOptionDurationOfActivation.RemoveAll();
  activeOptionActiveState.RemoveAll();
  activeOptionDurationOfStateActivation.RemoveAll();
  activeOptionNumberOfParameters.RemoveAll();

  activeBasicBehaviorName.Empty();
  activeBasicBehaviorDurationOfActivation.Empty();
  activeBasicBehaviorNumberOfParameters = 0;

  generatedAction.Empty();
}

void CXabsl2DlgBar::reload()
{
  if(currentEngineID != SolutionRequest::undefined)
  {
    CString filename = "Xabsl2/"; 
    filename.Insert(1000, SolutionRequest::getXabsl2EngineFileID(currentEngineID) );
    filename.Insert(1000, "-ds.dat");
    
    InConfigFile fin(filename);
    if(fin.exists())
    {
      char symbol[200];
      char type[2];
      int i,j;
      
      fin >> numberOfInputSymbols;
      for(i = 0; i < numberOfInputSymbols; i++)
      {
        fin >> type;
        switch(type[0])
        {
        case 'd':
          typeOfInputSymbol.Add(decimal);
          break;
        case 'b':
          typeOfInputSymbol.Add(boolean);
          break;
        case 'e':
          typeOfInputSymbol.Add(enumerated);
          break;
        }
        fin >> symbol;
        inputSymbolName.Add(symbol);
        inputSymbolValue.Add("0");
        CString folderString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\InputSymbols\\");
        showInputSymbol.Add(
          AfxGetApp()->GetProfileInt(
          folderString,
          symbol, 0) != 0  );
        if(showInputSymbol[i]) numberOfShownInputAndOutputSymbols++;
        inputSymbolIndex[symbol] = i;
      }
      
      fin >> numberOfOutputSymbols;
      for(i = 0; i < numberOfOutputSymbols; i++)
      {
        fin >> symbol;
        outputSymbolName.Add(symbol);
        outputSymbolValue.Add("0");
        showOutputSymbol.Add(
          AfxGetApp()->GetProfileInt(
          CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\OutputSymbols\\"), symbol, 0) != 0  );
        if(showOutputSymbol[i]) numberOfShownInputAndOutputSymbols++;
        outputSymbolIndex[symbol] = i;
        
        int z;
        fin >> z;
        numberOfOutputSymbolAlternatives.Add(z);
        
        char value[200];
        CStringArray* alternatives = new CStringArray();
        for(j = 0; j < numberOfOutputSymbolAlternatives[i]; j++)
        {
          fin >> value;
          alternatives->Add(value);
        }
        outputSymbolAlternatives.Add(alternatives);
        selectedOutputSymbolAlternative.Add(-1); // "unchanged"
      }
      
      fin >> numberOfOptions;
      char option[200];
      for(i = 0; i < numberOfOptions; i++)
      {
        fin >> option;
        optionName.Add(option);
        int z;
        fin >> z;
        numberOfOptionParameters.Add(z);
        char value[200];
        CStringArray* parameters = new CStringArray();
        for(j = 0; j < numberOfOptionParameters[i]; j++)
        {
          fin >> value;
          parameters->Add(value);
        }
        optionParameterNames.Add(parameters);
      }
      fin >> numberOfBasicBehaviors;
      char basicBehavior[200];
      for(i = 0; i < numberOfBasicBehaviors; i++)
      {
        fin >> basicBehavior;
        basicBehaviorName.Add(basicBehavior);
        int z;
        fin >> z;
        numberOfBasicBehaviorParameters.Add(z);
        char value[200];
        CStringArray* parameters = new CStringArray();
        for(j = 0; j < numberOfBasicBehaviorParameters[i]; j++)
        {
          fin >> value;
          parameters->Add(value);
        }
        basicBehaviorParameterNames.Add(parameters);
      }
    }
    else
    {
      getQueues().toGUI.out.text << filename << " could not be opened.";
      getQueues().toGUI.out.finishMessage(idText);
    }
    
    //GUI
    int i;
    
    for(i = 0; i < numberOfOutputSymbols; i++)
    {
      m_OutputSymbolCombo.AddString(outputSymbolName[i]);
    }
    
    m_OptionsCombo.AddString("unchanged");
    for(i = 0; i < numberOfOptions; i++)
    {
      m_OptionsCombo.AddString(optionName[i]);
    }
    m_OptionsCombo.AddString("---------------");
    for(i = 0; i < numberOfBasicBehaviors; i++)
    {
      m_OptionsCombo.AddString(basicBehaviorName[i]);
    }
    m_OutputSymbolCombo.SetCurSel(0);
    m_OptionsCombo.SetCurSel(0);
    OnSelchangeXabsl2OutputSymbolCombo();
    OnSelchangeXabsl2OptionsCombo();

    countNumberOfShownInputAndOutputSymbols();

    sendDebugRequest();

  }// if(currentEngineID != SolutionRequest::undefined)
}

void CXabsl2DlgBar::OnXabsl2TestOnRobotCheck() 
{
  sendDebugRequest();
}

void CXabsl2DlgBar::OnXabsl2TestHeadcontrolCheck() 
{
  sendDebugRequest();
}

void CXabsl2DlgBar::OnXabsl2SendButton() 
{
  sendDebugRequest();
}

void CXabsl2DlgBar::OnSelchangeXabsl2OptionsCombo() 
{
  int i = m_OptionsCombo.GetCurSel() - 1;
  currentSelection = nothing;
  
  if(0 <= i && i < numberOfOptions) currentSelection = option;
  if((numberOfOptions + 1) <= i && i < (numberOfOptions + numberOfBasicBehaviors + 1) )
  {
    currentSelection = basicBehavior;
    i -= (numberOfOptions + 1);
  }
  
  CString parameterString = "Parameter(s): ";
  CString valueString[6] = {"","","","","",""};
  
  if(currentSelection == option)
  {
    for(int j = 0; j < min(numberOfOptionParameters[i],5); j++)
    {
      CString keyString;
      keyString.Format("Param%d",(j+1));
      
      CString paramName = optionParameterNames[i]->ElementAt(j);
      int endOfPrefix = paramName.Find('.');  
      paramName.Delete(0,endOfPrefix + 1);
      
      parameterString.Insert(1000, paramName+ " ");
      valueString[j + 1] = AfxGetApp()->GetProfileString(
        CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\Options\\") + optionName[i],
        keyString, 
        "0"); 
    }
  }// if(currentSelection == option)
  
  if(currentSelection == basicBehavior)
  {
    for(int j = 0; j < min(numberOfBasicBehaviorParameters[i],5); j++)
    {
      CString keyString;
      keyString.Format("Param%d",(j+1));
      
      CString paramName = basicBehaviorParameterNames[i]->ElementAt(j);
      int endOfPrefix = paramName.Find('.');  
      paramName.Delete(0,endOfPrefix + 1);
      parameterString.Insert(1000, paramName);
      if (j < numberOfBasicBehaviorParameters[i] - 1)
        parameterString.Insert(1000, ", ");
      
      valueString[j + 1] = AfxGetApp()->GetProfileString(
        CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\BasicBehaviors\\") + basicBehaviorName[i], 
        keyString, 
        "0"); 
    }
  } // if(currentSelection == basicBehavior)
  
  m_Param1Edit.SetWindowText(valueString[1]);
  m_Param2Edit.SetWindowText(valueString[2]);
  m_Param3Edit.SetWindowText(valueString[3]);
  m_Param4Edit.SetWindowText(valueString[4]);
  m_Param5Edit.SetWindowText(valueString[5]);
  
  if(parameterString == "Parameter(s): ")  
  {
    if(currentSelection == basicBehavior)
      parameterString = "(no parameters available for basic behavior)";
    else parameterString = "(option selected - no parameters available)";
  }
  
  
  m_ParameterStatic.SetWindowText(parameterString);
  enableButtons();
  sendDebugRequest();
}

void CXabsl2DlgBar::OnSelchangeXabsl2OutputSymbolCombo() 
{
  int i = m_OutputSymbolCombo.GetCurSel();
  if (i < 0) return;
  m_OutputSymbolValueCombo.ResetContent();
  
  m_OutputSymbolValueCombo.AddString("unchanged");
  for(int j = 0; j < numberOfOutputSymbolAlternatives[i]; j++)
  {
    m_OutputSymbolValueCombo.AddString(outputSymbolAlternatives[i]->ElementAt(j));
  }
  m_OutputSymbolValueCombo.SetCurSel(selectedOutputSymbolAlternative[i] + 1);
}

void CXabsl2DlgBar::OnSelchangeXabsl2OutputSymbolValueCombo() 
{
  int symbolSelection = m_OutputSymbolCombo.GetCurSel();
  int valueSelection  = m_OutputSymbolValueCombo.GetCurSel();
  selectedOutputSymbolAlternative[symbolSelection] = valueSelection - 1;
  sendDebugRequest();
}

void CXabsl2DlgBar::enableButtons()
{
  switch(currentSelection)
  {
  case option:
    {
      int option = m_OptionsCombo.GetCurSel() - 1;
      enableParam1Edit = numberOfOptionParameters[option] > 0;
      enableParam2Edit = numberOfOptionParameters[option] > 1;
      enableParam3Edit = numberOfOptionParameters[option] > 2;
      enableParam4Edit = numberOfOptionParameters[option] > 3;
      enableParam5Edit = numberOfOptionParameters[option] > 4;
    }
    break;
  case basicBehavior:
    {
      int basicBehavior = m_OptionsCombo.GetCurSel() - 1 - (numberOfOptions + 1);
      enableParam1Edit = numberOfBasicBehaviorParameters[basicBehavior] > 0;
      enableParam2Edit = numberOfBasicBehaviorParameters[basicBehavior] > 1;
      enableParam3Edit = numberOfBasicBehaviorParameters[basicBehavior] > 2;
      enableParam4Edit = numberOfBasicBehaviorParameters[basicBehavior] > 3;
      enableParam5Edit = numberOfBasicBehaviorParameters[basicBehavior] > 4;
    }
    break;
  default:
    enableParam1Edit = false;
    enableParam2Edit = false;
    enableParam3Edit = false;
    enableParam4Edit = false;
    enableParam5Edit = false;
  }
}

void CXabsl2DlgBar::sendDebugRequest()
{
  CString valueString[6] = {"","","","","",""};
  m_Param1Edit.GetWindowText(valueString[1]); 
  m_Param2Edit.GetWindowText(valueString[2]);
  m_Param3Edit.GetWindowText(valueString[3]);
  m_Param4Edit.GetWindowText(valueString[4]);
  m_Param5Edit.GetWindowText(valueString[5]);
  
  int i = m_OptionsCombo.GetCurSel() - 1;
  
  if(currentSelection == option)
  {
    for(int j = 0; j < numberOfOptionParameters[i]; j++)
    {
      CString keyString;
      keyString.Format("Param%d",(j+1));
      AfxGetApp()->WriteProfileString(
        CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + 
        "\\Options\\") + optionName[i],
        keyString, 
        valueString[j + 1]); 
    }
  }
  if(currentSelection == basicBehavior)
  {
    i -= (numberOfOptions + 1);
    for(int j = 0; j < numberOfBasicBehaviorParameters[i]; j++)
    {
      CString keyString;
      keyString.Format("Param%d",(j+1));
      AfxGetApp()->WriteProfileString(
        CString("Xabsl2BehaviorTester\\" + selectedBehaviorControlName + "\\BasicBehaviors\\") + basicBehaviorName[i], 
        keyString, 
        valueString[j + 1]); 
    }
  }
  
  getDebugKeyTables().forSimulatedRobots.set(
    DebugKeyTable::sendXabsl2DebugMessagesForBehaviorControl,
    !m_TestOnRobotCheck.GetCheck() && !m_testHeadControlCheck.GetCheck() ? DebugKey::every_n_ms : DebugKey::disabled, 100);

  getDebugKeyTables().forSimulatedRobots.set(
    DebugKeyTable::sendXabsl2DebugMessagesForHeadControl,
    !m_TestOnRobotCheck.GetCheck() && m_testHeadControlCheck.GetCheck() ? DebugKey::every_n_ms : DebugKey::disabled, 100);

  getDebugKeyTables().forPhysicalRobots.set(
    DebugKeyTable::sendXabsl2DebugMessagesForBehaviorControl,
    m_TestOnRobotCheck.GetCheck() && !m_testHeadControlCheck.GetCheck() ? DebugKey::every_n_ms : DebugKey::disabled, 100);

  getDebugKeyTables().forPhysicalRobots.set(
    DebugKeyTable::sendXabsl2DebugMessagesForHeadControl,
    m_TestOnRobotCheck.GetCheck() && m_testHeadControlCheck.GetCheck() ? DebugKey::every_n_ms : DebugKey::disabled, 100);
  
  getDebugKeyTables().sendForPhysicalRobots();
  getDebugKeyTables().sendForSimulatedRobots();
  
  
  OutMessage& out = m_TestOnRobotCheck.GetCheck()==1
    ?getQueues().toPhysical.selectedRobot.out
    :getQueues().toSimulated.selectedRobot.out;
  
  countNumberOfShownInputAndOutputSymbols();
  
  out.bin << (char)currentEngineID;

  out.bin << numberOfShownInputSymbols;

  for(i = 0; i < numberOfInputSymbols; i++)
  {
    if(showInputSymbol[i])
    {
      switch(typeOfInputSymbol[i])
      {
      case decimal: out.bin << 'd'; break;
      case boolean: out.bin << 'b'; break;
      case enumerated: out.bin << 'e'; break;
      }
      out.bin << inputSymbolName[i];
    } //if(showInputSymbol[i])
  } //for(i = 0; i < numberOfInputSymbols; i++)
  
  out.bin << numberOfShownOutputSymbols;
  for(i = 0; i < numberOfOutputSymbols; i++)
  {
    if(showOutputSymbol[i])
    {
      out.bin << outputSymbolName[i];
    } //if(showOutputSymbol[i])
  } //for(i = 0; i < numberOfOutputSymbols; i++)
  
  // test b: basic behavior, o: option, d: default
  switch (currentSelection)
  {
  case basicBehavior:
    {
      int index = m_OptionsCombo.GetCurSel() - 1 - (numberOfOptions + 1);
      out.bin << (char)'b' << basicBehaviorName[index];
      out.bin << (char)min(numberOfBasicBehaviorParameters[index],5);
      for(int k = 0; k < min(numberOfBasicBehaviorParameters[index],5); k++)
      {
        out.bin << basicBehaviorParameterNames[index]->ElementAt(k);
        float value;
        sscanf(valueString[k+1].GetBuffer(valueString[k+1].GetLength()), "%f", &value);
        out.bin << (double)value;
      }
      break;
    }
  case option:
    {
      int index = m_OptionsCombo.GetCurSel() - 1;
      out.bin << (char)'o' << optionName[index];
      out.bin << (char)min(numberOfOptionParameters[index],5);
      for(int k = 0; k < min(numberOfOptionParameters[index],5); k++)
      {
        out.bin << optionParameterNames[index]->ElementAt(k);
        float value;
        sscanf(valueString[k+1].GetBuffer(valueString[k+1].GetLength()), "%f", &value);
        out.bin << (double)value;
      }
      break;
    }
  case nothing:
    out.bin << (char)'d';
    break;
  }
  
  int numberOfChangedOutputSymbols = 0;
  for(i = 0; i < numberOfOutputSymbols; i++)
  {
    if(selectedOutputSymbolAlternative[i] != -1) numberOfChangedOutputSymbols ++;
  }
  //changed output symbols
  out.bin << (char)numberOfChangedOutputSymbols;
  for(int k = 0; k < numberOfOutputSymbols; k++)
  {
    if(selectedOutputSymbolAlternative[k] != -1)
    {
      CString s = outputSymbolName[k];
      out.bin << s.GetBuffer(s.GetLength());
      s = outputSymbolAlternatives[k]->ElementAt(selectedOutputSymbolAlternative[k]);
      out.bin << s.GetBuffer(s.GetLength());
      //      out.bin << selectedOutputSymbolAlternative;
    }
  }
  
  out.finishMessage(idXabsl2DebugRequest);
}

void CXabsl2DlgBar::sendIntermediateCode()
{
  CString filename = "Xabsl2/";
  filename.Insert(1000, SolutionRequest::getXabsl2EngineFileID(currentEngineID) );
  filename.Insert(1000, "-ic.dat");
  
  char s[500000];
  memset(s,0,500000);
  InBinaryFile* f = new InBinaryFile(filename);
  if(f->exists())
  {
    f->read(s,500000);
    delete(f);
    
    getQueues().toPhysical.selectedRobot.out.bin << (char)currentEngineID;
    getQueues().toPhysical.selectedRobot.out.bin.write(s,(int)strlen(s));
    getQueues().toPhysical.selectedRobot.out.finishMessage(idXabsl2IntermediateCode);

    getQueues().toSimulated.selectedRobot.out.bin << (char)currentEngineID;
    getQueues().toSimulated.selectedRobot.out.bin.write(s,(int)strlen(s));
    getQueues().toSimulated.selectedRobot.out.finishMessage(idXabsl2IntermediateCode);
  }
  else
  {
    getQueues().toGUI.out.text << filename << " could not be opened.";
    getQueues().toGUI.out.finishMessage(idText);
  }
}

void CXabsl2DlgBar::updateUI(CCmdUI* pCmdUI)
{
  switch(pCmdUI->m_nID)
  {
  case IDC_XABSL2_PARAM_EDIT1:
    pCmdUI->Enable(enableParam1Edit);
    return;
  case IDC_XABSL2_PARAM_EDIT2:
    pCmdUI->Enable(enableParam2Edit);
    return;
  case IDC_XABSL2_PARAM_EDIT3:
    pCmdUI->Enable(enableParam3Edit);
    return;
  case IDC_XABSL2_PARAM_EDIT4:
    pCmdUI->Enable(enableParam4Edit);
    return;
  case IDC_XABSL2_PARAM_EDIT5:
    pCmdUI->Enable(enableParam5Edit);
    return;
  default:
    return;
  }
}

/*
* Change log :
* 
* $Log: Xabsl2DlgBar.cpp,v $
* Revision 1.9  2004/06/08 17:42:51  jhoffman
* - xabsl dialog only requests debug keys to be send every 100 ms
* - added comments
* - look-at-ball misbehavior removed
*
* Revision 1.8  2004/05/25 19:43:33  jhoffman
* bug fix
*
* Revision 1.7  2004/05/25 17:33:32  loetzsch
* bug fix
*
* Revision 1.6  2004/05/25 15:57:09  loetzsch
* bug fix
*
* Revision 1.5  2004/05/25 14:40:49  loetzsch
* bug fix
*
* Revision 1.12  2004/05/18 14:28:21  loetzsch
* now Xabsl2 intermediate code can be sent to different modules
*
* Revision 1.11  2004/05/17 18:35:23  loetzsch
* continued support for multiple Xabsl engines in different modules
*
* Revision 1.10  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.9  2004/03/25 14:48:30  loetzsch
* short enum elment names
*
* Revision 1.8  2004/03/16 14:00:23  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.3  2004/03/11 17:32:52  loetzsch
* fixed some bugs related to the restriction for the number of "setable" option parameters
*
* Revision 1.2  2004/03/08 18:59:46  loetzsch
* now allows for up to 10 option parameters
*
* Revision 1.7  2004/03/01 10:03:31  jhoffman
* truncating prefix of basic behavior name to fit it in the width of the dialog, default messages for no paramters added
*
* Revision 1.6  2004/02/23 21:35:23  jhoffman
* - message handling bug removed
* - added camera toolbar functionality to menu
*
* Revision 1.5  2003/12/09 19:49:26  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.4  2003/12/06 06:31:20  loetzsch
* no message
*
* Revision 1.3  2003/11/30 01:53:20  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.2  2003/11/06 18:00:01  loetzsch
* fixed a bug in the ::OnPaint() method
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:40:13  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.27  2003/06/25 11:49:22  risler
* submenus added to input symbol context menu
* added clear all to context menus
*
* Revision 1.26  2003/06/19 14:58:40  risler
* input symbol menu size increased
*
* Revision 1.25  2003/05/11 23:46:36  dueffert
* Depend now works with RobotControl too
*
* Revision 1.24  2003/03/23 19:11:50  loetzsch
* OUTPUT not allowed in the RobotControl thread anymore.
* Use getQueues().toGUI.out instead.
*
* Revision 1.23  2003/03/15 22:23:33  juengel
* Bug removed.
*
* Revision 1.22  2003/03/05 17:09:28  loetzsch
* redesign of the queues and debug key tables in RobotControl
*
* Revision 1.21  2003/02/27 12:01:24  dueffert
* magic IDs changed to avoid conflicts
*
* Revision 1.20  2003/01/29 10:36:07  juengel
* Xabsl2Dialog sends the selected option/basic behavior, the parameters,
* and the output symbols to be changed.
*
* Revision 1.19  2003/01/26 19:39:16  loetzsch
* removed a useless '+' symbol.
*
* Revision 1.18  2003/01/22 18:09:23  loetzsch
* continued implementation of sending of intermediate code from xabsl2 dialog
* to a xabsl2 behavior control
*
* Revision 1.17  2003/01/22 13:09:51  juengel
* reload() is called, if behaviorControlId changes
* intermediate code is sent, if reload is selected in context menu.
*
* Revision 1.16  2003/01/20 08:14:16  loetzsch
* added xabsl2BehaviorControlID to the solutionRequest to distinct between the
* different Xabsl2BehaviorControls
*
* Revision 1.15  2003/01/15 14:54:04  juengel
* Watched enumerated input symbols and watched output symbols are displayed.
*
* Revision 1.14  2003/01/15 10:08:02  juengel
* Parameters are stored in registry.
* Added variables for "SetOutputSymbol".
*
* Revision 1.13  2003/01/15 08:50:20  juengel
* Corrected output of option parameters.
*
* Revision 1.12  2003/01/13 22:37:11  loetzsch
* matthias removed some code in ::OnPaint() because it crashed
*
* Revision 1.11  2003/01/13 18:28:44  juengel
* Layout changed.
*
* Revision 1.10  2003/01/13 10:31:43  juengel
* XabslDebugMessage extended. Xabsl2Dialog shows option activation path.
*
* Revision 1.9  2003/01/12 14:53:27  loetzsch
* sending of debug mode (test basic behaviors or options) added
*
* Revision 1.8  2003/01/11 21:58:11  roefer
* Unused variables removed
*
* Revision 1.7  2003/01/11 18:38:38  juengel
* Xabsl2ehavior sends Xabsl2DebugMessages now.
*
* Revision 1.6  2003/01/11 13:29:55  juengel
* Xabsl2Dialog sends Xabsl2DebugRequest now.
*
* Revision 1.5  2003/01/10 15:50:10  juengel
* Added OptionsCombo to Xabsl2DialogBar
*
* Revision 1.4  2003/01/08 18:57:29  juengel
* Added Controls for the Xabsl2 dialog.
*
* Revision 1.3  2002/12/09 09:50:14  juengel
* input and output symbols are stored in a CArray now
*
* Revision 1.2  2002/12/08 16:18:40  juengel
* Added method drawText to xabsl2 dialog.
* Xabsl2 dialog loads and shows input and output symbols.
* Added context menu for xabsl2 dialog.
*
* Revision 1.1  2002/12/06 11:30:13  juengel
* Added Xabsl2Dlg.
*
*/



