/**
* @file ValueHistoryDlgBar.cpp
*
* Implementation of class CValueHistoryDlgBar
*
* @author Martin Ltzsch
* @author Matthias Jngel
*/

#include "StdAfx.h"
#include "ValueHistoryDlgBar.h"

#include "Tools/Streams/OutStreams.h"
#include "Tools/Player.h"

#include "Representations/Perception/BodyPosture.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/PSDPercept.h"
#include "Representations/Perception/CollisionPercept.h"
#include "Representations/Perception/SpecialPercept.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/EdgesPercept.h"
#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/PlayersPercept.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/RobotState.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Representations/Motion/MotionInfo.h"

CValueHistoryDlgBar::CValueHistoryDlgBar()
: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CValueHistoryDlgBar)
  //}}AFX_DATA_INIT
  bmpOffScreen = NULL;
  oldBitmap = NULL;
  freeze= false;
}

CValueHistoryDlgBar::~CValueHistoryDlgBar()
{
  AfxGetApp()->WriteProfileInt("ValueHistory","scale",(int)(100*scale));
  AfxGetApp()->WriteProfileInt("ValueHistory","timeRange",timeRange);
  for (int i=0; i <numberOfValueIDs;i++)
    (AfxGetApp()->WriteProfileInt("ValueHistory",CString("view ") + getValueIDName((valueID)i),(int)displayValue[i]));
  
  if(bmpOffScreen)
  {
    dcOffScreen.SelectObject(oldBitmap);
    bmpOffScreen->DeleteObject();
    delete(bmpOffScreen);
  }
}


void CValueHistoryDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CValueHistoryDlgBar)
	DDX_Control(pDX, IDC_VALUE_HISTORY_SCALE_SLIDER, m_scaleSlider);
  DDX_Control(pDX, IDC_VALUE_HISTORY_TIME_SLIDER, m_timeRangeSlider);
  DDX_Control(pDX, IDC_VALUE_HISTORY_TIME_STATIC, m_timeRangeStatic);
  DDX_Control(pDX, IDC_VALUE_HISTORY_FREEZE_BUTTON, m_freezeButton);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CValueHistoryDlgBar, CDynamicBarDlg)
//{{AFX_MSG_MAP(CValueHistoryDlgBar)
ON_WM_PAINT()
ON_WM_SIZE()
ON_WM_CONTEXTMENU()
ON_WM_HSCROLL()
ON_COMMAND(IDC_VALUE_HISTORY_FREEZE_BUTTON,OnFreezeButton)
//}}AFX_MSG_MAP
END_MESSAGE_MAP()


BOOL CValueHistoryDlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();
  
  scale = 0.01*AfxGetApp()->GetProfileInt("ValueHistory","scale",100);
  timeRange = AfxGetApp()->GetProfileInt("ValueHistory","timeRange",10);
  m_timeRangeSlider.SetRange(1,60);
  
  char text[100];
  sprintf(text,"range: %i s",timeRange);
  m_timeRangeStatic.SetWindowText(text);
  
  m_scaleSlider.SetPos((int)(100*scale));
  m_timeRangeSlider.SetPos(timeRange);  
  AddSzControl(m_timeRangeSlider,mdResize,mdNone);  
  AddSzControl(m_scaleSlider,mdRepos,mdNone);  
  AddSzControl(m_timeRangeStatic,mdRepos,mdNone);  
  AddSzControl(m_freezeButton,mdRepos,mdNone);
  
  latestTimeStamp = 0;

  displayValue[ballPerceptDistance] = (AfxGetApp()->GetProfileInt("ValueHistory","view ballPercept.distance",1)==1?true:false);
  AfxGetApp()->WriteProfileInt("ValueHistory","view ballPercept.distance",(int)displayValue[ballPerceptDistance]);
  displayValue[ballPerceptAngle] = (AfxGetApp()->GetProfileInt("ValueHistory","view ballPercept.angle",1)==1?true:false);
  AfxGetApp()->WriteProfileInt("ValueHistory","view ballPercept.angle",(int)displayValue[ballPerceptAngle]);
  displayValue[robotPoseX] = (AfxGetApp()->GetProfileInt("ValueHistory","view robotPose.x",1)==1?true:false);
  AfxGetApp()->WriteProfileInt("ValueHistory","view robotPose.x",(int)displayValue[robotPoseX]);
  displayValue[robotPoseY] = (AfxGetApp()->GetProfileInt("ValueHistory","view robotPose.y",1)==1?true:false);
  AfxGetApp()->WriteProfileInt("ValueHistory","view robotPose.y",(int)displayValue[robotPoseY]);

  for (int i=0; i <numberOfValueIDs;i++)
  {
    displayValue[i]=
      (AfxGetApp()->GetProfileInt("ValueHistory",
      CString("view ") + getValueIDName((valueID)i),0) ==1?true:false);
  }
  paintStyles[cameraMatrixIsValid].penColor = RGB(0,255,0);
  paintStyles[cameraMatrixIsValid].penWidth = 3;
  paintStyles[cameraMatrixIsValid].shape = PaintMethodsWin32::PaintStyle::horizontalLine;
  paintStyles[cameraMatrixIsValid].scale = 0.1;

  paintStyles[ballPerceptX].brushColor = RGB(164,255,0);
  paintStyles[ballPerceptX].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[ballPerceptX].scale = 0.1;

  paintStyles[ballPerceptY].brushColor = RGB(255,255,0);
  paintStyles[ballPerceptY].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[ballPerceptY].scale = 0.1;

  paintStyles[ballPerceptDistance].brushColor = RGB(255,164,0);
  paintStyles[ballPerceptDistance].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[ballPerceptDistance].scale = 0.1;
  
  paintStyles[ballPerceptAngle].brushColor = RGB(255,30,100);
  paintStyles[ballPerceptAngle].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[ballPerceptAngle].scale = 0.9;

  paintStyles[ballPerceptDistanceBearingBased].brushColor = RGB(255,100,0);
  paintStyles[ballPerceptDistanceBearingBased].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[ballPerceptDistanceBearingBased].scale = 0.1;
  
  paintStyles[ballPerceptAngleBearingBased].brushColor = RGB(255,100,100);
  paintStyles[ballPerceptAngleBearingBased].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[ballPerceptAngleBearingBased].scale = 0.9;

  paintStyles[ballPerceptDistanceSizeBased].brushColor = RGB(255,30,0);
  paintStyles[ballPerceptDistanceSizeBased].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[ballPerceptDistanceSizeBased].scale = 0.1;
  
  paintStyles[ballPerceptAngleSizeBased].brushColor = RGB(255,164,100);
  paintStyles[ballPerceptAngleSizeBased].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[ballPerceptAngleSizeBased].scale = 0.9;
  
  paintStyles[ballPositionDistance].penColor = RGB(255,164,0);
  paintStyles[ballPositionDistance].brushColor = RGB(255,164,0);
  paintStyles[ballPositionDistance].shape = PaintMethodsWin32::PaintStyle::square;
  paintStyles[ballPositionDistance].scale = 0.1;
  
  paintStyles[ballPositionAngle].penColor = RGB(255,30,100);
  paintStyles[ballPositionAngle].shape = PaintMethodsWin32::PaintStyle::square;
  paintStyles[ballPositionAngle].scale = 0.9;

  paintStyles[ballPositionSpeedAbs].penColor = RGB(100,30,255);
  paintStyles[ballPositionSpeedAbs].shape = PaintMethodsWin32::PaintStyle::square;
  paintStyles[ballPositionSpeedAbs].scale = 0.1;

  paintStyles[ballPositionSpeedDir].penColor = RGB(255,100,0);
  paintStyles[ballPositionSpeedDir].shape = PaintMethodsWin32::PaintStyle::square;
  paintStyles[ballPositionSpeedDir].scale = 0.25;

  paintStyles[psdPerceptToFarAway].penColor = RGB(255,0,0);
  paintStyles[psdPerceptToFarAway].shape = PaintMethodsWin32::PaintStyle::horizontalLine;
  paintStyles[psdPerceptToFarAway].scale = 0.1;

  paintStyles[psdPerceptX].brushColor = RGB(255,164,164);
  paintStyles[psdPerceptX].shape = PaintMethodsWin32::PaintStyle::filledSquare;
  paintStyles[psdPerceptX].scale = 0.1;

  paintStyles[psdPerceptY].brushColor = RGB(164,255,164);
  paintStyles[psdPerceptY].shape = PaintMethodsWin32::PaintStyle::filledSquare;
  paintStyles[psdPerceptY].scale = 0.1;

  paintStyles[psdPerceptZ].brushColor = RGB(164,164,255);
  paintStyles[psdPerceptZ].shape = PaintMethodsWin32::PaintStyle::filledSquare;
  paintStyles[psdPerceptZ].scale = 0.1;

  paintStyles[robotPoseX].penColor = RGB(50,255,50);
  paintStyles[robotPoseX].shape = PaintMethodsWin32::PaintStyle::cross;
  paintStyles[robotPoseX].scale = 0.04;
  
  paintStyles[robotPoseY].penColor = RGB(50,50,255);
  paintStyles[robotPoseY].shape = PaintMethodsWin32::PaintStyle::cross;
  paintStyles[robotPoseY].scale = 0.04;

  paintStyles[robotPoseAngle].penColor = RGB(255,50,50);
  paintStyles[robotPoseAngle].shape = PaintMethodsWin32::PaintStyle::cross;
  paintStyles[robotPoseAngle].scale = 0.5;

  paintStyles[robotPoseSpeed].penColor = RGB(50,50,50);
  paintStyles[robotPoseSpeed].shape = PaintMethodsWin32::PaintStyle::cross;
  paintStyles[robotPoseSpeed].scale = 0.5;

  paintStyles[specialPerceptRobotPoseX].penColor = RGB(50,155,50);
  paintStyles[specialPerceptRobotPoseX].shape = PaintMethodsWin32::PaintStyle::vcross;
  paintStyles[specialPerceptRobotPoseX].scale = 0.04;
  
  paintStyles[specialPerceptRobotPoseY].penColor = RGB(50,50,155);
  paintStyles[specialPerceptRobotPoseY].shape = PaintMethodsWin32::PaintStyle::vcross;
  paintStyles[specialPerceptRobotPoseY].scale = 0.04;

  paintStyles[specialPerceptRobotPoseAngle].penColor = RGB(155,50,50);
  paintStyles[specialPerceptRobotPoseAngle].shape = PaintMethodsWin32::PaintStyle::vcross;
  paintStyles[specialPerceptRobotPoseAngle].scale = 0.5;

  paintStyles[walkSpeedX].brushColor = RGB(127,127,255);
  paintStyles[walkSpeedX].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[walkSpeedX].scale = 0.3;

  paintStyles[walkSpeedY].brushColor = RGB(255,255,0);
  paintStyles[walkSpeedY].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[walkSpeedY].scale = 0.3;
  
  paintStyles[walkSpeedRotation].brushColor = RGB(255,0,0);
  paintStyles[walkSpeedRotation].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[walkSpeedRotation].scale = 60;

  paintStyles[positionInWalkCycle].brushColor = RGB(255,127,65);
  paintStyles[positionInWalkCycle].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[positionInWalkCycle].scale = 60;


  paintStyles[neckHeightCalculatedFromLegSensors].brushColor = RGB(255,0,0);
  paintStyles[neckHeightCalculatedFromLegSensors].penColor = RGB(255,255,255);
  paintStyles[neckHeightCalculatedFromLegSensors].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[neckHeightCalculatedFromLegSensors].scale = 60;

  paintStyles[neckHeightProvidedByMotionControl].brushColor = RGB(0,255,0);
  paintStyles[neckHeightProvidedByMotionControl].penColor = RGB(255,255,255);
  paintStyles[neckHeightProvidedByMotionControl].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[neckHeightProvidedByMotionControl].scale = 60;

  paintStyles[bodyRollCalculatedFromLegSensors].brushColor = RGB(255,0,0);
  paintStyles[bodyRollCalculatedFromLegSensors].penColor = RGB(255,255,255);
  paintStyles[bodyRollCalculatedFromLegSensors].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[bodyRollCalculatedFromLegSensors].scale = 60;

  paintStyles[bodyRollProvidedByMotionControl].brushColor = RGB(0,255,0);
  paintStyles[bodyRollProvidedByMotionControl].penColor = RGB(255,255,255);
  paintStyles[bodyRollProvidedByMotionControl].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[bodyRollProvidedByMotionControl].scale = 60;

  paintStyles[bodyRollCalculatedFromAccelerationSensors].brushColor = RGB(0,0,255);
  paintStyles[bodyRollCalculatedFromAccelerationSensors].penColor = RGB(255,255,255);
  paintStyles[bodyRollCalculatedFromAccelerationSensors].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[bodyRollCalculatedFromAccelerationSensors].scale = 60;
  
  paintStyles[bodyTiltCalculatedFromLegSensors].brushColor = RGB(255,0,0);
  paintStyles[bodyTiltCalculatedFromLegSensors].penColor = RGB(255,255,255);
  paintStyles[bodyTiltCalculatedFromLegSensors].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[bodyTiltCalculatedFromLegSensors].scale = 60;

  paintStyles[bodyTiltProvidedByMotionControl].brushColor = RGB(0,255,0);
  paintStyles[bodyTiltProvidedByMotionControl].penColor = RGB(255,255,255);
  paintStyles[bodyTiltProvidedByMotionControl].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[bodyTiltProvidedByMotionControl].scale = 60;

  paintStyles[bodyTiltCalculatedFromAccelerationSensors].brushColor = RGB(0,0,255);
  paintStyles[bodyTiltCalculatedFromAccelerationSensors].penColor = RGB(255,255,255);
  paintStyles[bodyTiltCalculatedFromAccelerationSensors].shape = PaintMethodsWin32::PaintStyle::filledCircle;
  paintStyles[bodyTiltCalculatedFromAccelerationSensors].scale = 60;

  return TRUE;
}

bool CValueHistoryDlgBar::handleMessage(InMessage& message)
{
  switch(message.getMessageID())
  {
  case idWorldState:
    {
      Player player;
      BallModel ballModel;
      PlayerPoseCollection playerPoseCollection;
      ObstaclesModel obstaclesModel;
      RobotState robotState;
      RobotPose robotPose;
      CameraMatrix cameraMatrix;
      CameraInfo cameraInfo;

      message.bin >> RECEIVE_WORLDSTATE(robotPose,
        ballModel,playerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo);

      addValue(CValueHistoryDlgBar::ballPositionDistance,
        robotPose.frameNumber,Geometry::distanceTo(robotPose,ballModel.seen));
      addValue(CValueHistoryDlgBar::ballPositionAngle,
        robotPose.frameNumber,toDegrees(Geometry::angleTo(robotPose.getPose(),ballModel.seen)));
      addValue(CValueHistoryDlgBar::ballPositionSpeedAbs,
        robotPose.frameNumber,ballModel.seen.speed.abs());
      addValue(CValueHistoryDlgBar::ballPositionSpeedDir,
        robotPose.frameNumber,toDegrees(ballModel.seen.speed.angle()));
      addValue(CValueHistoryDlgBar::robotPoseX,
        robotPose.frameNumber,robotPose.translation.x,robotPose.getValidity()<0.5);
      addValue(CValueHistoryDlgBar::robotPoseY,
        robotPose.frameNumber,robotPose.translation.y,robotPose.getValidity()<0.5);
      addValue(CValueHistoryDlgBar::robotPoseAngle,
        robotPose.frameNumber,toDegrees(robotPose.rotation),robotPose.getValidity()<0.5);
      addValue(CValueHistoryDlgBar::robotPoseSpeed,
        robotPose.frameNumber,toDegrees(robotPose.speedbyDistanceToGoal),robotPose.getValidity()<0.5);
     
      return true;
    }
  case idPercepts:
    {
      Player player;
      CameraMatrix cameraMatrix;
      CameraInfo cameraInfo;
      LandmarksPercept landmarksPercept;
      BallPercept ballPercept;
      PlayersPercept playersPercept;
      ObstaclesPercept obstaclesPercept;
      PSDPercept psdPercept;
      CollisionPercept collisionPercept;
      LinesPercept linesPercept;
      EdgesPercept edgesPercept;
      message.bin >> RECEIVE_PERCEPTS(cameraMatrix, cameraInfo, ballPercept,
        landmarksPercept,linesPercept,edgesPercept,playersPercept, obstaclesPercept, psdPercept, collisionPercept);
      
      if (ballPercept.ballWasSeen)
      {
        Vector2<double> ballOffset;
        ballPercept.getOffsetBearingBased(ballOffset);
        
        addValue(CValueHistoryDlgBar::ballPerceptX,
          ballPercept.frameNumber,ballOffset.x);
        
        addValue(CValueHistoryDlgBar::ballPerceptY,
          ballPercept.frameNumber,ballOffset.y);
        
        
        addValue(CValueHistoryDlgBar::ballPerceptDistance,
          ballPercept.frameNumber,ballPercept.getDistance());
        
        addValue(CValueHistoryDlgBar::ballPerceptAngle,
          ballPercept.frameNumber,toDegrees(ballPercept.getAngle()));
        
        addValue(CValueHistoryDlgBar::ballPerceptDistanceBearingBased,
          ballPercept.frameNumber,ballPercept.getDistanceBearingBased());
        
        addValue(CValueHistoryDlgBar::ballPerceptAngleBearingBased,
          ballPercept.frameNumber,toDegrees(ballPercept.getAngleBearingBased()));
        
        addValue(CValueHistoryDlgBar::ballPerceptDistanceSizeBased,
          ballPercept.frameNumber,ballPercept.getDistanceSizeBased());
        
        addValue(CValueHistoryDlgBar::ballPerceptAngleSizeBased,
          ballPercept.frameNumber,toDegrees(ballPercept.getAngleSizeBased()));
        
      }
      for (int i=0;i<psdPercept.numOfPercepts;i++)
      {
        if (psdPercept[i].isValid)
        {
          addValue(CValueHistoryDlgBar::psdPerceptToFarAway, psdPercept[i].frameNumber,(!psdPercept[i].tooFarAway && psdPercept[i].z > 100)?300:0);
          addValue(CValueHistoryDlgBar::psdPerceptX, psdPercept[i].frameNumber,psdPercept[i].x);
          addValue(CValueHistoryDlgBar::psdPerceptY, psdPercept[i].frameNumber,psdPercept[i].y);
          addValue(CValueHistoryDlgBar::psdPerceptZ, psdPercept[i].frameNumber,psdPercept[i].z);
        }
      }
      
      addValue(CValueHistoryDlgBar::cameraMatrixIsValid, cameraMatrix.frameNumber, cameraMatrix.isValid?100:0);

      return true;
    }
  case idSpecialPercept:
    {
      Player player;
      SpecialPercept specialPercept;
      CameraMatrix cameraMatrix;
      message.bin >> player >> specialPercept >> cameraMatrix;
      
      if (specialPercept.type==SpecialPercept::checkerboard)
      {
        addValue(CValueHistoryDlgBar::specialPerceptRobotPoseX, 
          specialPercept.frameNumber, specialPercept.checkerPose.translation.x);
        addValue(CValueHistoryDlgBar::specialPerceptRobotPoseY, 
          specialPercept.frameNumber, specialPercept.checkerPose.translation.y);
        addValue(CValueHistoryDlgBar::specialPerceptRobotPoseAngle, 
          specialPercept.frameNumber, toDegrees(specialPercept.checkerPose.rotation));
      }
      return true;
    }
  case idMotionInfo:
    {
      MotionInfo motionInfo;
      message.bin >> motionInfo;
      addValue(CValueHistoryDlgBar::walkSpeedX, motionInfo.executedMotionRequest.frameNumber, motionInfo.executedMotionRequest.walkRequest.walkParams.translation.x);
      addValue(CValueHistoryDlgBar::walkSpeedY, motionInfo.executedMotionRequest.frameNumber, motionInfo.executedMotionRequest.walkRequest.walkParams.translation.y);
      addValue(CValueHistoryDlgBar::walkSpeedRotation, motionInfo.executedMotionRequest.frameNumber, motionInfo.executedMotionRequest.walkRequest.walkParams.rotation);
      addValue(CValueHistoryDlgBar::positionInWalkCycle, motionInfo.executedMotionRequest.frameNumber, motionInfo.positionInWalkCycle);
      return true;
    }
  case idBodyPosture:
    {
      BodyPosture bodyPosture;
      message.bin >> bodyPosture;
      addValue(CValueHistoryDlgBar::neckHeightCalculatedFromLegSensors, bodyPosture.frameNumber, bodyPosture.neckHeightCalculatedFromLegSensors);
      addValue(CValueHistoryDlgBar::neckHeightProvidedByMotionControl, bodyPosture.frameNumber, bodyPosture.neckHeightProvidedByMotionControl);

      addValue(CValueHistoryDlgBar::bodyRollCalculatedFromLegSensors, bodyPosture.frameNumber, toDegrees(bodyPosture.bodyRollCalculatedFromLegSensors));
      addValue(CValueHistoryDlgBar::bodyRollProvidedByMotionControl, bodyPosture.frameNumber, toDegrees(bodyPosture.bodyRollProvidedByMotionControl));
      addValue(CValueHistoryDlgBar::bodyRollCalculatedFromAccelerationSensors, bodyPosture.frameNumber, toDegrees(bodyPosture.bodyRollCalculatedFromAccelerationSensors));

      addValue(CValueHistoryDlgBar::bodyTiltCalculatedFromLegSensors, bodyPosture.frameNumber, toDegrees(bodyPosture.bodyTiltCalculatedFromLegSensors));
      addValue(CValueHistoryDlgBar::bodyTiltProvidedByMotionControl, bodyPosture.frameNumber, toDegrees(bodyPosture.bodyTiltProvidedByMotionControl));
      addValue(CValueHistoryDlgBar::bodyTiltCalculatedFromAccelerationSensors, bodyPosture.frameNumber, toDegrees(bodyPosture.bodyTiltCalculatedFromAccelerationSensors));
      return true;
    }
    
  default:
    return false;
  }
}


void CValueHistoryDlgBar::addValue(valueID id, unsigned long timeStamp, double value, bool halfSiz)
{
  if (!freeze && timeStamp != 0)
  {
    //reset when getting data at least 250 frames (2s) older than the last
    if (timeStamp+250 < latestTimeStamp)
    {
      clear();
    }
    //find newest timestamp
    if (timeStamp > latestTimeStamp)
    {
      latestTimeStamp = timeStamp;
    }

    timeStamps[id].add(timeStamp);
    valueHistory[id].add(value);
    halfSize[id].add(halfSiz);
    
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  }
}


void CValueHistoryDlgBar::OnPaint() 
{
  CPaintDC dc(this); // device context for painting
  
  //  CBrush backGroundBrush(GetSysColor(COLOR_WINDOWFRAME));
  CRect currentRect;
  currentRect.top = 0;
  currentRect.left = 0;
  currentRect.bottom = rect.bottom;
  currentRect.right = rect.right;
  
  CBrush backGroundBrush(RGB(255,255,255));
  dcOffScreen.FillRect(&currentRect, &backGroundBrush);
  
  int fontSize = 14;
  
  CFont font;
  CFont* oldFont;
  font.CreateFont(
    fontSize,                  // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    ANSI_CHARSET,              // nCharSet
    OUT_DEFAULT_PRECIS,        // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    DEFAULT_PITCH | FF_SWISS,  // nPitchAndFamily
    "Arial");                 // lpszFacename
  oldFont = dcOffScreen.SelectObject(&font);
  
  CPen blackPen;
  blackPen.CreatePen(PS_SOLID, 1, RGB(0,0,0));
  CPen* oldPen;
  
  oldPen = dcOffScreen.SelectObject(&blackPen);
  
  
  dcOffScreen.SetBkMode(TRANSPARENT);
  
  // the begin of the legend 
  int leftLegend = currentRect.right - 200;
  
  // paint the axis
  int arrowSize=(currentRect.right + currentRect.bottom) / 100;
  dcOffScreen.MoveTo(currentRect.left + 10, currentRect.top + 10);
  dcOffScreen.LineTo(currentRect.left + 10, currentRect.bottom - 10);
  dcOffScreen.MoveTo(currentRect.left + 10, currentRect.top + 10);
  dcOffScreen.LineTo(currentRect.left + 10 - (int)(arrowSize * 0.7), currentRect.top + 10 + arrowSize);
  dcOffScreen.MoveTo(currentRect.left + 10, currentRect.top + 10);
  dcOffScreen.LineTo(currentRect.left + 10 + (int)(arrowSize * 0.7), currentRect.top + 10 + arrowSize);
  dcOffScreen.MoveTo(currentRect.left + 8, currentRect.bottom / 2);
  dcOffScreen.LineTo(leftLegend, currentRect.bottom / 2);
  dcOffScreen.LineTo(leftLegend - arrowSize, currentRect.bottom / 2 + (int)(arrowSize * 0.7));
  dcOffScreen.MoveTo(leftLegend, currentRect.bottom / 2);
  dcOffScreen.LineTo(leftLegend - arrowSize, currentRect.bottom / 2 - (int)(arrowSize * 0.7));
  
  for (int i=0; i < timeRange; i++)
  {
    int x = leftLegend - 10 - (int)((double)(leftLegend - 20) * 
      ((double)i + (latestTimeStamp%125) / 125.0)
      / timeRange);
    dcOffScreen.MoveTo(x,currentRect.bottom / 2);
    dcOffScreen.LineTo(x,currentRect.bottom / 2 - (int)(arrowSize*0.5));
  }
  
  for (i=0;i < numberOfValueIDs; i++)
  {
    if (displayValue[i])
    {
      for (int j=0; j < maxNumberOfEntries; j++)
      {
        int x = leftLegend - 10 - (int)((double)(leftLegend - 20) * 
          ((double)((latestTimeStamp - timeStamps[i][j]) / 125.0) / (double)timeRange));
        
        int y = currentRect.bottom / 2 - 
          (int)((valueHistory[i][j] * paintStyles[i].scale * scale) 
          / 100.0 * ((double)(currentRect.bottom - 20.0) / 2.0));
        
        
        if (x <= 10) break;
        
        paintValue(x,y,(int)(arrowSize * 0.4/(1+halfSize[i][j])),paintStyles[i]);
      }
    }
  }
  
  
  // paint the legend 
  int position = 0;
  for (i=0; i<numberOfValueIDs; i++)
  {
    if (displayValue[i])
    {
      position++;
      dcOffScreen.TextOut(leftLegend+10,10+30*position,
        CString(getValueIDName((valueID)i)) + ":");
      paintValue(leftLegend+20,30+30*position,(int)(arrowSize * 0.4),paintStyles[i]);
      
      CString text;
      text.Format("%.3f",valueHistory[i][0]);
      dcOffScreen.TextOut(leftLegend+30,23+30*position,text );
    }
  }
  
  dcOffScreen.SelectObject(oldFont);
  dcOffScreen.SelectObject(oldPen);
  
  blackPen.DeleteObject();
  font.DeleteObject();
  
  dc.BitBlt(rect.left, rect.top, rect.right, rect.bottom,
    &dcOffScreen, 0, 0, SRCCOPY);
}

void CValueHistoryDlgBar::paintValue(int x, int y, int size, PaintMethodsWin32::PaintStyle style)
{
  CPen pen;
  pen.CreatePen(PS_SOLID, style.penWidth, style.penColor);
  CPen* oldPen;
  oldPen = dcOffScreen.SelectObject(&pen);

  CBrush brush;
  brush.CreateSolidBrush(style.brushColor);
  CBrush* oldBrush;

  switch(style.shape)
  {
  case PaintMethodsWin32::PaintStyle::horizontalLine:
    dcOffScreen.MoveTo(x-size,y);
    dcOffScreen.LineTo(x+size,y);
    break;
  case PaintMethodsWin32::PaintStyle::circle: 
    dcOffScreen.Ellipse(x - size,y - size, x+size, y+size);
    break;
  case PaintMethodsWin32::PaintStyle::filledCircle:
    oldBrush = dcOffScreen.SelectObject(&brush);
    dcOffScreen.Ellipse(x - size,y - size, x+size, y+size);
    dcOffScreen.SelectObject(oldBrush);
    break;
  case PaintMethodsWin32::PaintStyle::square: 
    dcOffScreen.MoveTo(x-size,y-size);
    dcOffScreen.LineTo(x-size,y+size);
    dcOffScreen.LineTo(x+size,y+size);
    dcOffScreen.LineTo(x+size,y-size);
    dcOffScreen.LineTo(x-size,y-size);
    break;
  case PaintMethodsWin32::PaintStyle::filledSquare:
    oldBrush = dcOffScreen.SelectObject(&brush);
    dcOffScreen.Rectangle(x - size,y - size, x+size, y+size);
    dcOffScreen.SelectObject(oldBrush);
    break;
  case PaintMethodsWin32::PaintStyle::vcross:
    dcOffScreen.MoveTo(x-size,y-size);
    dcOffScreen.LineTo(x+size,y+size);
    dcOffScreen.MoveTo(x-size,y+size);
    dcOffScreen.LineTo(x+size,y-size);
    break;
  case PaintMethodsWin32::PaintStyle::cross:
  default:
    dcOffScreen.MoveTo(x-size,y);
    dcOffScreen.LineTo(x+size,y);
    dcOffScreen.MoveTo(x,y-size);
    dcOffScreen.LineTo(x,y+size);
    break;
  }
  
  dcOffScreen.SelectObject(oldPen);
  pen.DeleteObject();

  brush.DeleteObject();
}

void CValueHistoryDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  CDynamicBarDlg::OnSize(nType,cx,cy);
  
  CPaintDC dc(this); // device context for painting
  
  rect.left = 0;
  rect.right = cx;
  rect.top = 17;
  rect.bottom = cy - 17;
  
  if(!dcOffScreen)
    dcOffScreen.CreateCompatibleDC(&dc);
  
  if(bmpOffScreen)
  {
    dcOffScreen.SelectObject(oldBitmap);
    /* @todo this line can crash while resizing during painting: */
    bmpOffScreen->DeleteObject();
    delete(bmpOffScreen);
  }
  bmpOffScreen = new CBitmap;
  bmpOffScreen->CreateCompatibleBitmap(
    &dc, cx, cy);
  oldBitmap = dcOffScreen.SelectObject(bmpOffScreen);
  
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CValueHistoryDlgBar::clear()
{
  //dont allow adds during clearance
  bool oldFreeze = freeze;
  freeze = true;
  for (int i=0; i<numberOfValueIDs; i++)
  {
    valueHistory[i].init();
    timeStamps[i].init();
    halfSize[i].init();
  }
  latestTimeStamp = 0;
  freeze = oldFreeze;
}

void CValueHistoryDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu menu;
  VERIFY( menu.CreatePopupMenu() );
  
  UINT nFlags = 0;
  
  for (int i=0;i<numberOfValueIDs;i++)
  {
    menu.AppendMenu((displayValue[i]?MF_CHECKED:0),i+3000,getValueIDName(valueID(i)));
  }
  menu.AppendMenu(MF_SEPARATOR );
  
  menu.AppendMenu(0,4000,"clear");
  menu.AppendMenu(0,4020,"Save all to CSV file (seperate time lines)");
  
  // Track menu
  int nID = menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this);
  
  if (nID == 4000)
  {
    clear();
  }
  else if (nID == 4020)
  { 
    //save as CSV
     saveAllCSV();
  }
  else
  {
    displayValue[nID - 3000] = !displayValue[nID - 3000];
  }
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}


void CValueHistoryDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar)  
{
  CDynamicBarDlg::OnHScroll(nSBCode, nPos, pScrollBar);
  
  timeRange=m_timeRangeSlider.GetPos();
  scale = m_scaleSlider.GetPos() / 100.0;
  
  char text[100];
  sprintf(text,"range: %i s",timeRange);
  m_timeRangeStatic.SetWindowText(text);
  
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}


void CValueHistoryDlgBar::OnFreezeButton()
{
  freeze = (m_freezeButton.GetCheck()==1);
}




void CValueHistoryDlgBar::saveAllCSV()
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("HistoryValueDialogViewer", "historyValueCSVsavePath", defaultPath);
  pathName += "*.csv";
  
  CFileDialog fileDialog(false, ".csv",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "csv files (*.csv)|*.csv||", this);
  
  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("JointViewer", "jointCSVsavePath", pathName);
    
    OutTextFile stream(pathAndFileName);





    
  /* write table titles */
  
  for (int s = 0; s < CValueHistoryDlgBar::numberOfValueIDs; s++) {
    stream << "T:" << CValueHistoryDlgBar::getValueIDName((CValueHistoryDlgBar::valueID)s) << ",";
    stream << "V:" << CValueHistoryDlgBar::getValueIDName((CValueHistoryDlgBar::valueID)s) << ",";
  }


  stream << endl;
  


  int currentValueID = 0;
  int currentTimeStep= 0;
  

  // no sensor values
  if (numberOfValueIDs == 0) return;

  
  
  //if (numberOfValueIDs > 0)
  
  
  //////Maximum erzeugen von TimeStamps noch nicht gemacht, erstmal 4

  // er soll solange die Werte ausgeben, wie Werte vorhanden sind
  {
    int maxNumberOfEntries = 0;
    int lastEntry[CValueHistoryDlgBar::numberOfValueIDs];
    
    for (int h = 0; h < CValueHistoryDlgBar::numberOfValueIDs; h++)
    {
      if (maxNumberOfEntries < valueHistory[h].getNumberOfEntries())
      {
        maxNumberOfEntries = valueHistory[h].getNumberOfEntries();
      }
      lastEntry[h] = valueHistory[h].getNumberOfEntries() - 1;
    }

    for (int j = 0; j < maxNumberOfEntries; j++) 
    {
  
      
  // er soll fuer alle ValueArten ausgeben
      for (int i = 0; i < CValueHistoryDlgBar::numberOfValueIDs; i++) 
      {

        
      if (j >= valueHistory[i].getNumberOfEntries())
      {
            stream << "-,-,";
      }
      else
                //sollte eigentlich offset abziehen (offset = timeStamps[i][0]????)
        {
        stream << timeStamps[i][lastEntry[i]]  - timeStamps[i][timeStamps[i].getNumberOfEntries() - 1] << ",";
        stream << valueHistory[i][lastEntry[i]] << ",";
        lastEntry[i]--;
        }

  
        
        
      /*  
        
      if (j >= valueHistory[i].getNumberOfEntries())
      {
            stream << "-,-,";
      }
      else
                //sollte eigentlich offset abziehen (offset = timeStamps[i][0]????)
        {
          stream << timeStamps[i][j]  - timeStamps[i][timeStamps[i].getNumberOfEntries() - 1] << ",";
          stream << valueHistory[i][j] << ",";
        }

      */

      }
      
      stream << endl;
      
    }
  
  }
  }
}








/*
* Change log :
* 
* $Log: ValueHistoryDlgBar.cpp,v $
* Revision 1.9  2004/06/17 14:15:26  spranger
* replaced executedMotionRequest-debugKeys and Messages with motionInfo from Motion
*
* Revision 1.8  2004/06/15 10:58:27  thomas
* added edge-specialist, edges-percept, debug-drawings etc. (not yet called from image-processor)
*
* Revision 1.7  2004/06/02 17:18:23  spranger
* MotionRequest cleanup
*
* Revision 1.6  2004/05/27 18:49:17  kerdels
* added a small 5 frames sliding average for the relative ballspeed,
* added new ballState Representation and adjusted the apropriate files
*
* Revision 1.5  2004/05/27 15:44:24  juengel
* bug fixed
*
* Revision 1.4  2004/05/27 09:55:17  juengel
* Increased table size
*
* Revision 1.3  2004/05/26 17:13:27  juengel
* Added bodyPosture.
*
* Revision 1.2  2004/05/22 22:52:04  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:28:17  cvsadm
* created new repository GT2004_WM
*
* Revision 1.22  2004/04/29 15:16:57  dueffert
* only show valid psd percepts
*
* Revision 1.21  2004/04/28 17:07:46  goehring
* Scale slider saves values
*
* Revision 1.20  2004/04/22 09:41:52  goehring
* Added BallSpeedDirection
*
* Revision 1.19  2004/04/21 20:13:14  goehring
* Added slider
*
* Revision 1.18  2004/03/16 14:00:23  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.2  2004/03/09 18:42:10  juengel
* Moved PaintStyle to PaintMethodsWin32
*
* Revision 1.17  2004/03/10 09:57:59  dueffert
* only use "good" specialPercepts
*
* Revision 1.16  2004/03/05 15:49:07  dueffert
* robotPoseSpeed added
*
* Revision 1.15  2004/02/25 13:34:32  dueffert
* half size marks for low validity added
*
* Revision 1.14  2004/02/10 11:28:18  dueffert
* scaling corrected
*
* Revision 1.13  2004/02/03 13:20:48  spranger
* renamed all references to  class BallP_osition to BallModel (possibly changed include files)
*
* Revision 1.12  2004/01/10 10:09:15  juengel
* Added CameraInfo to and removed Player from (SEND|RECEIVE)_(PERCEPTS|WORLDSTATE).
*
* Revision 1.11  2003/12/06 16:44:14  loetzsch
* renamed notify() to addValue()
* messages are now handled inside the dialog
*
* Revision 1.10  2003/11/24 15:25:00  dueffert
* resetting corrected
*
* Revision 1.9  2003/11/19 19:30:32  kerdels
* Klammer-zuviel-Compile-Error behoben
*
* Revision 1.8  2003/11/19 13:53:08  dueffert
* timestamp resetting
*
* Revision 1.7  2003/11/18 13:24:42  goehring
* Different BallDistance measures notified
*
* Revision 1.6  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.5  2003/11/10 13:32:26  dueffert
* special percept visualization added
*
* Revision 1.4  2003/11/06 16:42:59  goehring
* CSV button added (BETA)
*
* Revision 1.3  2003/11/05 13:41:25  goehring
* JointStates removed
*
* Revision 1.2  2003/10/21 14:40:37  goehring
* ballPerceptX and Y available
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.12  2003/06/04 12:22:07  goehring
* AverageJointSensorDifferences implemented
*
* Revision 1.11  2003/05/11 23:46:36  dueffert
* Depend now works with RobotControl too
*
* Revision 1.10  2003/04/06 20:34:22  juengel
* CameraMatrix.isValid is displayed in ValueHistoryDlgBar.
*
* Revision 1.9  2003/04/06 16:40:03  juengel
* Added PSDPercept.tooFarAway to ValueHistoryDlgBar.
*
* Revision 1.8  2003/03/28 14:28:08  juengel
* Added PSDPercept
*
* Revision 1.7  2003/03/08 11:56:12  juengel
* changed scale for ballDistance
*
* Revision 1.6  2003/03/07 19:04:07  juengel
* Added style horizontalLine, and values for motionRequest, and green level
*
* Revision 1.5  2003/03/05 12:16:19  loetzsch
* added the "freeze" button to the value history dialog bar
*
* Revision 1.4  2003/03/04 16:22:17  jhoffman
* martin hat sachen schoen gemacht
*
* Revision 1.3  2003/03/04 15:02:00  loetzsch
* no message
*
* Revision 1.2  2003/03/03 23:08:01  loetzsch
* first working version of the value history dialog bar
*
* Revision 1.1  2003/02/27 15:30:47  loetzsch
* added the value history dialog
*
*/

