/**
* @file UDParameterSetDlgBar.cpp
* Implementation of class UDParameterSetDlgBar.
* @author Uwe Dffert
*/

#include "StdAfx.h"
#include "UDParameterSetDlgBar.h"
#include "RobotControlQueues.h"
#include "Tools/Math/Common.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"

CUDParameterSetDlgBar::CUDParameterSetDlgBar()
: CRobotControlDialogBar(IDD),currentParametersNumber(UDParametersSet::no_turn_0_fast)
{
  currentParam=paramSet.getParameters(currentParametersNumber);
  ERS7EvolveWalkingParameters p;
  extraParam=p;
  //{{AFX_DATA_INIT(CUDParameterSetDlgBar)
  //}}AFX_DATA_INIT
}

void CUDParameterSetDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CUDParameterSetDlgBar)
	DDX_Control(pDX, IDC_UDPARAMETERSET_FOOTMODE, m_FootMode);
  DDX_Control(pDX, IDC_UDPARAMETERSET_CHOOSERATIO, m_ChooseRatio);
  DDX_Control(pDX, IDC_UDPARAMETERSET_CHOOSEDIRECTION, m_ChooseDirection);
  DDX_Control(pDX, IDC_UDPARAMETERSET_CHOOSESPEED, m_ChooseSpeed);
  DDX_Control(pDX, IDC_UDPARAMETERSET_AUTO, m_autoCheck);
  DDX_Control(pDX, IDC_UDPARAMETERSET_ROTLABEL, m_rotLabel);
  DDX_Control(pDX, IDC_UDPARAMETERSET_CORRECTLABEL, m_correctLabel);
  DDX_Text(pDX, IDC_UDPARAMETERSET_REQMOT_X, currentParam->requestedMotion.translation.x);
  DDX_Text(pDX, IDC_UDPARAMETERSET_REQMOT_Y, currentParam->requestedMotion.translation.y);
  DDX_Text(pDX, IDC_UDPARAMETERSET_REQMOT_R, currentParam->requestedMotion.rotation);
  DDX_Text(pDX, IDC_UDPARAMETERSET_CORMOT_X, currentParam->correctedMotion.translation.x);
  DDX_Text(pDX, IDC_UDPARAMETERSET_CORMOT_Y, currentParam->correctedMotion.translation.y);
  DDX_Text(pDX, IDC_UDPARAMETERSET_CORMOT_R, currentParam->correctedMotion.rotation);
  DDX_Text(pDX, IDC_UDPARAMETERSET_FORECENTERX, currentParam->foreCenterX);
  DDX_Text(pDX, IDC_UDPARAMETERSET_FOREHEIGHT, currentParam->foreHeight);
  DDX_Text(pDX, IDC_UDPARAMETERSET_FOREWIDTH, currentParam->foreWidth);
  DDX_Text(pDX, IDC_UDPARAMETERSET_HINDCENTERX, currentParam->hindCenterX);
  DDX_Text(pDX, IDC_UDPARAMETERSET_HINDHEIGHT, currentParam->hindHeight);
  DDX_Text(pDX, IDC_UDPARAMETERSET_HINDWIDTH, currentParam->hindWidth);
  DDX_Text(pDX, IDC_UDPARAMETERSET_FOREFOOTLIFT, currentParam->foreFootLift);
  DDX_Text(pDX, IDC_UDPARAMETERSET_HINDFOOTLIFT, currentParam->hindFootLift);
  DDX_Text(pDX, IDC_UDPARAMETERSET_FOREFOOTTILT, currentParam->foreFootTilt);
  DDX_Text(pDX, IDC_UDPARAMETERSET_HINDFOOTTILT, currentParam->hindFootTilt);
  DDX_Text(pDX, IDC_UDPARAMETERSET_STEPLEN, currentParam->stepLen);
  DDV_MinMaxInt(pDX, currentParam->stepLen, 4, 400);
  DDX_Text(pDX, IDC_UDPARAMETERSET_GROUNDPHASEF, currentParam->groundPhase[0]);
  DDX_Text(pDX, IDC_UDPARAMETERSET_GROUNDPHASEH, currentParam->groundPhase[1]);
  DDX_Text(pDX, IDC_UDPARAMETERSET_LEGPHASE0, currentParam->legPhase[0]);
  DDX_Text(pDX, IDC_UDPARAMETERSET_LEGPHASE1, currentParam->legPhase[1]);
  DDX_Text(pDX, IDC_UDPARAMETERSET_LEGPHASE2, currentParam->legPhase[2]);
  DDX_Text(pDX, IDC_UDPARAMETERSET_LEGPHASE3, currentParam->legPhase[3]);
  DDX_Text(pDX, IDC_UDPARAMETERSET_SPEED, speed);
  DDX_Text(pDX, IDC_UDPARAMETERSET_RATIO, ratio);
  DDX_Text(pDX, IDC_UDPARAMETERSET_DIRECTION, direction);

	//}}AFX_DATA_MAP
  if (pDX->m_bSaveAndValidate)
  {
    if (m_FootMode.GetCurSel()!=(int)currentParam->footMode)
    {
      currentParam->footMode=(UDParameters::FootMode)m_FootMode.GetCurSel();
    }
  }
  else
  {
    if (m_FootMode.GetCurSel()!=(int)currentParam->footMode)
    {
      m_FootMode.SetCurSel((int)currentParam->footMode);
    }
  }
}

BEGIN_MESSAGE_MAP(CUDParameterSetDlgBar, CDialog)
//{{AFX_MSG_MAP(CUDParameterSetDlgBar)
ON_BN_CLICKED(IDC_UDPARAMETERSET_SEND, OnSend)
ON_BN_CLICKED(IDC_UDPARAMETERSET_LOAD, OnLoad)
ON_BN_CLICKED(IDC_UDPARAMETERSET_SAVE, OnSave)
ON_BN_CLICKED(IDC_UDPARAMETERSET_MIRROR, OnMirror)
ON_BN_CLICKED(IDC_UDPARAMETERSET_MEASURE, OnMeasure)
ON_EN_CHANGE(IDC_UDPARAMETERSET_SPEED, OnChangeSpeed)
ON_EN_CHANGE(IDC_UDPARAMETERSET_RATIO, OnChangeRatio)
ON_EN_CHANGE(IDC_UDPARAMETERSET_DIRECTION, OnChangeDirection)
ON_CBN_SELCHANGE(IDC_UDPARAMETERSET_CHOOSERATIO, OnSelchangeChooseSet)
ON_CBN_SELCHANGE(IDC_UDPARAMETERSET_CHOOSEDIRECTION, OnSelchangeChooseSet)
ON_CBN_SELCHANGE(IDC_UDPARAMETERSET_CHOOSESPEED, OnSelchangeChooseSet)
ON_EN_CHANGE(IDC_UDPARAMETERSET_REQMOT_X, OnChangeRequest)
ON_EN_CHANGE(IDC_UDPARAMETERSET_REQMOT_Y, OnChangeRequest)
ON_EN_CHANGE(IDC_UDPARAMETERSET_REQMOT_R, OnChangeRequest)
ON_WM_CONTEXTMENU()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()

BOOL CUDParameterSetDlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();
  m_autoCheck.SetCheck(1);

  /*
  for (int i=0;i<UDParametersSet::numberOfParameters;i++)
  {
    m_ChooseSet.AddString(UDParametersSet::getIndexString(i));
  }
  m_ChooseSet.SetCurSel((int)UDParametersSet::no_turn_0_fast);
  OnSelchangeChooseSet();
  */
  m_ChooseRatio.AddString("much_rturn");
  m_ChooseRatio.AddString("med_rturn");
  m_ChooseRatio.AddString("few_rturn");
  m_ChooseRatio.AddString("no_turn");
  m_ChooseRatio.AddString("few_lturn");
  m_ChooseRatio.AddString("med_lturn");
  m_ChooseRatio.AddString("med_lturn");
  m_ChooseRatio.AddString("MR only && oneForAll");

  m_ChooseDirection.AddString("-180");
  m_ChooseDirection.AddString("-135");
  m_ChooseDirection.AddString("-90");
  m_ChooseDirection.AddString("-45");
  m_ChooseDirection.AddString("0");
  m_ChooseDirection.AddString("45");
  m_ChooseDirection.AddString("90");
  m_ChooseDirection.AddString("135");

  m_ChooseSpeed.AddString("0");
  m_ChooseSpeed.AddString("slow");
  m_ChooseSpeed.AddString("med");
  m_ChooseSpeed.AddString("fast");
  currentParametersNumber=(int)UDParametersSet::no_turn_0_fast;
  setParametersSelection();
  
  //AddSzControl(m_mofEdit,mdResize,mdResize);
  
  //smallFont.CreateFont(12,0,0,0,FW_NORMAL,FALSE,FALSE,0,ANSI_CHARSET,OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,DEFAULT_PITCH | FF_SWISS,"Arial");
  //GetDlgItem(IDC_INVKINPAR_TEXTFOREHEIGHT)->  SetFont( &smallFont );
  
  return TRUE;
}

void CUDParameterSetDlgBar::setParametersSelection()
{
  int u=currentParametersNumber;
  if (u<3)
  {
    m_ChooseRatio.SetCurSel(0);
    m_ChooseDirection.SetCurSel(4);
    m_ChooseSpeed.SetCurSel(3-u);
  }
  else if (u==3)
  {
    m_ChooseRatio.SetCurSel(3);
    m_ChooseDirection.SetCurSel(4);
    m_ChooseSpeed.SetCurSel(0);
  }
  else if (u<7)
  {
    m_ChooseRatio.SetCurSel(6);
    m_ChooseDirection.SetCurSel(4);
    m_ChooseSpeed.SetCurSel(u-3);
  }
  else if (u<127)
  {
    m_ChooseDirection.SetCurSel((u-7)%8);
    m_ChooseRatio.SetCurSel(1+(((u-7)/8)%5));
    m_ChooseSpeed.SetCurSel(1+((u-7)/40));
  }
  else
  {
    m_ChooseRatio.SetCurSel(7);
    m_ChooseDirection.SetCurSel(4);
    m_ChooseSpeed.SetCurSel(0);
  }
  UpdateData(false);
  OnChangeRequest();
}

void CUDParameterSetDlgBar::OnMirror()
{
  UpdateData(true);
  paramSet.mirrorLeftTurnToRightTurn();
  UpdateData(false);
}

void CUDParameterSetDlgBar::OnSend()
{
  UpdateData(true);
  
  getQueues().toPhysical.selectedRobot.out.bin << *currentParam;
  getQueues().toPhysical.selectedRobot.out.finishMessage(idUDParameters);
  
  getQueues().toSimulated.selectedRobot.out.bin << *currentParam;
  getQueues().toSimulated.selectedRobot.out.finishMessage(idUDParameters);
}

void CUDParameterSetDlgBar::OnLoad()
{
  int mode=3; // load parametersSet
  getQueues().toPhysical.selectedRobot.out.bin << mode;
  getQueues().toPhysical.selectedRobot.out.finishMessage(idUDEvolutionRequest);
  
  getQueues().toSimulated.selectedRobot.out.bin << mode;
  getQueues().toSimulated.selectedRobot.out.finishMessage(idUDEvolutionRequest);
  
  paramSet.load();
  //2do: output if successful or no file to load
  getQueues().toGUI.out.text << "UDParameterSetDlg: parametersSet loaded";
  getQueues().toGUI.out.finishMessage(idText);
  UpdateData(false);
}

void CUDParameterSetDlgBar::OnSave()
{
  UpdateData(true);
  
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = AfxGetApp()->GetProfileString("UDParametersSet", "savePath", defaultPath);
  pathName += "*.dat";
  CFileDialog fileDialog(false, ".dat",pathName, OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON, "UDParametersSets (*.dat)|*.dat||", this);
  if (fileDialog.DoModal()==IDOK)
  {
    pathName = fileDialog.GetPathName().Left(fileDialog.GetPathName().GetLength() - fileDialog.GetFileName().GetLength());
    AfxGetApp()->WriteProfileString("UDParametersSet", "savePath", pathName);

    int mode=2; // save parametersSet
    getQueues().toPhysical.selectedRobot.out.bin << mode;
    getQueues().toPhysical.selectedRobot.out.finishMessage(idUDEvolutionRequest);

    //do not send to simulated robots, we just save it ourself
    paramSet.save(fileDialog.GetPathName().GetBuffer(128));
  }
}

void CUDParameterSetDlgBar::OnMeasure() 
{
  UpdateData(true);
  
  int mode=1; // measure UDParameters #...
  int autoEvo=m_autoCheck.GetCheck();
  getQueues().toPhysical.selectedRobot.out.bin << mode << currentParametersNumber << autoEvo;
  if (currentParametersNumber==(int)UDParametersSet::numberOfParameters)
  {
    getQueues().toPhysical.selectedRobot.out.bin << currentParam->requestedMotion;
  }
  getQueues().toPhysical.selectedRobot.out.finishMessage(idUDEvolutionRequest);
  
  getQueues().toSimulated.selectedRobot.out.bin << mode << currentParametersNumber << autoEvo;;
  if (currentParametersNumber==(int)UDParametersSet::numberOfParameters)
  {
    getQueues().toSimulated.selectedRobot.out.bin << currentParam->requestedMotion;
  }
  getQueues().toSimulated.selectedRobot.out.finishMessage(idUDEvolutionRequest);
}

bool CUDParameterSetDlgBar::handleMessage(InMessage& message)
{
  MessageID messageID = message.getMessageID();
  switch(messageID)
  {
  case idUDParameters:
    {
      UDParameters param;
      message.bin >> param;
      UpdateData(true);
      currentParametersNumber=param.index;
      if (currentParametersNumber<UDParametersSet::numberOfParameters)
      {
        currentParam=paramSet.getParameters(currentParametersNumber);
        *currentParam=param;
        paramSet.mirrorThis(currentParametersNumber);
      }
      else
      {
        currentParam=&extraParam;
        *currentParam=param;
      }
      setParametersSelection();
      return true;
    }
  default:
    return false;
  }
}

void CUDParameterSetDlgBar::OnChangeRequest()
{
  UpdateData(true);
  speed=UDParametersSet::getSpeed(currentParam->requestedMotion);
  ratio=UDParametersSet::getRatio(currentParam->requestedMotion);
  direction=UDParametersSet::getDirection(currentParam->requestedMotion)*180/pi;
  UpdateData(false);
}

void CUDParameterSetDlgBar::OnChangeSpeed()
{
  UpdateData(true);
  if (speed<0.001)
  {
    speed=0.001;
  }
  UDParametersSet::setSpeed(currentParam->requestedMotion,speed);
  //set ratio and direction too for the case that speed was 0 before:
  UDParametersSet::setRatio(currentParam->requestedMotion,ratio);
  UDParametersSet::setDirection(currentParam->requestedMotion,direction*pi/180);
  UpdateData(false);
}

void CUDParameterSetDlgBar::OnChangeRatio()
{
  UpdateData(true);
  if (ratio<-1) {ratio=-1;}
  else if (ratio>1) {ratio=1;}
  UDParametersSet::setRatio(currentParam->requestedMotion,ratio);
  //set direction too for the case that abs(ratio) was 1 before:
  UDParametersSet::setDirection(currentParam->requestedMotion,direction*pi/180);
  UpdateData(false);
}

void CUDParameterSetDlgBar::OnChangeDirection()
{
  UpdateData(true);
  if (fabs(direction)>=180) {direction=-180;}
  UDParametersSet::setDirection(currentParam->requestedMotion,direction*pi/180);
  UpdateData(false);
}

void CUDParameterSetDlgBar::OnSelchangeChooseSet()
{
  UpdateData(true);
  int ratio=m_ChooseRatio.GetCurSel();
  int direction=m_ChooseDirection.GetCurSel();
  int speed=m_ChooseSpeed.GetCurSel();
  if ((ratio!=3)&&(ratio!=7)&&(speed==0))
  {
    speed=1;
    m_ChooseSpeed.SetCurSel(speed);
  }
  if ((ratio==0)||(ratio==6))
  {
    direction=4;
    m_ChooseDirection.SetCurSel(direction);
  }
  else if (ratio==7)
  {
    direction=4;
    speed=0;
    m_ChooseDirection.SetCurSel(direction);
    m_ChooseSpeed.SetCurSel(speed);
  }

  if (ratio==0)
  {
    currentParametersNumber=3-speed;
  }
  else if (ratio==6)
  {
    currentParametersNumber=3+speed;
  }
  else if ((ratio==3)&&(speed==0))
  {
    currentParametersNumber=3;
  }
  else if (ratio==7)
  {
    currentParametersNumber=127;
  }
  else
  {
    currentParametersNumber=7+40*(speed-1)+8*(ratio-1)+direction;
  }

  if (currentParametersNumber<UDParametersSet::numberOfParameters)
  {
    currentParam=paramSet.getParameters(currentParametersNumber);
    m_correctLabel.SetWindowText("correctedMotion");
    m_rotLabel.SetWindowText("rot");
  }
  else
  {
    currentParam=&extraParam;
    m_correctLabel.SetWindowText("maxStepSize");
    m_rotLabel.SetWindowText("");
  }
  UpdateData(false);
  OnChangeRequest();
}

void CUDParameterSetDlgBar::copyToClipboard(bool useInvKin)
{
  UpdateData(true);
  char buffer[1024];
  memset(buffer,0,1024);
  OutTextMemory outTextMemory(buffer);
  if (useInvKin)
  {
    InvKinWalkingParameters w = *currentParam;
    sprintf(buffer,
      "%d %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %d %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f %.3f",
      w.footMode,
      w.foreHeight, w.foreWidth, w.foreCenterX, w.hindHeight, w.hindWidth, w.hindCenterX,
      w.foreFootLift, w.hindFootLift, w.foreFootTilt, w.hindFootTilt,
      w.legSpeedFactorX, w.legSpeedFactorY, w.legSpeedFactorR, w.maxStepSizeX, w.maxStepSizeY,
      w.maxSpeedXChange, w.maxSpeedYChange, w.maxRotationChange, w.counterRotation,
      w.stepLen,
      w.groundPhase[0], w.liftPhase[0], w.loweringPhase[0], w.groundPhase[1], w.liftPhase[1], w.loweringPhase[1],
      w.legPhase[0], w.legPhase[1], w.legPhase[2], w.legPhase[3],
      w.bodyShiftX, w.bodyShiftY, w.bodyShiftOffset, w.bodyTiltOffset);
  }
  else
  {
    outTextMemory << *currentParam;
  }

  GLOBALHANDLE hGMem = GlobalAlloc(GHND, strlen(buffer)+1 );
  char* pchGM = (char*) ::GlobalLock(hGMem);
  strcpy(pchGM, buffer);

  OpenClipboard();
	EmptyClipboard();
	SetClipboardData( CF_TEXT, pchGM );
	CloseClipboard();

  GlobalUnlock(hGMem);
}

void CUDParameterSetDlgBar::pasteFromClipboard(bool useInvKin)
{
  GLOBALHANDLE hGMem;

  OpenClipboard();
  if ( hGMem = GetClipboardData( CF_TEXT ) )
  {
    char* buffer = (char*)hGMem;
    InTextMemory inTextMemory(buffer,strlen(buffer));
    if (useInvKin)
    {
      InvKinWalkingParameters p;
      int footMode;
      if(
        (!(inTextMemory >> footMode).eof()) &&
        footMode >= 0 &&
        footMode < InvKinWalkingParameters::numOfFootModes &&
        (!(inTextMemory >> p.foreHeight).eof()) &&
        (!(inTextMemory >> p.foreWidth).eof()) &&
        (!(inTextMemory >> p.foreCenterX).eof()) &&
        (!(inTextMemory >> p.hindHeight).eof()) &&
        (!(inTextMemory >> p.hindWidth).eof()) &&
        (!(inTextMemory >> p.hindCenterX).eof()) &&
        (!(inTextMemory >> p.foreFootLift).eof()) &&
        (!(inTextMemory >> p.hindFootLift).eof()) &&
        (!(inTextMemory >> p.foreFootTilt).eof()) &&
        (!(inTextMemory >> p.hindFootTilt).eof()) &&
        (!(inTextMemory >> p.legSpeedFactorX).eof()) &&
        (!(inTextMemory >> p.legSpeedFactorY).eof()) &&
        (!(inTextMemory >> p.legSpeedFactorR).eof()) &&
        (!(inTextMemory >> p.maxStepSizeX).eof()) &&
        (!(inTextMemory >> p.maxStepSizeY).eof()) &&
        (!(inTextMemory >> p.maxSpeedXChange).eof()) &&
        (!(inTextMemory >> p.maxSpeedYChange).eof()) &&
        (!(inTextMemory >> p.maxRotationChange).eof()) &&
        (!(inTextMemory >> p.counterRotation).eof()) &&
        (!(inTextMemory >> p.stepLen).eof()) &&
        (!(inTextMemory >> p.groundPhase[0]).eof()) &&
        (!(inTextMemory >> p.liftPhase[0]).eof()) &&
        (!(inTextMemory >> p.loweringPhase[0]).eof()) &&
        (!(inTextMemory >> p.groundPhase[1]).eof()) &&
        (!(inTextMemory >> p.liftPhase[1]).eof()) &&
        (!(inTextMemory >> p.loweringPhase[1]).eof()) &&
        (!(inTextMemory >> p.legPhase[0]).eof()) &&
        (!(inTextMemory >> p.legPhase[1]).eof()) &&
        (!(inTextMemory >> p.legPhase[2]).eof()) &&
        (!(inTextMemory >> p.legPhase[3]).eof()) &&
        (!(inTextMemory >> p.bodyShiftX).eof()) &&
        (!(inTextMemory >> p.bodyShiftY).eof()) &&
        (!(inTextMemory >> p.bodyShiftOffset).eof()) &&
        (inTextMemory >> p.bodyTiltOffset).eof()
        )
      {
        p.footMode = (InvKinWalkingParameters::FootMode)footMode;
        p.headTilt = p.headPan = p.headRoll = p.mouth = jointDataInvalidValue;
        currentParametersNumber=UDParametersSet::numberOfParameters;
        currentParam=&extraParam;
        *currentParam=p;
        setParametersSelection();
      }
    }
    else
    {
      UDParameters param;
      //we may want to do more sophisticated parsing here...:
      if((inTextMemory >> param).eof())
      {
        currentParametersNumber=param.index;
        if (currentParametersNumber<UDParametersSet::numberOfParameters)
        {
          currentParam=paramSet.getParameters(currentParametersNumber);
          *currentParam=param;
          paramSet.mirrorThis(currentParametersNumber);
        }
        else
        {
          currentParam=&extraParam;
          *currentParam=param;
        }
        setParametersSelection();
      }
    }
  }
  CloseClipboard();
  UpdateData(false);
}

void CUDParameterSetDlgBar::OnContextMenu(CWnd* pWnd, CPoint point)
{
  CMenu menu;
  VERIFY(menu.LoadMenu(IDP_UDPARAMETERSET_MENU));

  UINT nID = menu.GetSubMenu(0)->TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this);

  switch (nID)
  {
    case IDC_UDPARAMETERSET_COPY:
      copyToClipboard();
      break;

    case IDC_UDPARAMETERSET_PASTE:
      pasteFromClipboard();
      break;

    case IDC_UDPARAMETERSET_INVKIN_COPY:
      copyToClipboard(true);
      break;

    case IDC_UDPARAMETERSET_INVKIN_PASTE:
      pasteFromClipboard(true);
      break;
  }
}

/*
* Change log :
* 
* $Log: UDParameterSetDlgBar.cpp,v $
* Revision 1.4  2004/07/07 15:10:35  dueffert
* copy'n'paste invkin added
*
* Revision 1.3  2004/06/14 16:47:20  juengel
* Removed some WalkingEngineParameterSets.
*
* Revision 1.2  2004/06/09 08:18:35  dueffert
* UDParametersSetDialog improved
*
* Revision 1.1.1.1  2004/05/22 17:28:16  cvsadm
* created new repository GT2004_WM
*
* Revision 1.20  2004/05/20 17:21:28  dueffert
* UDParametersDialog improved
*
* Revision 1.19  2004/05/12 14:26:55  dueffert
* support for speed/ratio/direction improved
*
* Revision 1.18  2004/04/21 09:43:53  dueffert
* another update bug fixed
*
* Revision 1.17  2004/04/07 11:41:08  dueffert
* update bug fixed
*
* Revision 1.16  2004/04/01 12:12:18  dueffert
* update bug and copy'n'paste bug fixed
*
* Revision 1.15  2004/03/29 15:33:49  dueffert
* speed change added
*
* Revision 1.14  2004/03/27 08:46:36  dueffert
* copy'n'paste support via context menu added
*
* Revision 1.13  2004/03/26 09:18:03  dueffert
* beautified
*
* Revision 1.12  2004/03/19 16:41:54  dueffert
* update improved
*
* Revision 1.11  2004/03/16 10:30:43  dueffert
* UI update and output improved
*
* Revision 1.10  2004/03/03 08:29:35  dueffert
* two UDEvolutionRequests added
*
* Revision 1.9  2004/03/01 14:56:34  dueffert
* update for incomming UDParameters improved
*
* Revision 1.8  2004/02/29 17:30:18  dueffert
* UDParameters have char* names now
*
* Revision 1.7  2004/02/29 13:38:47  dueffert
* symmetries in UDParametersSet handled
*
* Revision 1.6  2004/02/27 16:40:56  dueffert
* UDEvolutionRequest introduced
*
* Revision 1.5  2004/02/18 14:57:38  dueffert
* adapted to changed UDParameters
*
* Revision 1.4  2003/12/09 19:49:26  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.3  2003/12/09 14:20:03  dueffert
* numerous UD improvements
*
* Revision 1.2  2003/12/05 11:57:57  dueffert
* UDParameterSetDlg improved
*
* Revision 1.1  2003/12/02 18:07:52  dueffert
* first working not yet calibrated version of UDWalkingEngine added
*
*
*/
