/**
* @file SimulatorObjectViewerDlgBar.cpp
* 
* Implementation of class CSimulatorObjectViewerDlgBar.
*
* @author Martin Ltzsch
*/

#include "StdAfx.h"
#include "SimulatorObjectViewerDlgBar.h"
#include "RobotControlSimulatedRobots.h"
#include "RobotControl/SimRobot/Controller.h"
#include "RobotControl/SimRobot/NonPortable.h"

#include "SimRob95/SimRobot/sim3DWorld.h"


BEGIN_MESSAGE_MAP(CSimulatorObjectViewerDlgBar, CDynamicBarDlg)
//{{AFX_MSG_MAP(CSimulatorObjectViewerDlgBar)
ON_WM_PAINT()
ON_WM_SIZE()
ON_WM_HSCROLL()
ON_WM_VSCROLL()
ON_WM_RBUTTONUP()
ON_WM_ERASEBKGND()
ON_WM_LBUTTONDOWN()
ON_WM_LBUTTONDBLCLK()
ON_WM_MOUSEMOVE()
ON_WM_LBUTTONUP()
ON_WM_DESTROY()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()

CSimulatorObjectViewerDlgBar::CSimulatorObjectViewerDlgBar()
: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CSimulatorObjectViewerDlgBar)
  //}}AFX_DATA_INIT

  getSimulatedRobots().getSimRobotDocument()->pSimulatorObjectViewerDlgBar = this;
}

CSimulatorObjectViewerDlgBar::~CSimulatorObjectViewerDlgBar()
{
  if(m_pBitmap)
    delete m_pBitmap;
}

void CSimulatorObjectViewerDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CSimulatorObjectViewerDlgBar)
  DDX_Control(pDX, IDC_SIMROBOT_OBJECT_VIEWER_VSCROLL, m_vScrollBar);
  DDX_Control(pDX, IDC_SIMROBOT_OBJECT_VIEWER_HSCROLL, m_hScrollBar);
  //}}AFX_DATA_MAP
}

CLICKINFO CSimulatorObjectViewerDlgBar::CalcClickInfo(CPoint point)
{
  CRect rect;
  GetClientRect(&rect);
  CClientDC dc(this);
  dc.SetMapMode(MM_ISOTROPIC);
  dc.SetViewportExt(rect.right,rect.bottom);
  dc.SetViewportOrg(rect.right/2,rect.bottom/2);
  dc.SetWindowExt(16384,-16384);
  dc.SetWindowOrg(0,0);
  dc.DPtoLP(&point);
  int x = m_vScrollBar.GetScrollPos(),
    y = m_hScrollBar.GetScrollPos();
  return CLICKINFO(GetDocument()->GetSimulation()->GetObject(""),
    VECTOR(point.x,point.y,0),(x+180) % 360,(y+90) % 360,
    1 << m_nZoom,m_nDistortion ? (16384 >> m_nDistortion) << m_nZoom : 1e12);
}

BOOL CSimulatorObjectViewerDlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();
  
  AddSzControl(m_hScrollBar,mdRepos,mdResize);
  AddSzControl(m_vScrollBar,mdResize,mdRepos);
  
  m_hScrollBar.SetScrollRange(0,360,FALSE);
  m_vScrollBar.SetScrollRange(0,360,FALSE);
  m_hScrollBar.SetScrollPos(225,FALSE);
  m_vScrollBar.SetScrollPos(180,FALSE);
  
  m_bTurn = true;
  m_nZoom = 2;
  m_nDistortion = 2;
  m_nDetailLevel = 9;
  m_bHidden = false;
  
  m_menuPopup.LoadMenu(IDP_SIMROBOT_OBJECT_VIEWER);
  
  m_pBitmap = 0;
  m_rect = CRect(0,0,0,0);
  
  return TRUE;
}

BOOL CSimulatorObjectViewerDlgBar::OnEraseBkgnd(CDC* pDC) 
{
  return TRUE;	
}

void CSimulatorObjectViewerDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  Repaint();
  CDynamicBarDlg::OnSize(nType, cx, cy);
}

void CSimulatorObjectViewerDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  OnScroll(nSBCode, nPos, pScrollBar);
}

void CSimulatorObjectViewerDlgBar::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  OnScroll(nSBCode, nPos, pScrollBar);
}

void CSimulatorObjectViewerDlgBar::OnScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar)
{
  int zOrig, z;   // z = x or y depending on pScrollBar
  int zMin, zMax;
  zOrig = z = pScrollBar->GetScrollPos();
  pScrollBar->GetScrollRange(&zMin,&zMax);
  switch (nSBCode)
  {
  case SB_TOP:
    z = 0;
    break;
  case SB_BOTTOM:
    z = zMax;
    break;
  case SB_LINEUP:
    z -= 10;
    break;
  case SB_LINEDOWN:
    z += 10;
    break;
  case SB_PAGEUP:
    z -= 90;
    break;
  case SB_PAGEDOWN:
    z += 90;
    break;
  case SB_THUMBTRACK:
    if(!m_bTurn)
      return;
    z = nPos;
    break;
  case SB_THUMBPOSITION:
    z = nPos;
    break;
  default:        // ignore other notifications
    return;
  }
  if (z < 0)
    z = 0;
  else if (z > zMax)
    z = zMax;
  if (z != zOrig)
  {
    pScrollBar->SetScrollPos(z);
    Repaint();
  }
}

void CSimulatorObjectViewerDlgBar::OnRButtonUp(UINT nFlags, CPoint point) 
{
  ClientToScreen(&point);
  m_menuPopup.GetSubMenu(0)
    ->TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RIGHTBUTTON,
    point.x,point.y,AfxGetMainWnd());
}

bool CSimulatorObjectViewerDlgBar::handleCommand(UINT command)
{
  switch (command)
  {
  case IDC_SIMROBOT_ZOOM0: 
  case IDC_SIMROBOT_ZOOM1: 
  case IDC_SIMROBOT_ZOOM2: 
  case IDC_SIMROBOT_ZOOM3: 
  case IDC_SIMROBOT_ZOOM4: 
  case IDC_SIMROBOT_ZOOM5: 
  case IDC_SIMROBOT_ZOOM6: 
  case IDC_SIMROBOT_ZOOM7: 
  case IDC_SIMROBOT_ZOOM8: 
  case IDC_SIMROBOT_ZOOM9:
    m_nZoom = command - IDC_SIMROBOT_ZOOM0;
    Repaint();
    return true;
  case IDC_SIMROBOT_DETAIL_LEVEL0:
  case IDC_SIMROBOT_DETAIL_LEVEL1:
  case IDC_SIMROBOT_DETAIL_LEVEL2:
  case IDC_SIMROBOT_DETAIL_LEVEL3:
  case IDC_SIMROBOT_DETAIL_LEVEL4:
  case IDC_SIMROBOT_DETAIL_LEVEL5:
  case IDC_SIMROBOT_DETAIL_LEVEL6:
  case IDC_SIMROBOT_DETAIL_LEVEL7:
  case IDC_SIMROBOT_DETAIL_LEVEL8:
  case IDC_SIMROBOT_DETAIL_LEVEL9:
    m_nDetailLevel = command - IDC_SIMROBOT_DETAIL_LEVEL0;
    Repaint();
    return true;
  case IDC_SIMROBOT_DISTORTION0:
  case IDC_SIMROBOT_DIST0RTION1:
  case IDC_SIMROBOT_DIST0RTION2:
  case IDC_SIMROBOT_DISTORTION3:
  case IDC_SIMROBOT_DIST0RTION4:
  case IDC_SIMROBOT_DIST0RTION5:
    m_nDistortion = command - IDC_SIMROBOT_DISTORTION0;
    Repaint();
    return true;
  case IDC_SIMROBOT_VIEW_HIDDEN_LINES:
    m_bHidden = !m_bHidden;
    Repaint();
    return true;
  default:
    return false;
  }
}


void CSimulatorObjectViewerDlgBar::Repaint()
{
  m_bRepaint = TRUE;
  Invalidate(FALSE);
}

void CSimulatorObjectViewerDlgBar::OnPaint() 
{
  CPaintDC dc(this); // device context for painting
  
  this->OnDraw(&dc);
}

void CSimulatorObjectViewerDlgBar::DoDraw(CDC& dc)
{
  if (GetDocument() != 0)
  {
    SIMULATION* pSim = GetDocument()->GetSimulation();
    m_obj = GetDocument()->GetSimulation()->GetObject("");
    dc.SetWindowExt(16384,-16384);
    dc.SetWindowOrg(0,0);
    MFCGRAPH gp(&dc);
    gp.RegisterColors(((SIM3DWorld*) pSim->GetObject(""))->ColorList);
    int x = m_vScrollBar.GetScrollPos(),
      y = m_hScrollBar.GetScrollPos();
    MOVEMATRIX M(TurnYZMatrix ((double) ((y+90) % 360))
      * TurnXYMatrix ((double) ((x+180) % 360))
      * MATRIX(VECTOR (1 << m_nZoom,0,0),VECTOR (0,1 << m_nZoom,0),VECTOR (0,0,1 << m_nZoom))
      ,VECTOR (0,0,0));
    gp.SetViewPoint (M, 1, m_nDistortion ? (16384 >> m_nDistortion) << m_nZoom : 1e12);
    gp.SetHidden(m_bHidden);
    gp.NewPicture();
    pSim->DrawObject(m_obj,&gp,m_nDetailLevel,20);
    gp.EndOfPicture();
  }
}

void CSimulatorObjectViewerDlgBar::OnDraw(CDC* pDC)
{
  CRect rect;
  GetClientRect(&rect);
  
  if(!GetParent()->IsIconic() && rect.right && rect.bottom)
  {
    if(!m_obj)
    {
      CBrush br(GetSysColor(COLOR_WINDOW));
      pDC->FillRect(&rect,&br);
    }
    else
    {
      if(!m_pBitmap || rect != m_rect)
      {
        m_rect = rect;
        if(m_pBitmap)
          delete m_pBitmap;
        m_pBitmap = new CBitmap;
        m_pBitmap->CreateCompatibleBitmap(pDC,rect.right,rect.bottom);
        m_bRepaint = TRUE;
      }
      CDC dcMem;
      dcMem.CreateCompatibleDC(pDC);
      CBitmap* pBitmap = dcMem.SelectObject(m_pBitmap);
      if(m_bRepaint)
      {
        m_bRepaint = FALSE;
        dcMem.SaveDC();
        CBrush br(GetSysColor(COLOR_WINDOW));
        dcMem.FillRect(&rect,&br);
        dcMem.SetMapMode(MM_ISOTROPIC);
        dcMem.SetViewportExt(rect.right,rect.bottom);
        dcMem.SetViewportOrg(rect.right/2,rect.bottom/2);
        DoDraw(dcMem);
        dcMem.RestoreDC(-1);
      }
      pDC->BitBlt(0,0,rect.right,rect.bottom,&dcMem,0,0,SRCCOPY);
      dcMem.SelectObject(pBitmap);
    }
  }
}


void CSimulatorObjectViewerDlgBar::OnLButtonDown(UINT nFlags, CPoint point) 
{
  if(GetDocument() && GetDocument()->GetSimulation()->GetObject(""))
  {
    CLICKINFO ci = CalcClickInfo(point);
    GetDocument()->OnMouseClick(ci);
    GetDocument()->OnBeginDrag(ci);
    SetCapture();
  }
}

void CSimulatorObjectViewerDlgBar::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
  if(GetDocument() && GetDocument()->GetSimulation()->GetObject(""))
    GetDocument()->OnMouseDoubleClick(CalcClickInfo(point));
}

void CSimulatorObjectViewerDlgBar::OnMouseMove(UINT nFlags, CPoint point) 
{
  if(GetCapture() == this)
    GetDocument()->OnDrag(CalcClickInfo(point));
}

void CSimulatorObjectViewerDlgBar::OnLButtonUp(UINT nFlags, CPoint point) 
{
  if(GetCapture() == this)
  {
    GetDocument()->OnEndDrag(CalcClickInfo(point));
    ReleaseCapture();
  }
}

void CSimulatorObjectViewerDlgBar::OnDestroy() 
{
  if(GetCapture() == this)
  {
    ReleaseCapture();
    GetDocument()->OnEndDrag(CalcClickInfo(CPoint(0,0)));
  }
}

void CSimulatorObjectViewerDlgBar::updateUI(CCmdUI* pCmdUI)
{
  switch (pCmdUI->m_nID)
  {
  case IDC_SIMROBOT_DETAIL_LEVEL:
  case IDC_SIMROBOT_ZOOM:
    pCmdUI->Enable(IsWindowVisible());
    return;
  case IDC_SIMROBOT_VIEW_HIDDEN_LINES:
    pCmdUI->Enable(IsWindowVisible());
    pCmdUI->SetCheck(m_bHidden);
    return;
  case IDC_SIMROBOT_ZOOM0:
  case IDC_SIMROBOT_ZOOM1:
  case IDC_SIMROBOT_ZOOM2:
  case IDC_SIMROBOT_ZOOM3:
  case IDC_SIMROBOT_ZOOM4:
  case IDC_SIMROBOT_ZOOM5:
  case IDC_SIMROBOT_ZOOM6:
  case IDC_SIMROBOT_ZOOM7:
  case IDC_SIMROBOT_ZOOM8:
  case IDC_SIMROBOT_ZOOM9:
    pCmdUI->Enable(true);
    pCmdUI->SetCheck(m_nZoom == pCmdUI->m_nID - IDC_SIMROBOT_ZOOM0);
    return;
  case IDC_SIMROBOT_DETAIL_LEVEL0:
  case IDC_SIMROBOT_DETAIL_LEVEL1:
  case IDC_SIMROBOT_DETAIL_LEVEL2:
  case IDC_SIMROBOT_DETAIL_LEVEL3:
  case IDC_SIMROBOT_DETAIL_LEVEL4:
  case IDC_SIMROBOT_DETAIL_LEVEL5:
  case IDC_SIMROBOT_DETAIL_LEVEL6:
  case IDC_SIMROBOT_DETAIL_LEVEL7:
  case IDC_SIMROBOT_DETAIL_LEVEL8:
  case IDC_SIMROBOT_DETAIL_LEVEL9:
    pCmdUI->Enable(true);
    pCmdUI->SetCheck(m_nDetailLevel == pCmdUI->m_nID - IDC_SIMROBOT_DETAIL_LEVEL0);
    return;
  case IDC_SIMROBOT_DISTORTION0:
  case IDC_SIMROBOT_DIST0RTION1:
  case IDC_SIMROBOT_DIST0RTION2:
  case IDC_SIMROBOT_DISTORTION3:
  case IDC_SIMROBOT_DIST0RTION4:
  case IDC_SIMROBOT_DIST0RTION5:
    pCmdUI->Enable(true);
    pCmdUI->SetCheck(m_nDistortion == pCmdUI->m_nID - IDC_SIMROBOT_DISTORTION0);
    return;
  default:
    return;
  }
}

void CSimulatorObjectViewerDlgBar::OnConfigurationLoad(CString sectionName)
{
  m_nZoom = AfxGetApp()->GetProfileInt(sectionName + "\\Simulator","Zoom",2);
  
  m_nDetailLevel = AfxGetApp()->GetProfileInt(sectionName + "\\Simulator", "Detail Level",9);
  
  m_nDistortion = AfxGetApp()->GetProfileInt(sectionName + "\\Simulator", "Distortion",2);
  
  m_bHidden = AfxGetApp()->GetProfileInt(sectionName + "\\Simulator", "View Hidden Lines",0)==1;
  
  m_hScrollBar.SetScrollPos(
    AfxGetApp()->GetProfileInt(sectionName + "\\Simulator", "H Scroll",225),true);
  
  m_vScrollBar.SetScrollPos(
    AfxGetApp()->GetProfileInt(sectionName + "\\Simulator", "V Scroll",180),true);
}

void CSimulatorObjectViewerDlgBar::OnConfigurationSave(CString sectionName)
{
  AfxGetApp()->WriteProfileInt(sectionName + "\\Simulator","Zoom",m_nZoom);
  
  AfxGetApp()->WriteProfileInt(sectionName + "\\Simulator", "Detail Level", m_nDetailLevel);
  
  AfxGetApp()->WriteProfileInt(sectionName + "\\Simulator", "Distortion", m_nDistortion);
  
  AfxGetApp()->WriteProfileInt(sectionName + "\\Simulator", "View Hidden Lines",m_bHidden?1:0);
  
  AfxGetApp()->WriteProfileInt(sectionName + "\\Simulator", "H Scroll",m_hScrollBar.GetScrollPos());
  
  AfxGetApp()->WriteProfileInt(sectionName + "\\Simulator", "V Scroll",m_vScrollBar.GetScrollPos());
}



CSimRobotDocument* CSimulatorObjectViewerDlgBar::GetDocument()
{
  return getSimulatedRobots().getSimRobotDocument();
}

/*
* Change Log:
*
* $Log: SimulatorObjectViewerDlgBar.cpp,v $
* Revision 1.3  2004/05/22 18:47:15  juengel
* Re-added SimulatorObjectViewer.
*
* Revision 1.1.1.1  2004/05/22 17:28:02  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2003/12/06 06:31:20  loetzsch
* no message
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/05/11 23:46:35  dueffert
* Depend now works with RobotControl too
*
* Revision 1.4  2003/03/22 22:37:52  loetzsch
* finished GUI and message routing for 8 simulated robots
* almost all functionallity for simulated robots now is encapsulated in class
*   CRobotControlSimulatedRobots
*
* Revision 1.3  2003/03/22 16:39:33  loetzsch
* continued GUI for simulated robots
*
* Revision 1.2  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:49:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.4  2002/08/29 13:57:06  dueffert
* includes in correct case, system includes in <>
*
* Revision 1.3  2002/08/07 10:50:21  loetzsch
* - removed direct access to the dialogs
* - faster starting of RobotControl application and main window
* - general clean up
*
*/
