/**
* @file SimulatedRobotsToolBar.cpp
* 
* Implementation of the simulator ui toolbar.
*
* @author Martin Ltzsch
*/

#include "StdAfx.h"

#include "SimulatedRobotsToolBar.h"
#include "resource.h"
#include "MfcTools/IEStyleToolBars/SdkDefinitions.h"

#include "RobotControlSimulatedRobots.h"
#include "SimRob95/SimRobot/sim3DWorld.h"

IMPLEMENT_DYNAMIC(CSimulatedRobotsToolBar, CToolBarEx )


void CSimulatedRobotsToolBar::Init()
{
  static TBBUTTONEX tbButtons[] =
  {
    {{ 1,IDC_SIMROBOT_ZOOM, TBSTATE_ENABLED,TBSTYLE_DROPDOWN | BTNS_WHOLEDROPDOWN,0,0},true},
    {{ 2,IDC_SIMROBOT_DETAIL_LEVEL, TBSTATE_ENABLED,TBSTYLE_DROPDOWN | BTNS_WHOLEDROPDOWN,0,0},true},
    {{ 0,0,                 TBSTATE_ENABLED,TBSTYLE_SEP,   0,0},true},
    {{ 3,IDC_SIMROBOT_RESET,    TBSTATE_ENABLED,TBSTYLE_BUTTON,0,0},true},
    {{ 4,IDC_SIMROBOT_START,    TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_CHECK,0,0},true},
    {{ 5,IDC_SIMROBOT_STEP,     TBSTATE_ENABLED,TBSTYLE_BUTTON,0,0},true},
    {{ 0,0,                 TBSTATE_ENABLED,TBSTYLE_SEP,   0,0},true},
    {{ 43,IDC_DISABLE_SIMULATED_ROBOTS, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_CHECK,0,0},true},
    {{ 0,0,                 TBSTATE_ENABLED,TBSTYLE_SEP,   0,0},true},
    {{ 11,IDC_SIMULATED_ROBOT1, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_DROPDOWN,0,0},true},
    {{ 12,IDC_SIMULATED_ROBOT2, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_DROPDOWN,0,0},true},
    {{ 13,IDC_SIMULATED_ROBOT3, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_DROPDOWN,0,0},true},
    {{ 14,IDC_SIMULATED_ROBOT4, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_DROPDOWN,0,0},true},
    {{ 15,IDC_SIMULATED_ROBOT5, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_DROPDOWN,0,0},true},
    {{ 16,IDC_SIMULATED_ROBOT6, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_DROPDOWN,0,0},true},
    {{ 17,IDC_SIMULATED_ROBOT7, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_DROPDOWN,0,0},true},
    {{ 18,IDC_SIMULATED_ROBOT8, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_DROPDOWN,0,0},true},
    {{ 0,0,                 TBSTATE_ENABLED,TBSTYLE_SEP,   0,0},true},
    {{ 6,IDC_SIMROBOT_SEND_ORACLED_WORLD_STATE_LOCAL, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_CHECK,0,0},true},
    {{ 0,0,                 TBSTATE_ENABLED,TBSTYLE_SEP,   0,0},true},
    {{ 7,IDC_SIMROBOT_CRASHED, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_CHECK,0,0},true},
    {{ 8,IDC_SIMROBOT_BACK_PRESSED, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_CHECK,0,0},true},
    {{ 9,IDC_SIMROBOT_HIND_HEAD_PRESSED, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_CHECK,0,0},true},
    {{ 10,IDC_SIMROBOT_FORE_HEAD_PRESSED, TBSTATE_ENABLED,TBSTYLE_BUTTON | TBSTYLE_CHECK,0,0},true},
  };
  
  SetBitmaps( IDB_SIMULATED_ROBOTS, IDB_SIMULATED_ROBOTS_COLD);
  SetButtons( sizeof( tbButtons ) / sizeof( tbButtons[ 0 ] ), tbButtons);
  
  for (int i=0;i<8;i++)
    setImage(i);
}

bool CSimulatedRobotsToolBar::HasButtonText( UINT nID )
{
  switch ( nID )
  {    
  case IDC_SIMROBOT_SEND_ORACLED_WORLD_STATE_LOCAL:
    return true;
  default:
    return false;
  }
}

bool CSimulatedRobotsToolBar::handleCommand(UINT command)
{
  switch (command)
  {
  case IDC_DISABLE_SIMULATED_ROBOTS:
    {
      CRobotControlSimulatedRobots& robots = getSimulatedRobots();
      
      robots.setSimulatedRobotsAreDisabled(!robots.getSimulatedRobotsAreDisabled());
      
      if (robots.getSimulatedRobotsAreDisabled())
      {
        if (robots.getSimRobotDocument()->IsRunning()) 
        {
          // Stop simulation if running
          getSimulatedRobots().OnSimStart();
        }
      }
    }
    return true;
  case IDC_SIMULATED_ROBOT1:
  case IDC_SIMULATED_ROBOT2:
  case IDC_SIMULATED_ROBOT3:
  case IDC_SIMULATED_ROBOT4:
  case IDC_SIMULATED_ROBOT5:
  case IDC_SIMULATED_ROBOT6:
  case IDC_SIMULATED_ROBOT7:
  case IDC_SIMULATED_ROBOT8:
    {
      CRobotControlSimulatedRobots& simulatedRobots = getSimulatedRobots();
      
      int robot = command - IDC_SIMULATED_ROBOT1;
      if (simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::activeWithImages
        || simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::activeWithoutImages)
      {
        simulatedRobots.setSelectedRobot(robot);
      }
    }
    return true;
  case IDC_SIMROBOT_START:
    getSimulatedRobots().OnSimStart();
    return true;
  case IDC_SIMROBOT_RESET:
    getSimulatedRobots().OnSimReset();
    return true;
  case IDC_SIMROBOT_STEP:
    getSimulatedRobots().OnSimStep();
    return true;
  case IDC_SIMROBOT_CRASHED:
    getSimulatedRobots().getSimRobotDocument()->activateSwitch(SensorData::accelerationY,
      !getSimulatedRobots().getSimRobotDocument()->isSwitchActivated(SensorData::accelerationY));
    return true;
  case IDC_SIMROBOT_BACK_PRESSED:
    getSimulatedRobots().getSimRobotDocument()->activateSwitch(SensorData::back,
      !getSimulatedRobots().getSimRobotDocument()->isSwitchActivated(SensorData::back));
    return true;
  case IDC_SIMROBOT_HIND_HEAD_PRESSED:
    getSimulatedRobots().getSimRobotDocument()->activateSwitch(SensorData::headBack,
      !getSimulatedRobots().getSimRobotDocument()->isSwitchActivated(SensorData::headBack));
    return true;
  case IDC_SIMROBOT_FORE_HEAD_PRESSED:
    getSimulatedRobots().getSimRobotDocument()->activateSwitch(SensorData::headFront,
      !getSimulatedRobots().getSimRobotDocument()->isSwitchActivated(SensorData::headFront));
    return true;
  case IDC_SIMROBOT_SEND_ORACLED_WORLD_STATE_LOCAL:
    getSimulatedRobots().setSendOracle(!getSimulatedRobots().getSendOracle());
    return true;
  default: 
    return false;
  }
}



void CSimulatedRobotsToolBar::updateUI(CCmdUI* pCmdUI)
{
  switch (pCmdUI->m_nID)
  {
  case IDC_DISABLE_SIMULATED_ROBOTS:
    pCmdUI->SetCheck(getSimulatedRobots().getSimulatedRobotsAreDisabled());
    pCmdUI->Enable(true);
    return;
  case IDC_SIMULATED_ROBOT1:
  case IDC_SIMULATED_ROBOT2:
  case IDC_SIMULATED_ROBOT3:
  case IDC_SIMULATED_ROBOT4:
  case IDC_SIMULATED_ROBOT5:
  case IDC_SIMULATED_ROBOT6:
  case IDC_SIMULATED_ROBOT7:
  case IDC_SIMULATED_ROBOT8:
    {
      int robot = pCmdUI->m_nID - IDC_SIMULATED_ROBOT1;
      pCmdUI->SetCheck(getSimulatedRobots().getSelectedRobot() == robot);
      
      pCmdUI->Enable(!getSimulatedRobots().getSimulatedRobotsAreDisabled());
      return;
    }
  case IDC_SIMROBOT_START:
    pCmdUI->Enable(!getSimulatedRobots().getDoRestart() 
      && !getSimulatedRobots().getSimulatedRobotsAreDisabled());

    if (getSimulatedRobots().getSimRobotDocument()!=0)
    {
      pCmdUI->SetCheck(getSimulatedRobots().getSimRobotDocument()->IsRunning()?true:false);
    }
    else
    {
      pCmdUI->SetCheck(false);
    }
    return;
  case IDC_SIMROBOT_STEP:
    if (getSimulatedRobots().getSimRobotDocument())
    {
      pCmdUI->Enable(
        !getSimulatedRobots().getSimRobotDocument()->IsRunning()
        && !getSimulatedRobots().getDoRestart()
        && !getSimulatedRobots().getSimulatedRobotsAreDisabled());
    }
    else
    {
      pCmdUI->Enable(false);
    }
    return;
  case IDC_SIMROBOT_RESET:
    pCmdUI->Enable(!getSimulatedRobots().getSimulatedRobotsAreDisabled());
    return;
  case IDC_SIMROBOT_SEND_ORACLED_WORLD_STATE_LOCAL:
    pCmdUI->SetCheck(getSimulatedRobots().getSendOracle());
    pCmdUI->Enable(!getSimulatedRobots().getDoRestart()
      && !getSimulatedRobots().getSimulatedRobotsAreDisabled());
    return;
  case IDC_STATUS_BAR_SIMULATION_STEPS:
    if (getSimulatedRobots().getSimRobotDocument()!=0 && 
      getSimulatedRobots().getSimRobotDocument()->GetSimulation())
    {
      pCmdUI->Enable(true);
      char buf[20];
      sprintf(buf,"%06d",((SIM3DWorld*) getSimulatedRobots().getSimRobotDocument()->
        GetSimulation()->GetObject(""))->GetSteps());
      pCmdUI->SetText(buf);
    }
    else
    {
      pCmdUI->Enable(FALSE);
      pCmdUI->SetText("000000");
    }
    return;
  case IDC_SIMROBOT_CRASHED:
    {
      CRobotControlSimulatedRobots& robots = getSimulatedRobots();
      if (robots.getSimRobotDocument() 
        && robots.getSelectedRobot() != -1
        && !robots.getDoRestart()
        && !robots.getSimulatedRobotsAreDisabled())
      {
        pCmdUI->Enable(robots.getState(robots.getSelectedRobot())
          == CRobotControlSimulatedRobots::activeWithImages);
        pCmdUI->SetCheck(robots.getSimRobotDocument()->isSwitchActivated(SensorData::accelerationY));
      }
      else
      {
        pCmdUI->Enable(false);
        pCmdUI->SetCheck(0);
      }
      return;
    }
  case IDC_SIMROBOT_BACK_PRESSED:
     {
      CRobotControlSimulatedRobots& robots = getSimulatedRobots();
      if (robots.getSimRobotDocument() 
        && robots.getSelectedRobot() != -1
        && !robots.getDoRestart()
        && !robots.getSimulatedRobotsAreDisabled())
      {
        pCmdUI->Enable(robots.getState(robots.getSelectedRobot())
          == CRobotControlSimulatedRobots::activeWithImages);
        pCmdUI->SetCheck(robots.getSimRobotDocument()->isSwitchActivated(SensorData::back));
      }
      else
      {
        pCmdUI->Enable(false);
        pCmdUI->SetCheck(0);
      }
      return;
    }
  case IDC_SIMROBOT_HIND_HEAD_PRESSED:
    {
      CRobotControlSimulatedRobots& robots = getSimulatedRobots();
      if (robots.getSimRobotDocument() 
        && robots.getSelectedRobot() != -1
        && !robots.getDoRestart()
        && !robots.getSimulatedRobotsAreDisabled())
      {
        pCmdUI->Enable(robots.getState(robots.getSelectedRobot())
          == CRobotControlSimulatedRobots::activeWithImages);
        pCmdUI->SetCheck(robots.getSimRobotDocument()->isSwitchActivated(SensorData::headBack));
      }
      else
      {
        pCmdUI->Enable(false);
        pCmdUI->SetCheck(0);
      }
      return;
    }
  case IDC_SIMROBOT_FORE_HEAD_PRESSED: 
    {
      CRobotControlSimulatedRobots& robots = getSimulatedRobots();
      if (robots.getSimRobotDocument() 
        && robots.getSelectedRobot() != -1
        && !robots.getDoRestart()
        && !robots.getSimulatedRobotsAreDisabled())
      {
        pCmdUI->Enable(robots.getState(robots.getSelectedRobot())
          == CRobotControlSimulatedRobots::activeWithImages);
        pCmdUI->SetCheck(robots.getSimRobotDocument()->isSwitchActivated(SensorData::headFront));
      }
      else
      {
        pCmdUI->Enable(false);
        pCmdUI->SetCheck(0);
      }
      return;
    }
  default:
    return;
  }
}

void CSimulatedRobotsToolBar::setImage(int robot)
{
  switch(getSimulatedRobots().getState(robot))
  {
  case CRobotControlSimulatedRobots::disabled:
    SetImage(IDC_SIMULATED_ROBOT1 + robot, 27 + robot);
    break;
  case CRobotControlSimulatedRobots::passive:
    SetImage(IDC_SIMULATED_ROBOT1 + robot, 35 + robot);
    break;
  case CRobotControlSimulatedRobots::activeWithImages:
    SetImage(IDC_SIMULATED_ROBOT1 + robot, 11 + robot);
    break;
  case CRobotControlSimulatedRobots::activeWithoutImages:
    SetImage(IDC_SIMULATED_ROBOT1 + robot, 19 + robot);
    break;
  }
}

bool CSimulatedRobotsToolBar::handleDropDown(NMHDR* pNotifyStruct)
{
  NMTOOLBAR* pNMToolBar = (NMTOOLBAR*) pNotifyStruct;
  
  switch (pNMToolBar->iItem)
  {
  case IDC_SIMROBOT_ZOOM:
    {
      CMenu menu;
      CMenu* pPopup = 0;
      VERIFY(menu.LoadMenu(IDP_SIMROBOT_ZOOM));
      pPopup = menu.GetSubMenu(0);
      CRect rc;
      ::SendMessage(pNMToolBar->hdr.hwndFrom,
        TB_GETRECT,pNMToolBar->iItem,(LPARAM) &rc);
      rc.top = rc.bottom;
      ::ClientToScreen(pNMToolBar->hdr.hwndFrom,&rc.TopLeft());
      pPopup->TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON,rc.left,rc.top,this);
    }
    return true;
  case IDC_SIMROBOT_DETAIL_LEVEL:
    {
      CMenu menu;
      CMenu* pPopup = 0;
      VERIFY(menu.LoadMenu(IDP_SIMROBOT_DETAIL_LEVEL));
      pPopup = menu.GetSubMenu(0);
      CRect rc;
      ::SendMessage(pNMToolBar->hdr.hwndFrom,
        TB_GETRECT,pNMToolBar->iItem,(LPARAM) &rc);
      rc.top = rc.bottom;
      ::ClientToScreen(pNMToolBar->hdr.hwndFrom,&rc.TopLeft());
      pPopup->TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON,rc.left,rc.top,this);
    }
    return true;
  case IDC_SIMULATED_ROBOT1:
  case IDC_SIMULATED_ROBOT2:
  case IDC_SIMULATED_ROBOT3:
  case IDC_SIMULATED_ROBOT4:
  case IDC_SIMULATED_ROBOT5:
  case IDC_SIMULATED_ROBOT6:
  case IDC_SIMULATED_ROBOT7:
  case IDC_SIMULATED_ROBOT8:
    {
      CRobotControlSimulatedRobots& simulatedRobots = getSimulatedRobots();
      
      int robot = pNMToolBar->iItem - IDC_SIMULATED_ROBOT1;
      CMenu m;
      VERIFY( m.CreatePopupMenu() );
      
      m.AppendMenu(simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::disabled?
        MF_CHECKED | MF_GRAYED : MF_UNCHECKED ,3006,"Disabled");
      
      m.AppendMenu(MF_SEPARATOR,1);
    
      m.AppendMenu(simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::activeWithImages
        && simulatedRobots.getDesign(robot) == RobotDesign::ERS7?
        MF_CHECKED | MF_GRAYED : MF_UNCHECKED ,3000,"ERS7, active, generate images");
      m.AppendMenu(simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::activeWithoutImages
        && simulatedRobots.getDesign(robot) == RobotDesign::ERS7?
        MF_CHECKED | MF_GRAYED : MF_UNCHECKED ,3001,"ERS7, active, don't generate images");
      m.AppendMenu(simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::passive
        && simulatedRobots.getDesign(robot) == RobotDesign::ERS7?
        MF_CHECKED | MF_GRAYED : MF_UNCHECKED ,3002,"ERS7, passive");

      m.AppendMenu(MF_SEPARATOR,1);

      m.AppendMenu(simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::activeWithImages
        && simulatedRobots.getDesign(robot) == RobotDesign::ERS210?
        MF_CHECKED | MF_GRAYED : MF_UNCHECKED ,3003,"ERS210, active, generate images");
      m.AppendMenu(simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::activeWithoutImages
        && simulatedRobots.getDesign(robot) == RobotDesign::ERS210?
        MF_CHECKED | MF_GRAYED : MF_UNCHECKED ,3004,"ERS210, active, don't generate images");
      m.AppendMenu(simulatedRobots.getState(robot) == CRobotControlSimulatedRobots::passive
        && simulatedRobots.getDesign(robot) == RobotDesign::ERS210?
        MF_CHECKED | MF_GRAYED : MF_UNCHECKED ,3005,"ERS210, passive");


      CRect rc;
      ::SendMessage(pNMToolBar->hdr.hwndFrom,
        TB_GETRECT,pNMToolBar->iItem,(LPARAM) &rc);
      rc.top = rc.bottom;
      ::ClientToScreen(pNMToolBar->hdr.hwndFrom,&rc.TopLeft());
      
      // Track menu
      int nID = m.TrackPopupMenu( TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
        rc.left, rc.top, this);
      
      CRobotControlSimulatedRobots::State newState;
      RobotDesign::Design newDesign;

      switch (nID)
      {
      case 3000: 
        newState = CRobotControlSimulatedRobots::activeWithImages;
        newDesign = RobotDesign::ERS7;
        break;
      case 3001:
        newState = CRobotControlSimulatedRobots::activeWithoutImages;
        newDesign = RobotDesign::ERS7;
        break;
      case 3002:
        newState = CRobotControlSimulatedRobots::passive;
        newDesign = RobotDesign::ERS7;
        break;
      case 3003: 
        newState = CRobotControlSimulatedRobots::activeWithImages;
        newDesign = RobotDesign::ERS210;
        break;
      case 3004:
        newState = CRobotControlSimulatedRobots::activeWithoutImages;
        newDesign = RobotDesign::ERS210;
        break;
      case 3005:
        newState = CRobotControlSimulatedRobots::passive;
        newDesign = RobotDesign::ERS210;
        break;
      case 3006:
        newState = CRobotControlSimulatedRobots::disabled;
        newDesign = simulatedRobots.getDesign(robot);
        break;
      default:
        return true;
      }
      
      if (simulatedRobots.getState(robot) != newState 
        || simulatedRobots.getDesign(robot) != newDesign)
      {
        simulatedRobots.setStateAndDesign(robot, newState, newDesign);
        setImage(robot);
      }
    }
    return true;
  default:
    return false;

  }
}

/*
* Change log :
* 
* $Log: SimulatedRobotsToolBar.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:28:01  cvsadm
* created new repository GT2004_WM
*
* Revision 1.8  2004/01/06 14:15:09  loetzsch
* bug fix
*
* Revision 1.7  2004/01/04 18:01:10  loetzsch
* bug fix
*
* Revision 1.6  2004/01/02 16:03:34  loetzsch
* added support for different robot designs to CRobotControlSimulatedRobots
* and to the tool bar
*
* Revision 1.5  2003/12/14 14:23:54  loetzsch
* the buttons for pressing the switches of the robot are disabled
* when the selected robot is "active without images" as they
* have no effect in this case
*
* Revision 1.4  2003/12/12 20:54:40  loetzsch
* The simulator window is now also opened through the main menu instead
* of a button in the simulated robots toolbar.
*
* Revision 1.3  2003/12/12 17:12:24  loetzsch
* With the button "Disable simulated robots" all simulated robots can be
* disabled. But this does not deactivate any simulated robots.
*
* If the simulated robots are disabled, messages from physical robots or a
* log file are directly sent to the queue "toGUI". (One does not need to
* deactivate all simulated robots anymore if one wants to view images from a log file
* without sending them to the selected simulated robot.)
*
* Revision 1.2  2003/12/06 06:31:20  loetzsch
* no message
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.8  2003/05/11 23:46:35  dueffert
* Depend now works with RobotControl too
*
* Revision 1.7  2003/03/24 15:36:27  loetzsch
* bug fixed
*
* Revision 1.6  2003/03/24 14:05:01  loetzsch
* removed crashes when no robot is active
*
* Revision 1.5  2003/03/23 19:14:21  loetzsch
* finished GUI for 8 simulated robots
* - added the passive state
* - added the state[8] variable to CRobotControlSimulatedRobots
* - Repaint of object viewer after reset
* - Resend debug key table after reset
* - don't stop simulation when selected robot changes
*
* Revision 1.4  2003/03/22 22:37:52  loetzsch
* finished GUI and message routing for 8 simulated robots
* almost all functionallity for simulated robots now is encapsulated in class
*   CRobotControlSimulatedRobots
*
* Revision 1.3  2003/03/22 16:39:33  loetzsch
* continued GUI for simulated robots
*
* Revision 1.2  2003/03/21 14:25:38  loetzsch
* continued support for 8 simulated robots
*
* Revision 1.1  2003/03/20 20:36:22  loetzsch
* renamed SimulatorToolbar to Simulated Robots toolbar
* added class CRobotControlSimulatedRobots
* started implementing the GUI for 8 simulated robots
*
* Revision 1.1  2002/09/10 15:49:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.4  2002/09/01 17:19:47  loetzsch
* tidied up the MfcTools/ directory. Created directories
* MfcTools/DockingControlBars and MfcTools/IEStyleToolbars.
*
* Revision 1.3  2002/08/13 12:55:10  loetzsch
* - Added base class CRobotControlToolBar for tool bars
* - Moved command handling from CRobotControlMainFrame to the tool bars
*
* Revision 1.2  2002/07/23 13:43:37  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - empty constructors in bars / dialogs
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
* Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.4  2002/03/29 09:32:42  roefer
* Buttons for switches and crash in simulator toolbar are now working
*
* Revision 1.3  2002/03/26 17:27:26  loetzsch
* Added a button for the back switch
*
* Revision 1.2  2002/02/07 16:26:17  loetzsch
* Added the "send oracle" button
*
* Revision 1.1  2002/01/14 22:40:49  loetzsch
* SimRobot / RobotControl integration started
*
*/
