/**
* @file RadarViewerDlgBar.cpp
*
* Implementation of class CRadarViewerDlgBar
*
* @author Martin Ltzsch
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/

#include "StdAfx.h"
#include "RadarViewerDlgBar.h"

#include "Visualization/DrawingMethods.h"
#include "Visualization/PaintMethodsWin32.h"

#include "Representations/Cognition/RobotState.h"

#include "RadarViewerDlgBar.h"


CRadarViewerDlgBar::CRadarViewerDlgBar()
	: CRobotControlDialogBar(IDD)
{
	//{{AFX_DATA_INIT(CRadarViewerDlgBar)
	//}}AFX_DATA_INIT
  m_LeftButtonDown = false;
  numberOfShownEntries = 1;
}


void CRadarViewerDlgBar::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CRadarViewerDlgBar)
	DDX_Control(pDX, IDC_RADAR_BUFFER_EDIT, m_bufferEdit);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CRadarViewerDlgBar, CDynamicBarDlg)
	//{{AFX_MSG_MAP(CRadarViewerDlgBar)
	ON_WM_PAINT()
	ON_WM_SIZE()
	ON_WM_CONTEXTMENU()
  ON_WM_MOUSEMOVE()
	//}}AFX_MSG_MAP
  ON_WM_LBUTTONDOWN()
  ON_WM_LBUTTONUP()
  ON_EN_CHANGE(IDC_RADAR_BUFFER_EDIT, OnEnChangeRadarBufferEdit)
END_MESSAGE_MAP()


BOOL CRadarViewerDlgBar::OnInitDialog() 
{
	CDynamicBarDlg::OnInitDialog();
  return TRUE;
}

void CRadarViewerDlgBar::OnPaint() 
{
  CPaintDC dc(this); // device context for painting
  CDC dcOffScreen;   // an in memory device context

  int currentWidth  = rect.Width();
  int currentHeight = rect.Height();

  CBitmap  bmpOffScreen, *oldBitmap;
  
  dcOffScreen.CreateCompatibleDC(&dc);
  bmpOffScreen.CreateCompatibleBitmap(&dc, currentWidth, currentHeight);
  oldBitmap = dcOffScreen.SelectObject(&bmpOffScreen);

  //Generate white background
  //
  CPen noPen;
  CPen* oldPen;
  noPen.CreatePen(PS_NULL, 0 , RGB(0,0,0));
  oldPen = dcOffScreen.SelectObject(&noPen);
  dcOffScreen.Rectangle(0, 0, currentWidth + 1, currentHeight + 1);
  dcOffScreen.SelectObject(oldPen);

  // Transformation matrix for turning
  //
  XFORM xform;
  xform.eM11 = 0.0;           xform.eM12 = (float)-scale;
  xform.eM21 = (float)-scale; xform.eM22 = 0.0;
  xform.eDx  = (float)(currentWidth / 2.0);
  xform.eDy  = (float)(currentHeight * 3.0 / 4.0);
  
  SetGraphicsMode(dcOffScreen.m_hDC, GM_ADVANCED);
  SetWorldTransform(dcOffScreen.m_hDC, &xform); 

  paintCurrentSelectionToCDC(dcOffScreen);

  ModifyWorldTransform(dcOffScreen, NULL, MWT_IDENTITY);

  // Copy the off screen dc to the dc for painting
  //
  dc.BitBlt(
    rect.left, rect.top + 22, 
    currentWidth, currentHeight - 22,
    &dcOffScreen, 0, 0, SRCCOPY
  );

  dcOffScreen.SelectObject(oldBitmap);
  dcOffScreen.DeleteDC();
}

void CRadarViewerDlgBar::paintCurrentSelectionToCDC(CDC& dc, bool paintFrame)
{
  int i;
  for(i = numberOfShownEntries - 1; i >= 0; i--)
  {
    if(showImageOnGround && imageBuffer.getNumberOfEntries() > i) 
      PaintMethodsWin32::paintImageProjectionOnGroundToCDC(cameraMatrixBuffer[i], cameraInfoBuffer[i], imageBuffer[i], dc);
  }

  if(paintFrame) PaintMethodsWin32::paintFrameForScalingToCDC(dc);
  if(showCoordinateSystem) PaintMethodsWin32::paintCoordinateSystemToCDC(dc);
  if(showRobot) PaintMethodsWin32::paintRobotToCDC(dc);
  if(showAngleOfView && cameraMatrixBuffer.getNumberOfEntries() > 0) PaintMethodsWin32::paintCameraMatrixToCDC(cameraMatrixBuffer[0], cameraInfoBuffer[0], dc);
  for(i = numberOfShownEntries - 1; i >= 0; i--)
  {
    if(showBallPerceptSizeBased && ballPerceptBuffer.getNumberOfEntries() > i) PaintMethodsWin32::paintBallPerceptToCDC(ballPerceptBuffer[i], false, ballAsCross, lineToBall, dc);
    if(showBallPerceptBearingBased && ballPerceptBuffer.getNumberOfEntries() > i) PaintMethodsWin32::paintBallPerceptToCDC(ballPerceptBuffer[i], true, ballAsCross, lineToBall, dc);
    if(showBallModel && ballModelBuffer.getNumberOfEntries() > i) PaintMethodsWin32::paintBallModelToCDC(ballModelBuffer[i], robotPoseBuffer[i], ballAsCross, lineToBall, dc);
    if(showLandmarksPercept && landmarksPerceptBuffer.getNumberOfEntries() > i) PaintMethodsWin32::paintLandmarksPerceptToCDC(landmarksPerceptBuffer[i], dc);
    if(showLinesPercept && linesPerceptBuffer.getNumberOfEntries() > i) PaintMethodsWin32::paintLinesPerceptToCDC(linesPerceptBuffer[i], dc);
    if(showObstaclesPercept && obstaclesPerceptBuffer.getNumberOfEntries() > i) PaintMethodsWin32::paintObstaclesPerceptToCDC(obstaclesPerceptBuffer[i], dc);
    if(showObstaclesModel && obstaclesModelBuffer.getNumberOfEntries() > i) PaintMethodsWin32::paintObstaclesModelToCDC(obstaclesModelBuffer[i], dc);
    if(showPSDPercept && psdPerceptBuffer.getNumberOfEntries() > i) PaintMethodsWin32::paintPSDPerceptToCDC(psdPerceptBuffer[i], dc);
  }

}

void CRadarViewerDlgBar::copyToClipBoard()
{
  HENHMETAFILE metaFileHandle;
  CMetaFileDC* metaFileDC;
  metaFileDC = new CMetaFileDC();

  metaFileDC->CreateEnhanced(NULL, NULL, NULL, NULL);
  SetGraphicsMode(metaFileDC->m_hDC, GM_ADVANCED); //needed for SetWorldTransform

  // Transformation matrix for turning
  //
  XFORM xform;
  xform.eM11 =  0.0f;  xform.eM12 =  -0.2f;
  xform.eM21 = -0.2f;  xform.eM22 =   0.0f;
  xform.eDx  =  0.0;   xform.eDy  =   0.0;
  SetWorldTransform(metaFileDC->m_hDC, &xform); 
  paintCurrentSelectionToCDC(*metaFileDC, true);

  ModifyWorldTransform(metaFileDC->m_hDC, NULL, MWT_IDENTITY);

  metaFileHandle = metaFileDC->CloseEnhanced();
  
  OpenClipboard();
	EmptyClipboard();
	::SetClipboardData(CF_ENHMETAFILE,metaFileHandle);
	CloseClipboard();

  DeleteEnhMetaFile(metaFileHandle);
  metaFileDC->DeleteDC();
  delete metaFileDC;
}

void CRadarViewerDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  rect.left = 0;
  rect.right = cx;
  rect.top = 0;
  rect.bottom = cy;
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CRadarViewerDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu menu;
  VERIFY( menu.CreatePopupMenu() );

  UINT flags = 0;

  if(showCoordinateSystem) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1900,"coordinate system");
  
  if(showRobot) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1901,"robot");

  menu.AppendMenu(MF_SEPARATOR );

  if(showBallPerceptSizeBased) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2000,"ball percept: size based");

  if(showBallPerceptBearingBased) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2001,"ball percept: bearing based");
  
  if(showBallModel) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2004,"ball model");
  
  if(ballAsCross) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2002,"show ball as cross");

  if(lineToBall) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2003,"show line to ball");

  menu.AppendMenu(MF_SEPARATOR );

  if(showLandmarksPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,3000,"landmarks percept");
  
  if(showPlayersPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,4000,"players percept");
  
  if(showLinesPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5000,"lines percept");

  if(showObstaclesPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5500,"obstacles percept");

  if(showObstaclesModel) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5510,"obstacles model");

  if(showSensorData) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5511,"sensor data");

  if(showJointData) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5512,"joint data");

  if(showAngleOfView) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,6000,"angle of view");

  if(showImage) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,7000,"image");

  if(showImageOnGround) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,8000,"image on ground");

  if(showPSDPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,9000,"PSD Percept");

  menu.AppendMenu(MF_SEPARATOR );

  flags = 0;
  menu.AppendMenu(flags,9100,"copy");

  UINT nID = menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this );
  
  switch(nID)
  {
  case 1900:
    showCoordinateSystem = !showCoordinateSystem;
    break;
  case 1901:
    showRobot = !showRobot;
    break;
  case 2000:
    showBallPerceptSizeBased = !showBallPerceptSizeBased;
    break;
  case 2001:
    showBallPerceptBearingBased = !showBallPerceptBearingBased;
    break;
  case 2004:
    showBallModel = !showBallModel;
    break;
  case 2002:
    ballAsCross = !ballAsCross;
    break;
  case 2003:
    lineToBall = !lineToBall;
    break;
  case 3000:
    showLandmarksPercept = !showLandmarksPercept;
    break;
  case 4000:
    showPlayersPercept = !showPlayersPercept;
    break;
  case 5000:
    showLinesPercept = !showLinesPercept;
    break;
  case 5500:
    showObstaclesPercept = !showObstaclesPercept;
    break;
  case 5510:
    showObstaclesModel = !showObstaclesModel;
    break;
  case 5511:
    showSensorData = !showSensorData;
    break;
  case 5512:
    showJointData = !showJointData;
    break;
  case 6000:
    showAngleOfView = !showAngleOfView;
    break;
  case 7000:
    showImage = !showImage;
    break;
  case 8000:
    showImageOnGround = !showImageOnGround;
    break;
  case 9000:
    showPSDPercept = !showPSDPercept;
    break;
  case 9100:
    copyToClipBoard();
    break;
  }

  InvalidateRect(NULL, false);
}


bool CRadarViewerDlgBar::handleMessage(InMessage& message) 
{
  if (message.getMessageID() == idPercepts)
  {
    Player player;
    LandmarksPercept landmarksPercept;
    BallPercept ballPercept;
    PlayersPercept playersPercept;
    ObstaclesPercept obstaclesPercept;
    PSDPercept psdPercept;
    CollisionPercept collisionPercept;
    LinesPercept linesPercept;
    EdgesPercept edgesPercept;
    CameraMatrix cameraMatrix;
    CameraInfo cameraInfo;

    message.bin >> RECEIVE_PERCEPTS(cameraMatrix, cameraInfo, ballPercept,
      landmarksPercept,linesPercept,edgesPercept,playersPercept,obstaclesPercept, psdPercept, collisionPercept);

    cameraMatrix.frameNumber = ballPercept.frameNumber;

    if(
      cameraMatrixBuffer.getNumberOfEntries() == 0 ||
      cameraMatrix.frameNumber != cameraMatrixBuffer[0].frameNumber)
    {
      cameraMatrixBuffer.add(cameraMatrix);
      cameraInfoBuffer.add(cameraInfo);
    }
    ballPerceptBuffer.add(ballPercept);
    landmarksPerceptBuffer.add(landmarksPercept);
    linesPerceptBuffer.add(linesPercept);
    obstaclesPerceptBuffer.add(obstaclesPercept);
    psdPerceptBuffer.add(psdPercept);
    InvalidateRect(NULL,FALSE);
    return true;
  }
  else if(message.getMessageID() == idWorldState)
  {
    Player player;
    BallModel ballModel;
    PlayerPoseCollection playerPoseCollection;
    ObstaclesModel obstaclesModel;
    RobotState robotState;
    RobotPose robotPose;
    CameraMatrix cameraMatrix;
    CameraInfo cameraInfo;

    message.bin >> RECEIVE_WORLDSTATE(robotPose,
      ballModel,playerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo);
    
    ballModelBuffer.add(ballModel);
    robotPoseBuffer.add(robotPose);
    obstaclesModelBuffer.add(obstaclesModel);
    InvalidateRect(NULL,FALSE);
    return true;
  }
  else if (message.getMessageID() == idImage)
  {
    Image image;
    CameraMatrix cameraMatrix;
    message.bin >> RECEIVE_IMAGE(image, cameraMatrix);
    imageBuffer.add(image);
    cameraMatrixBuffer.add(cameraMatrix);
    InvalidateRect(NULL,FALSE);
    return true;
  }
  return false;
}

void CRadarViewerDlgBar::OnMouseMove(UINT nFlags, CPoint point)
{
  if(m_LeftButtonDown)
  {
    CSize diff = m_LeftDownPos - point;
    m_LeftDownPos = point;
    for(int i = 0; i < abs(diff.cx); i++)
    {
      if(diff.cx >= 0) scale /= 1.1;
      else scale *= 1.1;
    }
    if(scale < 0.05) scale = 0.05;
    if(scale >    5)  scale =    5;
    InvalidateRect(NULL,FALSE);
  }
  CRobotControlDialogBar::OnMouseMove(nFlags, point);
}

void CRadarViewerDlgBar::OnEnChangeRadarBufferEdit()
{
  CString string;
  m_bufferEdit.GetWindowText(string);
  numberOfShownEntries = atoi(string);
  if(numberOfShownEntries > bufferSize ) numberOfShownEntries = bufferSize;
  InvalidateRect(NULL,FALSE);
}

void CRadarViewerDlgBar::OnLButtonDown(UINT nFlags, CPoint point)
{
  m_LeftButtonDown = true;
  m_LeftDownPos = point;
  CRobotControlDialogBar::OnLButtonDown(nFlags, point);
}

void CRadarViewerDlgBar::OnLButtonUp(UINT nFlags, CPoint point)
{
  m_LeftButtonDown = false;
  CRobotControlDialogBar::OnLButtonUp(nFlags, point);
}

void CRadarViewerDlgBar::OnConfigurationLoad(CString sectionName)
{
  showCoordinateSystem = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showCoordinateSystem",1)==1;
  showRobot            = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showRobot",1)==1;
  showBallPerceptSizeBased    = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showBallPerceptSizeBased",0)==1;
  showBallPerceptBearingBased = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showBallPerceptBearingBased",0)==1;
  showBallModel        = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showBallModel",0)==1;
  ballAsCross          = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "ballAsCross",0)==1;
  lineToBall           = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "lineToBall",0)==1;
  showLandmarksPercept = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showLandmarksPercept",0)==1;
  showPlayersPercept   = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showPlayersPercept",0)==1;
  showLinesPercept     = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showLinesPercept",0)==1;
  showObstaclesPercept = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showObstaclesPercept",0)==1;
  showObstaclesModel   = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showObstaclesModel",0)==1;
  showSensorData       = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showSensorData",0)==1;
  showJointData        = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showJointData",0)==1;
  showAngleOfView      = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showAngleOfView",0)==1;
  showImage            = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showImage",0)==1;
  showImageOnGround    = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showImageOnGround",0)==1;
  showPSDPercept       = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "showPSDPercept",0)==1;

  scale = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "scale",1000000 + 200);
  scale -= 1000000; scale /= 1000.0;

  numberOfShownEntries = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer", "bufferSize", 1);
  if(numberOfShownEntries > bufferSize ) numberOfShownEntries = bufferSize;
  CString string;
  string.Format("%i", numberOfShownEntries);
  m_bufferEdit.SetWindowText(string);

  InvalidateRect(NULL, false);
}


void CRadarViewerDlgBar::OnConfigurationSave(CString sectionName)
{
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showCoordinateSystem",showCoordinateSystem?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showRobot",showRobot?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showBallPerceptSizeBased",showBallPerceptSizeBased?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showBallPerceptBearingBased",showBallPerceptBearingBased?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showBallModel",showBallModel?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "ballAsCross",ballAsCross?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "lineToBall",lineToBall?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showLandmarksPercept",showLandmarksPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showPlayersPercept",showPlayersPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showLinesPercept",showLinesPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showObstaclesPercept",showObstaclesPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showObstaclesModel",showObstaclesModel?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showSensorData",showSensorData?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showJointData",showJointData?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showAngleOfView",showAngleOfView?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showImage",showImage?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showImageOnGround",showImageOnGround?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "showPSDPercept",showPSDPercept?1:0);

  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "scale", (int)(scale * 1000 + 1000000));
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer", "bufferSize", numberOfShownEntries);
}

/*
 * Change log :
 * 
 * $Log: RadarViewerDlgBar.cpp,v $
 * Revision 1.3  2004/06/20 14:27:24  kindler
 * - Removed arkward scale-parameter in paint functions, all drawings are now done in world coordinates.
 * - It's easy now to calculate the inverse transformation for interactive DebugDrawings (like the CeilingCamView)
 *
 * Revision 1.2  2004/06/15 10:58:27  thomas
 * added edge-specialist, edges-percept, debug-drawings etc. (not yet called from image-processor)
 *
 * Revision 1.1.1.1  2004/05/22 17:27:53  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.8  2004/03/25 21:22:27  juengel
 * Ball model is shown.
 *
 * Revision 1.7  2004/02/16 18:04:00  dueffert
 * include bug fixed
 *
 * Revision 1.6  2004/02/07 15:48:24  juengel
 * Added buffer edit box to RadarViewer.
 * Added some different modes of view for the BallPercept
 *
 * Revision 1.5  2004/02/05 14:45:05  wachter
 * Fixed "redifinition of i" error.
 *
 * Revision 1.4  2004/02/05 14:31:41  juengel
 * Changed order of drawings.
 *
 * Revision 1.3  2004/02/03 13:09:04  spranger
 * changed include to make DSP_Generation work
 *
 * Revision 1.2  2004/02/02 10:08:56  juengel
 * Redesign of RadarViewer.
 *
 * Revision 1.1  2003/10/07 10:09:38  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2003/05/11 23:46:35  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.2  2002/09/20 23:28:33  juengel
 * Moved instance of DebugDrawingManager to RobotControlMainFrame.
 *
 * Revision 1.1  2002/09/10 15:49:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.2  2002/07/23 13:43:37  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - 5 instead of 3 debug queues in RobotControl
 * - exchanged StaticQueue with MessageQueue
 * - new debug message handling
 * - empty constructors in bars / dialogs
 * - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
 * - general clean up
 *
 * Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.1  2002/02/08 13:28:08  loetzsch
 * Visualization of PerceptCollection and RadarViewer started
 * (not implemented yet)
 *
 */
