/**
* @file RadarViewer3DDlgBar.cpp
* 
* Implementation of class CRadarViewer3DDlgBar.
*
* @author Martin Ltzsch
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/

#include "StdAfx.h"
#include "RadarViewer3DDlgBar.h"
#include "Representations/Motion/JointDataBuffer.h"

#include <gl/gl.h>
#include <gl/glu.h>
#include "Visualization/OpenGLMethods.h"
#include "Tools/Math/Common.h"

#include "Tools/Player.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/PlayersPercept.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/EdgesPercept.h"
#include "Representations/Perception/PSDPercept.h"
#include "Representations/Perception/CollisionPercept.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/RobotState.h"
#include "Platform/Sensors.h"

CRadarViewer3DDlgBar::CRadarViewer3DDlgBar()
: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CRadarViewer3DDlgBar)
		// NOTE: the ClassWizard will add member initialization here
  //}}AFX_DATA_INIT
  m_hGLContext = NULL;
  
  m_xRotate = pi / 3.0;
  m_yRotate = 0;
  m_zRotate = 0;
    
  m_LeftButtonDown = false;
}

void CRadarViewer3DDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDynamicBarDlg::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CRadarViewer3DDlgBar)
	DDX_Control(pDX, IDC_R3D_BUFFER_SIZE_STATIC, m_static_buffer_size);
	DDX_Control(pDX, IDC_R3D_SLIDER_BUFFER_SIZE, m_slider_buffer_size);
	DDX_Control(pDX, IDC_R3D_SLIDER_X_POS, m_slider_x_pos);
	DDX_Control(pDX, IDC_R3D_SLIDER_Y_POS, m_slider_y_pos);
	DDX_Control(pDX, IDC_R3D_SLIDER_Z_POS, m_slider_z_pos);
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CRadarViewer3DDlgBar, CDynamicBarDlg)
ON_WM_CONTEXTMENU()
//{{AFX_MSG_MAP(CRadarViewer3DDlgBar)
ON_WM_PAINT()
ON_WM_CREATE()
ON_WM_DESTROY()
ON_WM_SIZE()
ON_WM_LBUTTONDOWN()
ON_WM_LBUTTONUP()
ON_WM_MOUSEMOVE()
ON_WM_CONTEXTMENU()
	ON_WM_HSCROLL()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CRadarViewer3DDlgBar message handlers

BOOL CRadarViewer3DDlgBar::OnInitDialog()
{
  CDynamicBarDlg::OnInitDialog();

  showBallPercept = true;
  showLandmarksPercept = true;
  showPlayersPercept = true;
  showObstaclesPercept = true;
  showObstaclesModel = true;
  showSensorData = true;
  showJointData = true;
  showLinesPercept = true;
  showAngleOfView = true;
  showImage = false;
  showImageOnGround = false;
  showPSDPercept = false;

  x_pos = 0;
  y_pos = 0;
  z_pos = 4.5;
  imageBufferSize = 1;
  setSliders();
  
  return TRUE;
}

void CRadarViewer3DDlgBar::OnPaint() 
{
  wglMakeCurrent(m_hDC,m_hGLContext);
  
  // ** Draw scene **
  CPaintDC dc(this);
  renderScene();
  glFinish();
  SwapBuffers(dc.m_hDC);
  wglMakeCurrent(NULL,NULL);
}


int CRadarViewer3DDlgBar::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
  //  HDC hDC = ::GetDC(m_hWnd);
  m_hDC = ::GetDC(GetSafeHwnd());
  if(OpenGLMethods::setWindowPixelFormat(m_hDC) == false)
    return 0;
  
  if(OpenGLMethods::createViewGLContext(m_hDC, m_hGLContext) == false)
    return 0;
  
  // Antialiasing
  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
  glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);
  glLineWidth(1.5); // required
  glPointSize(2.5);
  glPolygonMode(GL_FRONT,GL_LINE);
  glPolygonMode(GL_BACK,GL_LINE);
  glShadeModel(GL_SMOOTH);
  
  OpenGLMethods::paintCoordinateSystem(OpenGLMethods::radarViewer3DAxesList);
  
  return 0;
}

void CRadarViewer3DDlgBar::OnDestroy() 
{
  CDynamicBarDlg::OnDestroy();
  
  if(wglGetCurrentContext() != NULL)
    wglMakeCurrent(NULL,NULL);
  
  if(m_hGLContext != NULL)
  {
    wglDeleteContext(m_hGLContext);
    m_hGLContext = NULL;
  }
  
}

void CRadarViewer3DDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  CDynamicBarDlg::OnSize(nType, cx, cy);
  
  wglMakeCurrent(m_hDC,m_hGLContext);
  
  GLsizei width = cx;
  GLsizei height = cy;
  GLdouble aspect;
  
  if(cy==0)
    aspect = (GLdouble)width;
  else
    aspect = (GLdouble)width/(GLdouble)height;
  
  glViewport(0,0,width,height);
  
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  
  gluPerspective(30,aspect,1,100.0);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  glDrawBuffer(GL_BACK);
  glEnable(GL_DEPTH_TEST);
  
  InvalidateRect(NULL,FALSE);
}



void CRadarViewer3DDlgBar::renderScene()
{
  glClearColor(0.75, 0.75, 0.75, 1.0);
  
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  glPushMatrix();

  glTranslated(x_pos, -y_pos ,-z_pos);

  glRotated(m_xRotate, 1.0, 0.0, 0.0);
  glRotated(m_yRotate, 0.0, 1.0, 0.0);
  glRotated(m_zRotate, 0.0, 0.0, 1.0);
  glCallList(OpenGLMethods::radarViewer3DAxesList); 
  if(showBallPercept)
    glCallList(OpenGLMethods::radarViewer3DBallPerceptList); 
  if(showLandmarksPercept)
    glCallList(OpenGLMethods::radarViewer3DLandmarksPerceptList); 
//  if(showPlayersPercept)
//    glCallList(OpenGLMethods::radarViewer3DPlayersPerceptList); 
  if(showLinesPercept)
    glCallList(OpenGLMethods::radarViewer3DLinesPerceptList); 
  if(showObstaclesPercept)
    glCallList(OpenGLMethods::radarViewer3DObstaclesPerceptList); 
  if(showAngleOfView)
    glCallList(OpenGLMethods::radarViewer3DAngleOfViewList); 
  if(showImage)
    glCallList(OpenGLMethods::radarViewer3DImageList); 
  if(showImageOnGround)
    glCallList(OpenGLMethods::radarViewer3DImageOnGroundList); 
  if(showObstaclesModel)
    glCallList(OpenGLMethods::radarViewer3DObstaclesModelList);
  if(showSensorData)
    glCallList(OpenGLMethods::radarViewer3DSensorDataList);
  if(showJointData)
    glCallList(OpenGLMethods::radarViewer3DJointDataList);
  if(showPSDPercept)
    glCallList(OpenGLMethods::radarViewer3DPSDPerceptList);
  glPopMatrix();
}


// MOUSE INTERACTION

void CRadarViewer3DDlgBar::OnLButtonDown(UINT nFlags, CPoint point) 
{
  m_LeftButtonDown = TRUE;
  m_LeftDownPos = point;
  SetCapture();
  CDynamicBarDlg::OnLButtonDown(nFlags, point);
}
void CRadarViewer3DDlgBar::OnLButtonUp(UINT nFlags, CPoint point) 
{
  m_LeftButtonDown = FALSE;
  ReleaseCapture();
  CDynamicBarDlg::OnLButtonUp(nFlags, point);
}


void CRadarViewer3DDlgBar::OnMouseMove(UINT nFlags, CPoint point) 
{
  if(m_LeftButtonDown)
  {
    CSize rotate = m_LeftDownPos - point;
    m_LeftDownPos = point;
    //    m_yRotate -= rotate.cx;
    m_zRotate -= rotate.cx;
    m_xRotate -= rotate.cy;
    InvalidateRect(NULL,FALSE);
  }
  CDynamicBarDlg::OnMouseMove(nFlags, point);
}

bool CRadarViewer3DDlgBar::handleMessage(InMessage& message) 
{
  if (message.getMessageID() == idPercepts)
  {
    if (this->IsWindowVisible())
    {
      Player player;
      LandmarksPercept landmarksPercept;
      BallPercept ballPercept;
      PlayersPercept playersPercept;
      ObstaclesPercept obstaclesPercept;
      PSDPercept psdPercept;
      CollisionPercept collisionPercept;
      LinesPercept linesPercept;
      EdgesPercept edgesPercept;
      CameraMatrix cameraMatrix;
      CameraInfo cameraInfo;


      message.bin >> RECEIVE_PERCEPTS(cameraMatrix, cameraInfo, ballPercept,
        landmarksPercept,linesPercept,edgesPercept,playersPercept,obstaclesPercept, psdPercept, collisionPercept);

      //paint
      wglMakeCurrent(m_hDC,m_hGLContext);
      if(showBallPercept)
        OpenGLMethods::paintBallPerceptToOpenGLList(ballPercept, OpenGLMethods::radarViewer3DBallPerceptList);
      if(showLandmarksPercept)
        OpenGLMethods::paintLandmarksPerceptToOpenGLList(cameraMatrix, landmarksPercept, OpenGLMethods::radarViewer3DLandmarksPerceptList);
//      OpenGLMethods::paintPlayersPerceptToOpenGLList(playersPercept, OpenGLMethods::radarViewer3DPlayersPerceptList);
      if(showLinesPercept)
        OpenGLMethods::paintLinesPerceptToOpenGLList(linesPercept, OpenGLMethods::radarViewer3DLinesPerceptList);
      if(showObstaclesPercept)
        OpenGLMethods::paintObstaclesPerceptToOpenGLList(obstaclesPercept, OpenGLMethods::radarViewer3DObstaclesPerceptList);
      if(showAngleOfView)
        OpenGLMethods::paintAngleOfViewToOpenGLList(cameraMatrix, cameraInfo, OpenGLMethods::radarViewer3DAngleOfViewList);
	  if(showPSDPercept)
        OpenGLMethods::paintPSDPerceptToOpenGLList(cameraMatrix, psdPercept, OpenGLMethods::radarViewer3DPSDPerceptList);
      
      InvalidateRect(NULL,FALSE);
    }
    return true;
  }
  else if (message.getMessageID() == idImage)
  {
    if (this->IsWindowVisible() && (showImage || showImageOnGround) )
    {
      Image image;
      CameraMatrix cameraMatrix;
      message.bin >> RECEIVE_IMAGE(image, cameraMatrix);
      imageRingBuffer.add(image);
      cameraMatrixRingBuffer.add(cameraMatrix);
      if(showAngleOfView)
      {
        wglMakeCurrent(m_hDC,m_hGLContext);
        OpenGLMethods::paintAngleOfViewToOpenGLList(cameraMatrix, image.cameraInfo, OpenGLMethods::radarViewer3DAngleOfViewList);
      }
      if(showImage)
      {
        wglMakeCurrent(m_hDC,m_hGLContext);
        for(int i = 0; i < imageBufferSize; i++)
        {
          OpenGLMethods::paintRotatedImageToOpenGLList(
//            sensorDataBasedCameraMatrix,
            cameraMatrixRingBuffer[i], 
            imageRingBuffer[i], 
            false, 
            OpenGLMethods::radarViewer3DImageList, i==0, i==imageBufferSize-1);
        }
      }
      if(showImageOnGround)
      {
        wglMakeCurrent(m_hDC,m_hGLContext);
        for(int i = 0; i < imageBufferSize; i++)
        {
          OpenGLMethods::paintRotatedImageToOpenGLList(
//            sensorDataBasedCameraMatrix,
            cameraMatrixRingBuffer[i], 
            imageRingBuffer[i], 
            true, 
            OpenGLMethods::radarViewer3DImageOnGroundList, i==0, i==imageBufferSize-1);
        }
      }
      InvalidateRect(NULL,FALSE);
    }
    return true;
  }
  else if(message.getMessageID() == idWorldState)
  {
    if (this->IsWindowVisible())
    {
      Player player;
      BallModel ballModel;
      PlayerPoseCollection playerPoseCollection;
      ObstaclesModel obstaclesModel;
      RobotState robotState;
      RobotPose robotPose;
      CameraMatrix cameraMatrix;
      CameraInfo cameraInfo;

      message.bin >> RECEIVE_WORLDSTATE(robotPose,
        ballModel,playerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo);
      if(showObstaclesModel)
      {
        wglMakeCurrent(m_hDC,m_hGLContext);
        OpenGLMethods::paintObstaclesModelToOpenGLList(obstaclesModel, OpenGLMethods::radarViewer3DObstaclesModelList);
        InvalidateRect(NULL,FALSE);
      }
    }
    return true;
  }
  else if(message.getMessageID() == idJointData)
  {
    if (this->IsWindowVisible())
    {
      JointDataBuffer jointDataBuffer;
      message.bin >> jointDataBuffer;
      jointDataRingBuffer.add(jointDataBuffer.frame[0]);
      if(showJointData)
      {
        wglMakeCurrent(m_hDC,m_hGLContext);
        OpenGLMethods::paintJointDataToOpenGLList(jointDataRingBuffer, OpenGLMethods::radarViewer3DJointDataList);
        InvalidateRect(NULL,FALSE);
      }
    }
    return true;
  }
  else if(message.getMessageID() == idSensorData)
  {
    if (this->IsWindowVisible())
    {
      SensorDataBuffer sensorDataBuffer;
      message.bin >> sensorDataBuffer;
      for(int i = 0; i < 4; ++i)
        sensorDataRingBuffer.add(sensorDataBuffer.frame[i]);
      if(showSensorData)
      {
        wglMakeCurrent(m_hDC,m_hGLContext);
        OpenGLMethods::paintSensorDataToOpenGLList(sensorDataRingBuffer, OpenGLMethods::radarViewer3DSensorDataList);
        //OpenGLMethods::paintSensorDataToOpenGLList(sensorDataBuffer, OpenGLMethods::radarViewer3DSensorDataList);
        InvalidateRect(NULL,FALSE);
      }

      // This comment can be used for calibrating the bodyTilt ...
/*
      sensorDataBasedCameraMatrix;

      // calculate camera matrix
      double tilt = fromMicroRad(sensorDataBuffer.frame[0].data[SensorData::neckTilt]);
      double pan = fromMicroRad(sensorDataBuffer.frame[0].data[SensorData::headPan]);
      double roll = fromMicroRad(sensorDataBuffer.frame[0].data[SensorData::headTilt]);

      double bodyRoll = 0;
      double bodyTilt = fromDegrees(9);
      double neckHeight = 100;

      RobotDimensionsERS7 r;

      (Pose3D&) sensorDataBasedCameraMatrix =
        Pose3D(0,0,neckHeight).
        rotateX(bodyRoll).
        rotateY(bodyTilt-tilt).
        translate(0,0,r.distanceNeckToPanCenter).
        rotateZ(pan).
        rotateY(-roll).
        translate(r.distancePanCenterToCameraX,0,-r.distancePanCenterToCameraZ);
*/      

    }
    return true;
  }
  return false;
}

void CRadarViewer3DDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu menu;
  VERIFY( menu.CreatePopupMenu() );

  UINT flags = 0;

  if(showBallPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2000,"ball percept");
  
  if(showLandmarksPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,3000,"landmarks percept");
  
  if(showPlayersPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,4000,"players percept");
  
  if(showLinesPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5000,"lines percept");

  if(showObstaclesPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5500,"obstacles percept");

  if(showObstaclesModel) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5510,"obstacles model");

  if(showSensorData) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5511,"sensor data");

  if(showJointData) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,5512,"joint data");

  if(showAngleOfView) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,6000,"angle of view");

  if(showImage) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,7000,"image");

  if(showImageOnGround) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,8000,"image on ground");

  if(showPSDPercept) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,9000,"PSD Percept");

  UINT nID = menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this );
  
  wglMakeCurrent(m_hDC,m_hGLContext);

  switch(nID)
  {
  case 2000:
    showBallPercept = !showBallPercept;
    break;
  case 3000:
    showLandmarksPercept = !showLandmarksPercept;
    break;
  case 4000:
    showPlayersPercept = !showPlayersPercept;
    break;
  case 5000:
    showLinesPercept = !showLinesPercept;
    break;
  case 5500:
    showObstaclesPercept = !showObstaclesPercept;
    break;
  case 5510:
    showObstaclesModel = !showObstaclesModel;
    break;
  case 5511:
    showSensorData = !showSensorData;
    break;
  case 5512:
    showJointData = !showJointData;
    break;
  case 6000:
    showAngleOfView = !showAngleOfView;
    break;
  case 7000:
    showImage = !showImage;
    break;
  case 8000:
    showImageOnGround = !showImageOnGround;
    break;
  case 9000:
    showPSDPercept = !showPSDPercept;
    break;
  }

  InvalidateRect(NULL, false);
}

void CRadarViewer3DDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  x_pos = (m_slider_x_pos.GetPos() - 50) / 20.0;
  y_pos = (m_slider_y_pos.GetPos() - 50) / 20.0;
  z_pos = m_slider_z_pos.GetPos() / 10.0 + 1.2;
  imageBufferSize = m_slider_buffer_size.GetPos();
  setSliders();

  wglMakeCurrent(m_hDC,m_hGLContext);
  InvalidateRect(NULL,FALSE);
  
  CDynamicBarDlg::OnHScroll(nSBCode, nPos, pScrollBar);
}

void CRadarViewer3DDlgBar::setSliders()
{
  m_slider_x_pos.SetPos((int)(x_pos * 20.0 + 50));
  m_slider_y_pos.SetPos((int)(y_pos * 20.0 + 50));
  m_slider_z_pos.SetPos((int)((z_pos - 1.2) * 10.0));
  m_slider_buffer_size.SetPos(imageBufferSize);
  
  CString string;
  
  string.Format("%d", imageBufferSize);
  m_static_buffer_size.SetWindowText(string);
}

void CRadarViewer3DDlgBar::OnConfigurationLoad(CString sectionName)
{
  showBallPercept      = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showBallPercept",0)==1;
  showLandmarksPercept = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showLandmarksPercept",0)==1;
  showPlayersPercept   = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showPlayersPercept",0)==1;
  showLinesPercept     = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showLinesPercept",0)==1;
  showObstaclesPercept = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showObstaclesPercept",0)==1;
  showObstaclesModel   = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showObstaclesModel",0)==1;
  showSensorData       = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showSensorData",0)==1;
  showJointData        = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showJointData",0)==1;
  showAngleOfView      = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showAngleOfView",0)==1;
  showImage            = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showImage",0)==1;
  showImageOnGround    = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "showImageOnGround",0)==1;

  x_pos                = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "x_pos",1000000);
  y_pos                = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "y_pos",1000000);
  z_pos                = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "z_pos",4500 + 1000000);

  x_pos -= 1000000; x_pos /= 1000.0;
  y_pos -= 1000000; y_pos /= 1000.0;
  z_pos -= 1000000; z_pos /= 1000.0;

  m_xRotate            = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "m_xRotate",1000000);
  m_yRotate            = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "m_yRotate",1000000);
  m_zRotate            = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "m_zRotate",1000 + 1000000);

  m_xRotate -= 1000000; m_xRotate /= 1000.0;
  m_yRotate -= 1000000; m_yRotate /= 1000.0;
  m_zRotate -= 1000000; m_zRotate /= 1000.0;

  imageBufferSize      = AfxGetApp()->GetProfileInt(sectionName + "\\RadarViewer3D", "imageBufferSize",1);

  setSliders();
  InvalidateRect(NULL, false);
}


void CRadarViewer3DDlgBar::OnConfigurationSave(CString sectionName)
{
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showBallPercept",showBallPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showLandmarksPercept",showLandmarksPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showPlayersPercept",showPlayersPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showLinesPercept",showLinesPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showObstaclesPercept",showObstaclesPercept?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showObstaclesModel",showObstaclesModel?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showSensorData",showSensorData?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showJointData",showJointData?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showAngleOfView",showAngleOfView?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showImage",showImage?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "showImageOnGround",showImageOnGround?1:0);


  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "x_pos", (int)(x_pos * 1000 + 1000000));
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "y_pos", (int)(y_pos * 1000 + 1000000));
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "z_pos", (int)(z_pos * 1000 + 1000000));

  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "m_xRotate", (int)(m_xRotate * 1000 + 1000000));
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "m_yRotate", (int)(m_yRotate * 1000 + 1000000));
  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "m_zRotate", (int)(m_zRotate * 1000 + 1000000));

  AfxGetApp()->WriteProfileInt(sectionName + "\\RadarViewer3D", "imageBufferSize", imageBufferSize);
}


/*
* Change log :
* $Log: RadarViewer3DDlgBar.cpp,v $
* Revision 1.5  2004/06/15 10:58:27  thomas
* added edge-specialist, edges-percept, debug-drawings etc. (not yet called from image-processor)
*
* Revision 1.4  2004/05/27 17:13:38  jhoffman
* - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
* - clipping included for setJoints
* - removed some microrad/rad-bugs
* - bodyPosture constructor and "=" operator fixed
*
* Revision 1.3  2004/05/26 17:32:17  dueffert
* better data types used
*
* Revision 1.2  2004/05/22 22:52:04  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:27:52  cvsadm
* created new repository GT2004_WM
*
* Revision 1.11  2004/02/14 13:05:57  roefer
* SensorData drawn for feet and knees
*
* Revision 1.10  2004/02/03 13:20:48  spranger
* renamed all references to  class BallP_osition to BallModel (possibly changed include files)
*
* Revision 1.9  2004/01/25 21:23:23  juengel
* AngleOfView is drawn, when an image is received.
* Added bodyTiltCalibration mechanism.
*
* Revision 1.8  2004/01/21 17:37:29  mellmann
* BallPercept 3DRadar-visualization  exist & works now :)
*
* Revision 1.7  2004/01/14 19:19:02  mellmann
* Visualization of PSDPercept.
*
* Revision 1.6  2004/01/10 12:39:57  juengel
* Selections are stored dependent on configuration.
*
* Revision 1.5  2004/01/10 10:09:14  juengel
* Added CameraInfo to and removed Player from (SEND|RECEIVE)_(PERCEPTS|WORLDSTATE).
*
* Revision 1.4  2004/01/04 18:20:21  juengel
* Bug fixed.
*
* Revision 1.3  2003/12/15 11:49:07  juengel
* Introduced CameraInfo
*
* Revision 1.2  2003/10/23 07:30:09  juengel
* Added image buffer to radar viewer 3d.
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.6  2003/09/26 21:23:20  loetzsch
* renamed class JointState to CollisionPercept
*
* Revision 1.5  2003/09/26 15:28:23  juengel
* Renamed DataTypes to representations.
*
* Revision 1.4  2003/09/11 15:54:57  juengel
* Added sliders for camera position.
* Added visualization of joint and sensor data.
*
* Revision 1.3  2003/09/01 10:23:14  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.2  2003/08/23 14:28:33  roefer
* fireGLHack added
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.12  2003/06/26 12:35:53  juengel
* Visualization of ObstaclesModel.
*
* Revision 1.11  2003/06/02 09:39:34  goehring
* Added JointState to SEND_PERCEPTS and RECEIVE_PERCEPTS.
*
* Revision 1.10  2003/05/11 23:46:34  dueffert
* Depend now works with RobotControl too
*
* Revision 1.9  2003/03/28 14:29:36  juengel
* Added PSDPercept to SEND_PERCEPTS and RECEIVE_PERCEPTS.
*
* Revision 1.8  2003/03/11 11:26:56  juengel
* Added obstaclesPercept to drawingMethods.
*
* Revision 1.7  2003/03/10 14:07:57  juengel
* Added ObstaclesPercept and ObstaclesModel.
*
* Revision 1.6  2002/11/28 18:53:52  juengel
* RadarViewer3D shows images projected on ground.
*
* Revision 1.5  2002/11/26 19:19:24  loetzsch
* JPEG images are put into local processes
*
* Revision 1.4  2002/11/26 14:17:36  juengel
* RadarViewer3DDlg displays images.
*
* Revision 1.3  2002/10/11 13:45:30  juengel
* Methods for visualization of percepts added.
*
* Revision 1.2  2002/10/10 16:18:39  loetzsch
* some minor improvements and better doxygen comments
*
* Revision 1.1  2002/10/10 13:09:49  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
*/
