/**
* @file MofTesterDlgBar.cpp
* Implementation of class CMofTesterDlgBar.
* @author Max Risler
*/

#include "StdAfx.h"
#include "MofTesterDlgBar.h"

#include "RobotControlQueues.h"
#include "RobotControlDebugKeyTables.h"
#include "Platform/Sensors.h"

#include "Tools/Debugging/Debugging.h"
#include "MessageViewerDlgBar.h"

CMofTesterDlgBar::CMofTesterDlgBar()
	: CRobotControlDialogBar(IDD)
{
	//{{AFX_DATA_INIT(CMofTesterDlgBar)
	//}}AFX_DATA_INIT
  waitForSensorData = false;
}


void CMofTesterDlgBar::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CMofTesterDlgBar)
	DDX_Control(pDX, IDC_MOFTESTER_LINES, m_lines);
	DDX_Control(pDX, IDC_MOFTESTER_STATIC1, m_static1);
	DDX_Control(pDX, IDC_MOFTESTER_STOP_BUTTON, m_stopButton);
	DDX_Control(pDX, IDC_MOFTESTER_MOF_EDIT, m_mofEdit);
	DDX_Control(pDX, IDC_MOFTESTER_READ_BUTTON, m_readButton);
	DDX_Control(pDX, IDC_MOFTESTER_LOOP, m_loopCheck);
  DDX_Control(pDX, IDC_MOFTESTER_EXECUTE_BUTTON, m_executeButton);
  DDX_Control(pDX, IDC_MOFTESTER_MIRROR_BUTTON, m_mirrorButton);
  DDX_Control(pDX, IDC_MOFTESTER_RAWDATA_BUTTON, m_rawdataButton);
  DDX_Control(pDX, IDC_MOFTESTER_SLOWMO_BUTTON, m_slowMoButton);
  DDX_Control(pDX, IDC_MOFTESTER_SLOWMO_EDIT, m_slowMoEdit);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CMofTesterDlgBar, CDynamicBarDlg)
	//{{AFX_MSG_MAP(CMofTesterDlgBar)
	ON_BN_CLICKED(IDC_MOFTESTER_EXECUTE_BUTTON, OnExecuteButton)
	ON_BN_CLICKED(IDC_MOFTESTER_READ_BUTTON, OnReadButton)
	ON_BN_CLICKED(IDC_MOFTESTER_STOP_BUTTON, OnStopButton)
  ON_BN_CLICKED(IDC_MOFTESTER_MIRROR_BUTTON, OnMirrorButton)
  ON_BN_CLICKED(IDC_MOFTESTER_RAWDATA_BUTTON, OnRawDataButton)
  ON_BN_CLICKED(IDC_MOFTESTER_SLOWMO_BUTTON, OnSlowMoButton)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

BOOL CMofTesterDlgBar::OnInitDialog() 
{
	CDynamicBarDlg::OnInitDialog();

  AddSzControl(m_mofEdit,mdResize,mdResize);
  AddSzControl(m_loopCheck,mdRepos,mdRepos);
  AddSzControl(m_lines,mdNone,mdRepos);
  AddSzControl(m_static1,mdNone,mdRepos);
  AddSzControl(m_readButton,mdNone,mdRepos);
  AddSzControl(m_stopButton,mdRepos,mdRepos);
  AddSzControl(m_executeButton,mdRepos,mdRepos);
  AddSzControl(m_mirrorButton,mdNone,mdRepos);
  AddSzControl(m_rawdataButton,mdNone,mdRepos);
  AddSzControl(m_slowMoButton,mdRepos,mdRepos);
  AddSzControl(m_slowMoEdit,mdRepos,mdRepos);

  m_slowMoEdit.SetWindowText("5");

  return TRUE;

}

void CMofTesterDlgBar::OnExecuteButton() 
{
  generateJointDataSequence();
  sendSequence();
}

void CMofTesterDlgBar::OnSlowMoButton() 
{
  CString factor;
  m_slowMoEdit.GetWindowText(factor);
  generateJointDataSequence(atoi(factor.GetBuffer(factor.GetLength())));
  sendSequence();
}


void CMofTesterDlgBar::OnReadButton() 
{
  getDebugKeyTables().forPhysicalRobots.set(DebugKeyTable::sendSensorData,DebugKey::n_times,1);
  getDebugKeyTables().sendForPhysicalRobots();

  waitForSensorData = true;
}

void CMofTesterDlgBar::OnStopButton() 
{
jointDataSequence.numOfEntries = 0;
sendSequence();
  }

void CMofTesterDlgBar::OnMirrorButton()
  {
  char numtostr[10],line[200];
  int i;
  CString input = "";
  
  /* puts the current text into "input" */
  m_mofEdit.GetWindowText(input);
  
  generateJointDataSequence();
  
  input += "\r\n//Mirrored Values:\r\n";
  for (i = 0; i < jointDataSequence.numOfEntries; i++)
    {
    sprintf(line,
      "%5d %5d %5d   %5d %5d %5d   %5d %5d %5d   %5d %5d %5d   %5d %5d %5d   %5d %5d %5d   %d %d",
      jointDataSequence.entry[i].data[JointData::neckTilt] / 1000,
      -jointDataSequence.entry[i].data[JointData::headPan] / 1000,
      -jointDataSequence.entry[i].data[JointData::headTilt] / 1000,
      jointDataSequence.entry[i].data[JointData::mouth] / 1000,
      -jointDataSequence.entry[i].data[JointData::tailPan] / 1000,
      jointDataSequence.entry[i].data[JointData::tailTilt] / 1000,
      jointDataSequence.entry[i].data[JointData::legFL1] / 1000,
      jointDataSequence.entry[i].data[JointData::legFL2] / 1000,
      jointDataSequence.entry[i].data[JointData::legFL3] / 1000,
      jointDataSequence.entry[i].data[JointData::legFR1] / 1000,
      jointDataSequence.entry[i].data[JointData::legFR2] / 1000,
      jointDataSequence.entry[i].data[JointData::legFR3] / 1000,
      jointDataSequence.entry[i].data[JointData::legHL1] / 1000,
      jointDataSequence.entry[i].data[JointData::legHL2] / 1000,
      jointDataSequence.entry[i].data[JointData::legHL3] / 1000,
      jointDataSequence.entry[i].data[JointData::legHR1] / 1000,
      jointDataSequence.entry[i].data[JointData::legHR2] / 1000,
      jointDataSequence.entry[i].data[JointData::legHR3] / 1000,
      (jointDataSequence.interpolate[i])?1:0,
      jointDataSequence.repeats[i]);
      input += line;
      input += "\r\n";
  }
  sprintf(numtostr,"%5d",-jointDataInvalidValue/1000);
  input.Replace(numtostr,"~");
  sprintf(numtostr,"%5d",jointDataInvalidValue/1000);
  input.Replace(numtostr,"~");
  m_mofEdit.SetWindowText(input);
 
  input.Empty();
  
  
}

/*Prints the JointDataValue-C-Code into m_mofEdit*/
void CMofTesterDlgBar::OnRawDataButton()
  {
  char numtostr[10],line[200];
  int i;
  CString input = "";
  CString isLoop;
  CString isInterpol;
  char semi=';';
  
  /* puts the current text into "input" */
  m_mofEdit.GetWindowText(input);
  
  generateJointDataSequence();
  
  input += "\r\n// ------------ Raw Data: ------------\r\n";
  
  if (jointDataSequence.loop==0)
    {
    isLoop="false";
    }
  else
    {
    isLoop="true";
    }
  
  
  
  sprintf(line, "// sequence[NameOfState].numOfEntries = %i%c", jointDataSequence.numOfEntries,semi);
  input += line;
  input += "\r\n";
  
  sprintf(line, "// sequence[NameOfState].loop = %s%c",isLoop,semi);
  input += line;
  input += "\r\n";
  input += "\r\n";
  for (i = 0; i < jointDataSequence.numOfEntries; i++)
    {
    sprintf(line,"// sequence[NameOfState].entry[%i] = JointData(",i);
    input += line;
    sprintf(line,
      "%5d, %5d, %5d,  I, I, I,   %5d, %5d, %5d,   %5d, %5d, %5d,   %5d, %5d, %5d,   %5d, %5d, %5d",
      
      jointDataSequence.entry[i].data[JointData::neckTilt] ,
      jointDataSequence.entry[i].data[JointData::headPan] ,
      jointDataSequence.entry[i].data[JointData::headTilt] ,
      
      //the position of mouth may be included
      //      jointDataSequence.entry[i].data[JointData::mouth] , 
      //      jointDataSequence.entry[i].data[JointData::tailPan] ,
      //      jointDataSequence.entry[i].data[JointData::tailTilt] ,
      
      jointDataSequence.entry[i].data[JointData::legFR1] ,
      jointDataSequence.entry[i].data[JointData::legFR2] ,
      jointDataSequence.entry[i].data[JointData::legFR3] ,
      
      jointDataSequence.entry[i].data[JointData::legFL1] ,
      jointDataSequence.entry[i].data[JointData::legFL2] ,
      jointDataSequence.entry[i].data[JointData::legFL3] ,
      
      jointDataSequence.entry[i].data[JointData::legHR1] ,
      jointDataSequence.entry[i].data[JointData::legHR2] ,
      jointDataSequence.entry[i].data[JointData::legHR3] ,
      
      jointDataSequence.entry[i].data[JointData::legHL1] ,
      jointDataSequence.entry[i].data[JointData::legHL2] ,
      jointDataSequence.entry[i].data[JointData::legHL3]);
      
      
      input += line;
      
      sprintf(line,")%c",semi);
      
      input += line;
      input += "\r\n";
      
      if (jointDataSequence.interpolate[i]==0)
        {
        isInterpol="false";
        }
      else
        {
        isInterpol="true";
        }
      
      sprintf(line,"// sequence[NameOfState].interpolate[%i] = %s%c", i, isInterpol,semi);
      input += line;
      input += "\r\n";
      
      sprintf(line,"// sequence[NameOfState].repeats[%i] = %d%c", i, jointDataSequence.repeats[i],semi);
      input += line;
      input += "\r\n";
      input += "\r\n";
  }
  
  input +="// ------------ End of Raw Data ------------";
  input += "\r\n";
  sprintf(numtostr,"%5d",-jointDataInvalidValue);
  // input.Replace("~","I");
  sprintf(numtostr,"%5d",jointDataInvalidValue);
  // input.Replace("~","I");
  m_mofEdit.SetWindowText(input);
  
  input.Empty();
  
}


bool CMofTesterDlgBar::handleMessage(InMessage& message)
{
  if (message.getMessageID() == idSensorData)
  {
    SensorDataBuffer sensorDataBuffer;
    message.bin >> sensorDataBuffer;
    if (waitForSensorData)
    {
      CString input;
      m_mofEdit.GetWindowText(input);
      
      sprintf(input.GetBuffer(input.GetLength()+200)+input.GetLength(),
        "%d %d %d  ~ ~ ~  %d %d %d  %d %d %d  %d %d %d %d %d %d  0 25\r\n",
        sensorDataBuffer.lastFrame().data[SensorData::neckTilt]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::headPan]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::headTilt]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legFR1]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legFR2]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legFR3]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legFL1]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legFL2]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legFL3]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legHR1]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legHR2]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legHR3]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legHL1]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legHL2]/1000,
        sensorDataBuffer.lastFrame().data[SensorData::legHL3]/1000);
      
      m_mofEdit.SetWindowText(input);
      
      waitForSensorData = false;
    }
    return true;
  }
  return false;
}

void CMofTesterDlgBar::generateJointDataSequence(int slowFactor)
{
  CString input,strLine;
  char numtostr[10];
  int i,start,end;
  long d[20];

  sprintf(numtostr,"%d",jointDataInvalidValue/1000);

  m_mofEdit.GetSel(start,end);
  m_mofEdit.GetWindowText(input);
  if (start!=end) 
  {
    input = input.Left(end);
    input.Delete(0,start);
  }

  input+="\n";
  
  jointDataSequence.numOfEntries = 0;

  while(input.GetLength()>0 && jointDataSequence.numOfEntries<jointDataSequenceMaxNumOfEntries) {
    i=input.Find("\n");
    if (i==-1 || i>1024) break;
    strLine = input.Left(i);
    input.Delete(0,i+1);
    if (i==0 || strLine.GetAt(0)=='/') continue;
    
    strLine.Replace("~",numtostr);
    
    if (sscanf(strLine.GetBuffer(strLine.GetLength()),"%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
      &d[0],&d[1],&d[2],&d[3],&d[4],&d[5],&d[6],&d[7],&d[8],&d[9],
      &d[10],&d[11],&d[12],&d[13],&d[14],&d[15],&d[16],&d[17],&d[18],&d[19]
      )==20) 
    {
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::neckTilt] = d[0] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::headPan] = d[1] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::headTilt] = d[2] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::mouth] = d[3] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::tailPan] = d[4] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::tailTilt] = d[5] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legFR1] = d[6] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legFR2] = d[7] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legFR3] = d[8] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legFL1] = d[9] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legFL2] = d[10] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legFL3] = d[11] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legHR1] = d[12] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legHR2] = d[13] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legHR3] = d[14] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legHL1] = d[15] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legHL2] = d[16] * 1000;
      jointDataSequence.entry[jointDataSequence.numOfEntries].data[JointData::legHL3] = d[17] * 1000;
      jointDataSequence.interpolate[jointDataSequence.numOfEntries] = (d[18] == 1) || (slowFactor != 1);
      jointDataSequence.repeats[jointDataSequence.numOfEntries] = d[19] * slowFactor;
      jointDataSequence.numOfEntries++;
    }
  }
  jointDataSequence.loop = (m_loopCheck.GetCheck()!=0);
  
  sprintf(numtostr,"%d",jointDataSequence.numOfEntries);
  m_lines.SetWindowText(numtostr);
}

void CMofTesterDlgBar::sendSequence()
{
  getQueues().toPhysical.selectedRobot.out.bin << jointDataSequence;
  getQueues().toPhysical.selectedRobot.out.finishMessage(idJointDataSequence);

  getQueues().toSimulated.selectedRobot.out.bin << jointDataSequence;
  getQueues().toSimulated.selectedRobot.out.finishMessage(idJointDataSequence);
}

void CMofTesterDlgBar::updateUI(CCmdUI* pCmdUI)
{
  switch(pCmdUI->m_nID)
  {
  case IDC_MOFTESTER_EXECUTE_BUTTON:
  case IDC_MOFTESTER_LOOP:
    {
      CString input;
      m_mofEdit.GetWindowText(input);
      pCmdUI->Enable( input.GetLength()==0 ? false : true);
    } 
    break;
  default:
    pCmdUI->Enable(true);
  }
}


bool CMofTesterDlgBar::handleCommand(UINT command)
{
  if (command == IDC_SEND_MOTION_NET)
  {
    CString filename("spec_act.dat");
    
    char s[200000];
    memset(s,0,200000);
    InBinaryFile* f = new InBinaryFile(filename);
    if(f->exists())
    {
      f->read(s,200000);
      delete(f);
      
      getQueues().toPhysical.selectedRobot.out.bin.write(s,(int)strlen(s));
      getQueues().toPhysical.selectedRobot.out.finishMessage(idMotionNet);
      getQueues().toSimulated.selectedRobot.out.bin.write(s,(int)strlen(s));
      getQueues().toSimulated.selectedRobot.out.finishMessage(idMotionNet);
    }
    else
    {
      getQueues().toGUI.out.text << filename << " could not be opened.";
      getQueues().toGUI.out.finishMessage(idText);
    }
    return true;
  }
  else
  {
    return false;
  }
}

/*
 * Change log :
 * 
 * $Log: MofTesterDlgBar.cpp,v $
 * Revision 1.2  2004/05/27 17:13:38  jhoffman
 * - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
 * - clipping included for setJoints
 * - removed some microrad/rad-bugs
 * - bodyPosture constructor and "=" operator fixed
 *
 * Revision 1.1.1.1  2004/05/22 17:27:49  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.15  2004/03/17 10:26:07  jhoffman
 * - removed display problems in dialogs
 * - brushed up ugly dialogs
 *
 * Revision 1.14  2004/03/16 14:10:39  heinze
 * removed true-false-bug at "convert to raw data function"
 *
 * Revision 1.13  2004/03/10 17:15:35  risler
 * added slow motion button
 *
 * Revision 1.12  2004/03/03 11:52:32  heinze
 * extended and improved the raw data print
 * - now the complete c-code for the joint data values will be created
 *
 * Revision 1.11  2004/02/29 20:45:17  heinze
 * adding some comments
 *
 * Revision 1.10  2004/02/29 20:34:08  heinze
 * removed warnings
 *
 * Revision 1.9  2004/02/29 20:13:03  loetzsch
 * bug fix (missing bracket in function call) (2)
 *
 * Revision 1.8  2004/02/29 20:11:03  loetzsch
 * bug fix
 *
 * Revision 1.7  2004/02/29 18:22:19  heinze
 * - added the RawData-Button which returns the joint-data-values
 *   for adding them to c-code
 * - removed mirror bug, which deletes all values in textbox
 *
 * Revision 1.6  2003/12/19 17:36:30  loetzsch
 * beautified the mof tester
 *
 * Revision 1.5  2003/12/16 19:02:45  loetzsch
 * The motion net file Config/spec_act.dat can be sent through WLAN to a robot.
 *
 * Revision 1.4  2003/12/16 10:08:43  loetzsch
 * made resizable again
 *
 * Revision 1.3  2003/12/09 19:49:25  loetzsch
 * Renamed some of the main queues of RobotControl.
 *
 * Added possibility to send messages to specific simulated or physical robots.
 *
 * Revision 1.2  2003/12/09 18:14:44  loetzsch
 * changed comments
 *
 * Revision 1.1  2003/10/07 10:09:38  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/05/11 23:46:34  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.5  2003/05/05 14:47:58  risler
 * idJointData debug message sends JointDataBuffer
 * JointViewerDlg shows reference values
 *
 * Revision 1.4  2003/03/05 17:09:27  loetzsch
 * redesign of the queues and debug key tables in RobotControl
 *
 * Revision 1.3  2003/02/01 17:05:03  risler
 * added mirror function to MofTester
 *
 * Revision 1.2  2002/10/01 11:14:34  loetzsch
 * Redesigned DebugKey and DebugKeyTable
 *
 * Revision 1.1  2002/09/10 15:49:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.2  2002/07/23 13:43:37  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - 5 instead of 3 debug queues in RobotControl
 * - exchanged StaticQueue with MessageQueue
 * - new debug message handling
 * - empty constructors in bars / dialogs
 * - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
 * - general clean up
 *
 * Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.3  2002/04/23 17:45:16  loetzsch
 * - splitted debugKeyTable into debugKeyTableForRobot and debugKeyTableForLocalProcesses
 * - removed Modules Toolbar
 * - removed access for dialogs and toolbars to solutionRequest.
 * - changed access for dialogs and toolbars to debug queues
 * - removed the instance of SolutionRequest in CRobotControlApp
 * - only the log player, local processes and the robot put messages into queueToRobotControl
 *
 * Revision 1.2  2002/02/12 16:34:49  risler
 * finished MofTester
 *
 * Revision 1.1  2002/02/12 01:02:31  risler
 * added MofTester
 *
 */

