/**
* @file KickEditorDlgBar.h
*
* Declaration of class CKickEditorDlgBar
*
* @author Matthias Jngel
*/

#ifndef __KickEditorDlgBar_h_
#define __KickEditorDlgBar_h_

#include "StdAfx.h"
#include "RobotControlDialogBar.h"

#include "Tools/KickSelectionTable.h"
#include "Visualization/PaintMethodsWin32.h"
#include "Representations/Motion/MotionRequest.h"

/**
* @class CKickEditorDlgBar 
* 
* A dialog bar for creating the data base for the kick selector
*
* @author Matthias Jngel
*/
class CKickEditorDlgBar : public CRobotControlDialogBar
{
public:
  /** Default constructor.*/
  CKickEditorDlgBar();

  /** Destructor */
  ~CKickEditorDlgBar();

  /** 
  * Called from a MessageQueue to distribute messages 
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);

  //{{AFX_DATA(CKickEditorDlgBar)
	enum { IDD = IDD_DIALOG_BAR_KICK_EDITOR };
	CSliderCtrl	m_sizeSlider;
  CButton m_paintComparisonCheck;
	CComboBox	m_tableCombo;
	CComboBox	m_actionCombo;
	//}}AFX_DATA

  //{{AFX_VIRTUAL(CKickEditorDlgBar)
  protected:
    virtual void DoDataExchange(CDataExchange* pDX);    
    virtual BOOL OnInitDialog();
  //}}AFX_VIRTUAL
  
protected:
  /** A case base of recored kick experiments */
  CArray<KickCase,KickCase&> caseBase[KickSelectionTable::numberOfActions];

  //{{AFX_MSG(CKickEditorDlgBar)
	afx_msg void OnPaint();
	afx_msg void OnSize(UINT nType, int cx, int cy);
	afx_msg void OnLButtonDown(UINT nFlags, CPoint point);
	afx_msg void OnLButtonUp(UINT nFlags, CPoint point);
	afx_msg void OnMouseMove(UINT nFlags, CPoint point);
	afx_msg void OnContextMenu(CWnd* pWnd, CPoint point);
	afx_msg void OnSave();
	afx_msg void OnSelchangeActionCombo();
	afx_msg void OnSelchangeTableCombo();
	afx_msg void OnOpen();
	afx_msg void OnLButtonDblClk(UINT nFlags, CPoint point);
	afx_msg void OnSend();
	afx_msg void OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
  	afx_msg void OnPaintComparisonFirstCheck();
	//}}AFX_MSG
  
  DECLARE_MESSAGE_MAP();

private:
  /** 
  * Is called when a configuration is loaded. Can be used by the dialogs and 
  * bars to load configuration dependand settings.
  * @param sectionName The registry section for the current configuration
  */
  virtual void OnConfigurationLoad(CString sectionName);

  /** 
  * Is called when a configuration is saved. Can be used by the dialogs and 
  * bars to save configuration dependand settings.
  * @param sectionName The registry section for the current configuration
  */
  virtual void OnConfigurationSave(CString sectionName);

  void setSliders();

  void paintKickSelectionTableToCDC(
    CDC& dc, 
    const KickSelectionTable& kickSelectionTable,
    KickSelectionTable::KickSelectionTableID kickSelectionTableID,
    int sector,
    bool paintGrayed
    );

  void paintLegend(CDC& dc);
  void changeKickPerLegend(CPoint point);

  void paintDot(CDC& dc, int x, int y, PaintMethodsWin32::PaintStyle style);


  void paintCurrentSelectionToCDC(CDC& dc, bool paintFrame = false);
  void copyToClipBoard();

  /** The scale for all drawings */
  double scale;

  // Mouse
	CPoint m_LeftDownPos;
	bool m_LeftButtonDown;
  
  int highlightedSector;
  int selectedSector;

  int highlightedX, highlightedY;

  /** The rect where the image is painted*/
  CRect rect;

  bool showCoordinateSystem;
  bool showRobot;
  bool showKickCases;

  int currentWidth;
  int currentHeight;
  int horizontalOffset;
  int verticalOffset;

  KickSelectionTable kickSelectionTable;
  KickSelectionTable kickSelectionTableForComparison;
  KickSelectionTable kickSelectionTableForCopies;

  /** The paint style for each action */
  PaintMethodsWin32::PaintStyle paintStyles[KickSelectionTable::numberOfActions];

private:
  KickSelectionTable::KickSelectionTableID currentTable;
  KickSelectionTable::ActionID currentAction;

  int frameSize;


  // for legend and changeKickPerLegend
  int columnsCount;
  int columnWidth;
  int xOffset, yOffset, ySpacing;
  int actionsPerColumn;

  bool kickRecordWasSent;

  int selectedSectorWhenCopied;
  void copy();
  void pasteForCurrentSector();
  void pasteForAllSectors();
  void mirrorForCurrentSector();
  void mirrorForAllSectors();

  /** Clears the case base and reads all log files which can be found in 
  * "Config/Kick-logs/". */
  void reloadCaseBase();

  void clearCaseBase();

  void loadCaseBaseLogfile(CString fileName);

  void paintKickCase(CDC& dc, KickCase& kickCase, bool highlight);
  void paintCaseBase(CDC& dc, KickSelectionTable::ActionID action);

  void floodFill(int x, int y, KickSelectionTable::ActionID oldAction, KickSelectionTable::ActionID newAction);
  KickSelectionTable::ActionID oldAction;  
  void setFloodFillColor(int x, int y, KickSelectionTable::ActionID newAction);
};

//{{AFX_INSERT_LOCATION}}

#endif //__KickEditorDlgBar_h_

/*
 * Change log :
 * 
 * $Log: KickEditorDlgBar.h,v $
 * Revision 1.12  2004/06/30 22:21:12  risler
 * improved usability (yes indeed)
 * single logfiles can be loaded, not all files read at startup
 *
 * Revision 1.11  2004/06/29 16:08:06  spranger
 * improved kickeditor dialog
 *
 * Revision 1.10  2004/06/23 20:49:58  loetzsch
 * now automatically loads all logfiles in "config/kick-logs".
 *
 * Revision 1.9  2004/06/23 18:19:04  risler
 * kick case drawing improved
 *
 * Revision 1.8  2004/06/20 15:22:36  risler
 * increased kick editor usability:
 * KickLogger now sends kick record via debug key instead of writing to file
 * KickEditor automatically displays sent kick records
 * KickCase moved to KickSelectionTable
 *
 * Revision 1.7  2004/06/20 14:27:24  kindler
 * - Removed arkward scale-parameter in paint functions, all drawings are now done in world coordinates.
 * - It's easy now to calculate the inverse transformation for interactive DebugDrawings (like the CeilingCamView)
 *
 * Revision 1.6  2004/06/02 17:18:23  spranger
 * MotionRequest cleanup
 *
 * Revision 1.5  2004/05/27 16:22:52  goehring
 * renamed some special actions
 *
 * Revision 1.4  2004/05/27 13:07:45  loetzsch
 * bug fix
 *
 * Revision 1.3  2004/05/27 12:34:54  goehring
 * work in progress
 *
 * Revision 1.2  2004/05/26 20:47:13  goehring
 * removed some special actions
 *
 * Revision 1.1.1.1  2004/05/22 17:27:44  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.10  2004/05/07 17:25:39  goetzke
 * improved floodfill
 *
 * Revision 1.9  2004/05/04 18:10:25  juengel
 * Added msh kicks
 *
 * Revision 1.8  2004/04/21 05:50:34  juengel
 * improved
 *
 * Revision 1.7  2004/04/05 17:56:50  loetzsch
 * merged the local German Open CVS of the aibo team humboldt with the tamara CVS
 *
 * Revision 1.2  2004/04/03 02:30:36  jumped
 * added more kicks
 *
 * Revision 1.1.1.1  2004/03/31 11:16:56  loetzsch
 * created ATH repository for german open 2004
 *
 * Revision 1.6  2004/03/19 23:01:52  altmeyer
 * added changeKickPerLegend
 *
 * Revision 1.5  2004/03/19 21:18:28  altmeyer
 * SideHeadKick and SideHeadKick2 changed and renamed to SideHeadKick L and SideHeadKickR
 * a few ATH ERS7-kicks removed
 *
 * Revision 1.4  2004/03/19 11:26:22  juengel
 * recursion depth of floodFill bounded
 *
 * Revision 1.3  2004/03/17 19:55:26  juengel
 * Added OnSend() and floodFill()
 *
 * Revision 1.2  2004/03/16 16:25:02  altmeyer
 * improved visualization using columns and different colors/symbol for each special action
 *
 * Revision 1.1  2004/03/16 14:00:22  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.11  2004/03/11 17:28:02  juengel
 * Added visualization of KickCases.
 *
 * Revision 1.10  2004/03/11 11:41:21  loetzsch
 * implemented loading of case base
 *
 * Revision 1.9  2004/03/10 22:02:03  juengel
 * Added copy and mirror methods.
 *
 * Revision 1.8  2004/03/10 21:17:00  juengel
 * Added Buttons to KickEditor.
 *
 * Revision 1.7  2004/03/10 20:56:52  juengel
 * Moved origin of KickEditor.
 *
 * Revision 1.6  2004/03/10 16:37:58  juengel
 * Added combo boxes to KickEditor.
 *
 * Revision 1.5  2004/03/10 13:47:49  juengel
 * Added save button to KickEditorDlg.
 *
 * Revision 1.4  2004/03/09 18:44:11  juengel
 * Some improvements.
 *
 * Revision 1.3  2004/03/09 13:41:54  juengel
 * Moved KickSelectionTable to Tools.
 *
 * Revision 1.2  2004/03/09 01:14:44  juengel
 * Added class KickDirections.
 *
 * Revision 1.1  2004/03/08 00:17:04  juengel
 * Added the KickEditorDlgBar.
 *
*/
