/**
* @file KalmanSettingsDlgBar.h
* Contains the GUI debugging interface for the KalmanBallLocator
*
* @author <a href="mailto:stefanuhrig@gmx.net">Stefan Uhrig</a>
*/
//------------------------------------------------------------------------------
#ifndef AFX_KALMANSETTINGSDLGBAR_H__3F005660_CDAD_4BC1_98A3_7F7E4855A1A6__INCLUDED_
#define AFX_KALMANSETTINGSDLGBAR_H__3F005660_CDAD_4BC1_98A3_7F7E4855A1A6__INCLUDED_

#include "RobotControlDialogBar.h"
#include "Tools/Streams/InOut.h"

#include <string>
#include <vector>
//------------------------------------------------------------------------------
/**
* @class KalmanModelParameters
* Class that contains the individual model parameters
*/
class KalmanModelParameters
{
public:
  /**
  * Returns the process covariance matrix as a string
  * @return process covarinace matrix as string
  */
  std::string GetQString() const;
  /**
  * Returns the measurement covariance matrix as a string
  * @return measurement covarinace matrix as string
  */
  std::string GetRString() const;
  /**
  * Returns a string represantation of a passed matrix
  * @param n dimension of the matrix
  * @param M vector containing the entries of the matrix
  * @return string represantation of the matrix
  */   
  static std::string GetMatrixString(int n, const std::vector<double>& M);

  /**
  * Parses a matrix string and stores the result to member variable Q
  * @param s matrix string
  * @return true if successful
  */
  bool ParseQString(const std::string& s);
  /**
  * Parses a matrix string and stores the result to member variable R
  * @param s matrix string
  * @return true if successful
  */
  bool ParseRString(const std::string& s);
  /**
  * Parses a matrix string and stores the result to pDest
  * @param n dimension of matrix
  * @param s the matrix string to be parsed
  * @param pDest pointer to array of double values to store result to
  * @return true if parsing was successful
  */
  static bool ParseMatrixString(int n, const std::string s, double* pDest);

public:
  std::string modelName;  ///< name of the model
  bool bUseModel;         ///< true if the model is to be used
  int stateDim;           ///< dimension of the state (and matrix)
  std::vector<double> Q;  ///< process covariance matrix
  std::vector<double> R;  ///< measurement convarinace matrix
};
/**
* Operator to write Kalman model parameters to a stream (processed by the
* KalmanBallLocator)
* @param stream the out stream
* @param kmp reference to a KalmanModelParameters object
* @return reference of out stream
*/
Out& operator<<(Out& stream, const KalmanModelParameters& kmp);
/**
* Operator to read Kalman model parameters from a stream (sent by the
* KalmanBallLocator)
* @param stream the in stream
* @param kmp reference to a KalmanModelParameters object
* @return reference to the in stream
*/
In& operator>>(In& stream, KalmanModelParameters& kmp);
//------------------------------------------------------------------------------
/**
* @class KalmanModelState
* Class that contains the state of a model
*/
class KalmanModelState
{
public:
  KalmanModelState();

public:
  /**
  * Returns string representation of the internal filter covariance matrix
  * @return string representation of the internal filter covariance matrix
  **/
  std::string GetPString() const;
  /**
  * Returns a string represantation of a passed matrix
  * @param n dimension of the matrix
  * @param M vector containing the entries of the matrix
  * @return string represantation of the matrix
  */   
  static std::string GetMatrixString(int n, const std::vector<double>& M);

public:
  double x_rel;        ///< ball x-position relative to robot in millimeters
  double y_rel;        ///< ball y-position relative to robot in millimeters
  double vx;           ///< speed in x-direction in millimeters per second
  double vy;           ///< speed in y-direction in millimeters per second
  double probability;  ///< probability of the model

  int stateDim;           ///< dimension of the state
  std::vector<double> P;  ///< internal filter covariance matrix
};
/**
* Operator to read Kalman model states from a stream (sent by the
* KalmanBallLocator)
* @param stream the in stream
* @param kms reference to a KalmanModelState object
* @return reference to the in stream
*/
In& operator>>(In& stream, KalmanModelState& kms);
//------------------------------------------------------------------------------
/**
* @class CKalmanSettingsDlgBar
* Dialog containing controls to set parameters of Kalman process models
* and to display the states of the process models
*/
class CKalmanSettingsDlgBar : public CRobotControlDialogBar
{
// Konstruktion
public:
	CKalmanSettingsDlgBar();

  /**
  * Handles messages with ID idKalmanData
  * @param message the message to be handled
  * @return true if message was handled
  */
  virtual bool handleMessage(InMessage& message);

	//{{AFX_DATA(CKalmanSettingsDlgBar)
	enum { IDD = IDD_DIALOG_BAR_KALMAN_SETTINGS };
	CEdit	m_editModelY;
	CEdit	m_editModelX;
	CEdit	m_editModelVY;
	CEdit	m_editModelVX;
	CEdit	m_editGlobalY;
	CEdit	m_editGlobalX;
	CEdit	m_editGlobalVY;
	CEdit	m_editGlobalVX;
	CButton	m_checkPhysicalRobot;
  CButton m_checkUseModel;
	CEdit	m_editProbability;
	CEdit	m_editMessages;
	CEdit	m_editMatrixR;
	CEdit	m_editMatrixQ;
	CEdit	m_editMatrixP;
	CComboBox	m_comboKalmanModel;
	//}}AFX_DATA


	//{{AFX_VIRTUAL(CKalmanSettingsDlgBar)
	protected:
	virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV-Untersttzung
	//}}AFX_VIRTUAL

protected:
	//{{AFX_MSG(CKalmanSettingsDlgBar)
	virtual BOOL OnInitDialog();
	afx_msg void OnKsGet();
	afx_msg void OnSelchangeKsKalmanModel();
	afx_msg void OnKsSend();
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()

private:
	/**
  * Reads Kalman parameters sent by a KalmanBallLocator from message stream
  * @param message the message
  */
  void ReadKalmanParameters(InMessage& message);
  /**
  * Request the Kalman parameters from a KalmanBallLocator on the selected
  * robot
  */
  void RequestKalmanParameters();
  /**
  * Reads the states of the Kalman process models from the message stream 
  * (sent by a KalmanBallLocator)
  */
  void ReadFilterStates(InMessage& message);
	/**
  * Displays the parameters of the chosen model
  */
  void ShowSelKalmanParametersEntry();
  /**
  * Displays available Kalman models.
  */
	void ShowKalmanParameters();
	/**
  * Shows the state of the selected Kalman process model
  */
  void ShowSelKalmanState();

private:  
  bool bParamRequestSent; ///< true if request for parameters was sent
  /// Parameters of the kalman process model
  std::vector<KalmanModelParameters> kalmanModelParameters;
  
  unsigned long kalmanStateTime; ///< time calculations where performed at
  RobotPose robotPose;           ///< robot pose
  double global_x_rel; ///< global x-position relative to robot in millimeters
  double global_y_rel; ///< global y-position relative to robot in millimeters
  double global_vx;    ///< global x-direction speed in millimeters per second
  double global_vy;    ///< global y-direction speed in millimeters per second
  bool ballWasSeen;    ///< true if ball was seen
  /// states of the Kalman process models
  std::vector<KalmanModelState> kalmanModelStates;
};
//------------------------------------------------------------------------------
//{{AFX_INSERT_LOCATION}}
// Microsoft Visual C++ fgt unmittelbar vor der vorhergehenden Zeile zustzliche Deklarationen ein.
//------------------------------------------------------------------------------
#endif // AFX_KALMANSETTINGSDLGBAR_H__3F005660_CDAD_4BC1_98A3_7F7E4855A1A6__INCLUDED_
//------------------------------------------------------------------------------
