//------------------------------------------------------------------------------
#include "StdAfx.h"
#include "RobotControl.h"
#include "KalmanSettingsDlgBar.h"
#include "Tools/Debugging/KalmanDebugData.h"
#include <iostream>
#include <sstream>

//------------------------------------------------------------------------------
std::string KalmanModelParameters::GetQString() const
{
  return GetMatrixString(stateDim, Q);
}
//------------------------------------------------------------------------------
std::string KalmanModelParameters::GetRString() const
{
  return GetMatrixString(stateDim, R);
}
//------------------------------------------------------------------------------
std::string
  KalmanModelParameters::GetMatrixString(int n,
                                         const std::vector<double>& M)
{
  std::ostringstream os;
  int i, j;
  for (i = 0; i < n; ++i)
  {
    for (j = 0; j < n; ++j)
    {
      os << M[i*n + j];
      if (j != (n-1))
        os << ", ";
    }
    os << ";\r\n";
  }
  return os.str();
}
//------------------------------------------------------------------------------
bool KalmanModelParameters::ParseQString(const std::string& s)
{
  std::vector<double> tempQ(Q);
  bool bResult = ParseMatrixString(stateDim, s, &tempQ[0]);
  if (bResult)
    Q = tempQ;
  return bResult;
}
//------------------------------------------------------------------------------
bool KalmanModelParameters::ParseRString(const std::string& s)
{
  std::vector<double> tempR(Q);
  bool bResult = ParseMatrixString(stateDim, s, &tempR[0]);
  if (bResult)
    R = tempR;
  return bResult;
}
//------------------------------------------------------------------------------
bool KalmanModelParameters::ParseMatrixString(int n, const std::string s,
                                              double* pDest)
{
  std::string::size_type index = 0;
  std::string::size_type pos;
  int nEntries = 0;

  std::string sub;
  
  while ((nEntries < n*n) &&
         (index < s.length()))
  {
    pos = s.find_first_of(",;", index);
    if (pos == std::string::npos)
      return false;

    sub = s.substr(index, pos-index);
    double v = atof(sub.c_str());
    *pDest = v;
    pDest++;
    
    nEntries++;
    index = pos + 1;    
  }

  if (nEntries != n*n)
    return false;

  return true;
}
//------------------------------------------------------------------------------
Out& operator<<(Out& stream, const KalmanModelParameters& kmp)
{
  int nUse = kmp.bUseModel? 1: 0;
  stream << nUse;
  
  int n = kmp.stateDim;
  stream << n;

  int i;
  for (i = 0; i < n*n; ++i)  
    stream << kmp.Q[i];
  for (i = 0; i < n*n; ++i)
    stream << kmp.R[i];

  return stream;
}
//------------------------------------------------------------------------------
In& operator>>(In& stream, KalmanModelParameters& kmp)
{
  int nLength;
  stream >> nLength;
  std::vector<char> modelName(nLength+1);
  stream >> &modelName[0];
  kmp.modelName = &modelName[0];

  int nUse;
  stream >> nUse;
  kmp.bUseModel = (nUse != 0);

  int n;
  stream >> n;
  kmp.stateDim = n;
  kmp.Q.resize(n*n);
  kmp.R.resize(n*n);

  int i;
  for (i = 0; i < n*n; ++i)
    stream >> kmp.Q[i];
  for (i = 0; i < n*n; ++i)
    stream >> kmp.R[i];
  
  return stream;
}
//------------------------------------------------------------------------------
KalmanModelState::KalmanModelState()
{
  x_rel = 0.0;
  y_rel = 0.0;
  vx = 0.0;
  vy = 0.0;
  probability = 0.0;
  stateDim = 0;
}
//------------------------------------------------------------------------------
std::string KalmanModelState::GetPString() const
{
  return GetMatrixString(stateDim, P);
}
//------------------------------------------------------------------------------
std::string
  KalmanModelState::GetMatrixString(int n,
                                    const std::vector<double>& M)
{
  std::ostringstream os;
  int i, j;
  for (i = 0; i < n; ++i)
  {
    for (j = 0; j < n; ++j)
    {
      os << M[i*n + j];
      if (j != (n-1))
        os << ", ";
    }
    os << ";\r\n";
  }
  return os.str();
}
//------------------------------------------------------------------------------
In& operator>>(In& stream, KalmanModelState& kms)
{
  stream >> kms.x_rel >> kms.y_rel;
  stream >> kms.vx >> kms.vy;
  stream >> kms.probability;
  
  int i, n;
  stream >> n;
  kms.stateDim = n;
  kms.P.resize(n*n);

  for (i = 0; i < n*n; ++i)
    stream >> kms.P[i];
  
  return stream;
}
//------------------------------------------------------------------------------
CKalmanSettingsDlgBar::CKalmanSettingsDlgBar()
	: CRobotControlDialogBar(IDD)
{
  bParamRequestSent = false;
	//{{AFX_DATA_INIT(CKalmanSettingsDlgBar)
	//}}AFX_DATA_INIT
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CKalmanSettingsDlgBar)
	DDX_Control(pDX, IDC_KS_MODEL_Y, m_editModelY);
	DDX_Control(pDX, IDC_KS_MODEL_X, m_editModelX);
	DDX_Control(pDX, IDC_KS_MODEL_VY, m_editModelVY);
	DDX_Control(pDX, IDC_KS_MODEL_VX, m_editModelVX);
	DDX_Control(pDX, IDC_KS_GLOBAL_Y, m_editGlobalY);
	DDX_Control(pDX, IDC_KS_GLOBAL_X, m_editGlobalX);
	DDX_Control(pDX, IDC_KS_GLOBAL_VY, m_editGlobalVY);
	DDX_Control(pDX, IDC_KS_GLOBAL_VX, m_editGlobalVX);
	DDX_Control(pDX, IDC_PHYSICAL_ROBOT, m_checkPhysicalRobot);
  DDX_Control(pDX, IDC_KS_USE_MODEL, m_checkUseModel);
	DDX_Control(pDX, IDC_KS_PROBABILITY, m_editProbability);
	DDX_Control(pDX, IDC_KS_MESSAGES, m_editMessages);
	DDX_Control(pDX, IDC_KS_MATRIX_R, m_editMatrixR);
	DDX_Control(pDX, IDC_KS_MATRIX_Q, m_editMatrixQ);
	DDX_Control(pDX, IDC_KS_MATRIX_P, m_editMatrixP);
	DDX_Control(pDX, IDC_KS_KALMAN_MODEL, m_comboKalmanModel);
	//}}AFX_DATA_MAP
}
//------------------------------------------------------------------------------
BEGIN_MESSAGE_MAP(CKalmanSettingsDlgBar, CDynamicBarDlg)
	//{{AFX_MSG_MAP(CKalmanSettingsDlgBar)
	ON_BN_CLICKED(IDC_KS_GET, OnKsGet)
	ON_CBN_SELCHANGE(IDC_KS_KALMAN_MODEL, OnSelchangeKsKalmanModel)
	ON_BN_CLICKED(IDC_KS_SEND, OnKsSend)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()
//------------------------------------------------------------------------------
BOOL CKalmanSettingsDlgBar::OnInitDialog() 
{
	CDynamicBarDlg::OnInitDialog();	
	return TRUE;
}
//------------------------------------------------------------------------------
bool CKalmanSettingsDlgBar::handleMessage(InMessage& message)
{
  if (message.getMessageID() != idKalmanData)
    return false;

  if (!CRobotControlQueues::isFromSelectedOrUndefinedRobot(message))
    return true;

  char action;
  message.bin >> action;

  switch (action)
  {
    case KalmanDebugData::idSendingParameters:
      ReadKalmanParameters(message);
      ShowKalmanParameters();
      return true;

    case KalmanDebugData::idFilterState:
      if (!bParamRequestSent)
      {
        RequestKalmanParameters();
        return true;
      }
      if (kalmanModelParameters.size() == 0)
        return true;
      ReadFilterStates(message);
      ShowSelKalmanState();
      return true;
  }
  
  message.resetReadPosition();
  return false;
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::ReadKalmanParameters(InMessage& message)
{
  kalmanModelParameters.clear();
  int modelNum;
  message.bin >> modelNum;
  kalmanModelParameters.resize(modelNum);
  kalmanModelStates.resize(modelNum);
    
  int i;
  for (i = 0; i < modelNum; ++i)
    message.bin >> kalmanModelParameters[i];
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::ReadFilterStates(InMessage& message)
{
  message.bin >> kalmanStateTime;
  message.bin >> robotPose;
  message.bin >> global_x_rel >> global_y_rel;
  message.bin >> global_vx >> global_vy;
  char bws;
  message.bin >> bws;
  ballWasSeen = (bws != 0);
  
  int i, n;
  message.bin >> n;
  for (i = 0; i < n; ++i)
    message.bin >> kalmanModelStates[i];
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::OnKsGet() 
{
  RequestKalmanParameters();
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::ShowKalmanParameters()
{
  m_comboKalmanModel.ResetContent();
  m_editMatrixQ.SetWindowText("");
  m_editMatrixR.SetWindowText("");

  if (kalmanModelParameters.size() == 0)
    return;

  unsigned i;
  for (i = 0; i < kalmanModelParameters.size(); ++i)
    m_comboKalmanModel.AddString(kalmanModelParameters[i].modelName.c_str());

  m_comboKalmanModel.SetCurSel(0);
  ShowSelKalmanParametersEntry();
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::ShowSelKalmanParametersEntry()
{
  int nSel = m_comboKalmanModel.GetCurSel();
  m_editMatrixQ.SetWindowText(kalmanModelParameters[nSel].GetQString().c_str());
  m_editMatrixR.SetWindowText(kalmanModelParameters[nSel].GetRString().c_str());
  m_checkUseModel.SetCheck(kalmanModelParameters[nSel].bUseModel? 1: 0);
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::OnSelchangeKsKalmanModel() 
{
  ShowSelKalmanParametersEntry();	
  ShowSelKalmanState();
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::RequestKalmanParameters()
{
  bParamRequestSent = true;
  bool bToPhysical = m_checkPhysicalRobot.GetCheck() == 1;
  OutMessage& message = bToPhysical? getQueues().toPhysical.selectedRobot.out:
                                     getQueues().toSimulated.selectedRobot.out;

  message.bin << (char)KalmanDebugData::idRequestParameters;
  message.finishMessage(idKalmanData);
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::ShowSelKalmanState()
{
  m_editMatrixP.SetWindowText("");
  int nSel = m_comboKalmanModel.GetCurSel();
  if (kalmanModelStates[nSel].stateDim == 0)
    return;

  char szVal[128];
  
  m_editMatrixP.SetWindowText(kalmanModelStates[nSel].GetPString().c_str());

  sprintf(szVal, "%g", kalmanModelStates[nSel].x_rel);
  m_editModelX.SetWindowText(szVal);
  sprintf(szVal, "%g", kalmanModelStates[nSel].y_rel);
  m_editModelY.SetWindowText(szVal);
  sprintf(szVal, "%g", kalmanModelStates[nSel].vx);
  m_editModelVX.SetWindowText(szVal);
  sprintf(szVal, "%g", kalmanModelStates[nSel].vy);
  m_editModelVY.SetWindowText(szVal);

  sprintf(szVal, "%g", kalmanModelStates[nSel].probability);
  m_editProbability.SetWindowText(szVal);

  sprintf(szVal, "%g", global_x_rel);
  m_editGlobalX.SetWindowText(szVal);
  sprintf(szVal, "%g", global_y_rel);
  m_editGlobalY.SetWindowText(szVal);
  sprintf(szVal, "%g", global_vx);
  m_editGlobalVX.SetWindowText(szVal);
  sprintf(szVal, "%g", global_vy);
  m_editGlobalVY.SetWindowText(szVal);
}
//------------------------------------------------------------------------------
void CKalmanSettingsDlgBar::OnKsSend() 
{
  int nSel = m_comboKalmanModel.GetCurSel();
  if (nSel < 0)
    return;
  
  CString s;
  m_editMatrixQ.GetWindowText(s);
  std::string sQ = (LPCTSTR)s;
  if (!kalmanModelParameters[nSel].ParseQString(sQ))
  {
    MessageBox("Invalid Matrix Q", "Parse Error", MB_OK | MB_ICONERROR);
    return;
  }

  m_editMatrixR.GetWindowText(s);
  std::string sR = (LPCTSTR)s;
  if (!kalmanModelParameters[nSel].ParseRString(sR))
  {
    MessageBox("Invalid Matrix R", "Parse Error", MB_OK | MB_ICONERROR);
    return;
  }  

  kalmanModelParameters[nSel].bUseModel = m_checkUseModel.GetCheck() == 1;

  bool bToPhysical = m_checkPhysicalRobot.GetCheck() == 1;
  OutMessage& message = bToPhysical? getQueues().toPhysical.selectedRobot.out:
                                     getQueues().toSimulated.selectedRobot.out;

  message.bin << (char)KalmanDebugData::idSetParameters;
  message.bin << nSel;
  message.bin << kalmanModelParameters[nSel];
  message.finishMessage(idKalmanData);  
}
//------------------------------------------------------------------------------
