/**
* @file JointViewerDlgBar.cpp
* 
* Implementation of class CJointViewerDlgBar.
*
* @author Martin Ltzsch
*/

#include "StdAfx.h"
#include "JointViewerDlgBar.h"

#include "Tools/Math/FourierCoefficient.h"
#include "Tools/Streams/OutStreams.h"
#include "Platform/Sensors.h"

static const COLORREF lineColors[] = 
{
  RGB(255, 0, 0), //red
    RGB(0, 0, 255), // blue
    RGB(0, 150, 0),// dark green
    RGB(210, 30, 100),// dark red
    RGB(60, 200, 60), //green
    RGB(0, 20, 180),// dark blue
    RGB(150, 0, 150),// dark magenta
    RGB(0, 160, 160),// dark cyan
    RGB(120, 160, 220),// sky blue
    RGB(200, 200, 50),// dark yellow
};

static const int numOfLineColors = 9;

static const int maxJointValue=2200000;
static const int minJointValue=-2200000;


BEGIN_MESSAGE_MAP(CJointViewerDlgBar, CDynamicBarDlg)
//{{AFX_MSG_MAP(CJointViewerDlgBar)
ON_WM_PAINT()
ON_WM_SIZE()
ON_WM_HSCROLL()
ON_WM_VSCROLL()
ON_WM_RBUTTONUP()
ON_WM_CONTEXTMENU()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()

CJointViewerDlgBar::CJointViewerDlgBar()
: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CJointViewerDlgBar)
  //}}AFX_DATA_INIT
}

CJointViewerDlgBar::~CJointViewerDlgBar()
{
  AfxGetApp()->WriteProfileInt("JointViewer","pixels per second",pixelsPerSecond);
  
  for (int i=0;i<numOfSensorDataGroups;i++)
  {
    CString key = CString("view sensor ") 
      + CString(getSensorDataGroupName((CJointViewerDlgBar::sensorDataGroup)i)); 
    AfxGetApp()->WriteProfileInt("JointViewer", key, viewSensorDataGroup[i]);
  }
  for (i=0;i<numOfJointDataGroups;i++)
  {
    CString key = CString("view joint ") 
      + CString(getJointDataGroupName((CJointViewerDlgBar::jointDataGroup)i)); 
    AfxGetApp()->WriteProfileInt("JointViewer", key, viewJointDataGroup[i]);
  }
}

void CJointViewerDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CJointViewerDlgBar)
  DDX_Control(pDX, IDC_JOINT_VIEWER_VSCROLL, m_vScrollBar);
  DDX_Control(pDX, IDC_JOINT_VIEWER_HSCROLL, m_hScrollBar);
  //}}AFX_DATA_MAP
}

BOOL CJointViewerDlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();
  
  AddSzControl(m_vScrollBar,mdRepos,mdResize);
  AddSzControl(m_hScrollBar,mdResize,mdRepos);
  
  pixelsPerSecond = AfxGetApp()->GetProfileInt("JointViewer","pixels per second",50);
  
  m_vScrollBar.SetScrollRange(10,250,FALSE);
  m_vScrollBar.SetScrollPos(pixelsPerSecond,FALSE);
  m_hScrollBar.SetScrollRange(0,10000,FALSE);
  
  for (int i=0;i<numOfSensorDataGroups;i++)
  {
    CString key = CString("view sensor ") 
      + CString(getSensorDataGroupName((CJointViewerDlgBar::sensorDataGroup)i)); 
    viewSensorDataGroup[i] = (AfxGetApp()->GetProfileInt("JointViewer", key, 0)!=0);
  }
  CString key = CString("view sensor ") + CString(getSensorDataGroupName(sdLegFL)); 
  viewSensorDataGroup[sdLegFL] = (AfxGetApp()->GetProfileInt("JointViewer", key, 1)!=0);
  
  for (i=0;i<numOfJointDataGroups;i++)
  {
    CString key = CString("view joint ") 
      + CString(getJointDataGroupName((CJointViewerDlgBar::jointDataGroup)i)); 
    viewJointDataGroup[i] = (AfxGetApp()->GetProfileInt("JointViewer", key, 0)!=0);
  }
  key = CString("view joint ") + CString(getJointDataGroupName(jdLegFL)); 
  viewJointDataGroup[jdLegFL] = (AfxGetApp()->GetProfileInt("JointViewer", key, 1)!=0);
  
  sensorDataArray.SetSize(100,50);
  sensorDataTimeStampsArray.SetSize(100,50);
  
  clear();
  
  // initializations for joint data period time calculation
  jointDataPeriodTime = 10000;

  jointValue[0] = 0;
  jointValue[1] = 0;
  jointValue[2] = 0;

  lastOccurrenceOfJointValue0Time = 0;

  return TRUE;
}

bool CJointViewerDlgBar::handleMessage(InMessage& message)
{
  if (!this->IsWindowVisible())
  {
    // no necessity to collect data if the dialog is not visible
    return true;
  }

  switch(message.getMessageID())
  {
  case idJointData:
    {
      JointData jointData;
      message.bin >> jointData;

      unsigned long time = message.getTimeStamp();

      calculateJointDataPeriodTime(jointData, time);
      
      if (numOfJointData > 0)
      {
        if (time < jointDataTimeStampsArray[numOfJointData-1] 
          || time < lastTime - 2000 || time > lastTime + 10000)
        {
          // clear if the sensordata is newer than the last one 
          // or if there was a break of 10 seconds
          clear();
        }
      }
      if (firstTime==0) firstTime=time;
      
      jointDataArray.Add(jointData);
      jointDataTimeStampsArray.Add(time);
      
      if (lastDisplayedTime < time) lastDisplayedTime = time;
      lastTime = lastDisplayedTime;
      
      updateHScroll();
      
      numOfJointData++;

      Invalidate(true);
      RedrawWindow(NULL, NULL, RDW_INVALIDATE);
      
      return true;
    }
  case idSensorData:
    {
      
      SensorDataBuffer sensorDataBuffer;
      message.bin >> sensorDataBuffer;
      
      if (sensorDataBuffer.numOfFrames == 0) return true;

      unsigned long time = message.getTimeStamp();
      
      if (numOfSensorData > 0)
      {
        if (time + 2000 < sensorDataTimeStampsArray[numOfSensorData-1] 
          || time + 2000 < lastTime || time > lastTime + 10000)
        {
          // clear if the sensordata is newer than the last one 
          // or if there was a break of 5 seconds
          clear();
        }
      }
      if (firstTime==0) firstTime=time;
      
      for (int i = 0; i < sensorDataBuffer.numOfFrames; i++)
      {
        sensorDataArray.Add(sensorDataBuffer.frame[i]);
        time += 8;
        sensorDataTimeStampsArray.Add(time);
        numOfSensorData++;
      }
      
      if (lastDisplayedTime < time) lastDisplayedTime = time;
      lastTime = lastDisplayedTime;
      
      updateHScroll();
      
      Invalidate(true);
      RedrawWindow(NULL, NULL, RDW_INVALIDATE);
      return true;
    }
  default:
    return false;
  }
}

void CJointViewerDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  scopeRect.left = 0;
  scopeRect.right = cx - 170;
  scopeRect.top = 0;
  scopeRect.bottom = cy - 20;
  inscriptionRect.left = scopeRect.right + 15;
  inscriptionRect.right = cx - 20;
  inscriptionRect.top = scopeRect.top;
  inscriptionRect.bottom = scopeRect.bottom;
  updateHScroll();
  Invalidate(true);
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  CDynamicBarDlg::OnSize(nType, cx, cy);
}

void CJointViewerDlgBar::updateHScroll()
{
  // check if the scrollbar should be activated
  if (scopeRect.right - scopeRect.left 
    - (int)((double)(lastTime - firstTime) * ((double)pixelsPerSecond/double(1000.0))) 
    > 0)
  {
    // the screen is not filled 
    bHScrollEnabled = false;
    return;
  }
  
  bHScrollEnabled = true;
  
  // the time range that can be displayed on the screen
  unsigned long displayedTimeRange = (unsigned long)
    (((double)(scopeRect.right - scopeRect.left)/(double)pixelsPerSecond)*(double)1000.0);
  
  if (lastDisplayedTime < firstTime + displayedTimeRange) 
    lastDisplayedTime = firstTime + displayedTimeRange;
  
  m_hScrollBar.SetScrollPos((int)((double)(lastDisplayedTime - firstTime - displayedTimeRange)
    *10000.0/(double)(lastTime-firstTime - displayedTimeRange)));
}

void CJointViewerDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  // the time that can be displayed on the screen
  unsigned long displayedTimeRange = (unsigned long)
    (((double)(scopeRect.right - scopeRect.left)/(double)pixelsPerSecond)*(double)1000.0);
  
  int zOrig, z;   // z = x or y depending on pScrollBar
  int zMin, zMax;
  // determines how much is scrolled on SB_PAGERIGHT and SB_PAGELEFT
  int pageWidth = (int)((double)displayedTimeRange * 0.9 
    / (double)(lastTime - firstTime - displayedTimeRange) * 10000.0);
  
  // determines how much is scrolled on SB_LINELEFT and SB_LINERIGHT
  int lineWidth = pageWidth/10;
  
  zOrig = z = pScrollBar->GetScrollPos();
  pScrollBar->GetScrollRange(&zMin,&zMax);
  switch (nSBCode)
  {
  case SB_LEFT: z = zMin;break;
  case SB_RIGHT: z = zMax;break;
  case SB_LINELEFT:z -= lineWidth;break;
  case SB_LINERIGHT:z += lineWidth;break;
  case SB_PAGELEFT: z -= pageWidth; break;
  case SB_PAGERIGHT: z += pageWidth; break;
  case SB_THUMBTRACK: z = nPos; break;
  case SB_THUMBPOSITION: z = nPos; break;
  default: break;// ignore other notifications
    
  }
  if (z < zMin) z = zMin; else if (z > zMax) z = zMax;
  
  if (z != zOrig)
  {
    pScrollBar->SetScrollPos(z);
  }
  
  lastDisplayedTime = (unsigned long)
    ((double)z * (double)(lastTime - firstTime - displayedTimeRange)/10000.0)
    + firstTime + displayedTimeRange;
  
  Invalidate(true);
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CJointViewerDlgBar::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  int zOrig, z;   // z = x or y depending on pScrollBar
  int zMin, zMax;
  zOrig = z = pScrollBar->GetScrollPos();
  pScrollBar->GetScrollRange(&zMin,&zMax);
  switch (nSBCode)
  {
  case SB_TOP:z = zMin;break;
  case SB_BOTTOM:z = zMax;break;
  case SB_LINEUP:z -= 2;break;
  case SB_LINEDOWN:z += 2;break;
  case SB_PAGEUP: z -= 10; break;
  case SB_PAGEDOWN: z += 10; break;
  case SB_THUMBTRACK: z = nPos; break;
  case SB_THUMBPOSITION: z = nPos; break;
  default: return;// ignore other notifications
    
  }
  if (z < zMin) z = zMin; else if (z > zMax) z = zMax;
  
  if (z != zOrig)
  {
    pScrollBar->SetScrollPos(z);
  }
  pixelsPerSecond = z;
  updateHScroll();
  Invalidate(true);
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CJointViewerDlgBar::OnPaint() 
{
 	CPaintDC dc(this); // device context for painting
  
  // the currently used color
  int colorIndex = 0;
  
  // Paint the scope
  
  // the x and y for the next LineTo
  int x, y;
  
  // paint the sensor data
  
  // the index where the sensor data painting starts
  int sensorDataStartIndex = 0;
  
  if (numOfSensorData > 0)
  {
    bool paintVerticalLine = false;
    
    for (int i=numOfSensorData-1; i> 0; i--)
    {
      if (sensorDataTimeStampsArray[i] <= lastDisplayedTime) break;
    }
    sensorDataStartIndex = i;
    
    for (int s=0;s < SensorData::numOfSensor_ERS7;s++)
    {
      SensorData::sensors sensor = (SensorData::sensors) s;
      
      if( viewSensorDataGroup[getSensorDataGroup(sensor)])
      {
        paintVerticalLine = true;
        
        CPen linePen;
        linePen.CreatePen(PS_SOLID, 1, lineColors[colorIndex%numOfLineColors]);
        dc.SelectObject(&linePen);
        colorIndex++;
        
        for (i=sensorDataStartIndex;i>=0;i--)
        {
          x = scopeRect.right 
            - (int)(((double)lastDisplayedTime - (double)sensorDataTimeStampsArray[i])
            * ((double)pixelsPerSecond/1000.0));
          
          y = (scopeRect.bottom - scopeRect.top)/2 
            - (int)(getSensorDataScaleFactor(sensor) * sensorDataArray[i].data[sensor]
            * ((double)(scopeRect.bottom - scopeRect.top)/(double)(maxJointValue-minJointValue)));
          
          if (x<scopeRect.left) break;
          
          if (i==sensorDataStartIndex)
          {
            dc.MoveTo(x,y);
          }
          else
          {
            dc.LineTo(x,y);
          }
        }
      }
    }
    for (s=0;s < SensorData::numOfSensor_ERS7;s++)
    {
      SensorData::sensors sensor = (SensorData::sensors) s;
      
      if( viewSensorDataGroup[getSensorDataGroup(sensor)] &&
          ( getSensorDataGroup(sensor) == sdHead ||
            getSensorDataGroup(sensor) == sdLegFR ||
            getSensorDataGroup(sensor) == sdLegFL ||
            getSensorDataGroup(sensor) == sdLegHL ||
            getSensorDataGroup(sensor) == sdLegHR ||
            getSensorDataGroup(sensor) == sdMouth)
        )
      {
        paintVerticalLine = true;
        
        CPen linePen;
        linePen.CreatePen(PS_SOLID, 1, lineColors[colorIndex%numOfLineColors]);
        dc.SelectObject(&linePen);
        colorIndex++;
        
        for (i=sensorDataStartIndex;i>=0;i--)
        {
          x = scopeRect.right 
            - (int)(((double)lastDisplayedTime - (double)sensorDataTimeStampsArray[i])
            * ((double)pixelsPerSecond/1000.0));
          
          y = (scopeRect.bottom - scopeRect.top)/2 
            - (int)(getSensorDataScaleFactor(sensor) * sensorDataArray[i].refValue[sensor]
            * ((double)(scopeRect.bottom - scopeRect.top)/(double)(maxJointValue-minJointValue)));
          
          if (x<scopeRect.left) break;
          
          if (i==sensorDataStartIndex)
          {
            dc.MoveTo(x,y);
          }
          else
          {
            dc.LineTo(x,y);
          }
        }
      }
    }
    if (paintVerticalLine)
    {
      // draw a vertical line to indicate for which position the inscriptions are for
      x = scopeRect.right -  
        (int)(((double)lastDisplayedTime - (double)sensorDataTimeStampsArray[sensorDataStartIndex])
        * ((double)pixelsPerSecond/1000.0));
      if (x > scopeRect.left)
      {
        CPen linePen;
        linePen.CreatePen(PS_SOLID, 1, RGB(110,110,110));
        dc.SelectObject(&linePen);
        dc.MoveTo(x,scopeRect.top);
        dc.LineTo(x,scopeRect.bottom);
      }
    }
  }
  
  
  // paint the joint data
  
  // the index where the joint data painting starts
  int jointDataStartIndex = 0;
  
  if (numOfJointData > 0)
  {
    bool paintVerticalLine = false;
    
    for (int i=numOfJointData-1; i> 0; i--)
    {
      if (jointDataTimeStampsArray[i] <= lastDisplayedTime) break;
    }
    jointDataStartIndex = i;
    
    for (int j=0;j < JointData::numOfJoint;j++)
    {
      JointData::JointID joint = (JointData::JointID)j;
      
      if( viewJointDataGroup[getJointDataGroup(joint)])
      {
        paintVerticalLine = true;
        
        CPen linePen;
        linePen.CreatePen(PS_SOLID, 1, lineColors[colorIndex%numOfLineColors]);
        dc.SelectObject(&linePen);
        colorIndex++;
        
        for (i=jointDataStartIndex;i>=0;i--)
        {
          x = scopeRect.right 
            - (int)(((double)lastDisplayedTime - (double)jointDataTimeStampsArray[i])
            * ((double)pixelsPerSecond/1000.0));
          
          y = (scopeRect.bottom - scopeRect.top)/2 
            - (int)(getJointDataScaleFactor(joint) * jointDataArray[i].data[joint]
            * ((double)(scopeRect.bottom - scopeRect.top)/(double)(maxJointValue-minJointValue)));
          
          if (x<scopeRect.left) break;
          
          if (i==jointDataStartIndex)
          {
            dc.MoveTo(x,y);
          }
          else
          {
            dc.LineTo(x,y);
          }
        }
      }
    }
    if (paintVerticalLine)
    {
      // draw a vertical line to indicate for which position the inscriptions are for
      x = scopeRect.right -  
        (int)(((double)lastDisplayedTime - (double)jointDataTimeStampsArray[jointDataStartIndex])
        * ((double)pixelsPerSecond/1000.0));
      if (x > scopeRect.left)
      {
        CPen linePen;
        linePen.CreatePen(PS_SOLID, 1, RGB(110,110,110));
        dc.SelectObject(&linePen);
        dc.MoveTo(x,scopeRect.top);
        dc.LineTo(x,scopeRect.bottom);
      }
    }
  }
  
  // Paint the inscription
  
  // the vertical position 
  int pos=0;
  colorIndex=0;
  
  dc.SetBkMode(TRANSPARENT);
  CFont font;
  font.CreateFont(
    14,                        // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    ANSI_CHARSET,              // nCharSet
    OUT_DEFAULT_PRECIS,        // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    DEFAULT_PITCH | FF_SWISS,  // nPitchAndFamily
    "Arial");                 // lpszFacename
  
  for (int s=0;s < SensorData::numOfSensor_ERS7;s++)
  {
    SensorData::sensors sensor = (SensorData::sensors) s;
    
    if( viewSensorDataGroup[getSensorDataGroup(sensor)])
    {
      dc.SetTextColor(lineColors[colorIndex%numOfLineColors]);
      dc.SelectObject(&font);
      colorIndex++;
      char str[100];
      strcpy(str,"S: ");
      strcat(str,SensorData::getSensorName(sensor));
      strcat(str,":");
      dc.TextOut(inscriptionRect.left,15*pos + inscriptionRect.top,str);
      if(sensorDataStartIndex>0)
      {
        ltoa(sensorDataArray[sensorDataStartIndex].data[sensor],str,10);
        dc.TextOut(inscriptionRect.left + 85,15*pos + inscriptionRect.top,str);
      }
      pos++;
    }
  }
  pos++;
  for (s=0;s < SensorData::numOfSensor_ERS7;s++)
  {
    SensorData::sensors sensor = (SensorData::sensors) s;
    
      if( viewSensorDataGroup[getSensorDataGroup(sensor)] &&
          ( getSensorDataGroup(sensor) == sdHead ||
            getSensorDataGroup(sensor) == sdLegFR ||
            getSensorDataGroup(sensor) == sdLegFL ||
            getSensorDataGroup(sensor) == sdLegHL ||
            getSensorDataGroup(sensor) == sdLegHR ||
            getSensorDataGroup(sensor) == sdMouth)
        )
    {
      dc.SetTextColor(lineColors[colorIndex%numOfLineColors]);
      dc.SelectObject(&font);
      colorIndex++;
      char str[100];
      strcpy(str,"R: ");
      strcat(str,SensorData::getSensorName(sensor));
      strcat(str,":");
      dc.TextOut(inscriptionRect.left,15*pos + inscriptionRect.top,str);
      if(sensorDataStartIndex>0)
      {
        ltoa(sensorDataArray[sensorDataStartIndex].refValue[sensor],str,10);
        dc.TextOut(inscriptionRect.left + 85,15*pos + inscriptionRect.top,str);
      }
      pos++;
    }
  }
  pos++;
  CString string;
  string.Format("leg joint period time: %d ms", jointDataPeriodTime);
  dc.SetTextColor(lineColors[1]);
  dc.SelectObject(&font);
  dc.TextOut(inscriptionRect.left,15*pos + inscriptionRect.top,string);
  pos++;
  pos++;
  for (int j=0;j < JointData::numOfJoint;j++)
  {
    JointData::JointID joint = (JointData::JointID)j;
    
    if( viewJointDataGroup[getJointDataGroup(joint)])
    {
      dc.SetTextColor(lineColors[colorIndex%numOfLineColors]);
      dc.SelectObject(&font);
      colorIndex++;
      char str[100];
      strcpy(str,"J: ");
      strcat(str,JointData::getJointName(joint));
      strcat(str,":");
      dc.TextOut(inscriptionRect.left,15*pos + inscriptionRect.top,str);
      if(jointDataStartIndex>0)
      {
        ltoa(jointDataArray[jointDataStartIndex].data[joint],str,10);
        dc.TextOut(inscriptionRect.left + 85,15*pos + inscriptionRect.top,str);
      }
      pos++;
    }
  }
  
  CDynamicBarDlg::OnPaint();
}

void CJointViewerDlgBar::clear()
{
  sensorDataArray.RemoveAll();
  sensorDataTimeStampsArray.RemoveAll();
  jointDataArray.RemoveAll();
  jointDataTimeStampsArray.RemoveAll();
  numOfSensorData = 0;
  numOfJointData = 0;
  
  firstTime = 0;
  lastTime = 0;
  lastDisplayedTime = 0;
  
  updateHScroll();
  Invalidate();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CJointViewerDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu menu;
  VERIFY( menu.CreatePopupMenu() );
  
  UINT flags = lastTime==firstTime?MF_GRAYED:0;
  
  menu.AppendMenu(flags,1000,"Clear");
  menu.AppendMenu(flags,1003,"Save all to CSV file (seperate time lines)");
  //menu.AppendMenu(flags,1004,"[ Save as CSV to file (time stamps buggy) ]");
  menu.AppendMenu(MF_SEPARATOR );
  menu.AppendMenu(flags,1005,"Save Fourier spectrum of leg motion (ASCII)");
  menu.AppendMenu(MF_SEPARATOR );
  menu.AppendMenu(0,1002,"All off");
  menu.AppendMenu(MF_SEPARATOR );
  for (int i=0; i< numOfSensorDataGroups; i++)
  {
    UINT flag = viewSensorDataGroup[i]?MF_BYCOMMAND | MF_CHECKED : MF_BYCOMMAND;
    menu.AppendMenu( flag, i, getSensorDataGroupName((sensorDataGroup)i));
  }
  menu.AppendMenu(MF_SEPARATOR );
  for (i=0; i< numOfJointDataGroups; i++)
  {
    UINT flag = viewJointDataGroup[i]?MF_BYCOMMAND | MF_CHECKED : MF_BYCOMMAND;
    menu.AppendMenu( flag, i+100, getJointDataGroupName((jointDataGroup)i));
  }
  
  UINT nID = menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this );
  
  switch(nID)
  {
  case 1000:
    clear();
    break;
  case 1003:
    saveAllCSV();
    break;
  case 1004:
    saveAsCSV();
    break;
  case 1005:
    saveFTSpectrum();
    break;
  case 1002:
    for (i=0;i<numOfSensorDataGroups;i++) viewSensorDataGroup[i] = false;
    for (i=0;i<numOfJointDataGroups;i++) viewJointDataGroup[i] = false;
    Invalidate(true);
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
    break;
  default:
    if (nID < numOfSensorDataGroups)
    {
      viewSensorDataGroup[nID] = ! viewSensorDataGroup[nID];
    }
    else if (nID < numOfJointDataGroups + 100)
    {
      viewJointDataGroup[nID-100] = ! viewJointDataGroup[nID-100];
    }
    Invalidate(true);
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
    
    break;
  }
}



void CJointViewerDlgBar::saveFTSpectrum()
{
  /** @todo make it variable */
  const int lengthOfPeriod=80;
  long functionValues[lengthOfPeriod];
  long i, t, joint;
  FourierCoefficient coefficients;
  
  if((jointDataArray.GetSize() - lengthOfPeriod) < 0)
  {
    AfxMessageBox("No complete period of data in the joint data array! Send more joint data from the robot or the local processes.", MB_OK);
    return;
  }

  for (joint = JointData::legFR1; joint <= JointData::legHL3; joint++)
  {
    /** create an one dimensional array that holds the 
    * values of the function over the last period. 
    */
    for (i = 0, t = (int)(jointDataArray.GetSize() - lengthOfPeriod); 
    i < lengthOfPeriod; 
    i++, t++)
    {
      functionValues[i] = jointDataArray[t].data[joint];
    }
    /** Use these values to calculate the (descrete) 
    * fourier transform (spectrum) of the function 
    */
    coefficients.calculate(functionValues, (JointData::JointID)joint);
  }
  
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = AfxGetApp()->GetProfileString("JointViewer", "FourierSpectrumSavePath", defaultPath);
  pathName += "*.fc";

  CFileDialog fileDialog(false, ".fc",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "fourier coefficient files (*.fc)|*.fc||", this);

  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("JointViewer", "FourierSpectrumSavePath", pathName);

    OutTextFile fout(pathAndFileName);
    fout << coefficients;
  }
}


void CJointViewerDlgBar::saveAsCSV()
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');

  CString pathName = 
    AfxGetApp()->GetProfileString("JointViewer", "jointCSVsavePath", defaultPath);
  pathName += "*.csv";
  
  CFileDialog fileDialog(false, ".csv",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "csv files (*.csv)|*.csv||", this);
  
  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("JointViewer", "jointCSVsavePath", pathName);
    
    OutTextFile stream(pathAndFileName);
    
    stream << "time,";
    for (int s=0;s<SensorData::numOfSensor_ERS7;s++)
    {
      SensorData::sensors sensor = (SensorData::sensors)s;
      
      if (viewSensorDataGroup[getSensorDataGroup(sensor)])
      {
        stream << "S:" << SensorData::getSensorName(sensor) << ",";
      }
    }
    for (s=0;s<SensorData::numOfSensor_ERS7;s++)
    {
      SensorData::sensors sensor = (SensorData::sensors)s;
      
      if( viewSensorDataGroup[getSensorDataGroup(sensor)] &&
          ( getSensorDataGroup(sensor) == sdHead ||
            getSensorDataGroup(sensor) == sdLegFR ||
            getSensorDataGroup(sensor) == sdLegFL ||
            getSensorDataGroup(sensor) == sdLegHL ||
            getSensorDataGroup(sensor) == sdLegHR ||
            getSensorDataGroup(sensor) == sdMouth)
        )
      {
        stream << "R:" << SensorData::getSensorName(sensor) << ",";
      }
    }
    for (int j=0;j<JointData::numOfJoint;j++)
    {
      JointData::JointID joint = (JointData::JointID)j;
      
      if (viewJointDataGroup[getJointDataGroup(joint)])
      {
        stream << "J:" << JointData::getJointName(joint) << ",";
      }
    }
    stream << endl;
    
    unsigned long currentTime = firstTime;
    int currentJointDataArrayPos = 0;
    int currentSensorDataArrayPos = 0;
    
    while (currentTime <= lastTime)
    {
      stream << currentTime - firstTime << ",";
      for (int s=0;s<SensorData::numOfSensor_ERS7;s++)
      {
        SensorData::sensors sensor = (SensorData::sensors)s;
        
        if (viewSensorDataGroup[getSensorDataGroup(sensor)])
        {
          if (numOfSensorData > currentSensorDataArrayPos)
          {
            if (sensorDataTimeStampsArray[currentSensorDataArrayPos] == currentTime)
            {
              stream << sensorDataArray[currentSensorDataArrayPos].data[sensor] << ",";
            }
            else
            {
              stream << ",";
            }
          }
          else
          {
            stream << ",";
          }
        }
      }
      for (s=0;s<SensorData::numOfSensor_ERS7;s++)
      {
        SensorData::sensors sensor = (SensorData::sensors)s;
        
        if( viewSensorDataGroup[getSensorDataGroup(sensor)] &&
            ( getSensorDataGroup(sensor) == sdHead ||
              getSensorDataGroup(sensor) == sdLegFR ||
              getSensorDataGroup(sensor) == sdLegFL ||
              getSensorDataGroup(sensor) == sdLegHL ||
              getSensorDataGroup(sensor) == sdLegHR ||
              getSensorDataGroup(sensor) == sdMouth)
          )
        {
          if (numOfSensorData > currentSensorDataArrayPos)
          {
            if (sensorDataTimeStampsArray[currentSensorDataArrayPos] == currentTime)
            {
              stream << sensorDataArray[currentSensorDataArrayPos].refValue[sensor] << ",";
            }
            else
            {
              stream << ",";
            }
          }
          else
          {
            stream << ",";
          }
        }
      }
      for (int j=0;j<JointData::numOfJoint;j++)
      {
        JointData::JointID joint = (JointData::JointID)j;
        
        if (viewJointDataGroup[getJointDataGroup(joint)])
        {
          if (numOfJointData > currentJointDataArrayPos)
          {
            if (jointDataTimeStampsArray[currentJointDataArrayPos] == currentTime)
            {
              stream << jointDataArray[currentJointDataArrayPos].data[joint] << ",";
            }
            else
            {
              stream << ",";
            }
          }
          else
          {
            stream << ",";
          }
        }
      }
      stream << endl;
      
      unsigned long nextJointDataTime = 0;
      if (numOfJointData > currentJointDataArrayPos + 1 )
      {
        nextJointDataTime=jointDataTimeStampsArray[currentJointDataArrayPos+1];
      }
      unsigned long nextSensorDataTime = 0;
      if(numOfSensorData > currentSensorDataArrayPos + 1)
      {
        nextSensorDataTime=sensorDataTimeStampsArray[currentSensorDataArrayPos+1];
      }
      
      if (nextSensorDataTime == 0 && nextJointDataTime != 0)
      {
        currentTime = nextJointDataTime;
        currentJointDataArrayPos++;
      }
      else if (nextSensorDataTime != 0 && nextJointDataTime == 0)
      {
        currentTime = nextSensorDataTime;
        currentSensorDataArrayPos++;
      }
      else if (nextSensorDataTime != 0 && nextJointDataTime != 0)
      {
        if (nextSensorDataTime < nextJointDataTime)
        {
          currentTime = nextSensorDataTime;
          currentSensorDataArrayPos++;
        }
        else if (nextSensorDataTime == nextJointDataTime)
        {
          currentTime = nextSensorDataTime;
          currentJointDataArrayPos++;
          currentSensorDataArrayPos++;
        }
        else
        {
          currentTime = nextJointDataTime;
          currentJointDataArrayPos++;
        }
      } 
      else
      {
        break;
      }
    }
  }
}


void CJointViewerDlgBar::saveAllCSV()
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("JointViewer", "jointCSVsavePath", defaultPath);
  pathName += "*.csv";
  
  CFileDialog fileDialog(false, ".csv",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "csv files (*.csv)|*.csv||", this);
  
  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("JointViewer", "jointCSVsavePath", pathName);
    
    OutTextFile stream(pathAndFileName);
  /* write table titles */
  stream << "time,"; // time stamp for sensor data
  for (int s = 0; s < SensorData::numOfSensor_ERS7; s++)
    stream << "S:" << SensorData::getSensorName((SensorData::sensors)s) << ",";

  for (s = 0; s < SensorData::numOfSensor_ERS7; s++)
    if( getSensorDataGroup((SensorData::sensors)s) == sdHead ||
        getSensorDataGroup((SensorData::sensors)s) == sdLegFR ||
        getSensorDataGroup((SensorData::sensors)s) == sdLegFL ||
        getSensorDataGroup((SensorData::sensors)s) == sdLegHL ||
        getSensorDataGroup((SensorData::sensors)s) == sdLegHR ||
        getSensorDataGroup((SensorData::sensors)s) == sdMouth
      )
    stream << "R:" << SensorData::getSensorName((SensorData::sensors)s) << ",";


  stream << "time,"; // time stamp for joint data (usually differs from sensor data time stamp)
  for (int j = 0; j < JointData::numOfJoint;j++)
    stream << "J:" << JointData::getJointName((JointData::JointID)j) << ",";
  stream << endl;
  
  int currentSensorDataArrayPos = 0;
  int currentJointDataArrayPos = 0;
  unsigned long currentSensorTime = 0;
  unsigned long currentJointTime = 0;

  if (numOfSensorData == 0 && numOfJointData == 0) return;

  if (numOfSensorData > 0)
    currentSensorTime = sensorDataTimeStampsArray[0];
  if (numOfJointData > 0)
    currentJointTime = jointDataTimeStampsArray[0];

  unsigned long timeOffset;
  if (currentSensorTime > currentJointTime)
    timeOffset = currentJointTime;  
  else 
    timeOffset = currentSensorTime;
  
  while ((currentSensorTime <= lastTime) || (currentJointTime <= lastTime))
  {
    /* output time stamp for sensor data*/
    if (currentSensorTime)
      stream << currentSensorTime - timeOffset;
    stream << ",";
    
    /* output all sensor data's */
    for (int s = 0; s < SensorData::numOfSensor_ERS7; s++)
    {    
      if (numOfSensorData > currentSensorDataArrayPos)
        if (sensorDataTimeStampsArray[currentSensorDataArrayPos] == currentSensorTime)
          stream << sensorDataArray[currentSensorDataArrayPos].data[(SensorData::sensors)s] << ",";
        else stream << ",";
        else stream << ",";
    }
    for (s = 0; s < SensorData::numOfSensor_ERS7; s++)
    {    
      if( getSensorDataGroup((SensorData::sensors)s) == sdHead ||
          getSensorDataGroup((SensorData::sensors)s) == sdLegFR ||
          getSensorDataGroup((SensorData::sensors)s) == sdLegFL ||
          getSensorDataGroup((SensorData::sensors)s) == sdLegHL ||
          getSensorDataGroup((SensorData::sensors)s) == sdLegHR ||
          getSensorDataGroup((SensorData::sensors)s) == sdMouth
        )
      {
        if (numOfSensorData > currentSensorDataArrayPos)
          if (sensorDataTimeStampsArray[currentSensorDataArrayPos] == currentSensorTime)
            stream << sensorDataArray[currentSensorDataArrayPos].refValue[(SensorData::sensors)s] << ",";
          else stream << ",";
        else stream << ",";
      }
    }
    
    /* output time stamp for joint data*/
    if (currentJointTime)
      stream << currentJointTime - timeOffset;
    stream << ",";
    
    /* output all joint (target) values */
    for (int j = 0; j < JointData::numOfJoint; j++)
    {
      if (numOfJointData > currentJointDataArrayPos)
        if (jointDataTimeStampsArray[currentJointDataArrayPos] == currentJointTime)
          stream << jointDataArray[currentJointDataArrayPos].data[(JointData::JointID)j] << ",";
        else stream << ",";
        else stream << ",";
    }
    stream << endl;
    
    unsigned long nextSensorDataTime = 0;
    if(numOfSensorData > currentSensorDataArrayPos + 1)
      nextSensorDataTime = sensorDataTimeStampsArray[currentSensorDataArrayPos + 1];
    
    if (nextSensorDataTime != 0)
    {
      currentSensorTime = nextSensorDataTime;
      currentSensorDataArrayPos++;
    }
    else currentSensorTime = 0;
    
    unsigned long nextJointDataTime = 0;
    if (numOfJointData > currentJointDataArrayPos + 1)
      nextJointDataTime = jointDataTimeStampsArray[currentJointDataArrayPos + 1];
    
    if (nextJointDataTime != 0)
    {
      currentJointTime = nextJointDataTime;
      currentJointDataArrayPos++;
    } 
    else currentJointTime = 0;
    
    if ((nextJointDataTime == 0) && (nextSensorDataTime == 0))
      break; 
  }  
  }
}

char* CJointViewerDlgBar::getSensorDataGroupName(sensorDataGroup group)
{
  switch (group)
  {
  case sdHead: return "Sensors: Head";
  case sdSwitches: return "Sensors: Switches";
  case sdLegFL: return "Sensors: LegFL"; 
  case sdLegHL: return "Sensors: LegHL"; 
  case sdLegFR: return "Sensors: LegFR"; 
  case sdLegHR: return "Sensors: LefHR"; 
  case sdPaws: return "Sensors: Paws";
  case sdAcceleration: return "Sensors: Acceleration"; 
  case sdTail: return "Sensors: Tail"; 
  case sdPsd: return "Sensors: Psd"; 
  case sdThermo: return "Sensors: Thermo"; 
  case sdMouth: return "Sensors: Mouth"; 
  case sdChin: return "Sensors: Chin"; 
  default: return "unknown Sensor";
  }
}

char* CJointViewerDlgBar::getJointDataGroupName(jointDataGroup group)
{
  switch (group)
  {
  case jdHead: return "Actorics: Head";
  case jdLegFR: return "Actorics: LegFR"; 
  case jdLegFL: return "Actorics: LegFL"; 
  case jdLegHR: return "Actorics: LegHR"; 
  case jdLegHL: return "Actorics: LegHL"; 
  case jdTail: return "Actorics: Tail"; 
  case jdMouth: return "Actorics: Mouth"; 
  case jdEars: return "Actorics: Ears";
  default: return "Actorics: Hallo";
  }
}

CJointViewerDlgBar::sensorDataGroup CJointViewerDlgBar::getSensorDataGroup(SensorData::sensors sensor)
{
  switch(sensor)
  {
  case SensorData::neckTilt: 
  case SensorData::headPan: 
  case SensorData::headTilt: return sdHead;
  case SensorData::headBack: 
  case SensorData::headFront: return sdSwitches;
  case SensorData::bodyPsd:
  case SensorData::headPsdFar:
  case SensorData::psd: return sdPsd;
  case SensorData::mouth: return sdMouth;
  case SensorData::chin: return sdChin;
  case SensorData::legFL1: 
  case SensorData::legFL2: 
  case SensorData::legFL3: return sdLegFL;
  case SensorData::pawFL: return sdPaws;
  case SensorData::legHL1: 
  case SensorData::legHL2: 
  case SensorData::legHL3: return sdLegHL;
  case SensorData::pawHL: return sdPaws;
  case SensorData::legFR1: 
  case SensorData::legFR2: 
  case SensorData::legFR3: return sdLegFR;
  case SensorData::pawFR: return sdPaws;
  case SensorData::legHR1: 
  case SensorData::legHR2: 
  case SensorData::legHR3: return sdLegHR;
  case SensorData::pawHR: return sdPaws;
  case SensorData::tailPan: 
  case SensorData::tailTilt: return sdTail;
  case SensorData::thermo: return sdThermo;
  case SensorData::back: return sdSwitches;
  case SensorData::accelerationY: 
  case SensorData::accelerationX: 
  case SensorData::accelerationZ: return sdAcceleration;
  default: return sdHead;
  }
}

CJointViewerDlgBar::jointDataGroup CJointViewerDlgBar::getJointDataGroup(JointData::JointID joint)
{
  switch (joint)
  {
  case JointData::neckTilt: 
  case JointData::headPan:  
  case JointData::headTilt: return jdHead; 
  case JointData::mouth: return jdMouth;
  case JointData::earL: 
  case JointData::earR: return jdEars; 
  case JointData::legFR1:  
  case JointData::legFR2:  
  case JointData::legFR3: return jdLegFR;
  case JointData::legFL1:  
  case JointData::legFL2:  
  case JointData::legFL3: return jdLegFL; 
  case JointData::legHR1:  
  case JointData::legHR2:  
  case JointData::legHR3: return jdLegHR; 
  case JointData::legHL1:  
  case JointData::legHL2:  
  case JointData::legHL3: return jdLegHL; 
  case JointData::tailPan:  
  case JointData::tailTilt: return jdTail; 
  default: return jdHead;
  }
}

double CJointViewerDlgBar::getSensorDataScaleFactor(SensorData::sensors sensor)
{
  switch (sensor)
  {
  case SensorData::headBack: 
  case SensorData::headFront: return 10.0;
  case SensorData::psd: return 2.0;
  case SensorData::mouth: return 10.0;
  case SensorData::chin: return 100.0;
  case SensorData::pawFL: 
  case SensorData::pawHL: 
  case SensorData::pawFR: 
  case SensorData::pawHR: return 1000000.0;
  case SensorData::tailPan: 
  case SensorData::tailTilt: return 5.0;
  case SensorData::thermo: return 0.01;
  case SensorData::back: return 10.0;
  case SensorData::accelerationY: 
  case SensorData::accelerationX: 
  case SensorData::accelerationZ: return 0.2;
    
  default: return 1.0;
  }
  
}

double CJointViewerDlgBar::getJointDataScaleFactor(JointData::JointID joint)
{
  switch (joint)
  {
  case JointData::tailPan:  
  case JointData::tailTilt: return 5.0;
  case JointData::mouth: return 10.0;
  case JointData::earL: 
  case JointData::earR: return 100.0; 
  default: return 1.0;
  }
}

void CJointViewerDlgBar::calculateJointDataPeriodTime
(
 JointData jointData,
 unsigned long time
 )
{
  if(jointValue[0] == jointData.data[JointData::legFR1])
  {
    jointDataPeriodTime = time - lastOccurrenceOfJointValue0Time;
    lastOccurrenceOfJointValue0Time = time;
  }
  if(time > lastOccurrenceOfJointValue0Time + jointDataPeriodTime + 5000)
  {
    jointValue[0] = jointData.data[JointData::legFR1];
    lastOccurrenceOfJointValue0Time = time;
  }
}

void CJointViewerDlgBar::updateUI(CCmdUI* pCmdUI)
{
  if (pCmdUI->m_nID == IDC_JOINT_VIEWER_HSCROLL)
  {
    pCmdUI->Enable(bHScrollEnabled);
  }
}

/* 
* Change Log:
* 
* $Log: JointViewerDlgBar.cpp,v $
* Revision 1.2  2004/05/27 17:13:38  jhoffman
* - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
* - clipping included for setJoints
* - removed some microrad/rad-bugs
* - bodyPosture constructor and "=" operator fixed
*
* Revision 1.1.1.1  2004/05/22 17:27:37  cvsadm
* created new repository GT2004_WM
*
* Revision 1.6  2004/01/05 10:04:20  juengel
* ERS7
*
* Revision 1.5  2003/12/31 20:16:14  roefer
* SensorData for ERS-7
*
* Revision 1.4  2003/12/06 06:31:20  loetzsch
* no message
*
* Revision 1.3  2003/12/04 17:33:18  loetzsch
* data are only collected when the dialog is visible
*
* Revision 1.2  2003/11/30 01:53:19  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.2  2003/09/26 11:40:12  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.13  2003/05/11 23:46:33  dueffert
* Depend now works with RobotControl too
*
* Revision 1.12  2003/05/05 14:47:57  risler
* idJointData debug message sends JointDataBuffer
* JointViewerDlg shows reference values
*
* Revision 1.11  2003/03/03 17:33:55  risler
* fixed bug when saving empty array to file
*
* Revision 1.10  2002/12/10 10:47:04  jhoffman
* debugged and pretty much working
*
* Revision 1.9  2002/12/10 08:04:50  dueffert
* no message
*
* Revision 1.8  2002/12/04 12:21:00  jhoffman
* no message
*
* Revision 1.7  2002/11/26 14:18:08  juengel
* Period detection for joint viewer added.
*
* Revision 1.6  2002/11/22 13:41:20  loetzsch
* - removed the FourierCoefficient::loadLegs and ::saveLegs functions
*   (streaming operators are now used)
* - .fcb files have text format now
* . moved the .fcb files from /Config to /Config/Fourier
*
* Revision 1.5  2002/11/19 17:14:14  risler
* coding conventions: renamed JointData::joint to JointID, GetName to getName
*
* Revision 1.4  2002/11/18 17:21:17  dueffert
* RobotControl should be startable in any path now
*
* Revision 1.3  2002/10/04 10:22:16  loetzsch
* Invalidate and RedrawWindow are only called in the handleMessage
* function when the dialog is visible
*
* Revision 1.2  2002/09/22 18:40:51  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:49:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.3  2002/08/10 17:09:46  roefer
* Sensor display error fixed
*
* Revision 1.2  2002/07/23 13:43:36  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - empty constructors in bars / dialogs
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
* Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.15  2002/04/30 14:33:16  risler
* removed bug with joint data coloring
*
* Revision 1.14  2002/04/25 14:50:34  kallnik
* changed double/float to double
* added several #include GTMath
*
* PLEASE use double
*
* Revision 1.13  2002/04/23 17:45:16  loetzsch
* - splitted debugKeyTable into debugKeyTableForRobot and debugKeyTableForLocalProcesses
* - removed Modules Toolbar
* - removed access for dialogs and toolbars to solutionRequest.
* - changed access for dialogs and toolbars to debug queues
* - removed the instance of SolutionRequest in CRobotControlApp
* - only the log player, local processes and the robot put messages into queueToRobotControl
*
* Revision 1.12  2002/04/23 15:00:48  jhoffman
* added FT save feature
*
* Revision 1.11  2002/03/19 16:45:13  jhoffman
* added fourier spectrum output
*
* Revision 1.10  2002/02/21 12:35:11  jhoffman
* _ added different CSV export that is more suitable for use in Excel (based on Martin's code)
* _ in doing so found that existing CSV output gets mixed up with time stamps and therefore needs to be debugged
*
* Revision 1.9  2002/02/12 16:34:49  risler
* finished MofTester
*
* Revision 1.8  2002/02/05 03:46:13  loetzsch
* renamed a registry key.
*
* Revision 1.7  2002/01/30 17:33:22  loetzsch
* Joint Viewer vorerst fertig
*
* Revision 1.6  2002/01/28 16:23:45  juengel
* Save joints as CSV hinzugefgt
*
* Revision 1.5  2002/01/28 14:00:33  loetzsch
* JointViewer continued
*
* Revision 1.4  2002/01/27 21:15:06  loetzsch
* JointViewer now displays SensorData
*
* Revision 1.3  2002/01/26 03:11:28  loetzsch
* JointViewer continued
*
* Revision 1.2  2002/01/25 16:52:58  loetzsch
* continued
*
*/
