/**
 * @file Representations/WLan/TeamMessageCollection.h
 *
 * Declaration of class TeamMessageCollection
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 * @author <A href=mailto:sebastian.schmidt@udo.edu>Sebastian Schmidt</A>
 * @author <A href=mailto:m_wachter@gmx.de>Michael Wachter</A>
 */ 

#ifndef __TeamMessageCollection_h_
#define __TeamMessageCollection_h_

#include "TeamMessage.h"

#include "Tools/RingBuffer.h"

/** 
 * @class TeamMessageCollection
 *
 * Represents a collection of all actual team messages received from other robots 
 * and has the ability to send to them too.
 *
 */

class TeamMessageCollection
{
  public:
	 
    /**
     * Timeout-constants for the processMasterSync method
     */
    enum {robotTimeoutTime = 4000, masterTimeoutTime = 2000, masterClaimTimeoutTime = 1500};
    /**
     * the maximum number of team-messages
     */
    enum {maxNumberOfTeamMessages = Player::numOfPlayerNumbers-1};

    int numberOfTeamMessages; /**< count of actual TeamMessages */
	

    /**
     * Constructor.
     */
    TeamMessageCollection() ;

    /**
     * Constant access operator to a certain team message. Only
     * actual team-messages are given back. (TeamMessage.isActual() == true)
     * @param index The index of the team message.
     * @return The team message with the requested index.
     */
    const TeamMessage& operator[](int index) const 
    {
      return *teamMessages[teamMessagesForActualTeamMessages[index]];
    }
                                                                              
    /**
	   * This method processes the incomming messages to calculate time-offsets for
	   * the time syncronisation, to find out which Robot is Master and to find out
     * if one robot is missing.
     */
	  void processMessages();

  	/**
	   * The function sets the internal references to the outgoing TeamMessages
	   * @param tm1 the first TeamMessage
	   * @param tm2 the second TeamMessage
	   * @param tm3 the third TeamMessage
	   */
   	void setOutTeamMessages(TeamMessage& tm);

    /**
	   * The function sets the internal references to the incoming TeamMessages
	   * @param tm1 the first TeamMessage
	   * @param tm2 the second TeamMessage
	   * @param tm3 the third TeamMessage
	   */
  	void setInTeamMessages(TeamMessage& tm);

	  /**
	   * This function sets the robotPose field in the outgoing TeamMessage 
	   * that is represented by the target
	   * @param robotPose The RobotPose to be sent
       * @param sendNow bool o indicate that the message needs to be sent now (default = false)
	   */
    void send(const RobotPose& robotPose, bool sendNow = false) const;

  	/**
	   * This function sets the BehaviorTeamMessage field in the outgoing TeamMessage 
	   * that is represented by the target
	   * @param behaviorTeamMessage The BehaviorTeamMessage to be sent
       * @param sendNow bool o indicate that the message needs to be sent now
	   */
    void send(const BehaviorTeamMessage& behaviorTeamMessage, bool sendNow = false) const;

    /**
	   * This function sets the BehaviorTeamMessage field in the outgoing TeamMessage 
	   * that is represented by the target
	   * @param seenBallPosition The SeenBallPosition to be sent
       * @param targets A bitfield representing the receivers (default = TGT-all)
    */
    void send(const SeenBallPosition& seenBallPosition, bool sendNow = false) const;

    /**
     * This function processes the outgoing messages.
     * @return A bool that is true when the messages should be send. 
     */
    bool processOutMessages();

    /**
     * get the delay between two subsequent sends
     */
    int getDelay() const ;
    
    /**
     * set the delay between two subsequent sends
     */
    void setDelay(int delay);

  private:
	int delay; /**< the delay between two subsequent sends */
    const TeamMessage* teamMessages[maxNumberOfTeamMessages]; /**< pointer to the incomming teamMessages */
    TeamMessage* outTeamMessages[maxNumberOfTeamMessages]; /**< pointer to outgoing teamMessages */
    int teamMessagesForActualTeamMessages[maxNumberOfTeamMessages]; /**< Array to translate teamMessage number for backwards compability */
	
    unsigned lastSendingTimeStamp;    /**< TimeStamp when sending the last TeamMessage */
    unsigned teamMessageTimeStamp[maxNumberOfTeamMessages]; /**< TimeStamps of the latest incoming TeamMessages */
    unsigned incomingTimeStamp[maxNumberOfTeamMessages];    /**< Local TimeStamp of the latest incoming TeamMessages */
    
    // Attrributes for time sync
    RingBuffer<signed long,100> offsets[maxNumberOfTeamMessages]  ; /**< Ringbuffer to save the last 100 time offsets for the 3 TeamMessages */
    signed long timeOffsetSum[maxNumberOfTeamMessages];     /**< Sum of all the values in the ringBuffers */

    // Attribute for better streaming of TeamMessage
    bool messagesSent;    /**< set true if the messages have been sent */
    mutable bool sendNow;        /**< pointer to a bool which shows that the robot has to send the messages */
    long lastSendTime;    /**< time when *sendNow was last true. Used to send after a delay */

	// Count of TeamMessages for initializing

	int numberOfInTeamMessages;
	int numberOfOutTeamMessages;

    /** 
     * This method is called by processMessages() to find out the time-offsets between the robots.
     * It saves the time-difference in the corresponding TeamMessage accesed by operator[]
     */
    void processTimeStamps();

    /**
     * This method is calls by processMessages() to set the fields of the teamMessageForTarget[] array.
     * This array is used by the send-methods to find out which teamMessages send to a target.
     */
    void processMessagePlayerRole();
};

#endif //__TeamMessageCollection_h_

/*
 * Change log :
 *
 * $Log: TeamMessageCollection.h,v $
 * Revision 1.1.1.1  2004/05/22 17:26:05  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/05/15 14:31:55  nistico
 * Didn't compile on VS6.
 *
 * Revision 1.2  2004/05/14 14:12:08  wachter
 * - Added communication support for 5 robots
 * - rewrote parts of team-communication to be faster and more stable
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/26 15:27:28  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.2  2003/07/02 19:14:23  loetzsch
 * bug fixes, removed unused functions
 *
 * Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.5  2003/05/02 12:57:13  loetzsch
 * TeamMessage now contains a SeenBallPosition instead of a BallPercept
 *
 * Revision 1.4  2003/02/28 17:02:55  wachter
 * Reenabled trainer-syncronisation for RuhrpottHellhound-Behavior
 *
 * Revision 1.3  2003/02/25 12:55:40  wachter
 * made some attributes muteable
 *
 * Revision 1.2  2003/02/18 13:24:36  wachter
 * added new TeamMessageCollection and TeamMessage
 *
 * Revision 1.1  2002/09/10 15:26:41  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.1  2002/06/03 15:28:06  roefer
 * BallLocator gets TeamMessageCollection
 *
 *
 */
