/**
* @file Representations/WLan/TeamMessage.cpp
*
* Implementation of class TeamMessage.
*
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
* @author <A href=mailto:sebastian.schmidt@udo.edu>Sebastian Schmidt</A>
* @author <A href=mailto:m_wachter@gmx.de>Michael Wachter</A>
*/

#include "TeamMessage.h"
#include "Platform/SystemCall.h"

bool TeamMessage::isActual() const
{
  return SystemCall::getTimeSince(lastReceivedTimeStamp) < 2000;
}

In& operator>>(In& stream,TeamMessage& teamMessage)
{
  stream >> teamMessage.timeStamp >> 
    teamMessage.lastReceivedTimeStamp >>
    teamMessage.incomingTimeStamp;
  char c;
  stream >> c;
  teamMessage.playerNumberOfSender = (Player::playerNumber)c;
  
  char fieldsInMessage;
  stream >> fieldsInMessage;
  teamMessage.sendRobotPose = ((fieldsInMessage & 1) == 1);
  teamMessage.sendSeenBallPosition = ((fieldsInMessage & 2) == 2);
  teamMessage.sendBehaviorTeamMessage = ((fieldsInMessage & 4) == 4);
  
  // robot pose
  if (teamMessage.sendRobotPose) 
  {
    TeamMessage::read(stream,teamMessage.robotPose);
    stream >> c;
    teamMessage.robotPose.setValidity(c / 100.0);
  }
  
  if (teamMessage.sendSeenBallPosition) 
  {
    stream.read(&teamMessage.seenBallPosition,sizeof(SeenBallPosition));
  }
  
  if (teamMessage.sendBehaviorTeamMessage) 
  {
    stream >> teamMessage.behaviorTeamMessage;
  }
    
  stream >> c;
  teamMessage.playerNumberOfSender = (Player::playerNumber)c;
  
  return stream;
}

Out& operator<<(Out& stream,const TeamMessage& teamMessage)
{
  stream << teamMessage.timeStamp << teamMessage.lastReceivedTimeStamp 
    << teamMessage.incomingTimeStamp;
  stream << (char)getPlayer().getPlayerNumber();
  
  // a bitfield which tells the receiver what fields are transmitted
  char fieldsInMessage = 
    teamMessage.sendRobotPose  
    + (teamMessage.sendSeenBallPosition << 1)
    + (teamMessage.sendBehaviorTeamMessage << 2);
  
  stream << fieldsInMessage;
  
  // robot pose
  if (teamMessage.sendRobotPose) {
    TeamMessage::write(stream,teamMessage.robotPose);
    stream << char(teamMessage.robotPose.getValidity() * 100.0 + 0.5);
  }
  
  if (teamMessage.sendBehaviorTeamMessage) 
  {
    stream.write(&teamMessage.seenBallPosition, sizeof(SeenBallPosition));
  }
  
  // BehaviorTeamMessage
  if (teamMessage.sendBehaviorTeamMessage) {
    stream << teamMessage.behaviorTeamMessage;
  }
  

  stream << (char) getPlayer().getPlayerNumber();
  return stream;
}

void TeamMessage::read(In& stream,Vector2<double>& v)
{
  short x,y;
  stream >> x >> y;
  v.x = x; 
  v.y = y;
}

void TeamMessage::read(In& stream,Pose2D& p)
{
  read(stream,p.translation);
  short rotation;
  stream >> rotation;
  p.rotation = rotation / 1000.0;
}

void TeamMessage::write(Out& stream,const Vector2<double>& v)
{
  stream << short(v.x + 0.5) << short(v.y + 0.5);
}

void TeamMessage::write(Out& stream,const Pose2D& p)
{
  write(stream,p.translation);
  stream << short(p.rotation * 1000 + 0.5);
}

long TeamMessage::getTimeStampInOwnTime() const
{
  return(timeStamp - timeOffset);
}

long TeamMessage::getTimeInOwnTime(long time) const
{
  return(time - timeOffset);
}

long TeamMessage::getTimeInRemoteTime(long time) const
{
  return(time + timeOffset);
}

/*
* Change log :
* 
* $Log: TeamMessage.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:26:05  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/07 10:09:36  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/26 15:27:27  juengel
* Renamed DataTypes to representations.
*
* Revision 1.2  2003/07/02 19:14:23  loetzsch
* bug fixes, removed unused functions
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.8  2003/05/02 12:57:13  loetzsch
* TeamMessage now contains a SeenBallPosition instead of a BallPercept
*
* Revision 1.7  2003/04/19 03:29:09  pruente
* Merged in changes by Michael: - send right playerRole
*
* Revision 1.6  2003/03/24 12:17:24  wachter
* fixed isActual so that it works like in revision 1.3
* ( returns false if the last message is older than 2000 ms)
*
* Revision 1.5  2003/02/28 17:02:55  wachter
* Reenabled trainer-syncronisation for RuhrpottHellhound-Behavior
*
* Revision 1.4  2003/02/18 13:24:37  wachter
* added new TeamMessageCollection and TeamMessage
*
* Revision 1.3  2002/09/22 18:40:51  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/17 23:55:20  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:26:41  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.5  2002/06/09 15:24:51  loetzsch
* Added TeamMessageCollection and BehaviorTeamMessage to the execute of BehaviorControl
*
* Revision 1.4  2002/06/08 09:26:32  Thomas Rfer
* Team ball position, first draft
*
* Revision 1.3  2002/06/06 16:10:46  roefer
* Robot pose and ball percept added
*
* Revision 1.2  2002/05/26 14:55:25  roefer
* Team communication is working
*
* Revision 1.1  2002/05/16 22:36:11  roefer
* Team communication and GTMath bugs fixed
*
*/
