/**
* @file GameControlData.h
*
* Declaration of class GameControlData
*
* @author Martin Ltzsch
*/ 

#ifndef __GameControlData_h_
#define __GameControlData_h_

#include "RoboCupGameControlData.h"
#include "Tools/Streams/InOut.h"
#include "Tools/Player.h"

/**
* @class GlobalGameControlData
*
* Specifies the game state for two teams of four robots.
* Different from the GameControlData sent from Sony's GameManager
* (which is specific for one robot) this class contains the game state
* for all robots.
*
* @author Martin Ltzsch
*/
class GlobalGameControlData
{
public:
  /** Constructor */
  GlobalGameControlData();
  
  /** Which team does the kickoff */
  enum Kickoff { kickoffRedTeam, kickoffBlueTeam } kickoff;
  
  /** The game state */
  enum State { initial = ROBOCUP_STATE_INITIAL, 
    ready = ROBOCUP_STATE_READY, 
    set = ROBOCUP_STATE_SET,
    playing = ROBOCUP_STATE_PLAYING, 
    penalized = ROBOCUP_STATE_PENALIZED, 
    finished = ROBOCUP_STATE_FINISHED
  } state;
  
  /** The score of the red team */
  short redScore;
  
  /** The score of the blue team */
  short blueScore;
  
  /** Possible penalties */
  enum Penalties { 
    notPenalized = PENALTY_NONE,
    illegalDefender = PENALTY_ILLEGAL_DEFENDER,
    illegalDefanse = PENALTY_ILLEGAL_DEFENSE,
    obstruction = PENALTY_OBSTRUCTION,
    keeperCharge = PENALTY_KEEPER_CHARGE, 
    playerCharge = PENALTY_FIELD_PLAYER_CHARGE,
    ballHolding = PENALTY_BALL_HOLDING, 
    requestForPickup = PENALTY_REQ_FOR_PICKUP
  };

  /** The current penalty states of the red team */
  Penalties penaltiesRedTeam[4];

  /** The current penalty states of the blue team */
  Penalties penaltiesBlueTeam[4];
};

/**
* Streaming operator that reads a GlobalGameControlData from a stream.
* @param stream The stream from which is read.
* @param globalGameControlData The GlobalGameControlData object.
* @return The stream.
*/ 
In& operator>>(In& stream,GlobalGameControlData& globalGameControlData);

/**
* Streaming operator that writes a GlobalGameControlData to a stream.
* @param stream The stream to write on.
* @param globalGameControlData The GlobalGameControlData object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const GlobalGameControlData& globalGameControlData);

/** 
* @class GameControlData
*
* Encapsulates the RoboCupGameControlData struct for the GT2003 project.
*
* @author Martin Loetzsch
*/
class GameControlData
{
public:
/**
* Constructor.
  */
  GameControlData();
  
  /** an instance of RoboCupGameControlData */
  RoboCupGameControlData data;
  
  /** 
  * The timestamp when the last request was received.
  * Note that this value is set in the In streaming operator.
  */
  unsigned long timeStamp;
  
  /** returns the name of the state */
  static const char* getStateName(RState state)
  {
    switch (state)
    {
    case ROBOCUP_STATE_INITIAL: return "initial";
    case ROBOCUP_STATE_READY: return "ready";
    case ROBOCUP_STATE_SET: return "set";
    case ROBOCUP_STATE_PLAYING: return "playing";
    case ROBOCUP_STATE_PENALIZED: return "penalized";
    case ROBOCUP_STATE_FINISHED: return "finished";
    default: return "unknown state";
    }
  }
  
  /** returns the name for the variable kickoff */
  static const char* getKickoffName(RKickOff kickoff)
  {
    switch (kickoff)
    {
    case ROBOCUP_KICKOFF_OWN: return "own";
    case ROBOCUP_KICKOFF_OPPONENT: return "opponent";
    case ROBOCUP_KICKOFF_INVALID: return "invalid";
    default: return "unknown kickoff";
    }
  }

  /** returns the name of the team color */
  static const char* getTeamColorName(RTeamColor teamColor)
  {
    switch (teamColor)
    {
    case ROBOCUP_TEAMCOLOR_RED: return "red";
    case ROBOCUP_TEAMCOLOR_BLUE: return "blue";
    default: return "unknown color";
    }
  }

  /** returns the name of the current penalty */
  static const char* getPenaltyName(RPenalty penalty)
  {
    switch (penalty)
    {
    case PENALTY_NONE: return "no penalty";
    case PENALTY_ILLEGAL_DEFENDER: return "illegal defender";
    case PENALTY_ILLEGAL_DEFENSE: return "illegal defense";
    case PENALTY_OBSTRUCTION: return "obstruction";
    case PENALTY_KEEPER_CHARGE: return "keeper charge";
    case PENALTY_FIELD_PLAYER_CHARGE: return "field player charge";
    case PENALTY_BALL_HOLDING: return "ball holding";
    case PENALTY_REQ_FOR_PICKUP: return "request for pick-up";
    default: return "unknown penalty";
    }
  }
  
  /** 
  * Translates a GlobalGameControlData to GameControlData.
  * @param globalGameControlData The global game state
  * @param player This player
  */
  void translateFromGlobalGameControlData( const GlobalGameControlData& globalGameControlData, 
    Player& player);
  
  /** the == operator */
  bool operator == (const GameControlData& other);
  
  /** the != operator */
  bool operator != (const GameControlData& other)
  {
    return ! (*this == other);
  }
};


/**
* Streaming operator that reads a GameControlData from a stream.
* @param stream The stream from which is read.
* @param gameControlData The GameControlData object.
* @return The stream.
*/ 
In& operator>>(In& stream,GameControlData& gameControlData);

/**
* Streaming operator that writes a GameControlData to a stream.
* @param stream The stream to write on.
* @param gameControlData The GameControlData object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const GameControlData& gameControlData);

#endif //__GameControlData_h_

/*
* Change log :
*
* $Log: GameControlData.h,v $
* Revision 1.1.1.1  2004/05/22 17:26:04  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/03/25 17:32:45  loetzsch
* ranamed state final to finished
*
* Revision 1.2  2004/03/15 12:50:32  tim
* Adaptions to new GameController
*
* Revision 1.1  2003/10/07 10:09:36  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/03/24 15:45:39  loetzsch
* added the new version of RoboCupGameControlData
*
* Revision 1.2  2003/03/05 11:10:05  loetzsch
* added class GlobalGameControlData
*
* Revision 1.1  2002/09/10 15:26:40  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.3  2002/07/23 13:32:57  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.2  2002/06/04 00:11:19  loetzsch
* added == and != operator
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.2  2002/05/05 22:12:42  loetzsch
* GameControlData can now be sent from the Game toolbar to BehaviorControl
*
* Revision 1.1  2002/05/05 18:52:01  loetzsch
* added
* - GameControlData,
* - Receivers for GameControlData
* - access by behavior to GameControlData
*
*/

