/**
* @file GameControlData.cpp
*
* Implementation of class GameControlData.
*
* @author Martin Loetzsch
*/

#include "GameControlData.h"
#include "Platform/SystemCall.h"
#include "Tools/Debugging/Debugging.h"

GlobalGameControlData::GlobalGameControlData()
: kickoff(kickoffRedTeam), state(initial), redScore(0), blueScore(0)
{
  for (int i=0; i<4; i++)
  {
    penaltiesBlueTeam[i] = notPenalized;
    penaltiesRedTeam[i] = notPenalized;
  }
}

In& operator>>(In& stream,GlobalGameControlData& globalGameControlData)
{
  stream.read(&globalGameControlData, sizeof(GlobalGameControlData));
  return stream;
}

Out& operator<<(Out& stream, const GlobalGameControlData& globalGameControlData)
{
  stream.write(&globalGameControlData, sizeof(GlobalGameControlData));
  return stream;
}

GameControlData::GameControlData()
: timeStamp(0)
{
  data.kickoff = ROBOCUP_KICKOFF_OPPONENT;
  data.opponentScore = 0;
  data.ownScore = 0;
  data.state = ROBOCUP_STATE_INITIAL;
  data.penalty = PENALTY_NONE;
  if (getPlayer().getTeamColor() == Player::red)
  {
    data.teamColor = ROBOCUP_TEAMCOLOR_RED;
  }
  else
  {
    data.teamColor = ROBOCUP_TEAMCOLOR_BLUE;
  }
}

void GameControlData::translateFromGlobalGameControlData(const GlobalGameControlData& globalGameControlData, 
                                                         Player& player)
{
  data.state = globalGameControlData.state;
  
  if (player.getTeamColor() == Player::red)
  {
    data.teamColor = ROBOCUP_TEAMCOLOR_RED;
    if (globalGameControlData.penaltiesRedTeam[player.getPlayerNumber()] 
      != GlobalGameControlData::notPenalized)
    {
      data.state = ROBOCUP_STATE_PENALIZED;
    }
  }
  else if (player.getTeamColor() == Player::blue)
  {
    data.teamColor = ROBOCUP_TEAMCOLOR_BLUE;
    if (globalGameControlData.penaltiesBlueTeam[player.getPlayerNumber()] 
      != GlobalGameControlData::notPenalized)
    {
      data.state = ROBOCUP_STATE_PENALIZED;
    }
  }

  data.ownScore = (player.getTeamColor() == Player::red ?
    globalGameControlData.redScore : globalGameControlData.blueScore);
  
  data.opponentScore = (player.getTeamColor() == Player::blue ?
    globalGameControlData.redScore : globalGameControlData.blueScore);
  
  if (player.getTeamColor() == Player::red)
  {
    if (globalGameControlData.kickoff == GlobalGameControlData::kickoffRedTeam)
      data.kickoff = ROBOCUP_KICKOFF_OWN;
    else
      data.kickoff = ROBOCUP_KICKOFF_OPPONENT;
  }
  else
  {
    if (globalGameControlData.kickoff == GlobalGameControlData::kickoffBlueTeam)
      data.kickoff = ROBOCUP_KICKOFF_OWN;
    else
      data.kickoff = ROBOCUP_KICKOFF_OPPONENT;
  }
  timeStamp = SystemCall::getCurrentSystemTime();
}

bool GameControlData::operator == (const GameControlData& other)
{
  if (data.state != other.data.state) return false;
  if (data.kickoff != other.data.kickoff) return false;
  if (data.ownScore != other.data.ownScore) return false;
  if (data.opponentScore != other.data.opponentScore) return false;
  if (data.penalty != other.data.penalty) return false;
  if (data.teamColor != other.data.teamColor) return false;
  return true;
}

In& operator>>(In& stream,GameControlData& gameControlData)
{
  gameControlData.timeStamp = SystemCall::getCurrentSystemTime();
  stream.read(&gameControlData.data,sizeof(RoboCupGameControlData));
  return stream;
}

Out& operator<<(Out& stream, const GameControlData& gameControlData)
{
  stream.write(&gameControlData.data,sizeof(RoboCupGameControlData));
  return stream;
}


/*
* Change log :
* 
* $Log: GameControlData.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:26:03  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/03/15 12:50:31  tim
* Adaptions to new GameController
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/07 10:09:36  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/03/06 11:40:46  dueffert
* re-order warning removed
*
* Revision 1.2  2003/03/05 11:10:05  loetzsch
* added class GlobalGameControlData
*
* Revision 1.1  2002/09/10 15:26:40  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/06/04 00:11:19  loetzsch
* added == and != operator
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.1  2002/05/05 18:52:01  loetzsch
* added
* - GameControlData,
* - Receivers for GameControlData
* - access by behavior to GameControlData
*
*/
