/**
* @file SpecialPercept.h
*
* Declaration of class SpecialPercept
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>, <A href=mailto:brunn@sim.tu-darmstadt.de>Ronnie Brunn</A>
*/ 

#ifndef __SpecialPercept_h_
#define __SpecialPercept_h_


#include "Tools/Streams/InOut.h"
#include "Tools/Math/Pose2D.h"
#include "Tools/Math/Vector2.h"
#include "Tools/Math/Geometry.h"
#include "Tools/ColorClasses.h"



class SpecialLandmark
{
public:
  colorClass sectors[8];
  Vector2<int> center;
  Vector2<int> size;
  Vector2<double> angles;
};

/** A class that contains a certain bitePoint at the open challenge bridge
  * @author <A href=mailto:jochen@kerdels.de>Jochen Kerdels</A>
  */
class BitePoint
{
  public:
    /** Constructor */
    BitePoint();

    /** The different bitePoints */
    enum Position
    {
      none = -1,
      frontleft = 0,
      frontright,
      behindleft,
      behindright,
      master,
      numOfPositions
    };

    /** postion of the bitePoint at the oc bridge */
    Position position;

    /** relative angle to the bitePoint */
    double angleTo;
    /** relative  distance to the bitePoint */
    double distanceTo;
    /** the relative position of the bitePoint */
    Vector2<double> relPos;
    /** the preciseShift to either left or right
      * while in front of the bytePoint
      */
    double preciseShift;

    /** time when the bitePoint was last seen by the own camera */
    unsigned long timeWhenLastSeen;

    /** method to add a new percept */
    void addPercept(double aAngleTo, double aDistanceTo, double aPreciseShift);

    /** indicates fresh-data */
    bool fresh;

    /** last odometry-data */
    Pose2D lastOdometry;

};

/**
 * Streaming operator that reads a BitePoint from a stream.
 * @param stream The stream from which is read.
 * @param bitePoint The BitePoint object.
 * @return The stream.
 */ 
In& operator>>(In& stream,BitePoint& bitePoint);

/**
 * Streaming operator that writes a BitePoint to a stream.
 * @param stream The stream to write on.
 * @param bitePoint The BitePoint object.
 * @return The stream.
*/ 
Out& operator<<(Out& stream, const BitePoint& bitePoint);




/** A class that contains data of the bridge for the openchallenge
  * @author <A href=mailto:jochen@kerdels.de>Jochen Kerdels</A>
  */
class OCBridge
{
  public:
    /** Constructor */
    OCBridge();

    /** The different Sides of the bridge */
    enum Side
    {
      none = 0,
      left,
      right,
      numOfSides
    };

    /** side of the bridge actually last seen by the camera */
    Side lastSeenSide;

    /** relative angle to the bridge (middle of the sidemark) */
    double angleTo;
    /** relative  distance to the bridge (middle of the sidemark) */
    double distanceTo;
    /** the relative position of the bridge (middle of the sidemark) */
    Vector2<double> relPos;

    /** time when the bridge was last seen by the own camera */
    unsigned long timeWhenLastSeen;

    /** the bitePoints of the bridge */
    BitePoint bitePoint[BitePoint::numOfPositions];

    /** method to add a new percept */
    void addPercept(double aAngleTo, double aDistanceTo);

    /** indicates fresh-data */
    bool fresh;

    /** last odometry-data */
    Pose2D lastOdometry;
};

/**
 * Streaming operator that reads a OCBridge from a stream.
 * @param stream The stream from which is read.
 * @param ocbridge The OCBridge object.
 * @return The stream.
 */ 
In& operator>>(In& stream,OCBridge& ocBridge);

/**
 * Streaming operator that writes a OCBridge to a stream.
 * @param stream The stream to write on.
 * @param ocbridge The OCBridge object.
 * @return The stream.
*/ 
Out& operator<<(Out& stream, const OCBridge& ocBridge);


/** A class that contains data of the bridge's red line for the open challenge
  */
class OCRedLine
{
  public:
    /** Constructor */
    OCRedLine();

	/** flag to know if red Line has been seen or not */
	bool seen;

    /** Startpoint and endpont of red line 
		The startpoint is nearer to the robot e.g. on the bottom of the image
	*/
	Vector2<int> lineStart;
	Vector2<int> lineEnd;
	Vector2<int> pointStart;
	Vector2<int> pointEnd;
	int angleInImage; // angle of red line in image. 0 is exactly vertical.
	unsigned long timeWhenLastSeen; /** time when the bridge was last seen by the own camera */


    /** method to add a new percept */
	void addPercept(Vector2<int> lineStart, Vector2<int> lineEnd, Vector2<int> pointStart,Vector2<int> pointEnd,int angleInImage);
};

/**
 * Streaming operator that reads a red line from a stream.
 * @param stream The stream from which is read.
 * @param ocRedLine the ocRedLine object.
 * @return The stream.
 */ 
In& operator>>(In& stream,OCRedLine& ocRedLine);

/**
 * Streaming operator that writes a red line to a stream.
 * @param stream The stream to write on.
 * @param ocRedLine The ocRedLine object.
 * @return The stream.
*/ 
Out& operator<<(Out& stream, const OCRedLine& ocRedLine);



/** A class that contains a certain specialPercept
  * @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
  * @author <A href=mailto:loetzsch@informatik.hu-berlin.de>Martin Ltzsch</A>
  */
class SpecialPercept
{
  public:
    /** Constructor */
    SpecialPercept();

    /** Reset special percept. */
    void reset(unsigned long frameNum=0);

    /** The different specialPercept types*/
    enum Type
    {
      motionDetection,
      none,
      checkerboard,
      barCode,
      autoShutter,
	  challengeOrientation,
	  openChallengeBridge,
	  openChallengeRedLine
    };

    /** The type of the specialPercept.*/
    Type type;

    /** test values (motion detection) */
    double pan, tilt;
    Vector2<long> mostMovement, leastMovement;

    /** values (checkerboard detection) */
    Pose2D checkerPose;
    unsigned long frameNumber;

    /** An id filled by the barCodeReader */
    int barCodeId;

	  /** Orientation of the robot in the "Almost-Slam-Challenge" **/
	  double orientation;
    /** angle of pink Flags seen relative to robot **/ 
#define MAXSEENFLAGS 100
#define MAXLEARNFLAGS 100
    double angleOfPinkFlags[MAXSEENFLAGS];
    int    countAngleOfPinkFlags;

    /** values for open challenge bridge detection */
    OCBridge ocBridge;

  	/** for red line detection */
	  OCRedLine ocRedLine;

    /** indicates if Shutter was selected */
    bool shutterSelected;
    
    SpecialLandmark specialLandmarks[6];
    int numOfSpecialLandmarks;
};

/**
 * Streaming operator that reads a SpecialPercept from a stream.
 * @param stream The stream from which is read.
 * @param specialPercept The SpecialPercept object.
 * @return The stream.
 */ 
In& operator>>(In& stream,SpecialPercept& specialPercept);

/**
 * Streaming operator that writes a SpecialPercept to a stream.
 * @param stream The stream to write on.
 * @param specialPercept The SpecialPercept object.
 * @return The stream.
*/ 
Out& operator<<(Out& stream, const SpecialPercept& specialPercept);

#endif //__SpecialPercept_h_

/*
* Change log :
* 
* $Log: SpecialPercept.h,v $
* Revision 1.6  2004/07/02 10:11:47  nistico
* Cloned main image processor and created
* SpecialLandmarks specialist for SLAM challenge
*
* Revision 1.5  2004/07/01 18:42:08  hamerla
* SLAM challenge find Orientation
*
* Revision 1.4  2004/06/29 15:10:57  kerdels
* Special Vision AutoShutter...
*
* Revision 1.3  2004/06/28 12:58:17  wachter
* Added special-percept for challenge and corresponding symbols.
*
* Revision 1.2  2004/06/01 13:58:14  schumann
* special percept type redLine added,
* part 1 of open challenge added
*
* Revision 1.1.1.1  2004/05/22 17:26:03  cvsadm
* created new repository GT2004_WM
*
* Revision 1.7  2004/05/21 16:32:43  hamerla
* no message
*
* Revision 1.6  2004/05/20 16:30:03  kerdels
* further work on open challenge
*
* Revision 1.5  2004/05/17 01:56:16  kerdels
* prepared some symbols for the open challenge bridge detection
*
* Revision 1.4  2004/01/19 14:53:46  dueffert
* all frameNumbers (and not only some of them) are unsigned long now
*
* Revision 1.3  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.2  2003/11/10 13:29:45  dueffert
* frame number added
*
* Revision 1.1  2003/10/07 10:09:36  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/07/30 14:55:54  dueffert
* checkerboard percept improved
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.13  2003/05/26 08:12:10  juengel
* Moved angleToFreePartOfGoal from specialPercept to obstaclesPercept.
*
* Revision 1.12  2003/05/01 10:20:58  roefer
* Added distance and width for freePartOfO*Goal
*
* Revision 1.11  2003/04/26 08:19:32  roefer
* reset() added
*
* Revision 1.10  2003/04/23 16:12:03  juengel
* Added angleToFreePartOfOwnGoal.
*
* Revision 1.9  2003/04/02 13:20:37  juengel
* Added SpecialPercept freePartOfOpponentGoal.
*
* Revision 1.8  2003/02/08 16:06:02  roefer
* ID sequence changed to be compatible with existing log files
*
* Revision 1.7  2003/01/29 10:37:21  juengel
* Added BarCodeReader.
*
* Revision 1.6  2003/01/22 15:20:25  dueffert
* checkerboard stuff added
*
* Revision 1.5  2002/11/28 14:43:48  jhoffman
* added a special percept for the motion detector
*
* Revision 1.4  2002/11/27 13:47:50  dueffert
* doxygen docu corrected
*
* Revision 1.3  2002/09/25 12:05:43  loetzsch
* removed BarPercept and PatternPercept
*
* Revision 1.2  2002/09/22 18:40:50  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:26:40  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.11  2002/07/23 13:32:57  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.10  2002/06/22 02:19:57  risler
* barIsNear added
*
* Revision 1.9  2002/06/21 12:09:07  juengel
* no message
*
* Revision 1.8  2002/06/17 19:24:13  mkunz
* BarPerceptor searches for own end only now
*
* Revision 1.7  2002/06/11 17:39:00  vigerske
* continued pattern perception
*
* Revision 1.6  2002/06/04 11:06:29  juengel
* added rectangle
*
* Revision 1.5  2002/06/02 19:07:23  mkunz
* no message
*
* Revision 1.4  2002/05/17 11:51:48  brunn
* first experimental version of the barPerceptor
*
* Revision 1.3  2002/05/15 11:24:31  brunn
* barEnd poses instead of complete bar pose
* and initialization added
*
* Revision 1.2  2002/05/11 06:03:32  juengel
* Constructor for PatternPercept added.
*
* Revision 1.1  2002/05/10 17:40:26  juengel
* Added SpecialVision and SpecialPercept.
*
*/
