/**
 * @file PlayersPercept.h
 * 
 * Declaration of classes PlayersPercept and SinglePlayerPercept
 * @author <A href=mailto:timlaue@informatik.uni-bremen.de>Tim Laue</A>
 */ 

#ifndef __PlayersPercept_h_
#define __PlayersPercept_h_


#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector2.h"


/**
* This class contains some information about a single perceived robot
*/
class SinglePlayerPercept
{
public:
  /** Simple constructor */
  SinglePlayerPercept() : validity(0) {}

  /** Copy constructor */
  SinglePlayerPercept(const SinglePlayerPercept& other)  {*this = other;};

  /** offset to the robot, relative to own robot */
  Vector2<double> offset;

  /** The direction of the perceived robot*/
  double direction;

  /** The validity of this percept */
  double validity;
};


/**
* This is a collection of SinglePlayerPercept for red and blue dressed robots.
*/
class PlayersPercept
{
public:
  /** Constants */
  enum {
    maxNumberOfRedPlayerPercepts = 13,
    maxNumberOfBluePlayerPercepts = 17,
    maxNumberOfOneRobotsPercepts = 17,
    maxNumberOfPercepts = 34
  };

  /** Constructor */
  PlayersPercept();

  /** Destructor */
  ~PlayersPercept();

  /** Initializes the percept */
  void reset(unsigned long frameNumber);

  /**
  * The function adds a red player percept.
  * @param percept The percept to add.
  */
  void addRedPlayer(const SinglePlayerPercept& percept);

  /**
  * The function adds a blue player percept.
  * @param percept The percept to add.
  */
  void addBluePlayer(const SinglePlayerPercept& percept);

  /** List of perceived red robots */
  SinglePlayerPercept redPlayers[maxNumberOfRedPlayerPercepts];

  /** List of perceived blue robots */
  SinglePlayerPercept bluePlayers[maxNumberOfBluePlayerPercepts];

  /** Number of perceived red robots */
  int numberOfRedPlayers;

  /** Number of perceived blue robots */
  int numberOfBluePlayers;
  
  /** The frame number when perceived. */
  unsigned long frameNumber;
};

/**
 * Streaming operator that reads a PlayersPercept from a stream.
 * @param stream The stream from which is read.
 * @param playersPercept The PlayersPercept object.
 * @return The stream.
 */ 
In& operator>>(In& stream,PlayersPercept& playersPercept);
 
/**
 * Streaming operator that writes a PlayersPercept to a stream.
 * @param stream The stream to write on.
 * @param playersPercept The PlayersPercept object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const PlayersPercept& playersPercept);


#endif //__PlayersPercept_h_

/*
 * Change log :
 * 
 * $Log: PlayersPercept.h,v $
 * Revision 1.1.1.1  2004/05/22 17:25:59  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/03/24 14:04:07  tim
 * - added member
 * - removed old unused members
 *
 * Revision 1.3  2004/01/19 14:53:46  dueffert
 * all frameNumbers (and not only some of them) are unsigned long now
 *
 * Revision 1.2  2003/11/12 16:19:35  goehring
 * frameNumber added to percepts
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.8  2003/05/20 12:15:43  reiss
 * Fit Variables.
 *
 * Revision 1.7  2003/03/26 09:46:06  roefer
 * LinesImageProcessor2 generates player percepts
 *
 * Revision 1.6  2003/03/06 11:53:16  dueffert
 * re-order warning removed
 *
 * Revision 1.5  2003/02/27 10:06:59  schmidt
 * Added three variants of a SensorFusionPlayersLocator.
 *
 * Revision 1.4  2002/11/27 13:52:48  dueffert
 * doxygen docu added
 *
 * Revision 1.3  2002/09/22 18:40:50  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.2  2002/09/17 23:55:20  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.8  2002/04/16 16:18:12  dueffert
 * no message
 *
 * Revision 1.2  2002/04/11 21:59:08  Joscha Bach
 * More players possible...
 *
 * Revision 1.7  2002/04/02 13:10:18  dueffert
 * big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
 *
 * Revision 1.6  2002/02/22 16:07:03  tim
 * Angle/Distance replaced by relative position
 *
 * Revision 1.5  2001/12/17 00:16:55  tim
 * validity hinzugefgt
 *
 * Revision 1.4  2001/12/15 18:21:39  tim
 * PlayersPercept fertig (voererst ;-)
 *
 * Revision 1.3  2001/12/10 17:47:06  risler
 * change log added
 *
 */
