/**
* @file PSDPercept.h
* 
* Declaration of classes PSDPercept
*
* @author Martin Ltzsch
*/ 

#ifndef __PSDPercept_h_
#define __PSDPercept_h_


#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector3.h"
#include "SensorDataBuffer.h"


/**
* @class SinglePSDPercept
*
* A spot relative to the robot that was detected by the PSD Sensor.
*
* @author Martin Ltzsch
*/
class SinglePSDPercept : public Vector3<double>
{
public:

  /** The frame number when perceived. */
  unsigned long frameNumber; 

  /**
  * The functions sets the frame number
  */
  void setFrameNumber(unsigned long frameNumber) {this->frameNumber = frameNumber;}

  /** 
  * If true, then the object was farer away than 90 cm 
  * The measured distance is then clipped to 90 cm
  */
  bool tooFarAway;

  /** 
  * The tilt of the robot's head. Necessary because 
  * sometimes you don't want to trust the PSD if
  * things are measured below a certain PSD angle
  */
  double neckTilt;

  /** The PSDPercept is only valid if motion was stable. */
  bool isValid;

  /**
  * value of the body psd sensor (if present)
  */
  double body;

  /** Constructor */
  SinglePSDPercept();
  
  /** Destructor */
  ~SinglePSDPercept();
};

/**
* @class PSDPercept
*
* Class containg all single PSD spots perceived in one frame.
*
* @author Max Risler
*
*/
class PSDPercept
{
public:

  /** number of psd percepts */
  int numOfPercepts;

  /** array of single psd percepts */
  SinglePSDPercept percepts[SensorDataBuffer::maxNumOfFrames];

  /** Constructor */
  PSDPercept();

  /** Destructor */
  ~PSDPercept();

  /** [] operator */
  SinglePSDPercept& operator[](int index)
  {
    return percepts[index];
  }

  /** [] operator */
  const SinglePSDPercept& operator[](int index) const
  {
    return percepts[index];
  }

};


/**
* Streaming operator that reads a PSDPercept from a stream.
* @param stream The stream from which is read.
* @param psdPercept The PSDPercept object.
* @return The stream.
*/ 
In& operator>>(In& stream,PSDPercept& psdPercept);

/**
* Streaming operator that writes a PSDPercept to a stream.
* @param stream The stream to write on.
* @param psdPercept The PSDPercept object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const PSDPercept& psdPercept);

/**
* Streaming operator that reads a SinglePSDPercept from a stream.
* @param stream The stream from which is read.
* @param psdPercept The PSDPercept object.
* @return The stream.
*/ 
In& operator>>(In& stream,SinglePSDPercept& psdPercept);

/**
* Streaming operator that writes a SinglePSDPercept to a stream.
* @param stream The stream to write on.
* @param psdPercept The PSDPercept object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const SinglePSDPercept& psdPercept);


#endif //__PSDPercept_h_

/*
* Change log :
* 
* $Log: PSDPercept.h,v $
* Revision 1.2  2004/05/27 17:13:38  jhoffman
* - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
* - clipping included for setJoints
* - removed some microrad/rad-bugs
* - bodyPosture constructor and "=" operator fixed
*
* Revision 1.1.1.1  2004/05/22 17:25:59  cvsadm
* created new repository GT2004_WM
*
* Revision 1.6  2004/03/10 14:16:33  risler
* body psd value added to PSDPercept and ObstaclesModel
*
* Revision 1.5  2004/01/19 14:53:46  dueffert
* all frameNumbers (and not only some of them) are unsigned long now
*
* Revision 1.4  2003/11/17 15:15:08  dueffert
* doxygen docu corrected
*
* Revision 1.3  2003/11/14 19:02:25  goehring
* frameNumber added
*
* Revision 1.2  2003/11/13 14:18:56  goehring
* frameNumber added
*
* Revision 1.1  2003/10/07 10:09:36  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.4  2003/06/21 12:50:53  juengel
* Added isValid.
*
* Revision 1.3  2003/05/14 19:54:42  risler
* PSDPercept contains all points from one frame
*
* Revision 1.2  2003/04/06 17:14:37  jhoffman
* added headTilt to PSDpercept and added debugging capabilities to microsectors
*
* Revision 1.1  2002/10/10 13:09:50  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
*/
