/**
* @file Representations/Perception/LandmarksPercept.h
*
* Contains the definition of class LandmarksPercept. 
*
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
* @author <A href=mailto:asbre01@tzi.de>Andreas Sztybryc</A>
*/ 

#ifndef __LandmarksPercept_h_
#define __LandmarksPercept_h_

#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector3.h"
#include "Tools/Math/Common.h"
#include "Tools/ColorClasses.h"
#include "Tools/Boundary.h"

/**
* The class represents a rectangular landmark boundary.
* It also encodes, which edges of the boundary touch the image border.
*/
class ConditionalBoundary : public Boundary<double>
{
private:
  Boundary<double> freeBorders; /**< Contains the edges that do not touch the image border. */
  
public:
/**
* Constructor.
* The boundary is empty.
  */
  ConditionalBoundary()
    : Boundary<double>(-pi,pi),
    freeBorders(-pi,pi) {}
  
  /**
   * The function adds a point to the boundary.
   * @param px The x-coordinate of a new x-boundary-candidate.
   * @param isOnBorder Is the point on the image border?
   */
  void addX(const double px,bool isOnBorder)
  {
    x.add(px);
    if(!isOnBorder)
      freeBorders.x.add(px);
  }
  
  
  /**
   * The function adds a point to the boundary.
   * @param py The y-coordinate of a new y-boundary-candidate.
   * @param isOnBorder Is the point on the image border?
   */
  void addY(const double py,bool isOnBorder)
  {
    y.add(py);
    if(!isOnBorder)
      freeBorders.y.add(py);
  }

  /**
  * The function adds another boundary to this one.
  * @param b The other boundary.
  */
  void add(const ConditionalBoundary& b)
  {
    Boundary<double>::add(b);
    freeBorders.add(b.freeBorders);
  }
  
  /**
  * The function determines whether a certain edge lies on the image border.
  * @param border This parameter specifies the edge to test. The parameter
  *               must be one of the following four members of this object:
  *               x.min, x.max, y.min, y.max. A typical call would be:
  *               b.isOnBorder(b.min.x)
  * @return Does the specified edge touch the image border?
  */
  bool isOnBorder(const double& border) const
  {
    if(&x.min == &border)
      return x.min != freeBorders.x.min;
    else if(&x.max == &border)
      return x.max != freeBorders.x.max;
    else if(&y.min == &border)
      return y.min != freeBorders.y.min;
    else if(&y.max == &border)
      return y.max != freeBorders.y.max;
    return false;
  }
};

/**
* The class represents a flag.
*/
class Flag : public ConditionalBoundary
{
public:
  enum FlagType
  {
    pinkAboveYellow, pinkAboveSkyblue, 
    yellowAbovePink, skyblueAbovePink,
    numberOfFlagTypes
  }; /**< The type is used to identify the six different flags on the field. */
  
  FlagType type; /**< The type of this flag. */
  Vector2<double> position; /**< The position of this flag on the field. */
  double distanceValidity; /**< The validity of the determined distance. Not used yet. */
  double angleValidity; /**< The validity of the determined direction. Not used yet. */
  
  /** distance to the flag, relative to robot */
  double distance;
  
  /** angle to the flag, relative to robot */
  double angle;
  
  /**
  * The function returns the color of the lower part of this flag.
  * @return The color of the lower half.
  */
  colorClass getLowerColor() const;
  
  /**
  * The function returns the color of the upper part of this flag.
  * @return The color of the upper half.
  */
  colorClass getUpperColor() const;
};

/**
* The class represents a goal.
*/
class Goal : public ConditionalBoundary
{
public:
  colorClass color; /**< The color of the goal. */
  Vector2<double> leftPost, /**< The position of the goal post that is <i>seen</i> left. */
    rightPost; /**< The position of the goal post that is <i>seen</i> right. */
  double distanceValidity; /**< The validity of the determined distance. Not used yet. */
  double angleValidity; /**< The validity of the determined direction. Not used yet. */
  
  /** distance to left corner of the goal, relative to robot */
  double distance;
  
  /** angle to the goal, relative to robot (meassured to the middle of the goalline) */
  double angle;
  
  /** rotation of the the goal, relative to robot (meassured to the middle of the goalline) */
  double rotation;
};

/**
* The class represents all detected landmark percepts.
*/
class LandmarksPercept
{
public:
  Flag flags[4]; /**< The array of up to 4 flags. */
  int numberOfFlags; /**< The number of flags actually stored in the array. */
  Goal goals[2]; /**< The array of up to 2 goals. */
  int numberOfGoals; /**< The number of goals actually stored in the array. */
  Vector3<double> cameraOffset; /**< The camera offset relative to position of the robot's neck */
  unsigned long frameNumber; /**< The frame number when perceived. */
  /**
  * Constructor.
  * Resets the object.
  */
  LandmarksPercept();
  
  /**
  * The function resets the object, i.e. the numbers of flags and goals are set to 0.
  */
  void reset(unsigned long frameNumber);
  
  /**
  * The function adds a flag to the flag array.
  */
  void addFlag(Flag::FlagType type,
    const Vector2<double>& position,
    const ConditionalBoundary& boundary);
  
  /**
  * The function adds a flag to the flag array.
  */
  void addFlag(Flag::FlagType type,
    bool ownTeamColorIsBlue,
    const ConditionalBoundary& boundary);

  /**
  * Calculates distance and angle for each flag.
  */
  void estimateOffsetForFlags
  (
   const Vector2<double>& cameraOffset
   );
  
  /**
  * The function adds a goal to the goal array.
  */
  void addGoal(colorClass color,
    const Vector2<double>& leftPost,
    const Vector2<double>& rightPost,
    const ConditionalBoundary& boundary);

  /**
  * The function adds a goal to the goal array.
  */
  void addGoal(colorClass color,
    bool ownTeamColorIsBlue,
    const ConditionalBoundary& boundary);

  /**
  * Calculates distance and angle for each goal.
  */
  void estimateOffsetForGoals
  (
   const Vector2<double>& cameraOffset
   );
};

/**
* Streaming operator that reads a landmarks percept from a stream.
* @param stream The stream from which is read.
* @param landmarksPercept The landmarks percept to read.
* @return The stream.
*/ 
In& operator>>(In& stream,LandmarksPercept& landmarksPercept);

/**
* Streaming operator that writes a landmarks percept to a stream.
* @param stream The stream to write on.
* @param landmarksPercept The landmarks percept to write.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const LandmarksPercept& landmarksPercept);


#endif //__LandmarksPercept_h_

/*
* Change log :
* 
* $Log: LandmarksPercept.h,v $
* Revision 1.3  2004/06/17 14:34:47  dassler
* GT2004HeadControl updated
* Now looks after propergated ball, followed up withe the communicated ball
* GT2004HeadPathPlanner work now with time optimized moves
* Middle Beacons removed of the Landmarkspercept
*
* Revision 1.2  2004/06/08 16:00:33  nistico
* Final(?) improvement to the beaconSpecialist: 3 or 4 columns (depending on size)
* are always scanned, and the results are merged based on validity, which is
* calculated from the number of edges found and consistency checks
*
* Revision 1.1.1.1  2004/05/22 17:25:51  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/01/19 14:53:46  dueffert
* all frameNumbers (and not only some of them) are unsigned long now
*
* Revision 1.2  2003/11/12 16:19:35  goehring
* frameNumber added to percepts
*
* Revision 1.1  2003/10/07 10:09:36  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/26 15:27:27  juengel
* Renamed DataTypes to representations.
*
* Revision 1.2  2003/09/26 11:37:23  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.10  2003/05/04 17:31:52  roefer
* Flag and goal specialists added to GT2003IP
*
* Revision 1.9  2003/03/22 16:53:10  juengel
* Added estimateOffsetForGoals.
*
* Revision 1.8  2003/03/15 13:28:01  juengel
* Added  void addGoal(colorClass color,  bool ownTeamColorIsBlue,  const ConditionalBoundary& boundary);
*
* Revision 1.7  2003/03/10 18:20:57  dueffert
* warning removed
*
* Revision 1.6  2003/03/07 11:35:01  juengel
* removed  smoothedDistance;
*
* added void addFlag(Flag::FlagType type, bool ownTeamColorIsBlue, const ConditionalBoundary& boundary);
* added void LandmarksPercept::estimateOffsetForFlags()
*
* Revision 1.5  2002/11/07 13:32:35  roefer
* isOnBorder repaired
*
* Revision 1.4  2002/10/14 13:14:24  dueffert
* doxygen comments corrected
*
* Revision 1.3  2002/09/22 18:40:49  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/17 23:55:20  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:26:40  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.3  2002/07/23 13:32:57  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.2  2002/05/14 12:39:04  dueffert
* corrected some documentation mistakes
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.14  2002/04/06 11:47:32  roefer
* ConditionalBoundary corrected
*
* Revision 1.13  2002/04/02 14:25:11  roefer
* smoothedDistance added
*
* Revision 1.12  2002/04/02 13:10:18  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.11  2002/03/07 12:12:59  brunn
* rumpelstilzchen
*
* Revision 1.10  2002/03/06 16:16:08  mkunz
* More unused validities.
*
* Revision 1.9  2002/02/18 15:40:55  loetzsch
* changed the offset vector in Flags and Goals to an angle and a destination
*
* Revision 1.8  2002/02/18 12:27:59  brunn
* distance and angle removed ;-)
*
* Revision 1.7  2002/02/18 12:14:28  brunn
* distance and angle added
*
* Revision 1.6  2002/02/18 12:05:17  loetzsch
* Percept visualization continued
*
* Revision 1.5  2002/02/13 22:43:02  roefer
* First working versions of DefaultLandmarksPerceptor and MonteCarloSelfLocator
*
* Revision 1.4  2002/02/12 09:45:17  roefer
* Progress in DefaultLandmarksPerceptor and MonteCarloSelfLocator
*
* Revision 1.2  2002/02/06 10:30:11  AndySHB
* MonteCarloLocalization First Draft
*
* Revision 1.3  2001/12/10 17:47:06  risler
* change log added
*
*/
