/** 
* @file ColorTableTSL.h
* Declaration of class ColorTableTSL.
*
* @author <A href=mailto:robocup@andreosterhues.de>Andr Osterhues</A>
*/

#ifndef _ColorTableTSL_h_
#define _ColorTableTSL_h_

// forward declaration
class ColorTableTSL;

#include "ColorTable.h"
#include "Tools/Streams/InOut.h"
#include "Tools/ColorClasses.h"
#include "ColorClassImage.h"
#include "Image.h"
#include "Modules/ImageProcessor/ImageProcessorTools/MSH2004ColorCorrector.h"

/**
* @class ColorTableTSL
*
* Contains a ColorTableTSL which can decode the color for
* every 8 * 4 * 4 cube in the 255 * 255 * 255 YUV color space.
*
* @author <A href=mailto:robocup@andreosterhues.de>Andr Osterhues</A>
*/
class ColorTableTSL : public ColorTable
{
public:
  /** Constructor */
  ColorTableTSL();

  /** Destructor */
  ~ColorTableTSL();

  /**
  * Clear color tables
  */
  void clear();

  /**
  * Reset to default TSL values
  */
  void reset();

  /** Generates the Image for the TSL-Dialog-Bar */
  void generateTSLDialogImage(const Image &image,ColorClassImage &colorClassImage);
  
  /** 
  * Add a TSL range color class to the color table
  * @param colorClass the color class
  * @param t_min the min t value
  * @param t_max the max t value
  * @param s_min the min s value
  * @param s_max the max s value
  * @param l_min the min l value
  * @param l_max the max l value
  */
  void addColorClass(
    colorClass colorClass,
    unsigned char t_min,
    unsigned char t_max,
    unsigned char s_min,
    unsigned char s_max,
    unsigned char l_min,
    unsigned char l_max
    );

  /** 
  * Calculates the lookup table 
  */
  void calculateLUT();  

  /** 
  * Returns the color class of a pixel.
  * @param y the y value of the pixel
  * @param u the u value of the pixel
  * @param v the v value of the pixel
  * @return the color class
  */
  virtual colorClass getColorClass(
    const unsigned char y, 
    const unsigned char u, 
    const unsigned char v
    ) const;

  /**
  * Segments an image to an color class image.
  * 
  * This doesn't need to used in the image processor, but is needed for visualisation
  * of color tables.
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  */
  virtual void generateColorClassImage(
    const Image& image, 
    ColorClassImage& colorClassImage
    ) const;

  //!@name Members to be used by the ColorTableTSL dialog
  //!@{

  /**
  * YUV->TSL conversion
  */
  void convertYUVToTSL(
    unsigned char y,
    unsigned char u,
    unsigned char v,
    unsigned char *t,
    unsigned char *s,
    unsigned char *l
	  ) const;

  /**
  * TSL->colorClass classification
  */
  colorClass classifyTSL(
    unsigned char t,
	  unsigned char s,
	  unsigned char l
	  ) const;

  /** 
  * The color table.
  * Each entry contains both the color class A (lower 4 bits) and the color class B 
  * (higher 4 bits) values.
  * As Y information is less relevant, only 5 bits are used for Y; U and V use 6 bits.
  */
  unsigned char colorClasses[32][64][64];
  
  /** 
  * The TSL threshold table.
  * This table stores the TSL thresholds as given by addColorClass() and 
  * updateColorClass().
  * The first index selects the color class.
  * The second index selects min (0) or max (1).
  * The third index selects t (0), s (1) or l (2).
  */
  unsigned char tsl_threshold[numOfColors][2][3];

  /** 
  * The TSL threshold index.
  */
  unsigned char tsl_index;

  /**
  * The TSL order table.
  * It contains a permutation of colorClass.
  */
  colorClass tsl_order[numOfColors];

  //!@}

  private: 
	  
  MSH2004ColorCorrector corrector;

};

/**
 * Streaming operator that reads a ColorTableTSL from a stream.
 * @param stream The stream from which is read.
 * @param colorTableTSL The ColorTableTSL object.
 * @return The stream.
 */ 
In& operator>>(In& stream, ColorTableTSL& colorTableTSL);
 
/**
 * Streaming operator that writes a ColorTableTSL to a stream.
 * @param stream The stream to write on.
 * @param colorTableTSL The ColorTableTSL object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const ColorTableTSL& colorTableTSL);

#endif   //  _ColorTableTSL_h_

/*
 * Change log :
 * 
 * $Log: ColorTableTSL.h,v $
 * Revision 1.1.1.1  2004/05/22 17:25:49  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/03/04 10:27:28  schmidtb
 * color correction integrated
 *
 * Revision 1.2  2003/12/09 21:57:23  schmidtb
 * speed up of generateColorTableImage() and
 * method generateTSLDialogImage() to ColorTableTSL
 * changes made in TSLColorTableToolDlgBar
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/07/21 13:43:51  juengel
 * generateColorClassImage returns void
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.8  2003/03/31 04:19:23  osterhues
 * Changed type of colorClasses from "colorClass" to "unsigned char" (drastically reduces memory usage)
 *
 * Revision 1.7  2003/03/04 16:20:02  dueffert
 * group in doxygen docu meant that way?
 *
 * Revision 1.6  2003/02/18 21:29:17  osterhues
 * Changed all instances of ColorTable64 to new base class ColorTable
 *
 * Revision 1.5  2003/02/06 20:33:15  deutsch
 * TSLTools converted for RobotControl
 *
 * Revision 1.4  2003/01/31 13:41:47  osterhues
 * Swapped U and V for compatibility reasons *sigh*
 *
 * Revision 1.3  2003/01/28 15:35:25  osterhues
 * Should be functional now
 *
 * Revision 1.2  2003/01/24 13:34:46  osterhues
 * New interface
 *
 * Revision 1.1  2003/01/21 16:47:52  osterhues
 * Added ColorTableTSL class for TSL color space support.
 *
 */
