/**
* @file CameraInfo.h
* 
* Declaration of class CameraInfo
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/ 

#ifndef __CameraInfo_h_
#define __CameraInfo_h_

#include "Tools/RobotConfiguration.h"
#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector2.h"

//!@name camera resolution (currently used mode rather then all modes)
//!@{
const int cameraResolutionWidth_ERS210  = 176;
const int cameraResolutionHeight_ERS210  = 144;
const int cameraResolutionWidth_ERS7  = 208;
const int cameraResolutionHeight_ERS7  = 160;
//!@}

//!@name opening angles of the camera
//!@{
const double openingAngleWidth_ERS210  = 1.012290966; // 58 * pi / 180
const double openingAngleHeight_ERS210 = 0.837758041; // 48 * pi / 180
const double openingAngleWidth_ERS7  =   0.993092344; // Sony: 56.9 * pi / 180
const double openingAngleHeight_ERS7 =   0.788888822; // Sony: 45.2 * pi / 180
//const double openingAngleWidth_ERS7  =   0.9653254990; // Thomas: 55.3 * pi / 180
//const double openingAngleHeight_ERS7 =   0.7696185530; // Thomas: 44.1 * pi / 180
//const double openingAngleWidth_ERS7  =   0.9512044423; // Walter: 54.5 * pi / 180
//const double openingAngleHeight_ERS7 =   0.7539822369; // Walter: 43.2 * pi / 180
//const double openingAngleWidth_ERS7  = 0.9075712110; // Matthias measured: 52 * pi / 180
//const double openingAngleHeight_ERS7 = 0.6981317008; // Matthias measured: 40 * pi / 180
//!@}

//!@name camera intrinsic parameters
//!@{
const double focalLength_ERS210 = 159.1;
const double opticalCenterX_ERS210 = 87.56;
const double opticalCenterY_ERS210 = 73.82;
const double secondOrderRadialDistortion_ERS210 = 0.0927;
const double fourthOrderRadialDistortion_ERS210 = -0.451;

const double focalLength_ERS7 = 201.75; 
const double opticalCenterX_ERS7 = 102.1; 
const double opticalCenterY_ERS7 = 82.4; 
const double secondOrderRadialDistortion_ERS7 = -0.1538; 
const double fourthOrderRadialDistortion_ERS7 = 0.2591; 
/* NOTE: 
* from intrinsic calculation, real angles seem to be: 
* openingAngleWidth: 54.5
* openingAngleHeight: 43.2
* The narrower field of view of ERS7 compared to ERS210 is justified by the significantly lower distortion 
* at borders and corners of the image (radial, mainly, maximum displacement is only 40% compared to old robot, 
* while tangential is a lot higher (300%) but still negligible (up to 0.7 pixel displacement))
* 
*/
//!@}

/**
* Matrix describing transformation from neck joint to camera.
*/
class CameraInfo
{
public:
  /** 
  * Default constructor. 
  * Sets the parameters for the ERS-7.
  */
  CameraInfo();

  /**
  * Constructor.
  * @param robotDesign Selects the robot the camera info is initialized for.
  */
  CameraInfo(RobotDesign::Design robotDesign);

  int resolutionWidth;
  int resolutionHeight;
  double openingAngleWidth;
  double openingAngleHeight;

  /** Intrinsic camera parameters: axis skew is modelled as 0 (90 perfectly orthogonal XY) 
  * and the same has been modeled for focal axis aspect ratio; distortion is considering
  * only 2nd and 4th order coefficients of radial model, which account for about 95% of total.
  */
  double focalLength;
  double focalLengthInv; // (1/focalLength) used to speed up certain calculations 
  Vector2<double> opticalCenter;
  double secondOrderRadialDistortion;
  double fourthOrderRadialDistortion;
  double focalLenPow2;  
  double focalLenPow4;

  /** Was the image generated by the simulator? */
  bool simulated;
};

/**
* Streaming operator that reads a CameraInfo from a stream.
* @param stream The stream from which is read.
* @param cameraInfo The CameraInfo object.
* @return The stream.
*/ 
In& operator>>(In& stream,CameraInfo& cameraInfo);

/**
* Streaming operator that writes a CameraInfo to a stream.
* @param stream The stream to write on.
* @param cameraInfo The CameraInfo object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const CameraInfo& cameraInfo);

#endif //__CameraInfo_h_

/*
* Change log :
* 
* $Log: CameraInfo.h,v $
* Revision 1.3  2004/06/14 20:12:10  jhoffman
* - numerous changes and additions to headcontrol
* - cameraInfo default constructor now creates ERS7 info
* - debug drawing "headcontrolfield" added
*
* Revision 1.2  2004/05/26 14:06:32  roefer
* Corrected initialization of addition y channels in default constructor
* Flag for simulated images added
*
* Revision 1.1.1.1  2004/05/22 17:25:44  cvsadm
* created new repository GT2004_WM
*
* Revision 1.9  2004/03/27 10:21:58  roefer
* Outcommented opening angles added
*
* Revision 1.8  2004/03/09 11:32:08  nistico
* - Intrinsic parameters based measurements can now be triggered through a single conditional compilation
* switch located in CameraInfo.h
* - Implemented fast (look-up table based) radial distortion correction
*
* Revision 1.7  2004/02/24 18:46:00  dueffert
* doxygen bug fixed
*
* Revision 1.6  2004/02/12 14:21:49  nistico
* Calculated refined (4 robots used) intrinsic camera parameters for ERS-7
* Fixed bug with ERS-7 logfiles and intrinsic camera parameters (formerly, they weren't correctly associated
* with the higher resolution image from the new robot)
*
* Revision 1.5  2004/02/11 10:57:24  nistico
* Calculated preliminary (only one robot used this far) intrinsic camera parameters for ERS-7
*
* Revision 1.4  2004/02/10 10:48:05  nistico
* Introduced Intrinsic camera parameters to perform geometric calculations (distance, angle, size...) without opening angle
* Implemented radial distortion correction function
* Implemented ball distance calculation based on size and intrinsic params (potentially more stable)
* To Be Done: calculate intrinsic params for ERS7, as soon as we get our puppies back
*
* Revision 1.3  2004/01/23 00:10:02  roefer
* New constructor added, opening angles from Sony
*
* Revision 1.2  2004/01/04 18:18:49  juengel
* Opening angles for ERS7.
*
* Revision 1.1  2003/12/15 11:13:15  juengel
* Introduced CameraInfo
*
*/
