/**
 * @file CalibrationRequest.h
 * 
 * Declaration of class CalibrationRequest
 * 
 * @author Matthias Juengel
 */ 

#ifndef __CalibrationRequest_h_
#define __CalibrationRequest_h_

#include "Tools/Streams/InOut.h"

/** 
* @class CalibrationRequest 
*
* A request for calibration.
*/
class CalibrationRequest
{
public:

  /** constructor */
  CalibrationRequest();

  /** destructor */
  ~CalibrationRequest();

  enum CalibrationMode
  {
    none,
      calibrateColors,
      calibrateBallSideDetector,
      numOfCalibrationModes
  };

  enum CalibrationFeedback
  {
    noFeedback,
      colorIsGreen,
      colorIsOrange,
      ballRolledByLeft,
      ballRolledByRight,
      ballDidNotRollBy,
      numOfCalibrationFeedbacks
  };

  //! Returns the name of a CalibrationMode. 
  static char* getCalibrationModeName(CalibrationMode calibrationMode)
  {
    switch(calibrationMode)
    {
    case none: return "none";
    case calibrateColors: return "calibrateColors";
    case calibrateBallSideDetector: return"calibrateBallSideDetector";
    default: return "undefined CalibrationMode";
    }
  };

  //! Returns the name of a CalibrationFeedback. 
  static char* getCalibrationFeedbackName(CalibrationFeedback calibrationFeedback)
  {
    switch(calibrationFeedback)
    {
    case noFeedback: return "noFeedback";
    case colorIsGreen: return "colorIsGreen";
    case colorIsOrange: return "colorIsOrange";
    case ballRolledByLeft: return"ballRolledByLeft";
    case ballRolledByRight: return"ballRolledByRight";
    case ballDidNotRollBy: return"ballDidNotRollBy";
    default: return "undefined CalibrationFeedback";
    }
  };

  CalibrationMode mode;
  CalibrationFeedback feedback;
};


/**
 * Streaming operator that reads a CalibrationRequest from a stream.
 * @param stream The stream from which is read.
 * @param calibrationRequest The CalibrationRequest object.
 * @return The stream.
 */ 
In& operator>>(In& stream,CalibrationRequest& calibrationRequest);
 
/**
 * Streaming operator that writes a CalibrationRequest to a stream.
 * @param stream The stream to write on.
 * @param calibrationRequest The CalibrationRequest object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const CalibrationRequest& calibrationRequest);



#endif //__CalibrationRequest_h_

/*
 * Change log :
 * 
 * $Log: CalibrationRequest.h,v $
 * Revision 1.1.1.1  2004/05/22 17:25:31  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2004/03/21 12:46:28  juengel
 * CalibrationRequest distinguishes between "mode" and "feedback".
 *
 * Revision 1.1  2003/10/12 11:44:58  juengel
 * Added CalibrationRequest.
 *
 */
