/** 
* @file MotionRequest.h
*
* This file contains the MotionRequest class as well as SpecialActionRequest and WalkingRequest classes.
*
* @author <A href=mailto:brunn@sim.informatik.tu-darmstadt.de>Ronnie Brunn</A>
* @author <A href=mailto:kuntze@sim.informatik.tu-darmstadt.de>Nicolai Kuntze</A>
* @author <A href=mailto:kunz@sim.informatik.tu-darmstadt.de>Michael Kunz</A>
* @author Max Risler
* @author Michael Spranger
*/

#ifndef __MOTIONREQUEST_H__
#define __MOTIONREQUEST_H__

#include "Tools/Streams/InOut.h"
#include "Tools/Math/Pose2D.h"


/**@name walking max speeds
*@{
* @attention These values are implementation independent maximum speeds.
* They dont reflect speeds reached with current walking engine implementations.
*
* They should be used e.g. for the Motion Tester and for the behavior to walk with
* maximum speed.
*/
/** the maximum forward speed for walk in mm/s */
static const double walkMaxForwardSpeed = 450.0;
/** the maximum backward speed for walk in mm/s */
static const double walkMaxBackwardSpeed = 350.0;
/** the maximum sideward left/right speed for walk in mm/s */
static const double walkMaxLeftRightSpeed = 400.0;
/** the maximum rotation speed for walk in rad/s */
static const double walkMaxRotationSpeed = fromDegrees(250);
/** 
*@} */

/*
 *	@class 
 */
class SpecialActionRequest{
public:
    /** ids for all special actions */
  enum SpecialActionID {
    //please sort alphabetically, but only these IDs here:
    anyLeft,
      armLeft,
      armRight,
      anyRight,
      bash,
      block,
      blockLeft,
      blockRight,
      cheerAssShake,
      cheerHappy1,
      cheerHappy2,
      cheerHappy3,
      cheerHappy4,
      cheerHappy5,
      cheerHappy6,
      cheerHeadStand,
      cheerHowl,
      cheerLegTrick,
      cheerNodHead,
      cheerSad1,
      cheerSad2,
      cheerSad3,
      cheerSad5,
      cheerSad4,
      cheerWaveLeft,
      cheerWaveRight,
      chestHard,
      chestSoft,
      demoBallOnBack1,
      demoBallOnBack2,
      demoPackHeadStand,
      demoRemoveEars,
      demoRollLeft,
      demoRollRight,
      demoScratchHead,
      demoSit,
      demoStandUpsideDown,
      ers210rollLeft,
      ers210rollRight,
      ers210armLeft,
      ers210armRight,
      ers210bash,
      ers210blockLeft,
      ers210blockRight,
      ers210bow,
      ers210chest,
      ers210headLeft,
      ers210headRight,
      ers210howl,
	  forwardKickFast,
	  forwardKickHard,
      headLeft,
	  headLeftSoft,
      headRight,
      headRightSoft,
      hookLeft,
      hookRight,
      leftPaw,
      pushSoft,
      pushStrong,
      putLeft,
      putRight,
      rightPaw,
	  slapLeft,
	  slapRight,
      snowPlow,
      swing,
      tunnelKick,
      numOfSpecialAction
  };
  
  SpecialActionRequest():specialActionType(swing){}

  SpecialActionID specialActionType;

  
  /** returns names for special action ids */
  static const char* getSpecialActionIDName(SpecialActionID id)
  {
    switch (id)
    {
    case anyLeft: return "anyLeft";
    case armLeft: return "armLeft";
    case armRight: return "armRight";
    case anyRight: return "anyRight";
    case bash: return "bash";
    case block: return "block";
    case blockLeft: return "blockLeft";
    case blockRight: return "blockRight";
    case cheerAssShake: return "cheerAssShake";
    case cheerHappy1: return "cheerHappy1";
    case cheerHappy2: return "cheerHappy2";
    case cheerHappy3: return "cheerHappy3";
    case cheerHappy4: return "cheerHappy4";
    case cheerHappy5: return "cheerHappy5";
    case cheerHappy6: return "cheerHappy6";
    case cheerHeadStand: return "cheerHeadStand";
    case cheerHowl: return "cheerHowl";
    case cheerLegTrick: return "cheerLegTrick";
    case cheerNodHead: return "cheerNodHead";
    case cheerSad1: return "cheerSad1";
    case cheerSad2: return "cheerSad2";
    case cheerSad3: return "cheerSad3";
    case cheerSad5: return "cheerSad5";
    case cheerSad4: return "cheerSad4";
    case cheerWaveLeft: return "cheerWaveLeft";
    case cheerWaveRight: return "cheerWaveRight";
    case chestHard: return "chestHard";
    case chestSoft: return "chestSoft";
    case demoBallOnBack1: return "demoBallOnBack1";
    case demoBallOnBack2: return "demoBallOnBack2";
    case demoPackHeadStand: return "demoPackHeadStand";
    case demoRemoveEars: return "demoRemoveEars";
    case demoRollLeft: return "demoRollLeft";
    case demoRollRight: return "demoRollRight";
    case demoScratchHead: return "demoScratchHead";
    case demoSit: return "demoSit";
    case demoStandUpsideDown: return "demoStandUpsideDown";
    case ers210rollLeft: return "ers210rollLeft";
    case ers210rollRight: return "ers210rollRight";
    case ers210armLeft: return "ers210armLeft";
    case ers210armRight: return "ers210armRight";
    case ers210bash: return "ers210bash";
    case ers210blockLeft: return "ers210blockLeft";
    case ers210blockRight: return "ers210blockRight";
    case ers210bow: return "ers210bow";
    case ers210chest: return "ers210chest";
    case ers210headLeft: return "ers210headLeft";
    case ers210headRight: return "ers210headRight";
    case ers210howl: return "ers210howl";
	case forwardKickFast: return "forwardKickFast";
	case forwardKickHard: return "forwardKickHard";
    case headLeft: return "headLeft";
	case headLeftSoft: return "headLeftSoft";
    case headRight: return "headRight";
    case headRightSoft: return "headRightSoft";
    case hookLeft: return "hookLeft";
    case hookRight: return "hookRight";
    case leftPaw: return "leftPaw";
    case pushSoft: return "pushSoft";
    case pushStrong: return "pushStrong";
    case putLeft: return "putLeft";
    case putRight: return "putRight";
    case rightPaw: return "rightPaw";
	case slapLeft: return "slapLeft";
	case slapRight: return "slapRight";
    case snowPlow: return "snowPlow";
    case swing: return "swing";
    case tunnelKick: return "tunnelKick";
      
    default: return "please edit MotionRequest::getSpecialActionName";
    }
  }

  /**
  * Returns the special action id for a name
  */
  static SpecialActionRequest::SpecialActionID getSpecialActionIDFromName(const char* name);
};

class WalkRequest{
public:
    /** ids for all walking types */
  enum WalkType {
      normal,
      upsideDown,
      turnWithBall,
      turnKick,
      dash,
      debug,
      numOfWalkType
  };

  WalkRequest():walkType(normal) {}

  /** walking type */
  WalkType walkType;
  
  /** walk parameters, in mm/s / rad/s */
  Pose2D walkParams;

  /** returns names for walking types */
  static const char* getWalkTypeName(WalkType id)
  {
    switch (id)
    {
    case normal: return "normal";
    case upsideDown: return "upsideDown";
    case turnWithBall: return "turnWithBall";
    case turnKick: return "turnKick";
    case dash: return "dash";
    case debug: return "debug";
    default: return "Unknown motion name, please edit MotionRequest::getWalkTypeName";
    }
  }
};

class TailRequest{
public:
    enum TailRequestID
  {
      wagHorizontal,
      wagVertical,
      wagLeftRightHorizontal,
      wagLeftRightVertical,
      wagUpDownLeft,
      wagUpDownRight,
      stayAsForced,
      stayAsForcedPan,
      stayAsForcedTilt,
      tailFollowsHead,
      tailParallelToGround,
      noTailWag,
      wagHorizontalFast,
      wagVerticalFast,
      twoPositionSwitchVertical,
      twoPositionSwitchHorizontal,
      threePositionSwitchVertical,
      threePositionSwitchHorizontal,
      fourPositionSwitchUpDownLeftRight,
      fourPositionSwitchCorners,
      fivePositionSwitch,
      eightPositionSwitch,
      tailLeftTop,    tailCenterTop,    tailRightTop,
      tailLeftCenter, tailCenterCenter, tailRightCenter,
      tailLeftBottom, tailCenterBottom, tailRightBottom,
      numOfTailRequests
  };
  
  TailRequest():tailRequestID(wagHorizontal){}

  /** tail request */
  TailRequestID tailRequestID;
    
  
  
  /** ids for tail request */
  /** returns names for tail requests */
  static const char* getTailRequestIDName(TailRequestID id)
  {
    switch (id)
    {
    case wagHorizontal: return "wagHorizontal";
    case wagVertical: return "wagVertical";
    case wagLeftRightHorizontal: return "wagLeftRightHorizontal";
    case wagLeftRightVertical: return "wagLeftRightVertical";
    case wagUpDownLeft: return "wagUpDownLeft";
    case wagUpDownRight: return "wagUpDownRight";
    case stayAsForced: return "stayAsForced";
    case stayAsForcedPan: return "stayAsForcedPan";
    case stayAsForcedTilt: return "stayAsForcedTilt";
    case tailFollowsHead: return "tailFollowsHead";
    case tailParallelToGround: return "tailParallelToGround";
    case noTailWag: return "noTailWag";
    case wagHorizontalFast: return "wagHorizontalFast";
    case wagVerticalFast: return "wagVerticalFast";
    case twoPositionSwitchVertical: return "twoPositionSwitchVertical";
    case twoPositionSwitchHorizontal: return "twoPositionSwitchHorizontal";
    case threePositionSwitchVertical: return "threePositionSwitchVertical";
    case threePositionSwitchHorizontal: return "threePositionSwitchHorizontal";
    case fourPositionSwitchUpDownLeftRight: return "fourPositionSwitchUpDownLeftRight";
    case fourPositionSwitchCorners: return "fourPositionSwitchCorners";
    case fivePositionSwitch: return "fivePositionSwitch";
    case eightPositionSwitch: return "eightPositionSwitch";
    case tailLeftTop: return "tailLeftTop";
    case tailCenterTop: return "tailCenterTop";
    case tailRightTop: return "tailRightTop";
    case tailLeftCenter: return "tailLeftCenter";
    case tailCenterCenter: return "tailCenterCenter";
    case tailRightCenter: return "tailRightCenter";
    case tailLeftBottom: return "tailLeftBottom";
    case tailCenterBottom: return "tailCenterBottom";
    case tailRightBottom: return "tailRightBottom";
      
    default: return "Unknown tail request, please edit MotionRequest::getTailRequestName";
    }
  }
  
};
/**
* This describes the MotionRequest
*/
class MotionRequest
{
public:
  
  unsigned long frameNumber;
  
  void setFrameNumber(unsigned long frameNumber) {this->frameNumber = frameNumber;}
  
  
  /** ids for all motion types */
  enum MotionID {
    stand,
    walk,
    specialAction,
    getup,
    playDead,
    numOfMotion
  };
  
  /** returns names for motion ids */
  static const char* getMotionName(MotionID id)
  {
    switch (id)
    {
    case playDead: return "playDead";
    case stand: return "stand";
    case walk: return "walk";
    case specialAction: return "specialAction";
    case getup: return "getup";
      
    default: return "Unknown motion name, please edit MotionRequest::getMotionName";
    }
  }
  
  /** motion type */
  MotionID motionType;

  /** SpecialActionRequest */
  SpecialActionRequest specialActionRequest;

  TailRequest tailRequest;

  WalkRequest walkRequest;
  
  
  
  
  
  /** constructor, startup motion defined here */
#ifdef _WIN32
  MotionRequest():motionType(playDead),stabilize(false),updateRP(true),breathe(false),frameNumber(0) {}
#else
  MotionRequest():motionType(playDead),stabilize(false),breathe(false),updateRP(true),frameNumber(0) {}
#endif
  /** == operator */
  bool operator==(const MotionRequest& other) const 
  {
    switch (motionType)
    {
    case stand:
    case getup:
    default:
      return (
        motionType == other.motionType &&
        stabilize == other.stabilize);
    case walk:
      return (walkRequest.walkParams == other.walkRequest.walkParams &&
        stabilize == other.stabilize &&
        motionType == other.motionType &&
        walkRequest.walkType == other.walkRequest.walkType);
    case specialAction:
      return (
        stabilize == other.stabilize &&
        motionType == other.motionType &&
        specialActionRequest.specialActionType == other.specialActionRequest.specialActionType);
    }
  }
  
  /** = operator */
  void operator=(const MotionRequest& other);
  
  /** determines if the motion is being stabilized */
  bool stabilize;
  
  /** turns on or off the propagation (by odometry) of the "robot pose"
  in the "motion" process */
  bool updateRP;
  
  /** switches robot breathing motion on/off */
  bool breathe;
  
  /** returns name of current motion */
  const char* getMotionName() const {return getMotionName(motionType);};
  
  /** returns name of current special action if any */
  const char* getSpecialActionName() const
  {
    if (motionType == specialAction)
      return specialActionRequest.getSpecialActionIDName(specialActionRequest.specialActionType);
    else
      return "";
  };
  
  /** return name of current walking type if any */
  const char* getWalkTypeName() const
  {
    if (motionType == walk)
      return walkRequest.getWalkTypeName(walkRequest.walkType);
    else
      return "";
  }
  
  /**
  * Prints the motion request to a readable string. (E.g. "NormalWalk 100,0,0")
  * @param destination The string to fill
  */
  void printOut(char* destination) const;
  
  int padding; /**< Without this padding, the gcc-version crashes. */
};

/**
* Streaming operator that reads a MotionRequest from a stream.
* @param stream The stream from which is read.
* @param motionRequest The MotionRequest object.
* @return The stream.
*/ 
In& operator>>(In& stream,MotionRequest& motionRequest);

/**
* Streaming operator that writes a MotionRequest to a stream.
* @param stream The stream to write on.
* @param motionRequest The MotionRequest object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const MotionRequest& motionRequest);

#endif // __MOTIONREQUEST_H__

/*
* Change log :
* 
* $Log: MotionRequest.h,v $
* Revision 1.12  2004/07/02 12:05:37  lohmann
* Added  headLeftSoft and headRightSoft
*
* Revision 1.11  2004/07/01 10:04:13  lohmann
* added slapLeft and slapRight
*
* Revision 1.10  2004/06/30 08:37:53  lohmann
* Added forwardKickHard and forwardKickFast
*
* Revision 1.9  2004/06/14 15:02:24  spranger
* removed playDead
*
* Revision 1.8  2004/06/04 16:36:44  juengel
* Added walkType turn-kick
*
* Revision 1.7  2004/06/04 13:50:46  dueffert
* unused turnwithball walk types removed
*
* Revision 1.6  2004/06/02 17:19:19  spranger
* split up motionrequest in tail-,specialaction-,walk-request
* removed walkcycle double (its in motioninfo)
*
* Revision 1.5  2004/05/27 16:22:52  goehring
* renamed some special actions
*
* Revision 1.4  2004/05/27 13:36:49  goehring
* bugfix
*
* Revision 1.3  2004/05/27 13:10:11  goehring
* some specialActions removed
*
* Revision 1.2  2004/05/26 20:47:13  goehring
* removed some special actions
*
* Revision 1.1.1.1  2004/05/22 17:25:30  cvsadm
* created new repository GT2004_WM
*
* Revision 1.48  2004/05/17 13:56:51  rossdeutscher
* Added mofs for open challenge
*
* Revision 1.47  2004/05/10 10:29:17  juengel
* Added positionInWalkCycle.
*
* Revision 1.46  2004/05/06 12:24:35  hamerla
* add some mofs
*
* Revision 1.45  2004/04/22 14:38:48  dueffert
* speeds adapted to new abilities
*
* Revision 1.44  2004/04/09 12:49:59  tim
* added kicks fom BB2004
*
* Revision 1.43  2004/04/08 15:33:07  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.42  2004/04/07 13:00:47  risler
* ddd checkin after go04 - second part
*
* Revision 1.10  2004/04/04 01:18:48  risler
* grab mof added
*
* Revision 1.9  2004/04/03 21:35:58  risler
* added dddErs7hold mof
* head kick from grab improved
*
* Revision 1.8  2004/04/03 21:21:05  dthomas
* added new mof dddErs7bande
*
* Revision 1.7  2004/04/02 17:01:25  Schmitt
* Added DDD SoccerKickLeft
*
* Revision 1.6  2004/04/01 14:32:09  Schmitt
* Changes removed
*
* Revision 1.5  2004/04/01 13:59:09  Schmitt
* Unified SpecialAction Names
*
* Revision 1.4  2004/03/31 18:28:26  dthomas
* updated kick-engine with tester, catch and catch-and-turn
*
* Revision 1.3  2004/03/31 10:50:58  risler
* walk type dash added
* adjusted max speed
*
* Revision 1.2  2004/03/30 16:26:23  risler
* ddd ers7 special actions renamed
*
* Revision 1.1.1.1  2004/03/29 08:28:45  Administrator
* initial transfer from tamara
*
* Revision 1.41  2004/04/05 17:56:50  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.40  2004/04/01 12:15:37  dueffert
* missing string added
* added athErs7GOShortDance
*
* Revision 1.4  2004/04/03 02:32:36  aldi
* added new athErs7 Cheers
*
* Revision 1.3  2004/04/01 21:59:10  aldi
* new athErs7 cheering added
*
* Revision 1.2  2004/04/01 20:49:44  loetzsch
* renamed athErs7UnswBash to athErs7ExactForwardKick
*
* Revision 1.1.1.1  2004/03/31 11:16:53  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.39  2004/03/29 17:50:29  loetzsch
* added athers7unswbashfromhold
*
* Revision 1.38  2004/03/29 15:30:22  dueffert
* new turn parameterset
*
* Revision 1.37  2004/03/29 15:14:42  loetzsch
* added athers7GrabBall
*
* Revision 1.36  2004/03/28 14:09:51  jhoffman
* - added error member variables to ball model
* - minor improvement in handling of updateRP in motiontesterdialog
*
* Revision 1.35  2004/03/28 11:07:35  thomas
* added more ddd-special-actions
*
* Revision 1.34  2004/03/19 21:18:00  altmeyer
* SideHeadKick and SideHeadKick2 changed and renamed to SideHeadKick L and SideHeadKickR
* a few ATH ERS7-kicks removed
*
* Revision 1.33  2004/03/19 18:58:23  jhoffman
* "update robot pose in motion" is now turned on by default
*
* Revision 1.32  2004/03/19 14:58:12  jess4279
* BB2004 Kicks  hinzugefgt
*
* Revision 1.31  2004/03/19 13:47:00  jess4279
* BB2004HeadKick hinzugefgt
*
* Revision 1.30  2004/03/18 11:59:59  thomas
* added: first ers7 special actions from ddd
*
* Revision 1.29  2004/03/17 21:08:39  cesarz
* Added turnWithBall3, which points to MSH2004TurnWithBallWalkingParameters.
*
* Revision 1.28  2004/03/17 19:52:09  juengel
* Added walkType turnWithBall2
*
* Revision 1.27  2004/03/17 15:38:51  thomas
* added walkType debug for debugging WalkingEngineParameterSets
* paramSets-messages send by debug-message are stored in the new debug-walkType only
*
* Revision 1.26  2004/03/16 14:00:21  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.2  2004/03/15 17:11:40  hoffmann
* - added ATH2004HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
* Revision 1.25  2004/03/16 13:49:25  altmeyer
* added new ATHErs7 Special Actions
*
* Revision 1.24  2004/03/16 13:46:07  altmeyer
* added
*
* Revision 1.23  2004/03/15 17:47:11  rossbacher
* Fixed naming problem with MSH7 and added another .mof
*
* Revision 1.22  2004/03/12 15:34:37  rossbacher
* Added new MSH7 Special Actions
*
* Revision 1.21  2004/03/11 11:23:04  schmitt
* Added ERS-7 headKick
*
* Revision 1.20  2004/03/10 15:10:10  rossdeutscher
* Adden some new MSH7 .mofs
*
* Revision 1.19  2004/03/10 11:16:17  rossbacher
* Added MSH7 mofs
*
* Revision 1.18  2004/03/04 18:32:38  rempe
* new goalie defend motions added
*
* Revision 1.17  2004/02/27 18:20:17  brueckne
* mofs added
*
* Revision 1.16  2004/02/27 17:26:06  heinze
* added 3 mofs
*
* Revision 1.15  2004/02/27 15:24:14  juengel
* Added some wake-up motions.
*
* Revision 1.14  2004/02/26 10:54:02  jhoffman
* - hopefully fixed ath-mof name problems
*
* Revision 1.13  2004/02/26 09:35:08  schumann
* removed compiler errors caused by misspelling of ath kick
*
* Revision 1.12  2004/02/25 15:28:01  jhoffman
* kick renamed
*
* Revision 1.11  2004/02/25 13:43:28  jhoffman
* ATH kick renamed
*
* Revision 1.10  2004/02/06 20:24:39  roefer
* Maximum rotation speed increased
*
* Revision 1.9  2004/01/19 14:53:46  dueffert
* all frameNumbers (and not only some of them) are unsigned long now
*
* Revision 1.8  2004/01/08 13:52:35  loetzsch
* added new special action ATH_kickForwardWithRightPaw
*
* Revision 1.7  2004/01/05 17:56:27  juengel
* Added HeadControlModes stayAsForcedPan and stayAsForcedTilt.
*
* Revision 1.6  2004/01/05 11:37:35  juengel
* Added new TailModes: wagUpDown* and wagLeftRight*.
*
* Revision 1.5  2003/12/21 15:43:12  kerdels
* einrckung..
*
* Revision 1.4  2003/12/16 18:52:55  loetzsch
* added a function for retrieving the id for as string.
*
* Revision 1.3  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.2  2003/10/29 19:38:41  lohmann
* New Headstand and new behavior for game-state-final (test-final.xml) added.
*
* Revision 1.1  2003/10/07 10:07:01  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.13  2003/09/25 11:23:52  juengel
* Removed BlobCollection.
*
* Revision 1.12  2003/09/25 10:16:18  juengel
* Removed some walk-types.
*
* Revision 1.11  2003/09/24 16:39:03  juengel
* comments removed
*
* Revision 1.10  2003/07/07 14:11:02  thomas
* remove old diveKick
*
* Revision 1.9  2003/07/07 08:56:02  brunn
* lanceKicks
*
* Revision 1.8  2003/07/05 22:14:56  dueffert
* kicks renamed and properly linked
*
* Revision 1.7  2003/07/05 15:25:20  dueffert
* escapePadua added
*
* Revision 1.6  2003/07/05 12:09:25  dueffert
* most important mof ever added
*
* Revision 1.5  2003/07/05 11:53:11  schumann
* challenge kicks added and grabbing behavior prepared
*
* Revision 1.4  2003/07/03 15:09:46  thomas
* fix: missing entries for new kicks
*
* Revision 1.3  2003/07/03 14:54:23  risler
* new kicks
*
* Revision 1.2  2003/07/02 10:21:11  dueffert
* missing kick added
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.42  2003/06/27 13:39:08  reiss
* New Kicks added (von Andreas Rei, Carsten Schuhmann, Michael Wachter, Wenchuan Ding)
*
* Revision 1.41  2003/06/23 12:56:07  schumann
* added special bicycle kick for challenge
*
* Revision 1.40  2003/06/19 20:19:28  risler
* diveKickFromHold added
*
* Revision 1.39  2003/06/19 18:26:24  thomas
* added: diveKick
*
* Revision 1.38  2003/06/18 18:08:50  jhoffman
* forward leaping kick reordered
*
* Revision 1.37  2003/06/13 17:06:17  jhoffman
* added new kick motion
*
* Revision 1.36  2003/05/06 16:50:51  juengel
* Added tail-mode eightPositionSwitch.
*
* Revision 1.35  2003/05/06 07:57:30  dueffert
* alphabetically reordered
*
* Revision 1.34  2003/04/15 15:52:12  risler
* DDD GO 2003 code integrated
*
* Revision 1.25  2003/04/11 23:12:33  max
* added howl mof
*
* Revision 1.33  2003/04/14 16:37:47  dueffert
* several cheer motions added
*
* Revision 1.4  2003/04/10 19:12:34  dueffert
* context sensitive reset in MotionTester
*
* Revision 1.3  2003/04/10 17:35:44  Daniel
* cheer motions implemented
*
* Revision 1.2  2003/04/10 08:25:05  dueffert
* fun stuff added
*
* Revision 1.32  2003/04/08 19:27:50  hebbel
* added bicycleKickFromHold
*
* Revision 1.31  2003/04/08 16:07:39  dueffert
* mofs added
*
* Revision 1.30  2003/04/05 15:08:46  loetzsch
* removed chestKick3 again
*
* Revision 1.29  2003/04/04 17:20:05  juengel
* Added chestKick3
*
* Revision 1.28  2003/04/04 16:06:08  cesarz
* kicks added
*
* Revision 1.27  2003/04/03 14:38:46  hebbel
* kickFromHold added
*
* Revision 1.26  2003/04/02 14:55:21  timrie
* Added BBwinner
*
* Revision 1.25  2003/04/02 14:10:39  dueffert
* walkType turnWithBall added, sorted alphabetically
*
* Revision 1.24  2003/04/02 13:54:36  loetzsch
* added lookForLandmarksWithBallCaught
*
* Revision 1.23  2003/04/01 22:45:34  cesarz
* added catchBallAndCheck special action
*
* Revision 1.22  2003/03/25 14:19:30  timrie
* Added Specialaction BBloser
*
* Revision 1.21  2003/03/24 14:45:49  juengel
* Added several tail modes
*
* Revision 1.20  2003/03/07 17:08:41  juengel
* initialized breathe with false
*
* Revision 1.19  2003/03/06 11:55:50  dueffert
* re-order warning removed
*
* Revision 1.18  2003/02/25 09:02:11  roefer
* breathe initialized under _WIN32
*
* Revision 1.17  2003/02/05 17:03:38  dueffert
* packs added, getUpsideDown renamed
*
* Revision 1.16  2003/02/03 16:40:41  risler
* increased maximum walking speeds
*
* Revision 1.15  2003/02/01 17:06:42  risler
* added rollRight, RollLeft
*
* Revision 1.14  2003/02/01 17:04:18  risler
* added rollRight, RollLeft
*
* Revision 1.13  2003/01/27 19:27:08  juengel
* added walkType upsideDown
*
* Revision 1.12  2003/01/21 23:33:32  juengel
* Added getUpsideDown.mof
*
* Revision 1.11  2003/01/15 16:20:25  loetzsch
* reanamed boundary kick to bicycle kick
*
* Revision 1.10  2002/12/01 18:52:28  roefer
* No playDead in simulation
*
* Revision 1.9  2002/11/26 19:09:19  loetzsch
* MotionRequest is initialized with play-dead
*
* Revision 1.8  2002/11/26 17:19:22  risler
* added wakeUp specialAction for smooth startup
*
* Revision 1.7  2002/11/25 14:49:08  jhoffman
* added "breathing" motion
*
* Revision 1.6  2002/11/20 18:02:29  risler
* added PID values to GT2001MotionNetSpecialActions
* added playDead motion
*
* Revision 1.5  2002/11/19 17:12:24  juengel
* "switch" tail modes added
*
* Revision 1.4  2002/10/11 16:59:15  roefer
* Fast tail wags added
*
* Revision 1.3  2002/09/22 18:40:54  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/17 23:55:19  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:26:40  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.26  2002/09/03 16:00:50  juengel
* HeadControlMode follwTail, TailMode stayAsForced.
*
* Revision 1.25  2002/08/22 14:41:03  risler
* added some doxygen comments
*
* Revision 1.24  2002/08/04 17:53:18  roefer
* SimGT2002 connection to physical robots added
*
* Revision 1.23  2002/07/23 13:32:57  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.22  2002/07/14 14:03:07  roefer
* First working gcc-version
*
* Revision 1.21  2002/06/27 11:18:59  risler
* added additional walk type for bar challenge
*
* Revision 1.20  2002/06/21 04:52:04  risler
* added walktype walkWithBar
*
* Revision 1.19  2002/06/22 07:57:27  RoboCup
* boundaryKickRight and -Left added.
*
* Revision 1.18  2002/06/19 11:30:25  jhoffman
* added heel kicks
*
* Revision 1.17  2002/06/13 21:51:21  loetzsch
* removed the comparisons for tail modes in the == operator
*
* Revision 1.16  2002/06/13 13:08:06  jhoffman
* added some kicks
*
* Revision 1.15  2002/06/10 11:05:17  risler
* added timestamp to motion request
* motioncontrol executes swing when no request was received
*
* Revision 1.14  2002/06/08 14:33:24  kosen
* openArm added
*
* Revision 1.13  2002/06/08 11:24:12  Cesarz
* Added GoalieJumps
*
* Revision 1.12  2002/06/06 16:15:14  jhoffman
* added unsw-bash-optimized (kick mof)
*
* Revision 1.11  2002/06/06 13:20:44  risler
* removed bug with MotionRequest == operator
*
* Revision 1.10  2002/06/06 12:51:44  risler
* removed bug with MotionRequest == operator / executed motion request
*
* Revision 1.9  2002/06/06 10:24:53  AndySHB
* Neue mof hinzugefuegt: grabBall.
*
* Revision 1.8  2002/06/04 16:44:55  risler
* tailRequest added
*
* Revision 1.7  2002/06/03 14:38:24  dueffert
* Jan's new paw kick
*
* Revision 1.6  2002/06/01 10:17:14  juengel
* Added catchBall2.
*
* Revision 1.5  2002/05/27 14:40:01  cesarz
* boundaryKick added
*
* Revision 1.4  2002/05/24 15:36:53  kosen
* Ball Challenge mofs added
*
* Revision 1.3  2002/05/16 15:05:49  risler
* added walk type walkWithBall
*
* Revision 1.2  2002/05/13 07:22:34  cesarz
* mantisKick, armKickleft and armKickRight added
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.41  2002/05/06 17:15:20  risler
* increased max walk speeds
*
* Revision 1.40  2002/05/05 15:13:36  risler
* changed stand implementation, removed not working special actions
*
* Revision 1.39  2002/05/04 18:18:06  risler
* made printOut const
*
* Revision 1.38  2002/05/04 12:43:38  loetzsch
* The currently executed MotionRequest is now sent from the MotionControl
* to the BehaviorControl via the OdometryData structure
*
* Revision 1.37  2002/05/03 16:34:36  loetzsch
* added == operator
*
* Revision 1.36  2002/05/03 11:49:56  juengel
* Removed secretSpecialActionID.
*
* Revision 1.35  2002/04/29 13:46:42  risler
* no message
*
* Revision 1.34  2002/04/24 07:49:14  risler
* changed normalWalkMaxSpeed to walkMaxSpeed
*
* Revision 1.33  2002/04/23 15:08:45  risler
* changed MotionRequest: walk instead of normalWalk,... and walkType added
*
* Revision 1.32  2002/04/18 10:25:31  roefer
* Bremen GO
*
* Revision 1.31  2002/04/17 17:04:40  risler
* Darmstadt GO
*
* Revision 1.30  2002/04/16 16:23:04  dueffert
* no message
*
* Revision 1.8  2002/04/08 18:30:23  dueffert
* no message
*
* Revision 1.29  2002/04/15 13:39:24  rentmeister
* DoWalkingEngine (Dortmunder WalkingEngine)
*
* Revision 1.28  2002/04/08 16:27:26  mierswa
* New input for FuzzyBeahviourControl.
*
* Revision 1.27  2002/04/04 15:07:59  rentmeister
* Walk with Ball hinzugefgt
*
* Revision 1.26  2002/04/04 10:21:33  risler
* changed normalWalkMaxRotationSpeed to angle
*
* Revision 1.25  2002/04/03 16:48:53  jhoffman
* added "stabilizeRobot" to motionControl (which is turned off as a default)
*
* Revision 1.24  2002/04/03 15:20:24  risler
* changed normalWalkMaxRotationSpeed to angle
*
* Revision 1.23  2002/04/02 16:06:27  loetzsch
* added printOut function
*/
