/**
* @file LEDRequest.h
* 
* Declaration of class LEDRequest
*
* @author Max Risler
*/ 

#ifndef __LEDRequest_h_
#define __LEDRequest_h_


#include "Tools/Streams/InOut.h"

/**
* LED settings requested by BehaviorControl
*/
class LEDRequest
{
public:
  /** enum defining settings for two red LEDs */
  enum RedLEDPair
  {
    bothOn, /*!< both leds are on */
      bothFastBlink, /*!< both leds blink fast */
      bothSlowBlink, /*!< both leds blink slow */
      bothOff, /*!< both leds are off */
      leftOnly, /*!< only left led is on */
      rightOnly, /*!< only right led ist on */
      leftFastBlink, /*!< the left led blinks fast */
      rightFastBlink, /*!< the right led blinks fast */
      leftSlowBlink, /*!< the left led blinks slow */
      rightSlowBlink, /*!< the right led blinks slow */
      alternateFast, /*!< both leds alternate fast */
      alternateSlow, /*!< both leds alternate slow */
      numOfRedLEDPairs
  };
  
  /** returns name of a RedLEDPair mode */
  static const char* getRedLEDPairName(RedLEDPair id)
  {
    switch (id)
    {
    case bothOn: return "bothOn";
    case bothFastBlink: return "bothFastBlink";
    case bothSlowBlink: return "bothSlowBlink";
    case bothOff: return "bothOff";
    case leftOnly: return "left";
    case rightOnly: return "right";
    case leftFastBlink: return "leftFastBlink";
    case rightFastBlink: return "rightFastBlink";
    case leftSlowBlink: return "leftSlowBlink";
    case rightSlowBlink: return "rightSlowBlink";
    case alternateFast: return "alternateFast";
    case alternateSlow: return "alternateSlow";
    default: return "Unknown RedLEDPair, please edit LEDRequest::getRedLEDPairName";
    }
  }
  
  /** enum defining different blink modi */
  enum BlinkMode
  {
    oooo,
      oool,
      oolo,
      ooll,
      oloo,
      olol,
      ollo,
      olll,
      looo,
      lool,
      lolo,
      loll,
      lloo,
      llol,
      lllo,
      llll,
      numOfBlinkModes
  };
  
  /** returns name of a blink mode */
  static const char* getBlinkModeName(BlinkMode id)
  {
    switch (id)
    {
    case oooo: return "oooo";
    case oool: return "oool";
    case oolo: return "oolo";
    case ooll: return "ooll";
    case oloo: return "oloo";
    case olol: return "olol";
    case ollo: return "ollo";
    case olll: return "olll";
    case looo: return "looo";
    case lool: return "lool";
    case lolo: return "lolo";
    case loll: return "loll";
    case lloo: return "lloo";
    case llol: return "llol";
    case lllo: return "lllo";
    case llll: return "llll";
    default: return "Unknown blink mode, please edit LEDRequest::BlinkModes";
    }
  }
  
  /** enum defining settings for tail LED modes */
  enum TailMode
  {
    on, /*!<  led is on */
      slowBlink, /*!< led blinks slow */
      fastBlink, /*!< led blinks slow */
      numOfTailModes
  };
  
  /** returns name of a TailMode */
  static const char* getTailModeName(TailMode id)
  {
    switch (id)
    {
    case on: return "on";
    case slowBlink: return "slowBlink";
    case fastBlink: return "fastBlink";
    default: return "Unknown TailMode, please edit LEDRequest::getTailModeName";
    }
  }

  /** The frameNumber */
  bool showCognitionFrameLostWarning;
  
  /** setting for the two top red leds */
  RedLEDPair redTopLEDs;
  
  /** setting for the two bottom red leds */
  RedLEDPair redBottomLEDs;
  
  /** setting for the tail led */
  TailMode tailMode;
  
  /** setting for the face leds */
  BlinkMode faceLED[14];
  
  /** setting for the back front white led */
  BlinkMode backFrontWhiteLED;
  
  /** setting for the back front blue led */
  BlinkMode backFrontBlueLED;
  
  /** setting for the back middle white led */
  BlinkMode backMiddleWhiteLED;
  
  /** setting for the back middle orange led */
  BlinkMode backMiddleOrangeLED;
  
  /** setting for the back rear white led */
  BlinkMode backRearWhiteLED;
  
  /** setting for the back rear red led */
  BlinkMode backRearRedLED;

  /** setting for the head orange led */
  BlinkMode headOrangeLED;
  
  /** setting for the head white led */
  BlinkMode headWhiteLED;

  /** Constructor */
  LEDRequest();
};

/**
* Streaming operator that reads a LEDRequest from a stream.
* @param stream The stream from which is read.
* @param ledRequest The LEDRequest object.
* @return The stream.
*/ 
In& operator>>(In& stream,LEDRequest& ledRequest);

/**
* Streaming operator that writes a LEDRequest to a stream.
* @param stream The stream to write on.
* @param ledRequest The LEDRequest object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const LEDRequest& ledRequest);

#endif //__LEDRequest_h_

/*
* Change log :
* 
* $Log: LEDRequest.h,v $
* Revision 1.2  2004/06/15 17:48:00  juengel
* Added showCognitionFrameLostWarning.
*
* Revision 1.1.1.1  2004/05/22 17:25:18  cvsadm
* created new repository GT2004_WM
*
* Revision 1.8  2004/03/25 21:20:19  juengel
* Added head LEDs.
*
* Revision 1.7  2004/03/25 19:56:28  loetzsch
* again changed the led request
*
* Revision 1.6  2004/03/25 18:07:01  loetzsch
* added single variables for the back colored leds
*
* Revision 1.5  2004/03/25 17:39:17  loetzsch
* renamed initial-setup-mode to back-mode
*
* Revision 1.4  2004/03/21 19:08:15  juengel
* Added back*White LED support.
*
* Revision 1.3  2004/03/20 19:24:07  juengel
* Added face LEDs to LEDRequest
*
* Revision 1.2  2004/03/01 12:37:22  wachter
* Added LED-Request for button-interface
*
* Revision 1.1  2003/10/07 10:07:01  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/25 11:23:52  juengel
* Removed BlobCollection.
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.4  2003/05/08 14:27:33  risler
* added tailMode to LEDRequest
*
* Revision 1.3  2003/05/06 07:58:01  dueffert
* output of LEDRequest names added
*
* Revision 1.2  2002/11/18 17:20:22  loetzsch
* removed "old version" of LED request
*
* Revision 1.1  2002/09/10 15:26:40  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.3  2002/08/22 14:41:03  risler
* added some doxygen comments
*
* Revision 1.2  2002/07/23 13:32:57  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.10  2002/05/07 05:49:56  dueffert
* new LED handling implemented
*
* Revision 1.9  2002/05/04 17:42:27  loetzsch
* extended the request by a new version
*
* Revision 1.8  2002/04/17 17:04:40  risler
* Darmstadt GO
*
* Revision 1.7  2002/02/28 14:35:53  risler
* changed names in LEDRequest
*
* Revision 1.6  2002/02/07 19:57:06  risler
* DefaultLEDControl uses getPlayer now, removed team and role from LEDRequest
*
* Revision 1.5  2001/12/10 17:47:05  risler
* change log added
*
*/
