/**
 * @file SampleSet.h
 *
 * The file contains the definition of the class SampleSet and SampleSetProxy.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#ifndef __SampleSet_h_
#define __SampleSet_h_

#include "Tools/Math/Pose2D.h"

/**
* The class represents a 2D sample.
*/
class Vector2Sample : public Vector2<double>
{
  public:
    double probability;

    /**
    * Constructor.
    */
    Vector2Sample() {probability = 0.5;}
    Vector2Sample(const Vector2<double>& v) : Vector2<double>(v) {probability = 0.5;}
};

/**
* The class represents a pose sample.
*/
class PoseSample : public Pose2D
{
  public:
    double probability;

    /**
    * Constructor.
    */
    PoseSample() {probability = 0.5;}
    PoseSample(const Pose2D& pose) : Pose2D(pose) {probability = 0.5;}
};

/**
 * @class SampleSetProxyBase
 * The base of class SampleSetProxy.
 */
class SampleSetProxyBase
{
  protected:
    const char* data;
    int numberOfSamples,
        sizeOfEntries;
        
  public:
    /**
     * Constructor.
     */
    SampleSetProxyBase() {numberOfSamples = 0;}

    /**
    * The function returns the number of samples in the set.
    * @return The number of samples.
    */
    int getNumberOfSamples() const {return numberOfSamples;}

    /**
    * The function links the proxy to a sample set.
    */
    void link(const char* start, int number, int size)
    {
      data = (const char*) start;
      numberOfSamples = number;
      sizeOfEntries = size;
    }
};

/**
 * @class SampleSetProxy
 * The SampleSetProxy can be linked to instances of the class SampleSet.
 */
template<class T> class SampleSetProxy : public SampleSetProxyBase
{
  public:
    /**
     * Constant access operator.
     * @param index The index of the sample to access.
     */
    const T& operator[](int index) const {return *(const T*) (data + index * sizeOfEntries);}
};

/**
 * @class SampleSet
 * A container for samples. Two independant sets are maintained.
 * As the sample set can be used by different modules that require
 * a different number of samples, the size of the set can be changed
 * at runtime.
 */
template<class T, int N> class SampleSet
{
  private:
    T* current, /**< The actual sample set. */
     * other; /**< The secondary sample set. */

  public:
    /**
     * Constructor.
     */
    SampleSet()
    {
      current = new T[N];
      other = new T[N];
    }

    /**
     * Destructor.
     */
    ~SampleSet()
    {
      delete [] current;
      delete [] other;
    }

    /**
    * The function links a proxy to this sample set.
    * @param sampleSetProxy The proxy.
    */
    void link(SampleSetProxyBase& sampleSetProxy) const 
    {
      sampleSetProxy.link((const char*) current, N, sizeof(T));
    }

    /**
    * The function returns the number of samples in the set.
    * @return The number of samples.
    */
    int getNumberOfSamples() const {return N;}

    /**
     * Access operator.
     * @param index The index of the sample to access.
     */
    T& operator[](int index) {return current[index];}

    /**
     * Constant access operator.
     * @param index The index of the sample to access.
     */
    const T& operator[](int index) const {return current[index];}

    /**
     * The function swaps the primary and secondary sample set.
     * @return The address of the previous sample set;
     */
    T* swap()
    {
      T* temp = current;
      current = other;
      other = temp;
      return other;
    }
};

#endif //__SampleSet_h_

/*
 * Change log :
 * 
 * $Log: SampleSet.h,v $
 * Revision 1.1.1.1  2004/05/22 17:25:16  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2003/06/21 11:59:09  roefer
 * SampleBallLocator added, does not work!
 *
 * Revision 1.1  2003/05/08 23:52:24  roefer
 * SampleSet and SampleSetProxy added
 *
 */
