/**
 * @file RobotState.h
 * 
 * Declaration of class RobotState
 */ 

#ifndef __RobotState_h_
#define __RobotState_h_


#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector3.h"
#include "Representations/Perception/BodyPercept.h"

/**
 * A class describing detected button messages and whether robot is crashed
 */
class RobotState
{
public:
  enum States { undefined, standing, crashed, rollLeft, rollRight, pickedUp };
  enum MouthStates { mouthOpen, mouthClosed };
  
  Vector3<double> acceleration;

  RobotState();
  ~RobotState();

  unsigned long frameNumber;
  void setFrameNumber(unsigned long frameNumber) {this->frameNumber = frameNumber;}

  void setState(const States s) {state = s;}
  States getState() const {return state;}

  void setMouthState(const MouthStates t) {mouthState = t;}
  MouthStates getMouthState() const {return mouthState;}

  void setSomethingInFrontOfChest(bool value) {somethingInFrontOfChest = value;}
  bool getSomethingInFrontOfChest() const {return somethingInFrontOfChest;}

  void setDistanceToSIFOC(double value) {distanceToSIFOC = value;}
  double getDistanceToSIFOC() const {return distanceToSIFOC;}

  void setTimeWhenSomethingWasInFrontOfChestLast(unsigned long int time) {timeWhenSomethingWasInFrontOfChestLast = time;}
  unsigned long int getTimeWhenSomethingWasInFrontOfChestLast() const {return timeWhenSomethingWasInFrontOfChestLast;}

  void setButtonPressed(bool* buttonPressed)
  {
    for(int i = 0; i < BodyPercept::numOfSwitches; i++)
    {
      this->buttonPressed[i] = buttonPressed[i];
    }
  }
  bool getButtonPressed(BodyPercept::Switches button) const 
  {
    return buttonPressed[button];
  }

  void setButtonTime(long* buttonTime)
  {
    for(int i = 0; i < BodyPercept::numOfSwitches; i++)
    {
      this->buttonTime[i] = buttonTime[i];
    }
  }
  long getButtonTime(BodyPercept::Switches button) const 
  {
    return buttonTime[button];
  }

  void setButtonPressedTime(long* buttonPressedTime)
  {
    for(int i = 0; i < BodyPercept::numOfSwitches; i++)
    {
      this->buttonPressedTime[i] = buttonPressedTime[i];
    }
  }
  long getButtonPressedTime(BodyPercept::Switches button) const
  {
    return buttonPressedTime[button];
  }

  void setButtonDuration(long* buttonDuration)
  {
    for(int i = 0; i < BodyPercept::numOfSwitches; i++)
    {
      this->buttonDuration[i] = buttonDuration[i];
    }
  }
  long getButtonDuration(BodyPercept::Switches button) const
  {
    return buttonDuration[button];
  }

  void setAnyBackButtonPressed(bool anyBackButtonPressed)
  {
    this->anyBackButtonPressed = anyBackButtonPressed;
  }
  bool getAnyBackButtonPressed() const 
  {
    return anyBackButtonPressed;
  }

  void setAnyBackButtonTime(long anyBackButtonTime)
  {
    this->anyBackButtonTime = anyBackButtonTime;
  }
  long getAnyBackButtonTime() const 
  {
    return anyBackButtonTime;
  }

  void setAnyBackButtonPressedTime(long anyBackButtonPressedTime)
  {
    this->anyBackButtonPressedTime = anyBackButtonPressedTime;
  }
  long getAnyBackButtonPressedTime() const
  {
    return anyBackButtonPressedTime;
  }

  void setAnyBackButtonDuration(long anyBackButtonDuration)
  {
    this->anyBackButtonDuration = anyBackButtonDuration;
  }
  long getAnyBackButtonDuration() const
  {
    return anyBackButtonDuration;
  }

  /** copies another RobotState to this one */
  void operator = (const RobotState& other);

  static const char* getStateName(States s)
  {
    switch(s)
    {
      case undefined: return "undefined";
      case standing: return "standing";
      case crashed: return "crashed";
      case rollLeft: return "rollLeft";
      case rollRight: return "rollRight";
      case pickedUp: return "pickedUp";
      default: return "please edit RobotState::getStateName";
    }
  }

  void setCollisionFrontLeft(bool c) {collisionFrontLeft = c;}
  bool getCollisionFrontLeft() const {return collisionFrontLeft;}

  void setCollisionFrontRight(bool c) {collisionFrontRight = c;}
  bool getCollisionFrontRight() const {return collisionFrontRight;}

  void setCollisionHindLeft(bool c) {collisionHindLeft = c;}
  bool getCollisionHindLeft() const {return collisionHindLeft;}

  void setCollisionHindRight(bool c) {collisionHindRight = c;}
  bool getCollisionHindRight() const {return collisionHindRight;}

  void setCollisionHead(bool c) {collisionHead = c;}
  bool getCollisionHead() const {return collisionHead;}

  void setCollisionAggregate(bool c) {collisionAggregate = c;}
  bool getCollisionAggregate() const {return collisionAggregate;}


  void setConsecutiveCollisionTimeFrontLeft(long c) {consecutiveCollisionTimeFrontLeft = c;}
  long getConsecutiveCollisionTimeFrontLeft() const {return consecutiveCollisionTimeFrontLeft;}

  void setConsecutiveCollisionTimeFrontRight(long c) {consecutiveCollisionTimeFrontRight = c;}
  long getConsecutiveCollisionTimeFrontRight() const {return consecutiveCollisionTimeFrontRight;}

  void setConsecutiveCollisionTimeHindLeft(long c) {consecutiveCollisionTimeHindLeft = c;}
  long getConsecutiveCollisionTimeHindLeft() const {return consecutiveCollisionTimeHindLeft;}

  void setConsecutiveCollisionTimeHindRight(long c) {consecutiveCollisionTimeHindRight = c;}
  long getConsecutiveCollisionTimeHindRight() const {return consecutiveCollisionTimeHindRight;}

  void setConsecutiveCollisionTimeHead(long c) {consecutiveCollisionTimeHead = c;}
  long getConsecutiveCollisionTimeHead() const {return consecutiveCollisionTimeHead;}

  void setConsecutiveCollisionTimeAggregate(long c) {consecutiveCollisionTimeAggregate = c;}
  long getConsecutiveCollisionTimeAggregate() const {return consecutiveCollisionTimeAggregate;}

  void setCollisionSide(int s) {collisionSide = s;}
  long getCollisionSide() const {return collisionSide;}


private:

  /**
   * current state of robot position
   * if crashed or standing
   */
  States state;

  /**
   * current state of the mouth
   */
  MouthStates mouthState;

   /**
   * collision indicator of front left leg
   */
  bool collisionFrontLeft;

  /**
   * collision indicator of front right leg
   */
  bool collisionFrontRight;

  /**
   * collision indicator of hind left leg
   */
  bool collisionHindLeft;

  /**
   * collision indicator of hind right leg
   */
  bool collisionHindRight;

  /**
   * collision indicator of head
   */
  bool collisionHead;

   /**
   * collision indicator of collision occurrence
   */
  bool collisionAggregate;

   /**
   * Time for last consecutive collision on front left leg
   */
  long consecutiveCollisionTimeFrontLeft;

  /**
   * Time for last consecutive collision on front right leg
   */
  long consecutiveCollisionTimeFrontRight;

   /**
   * Time for last consecutive collision on hind left leg
   */
  long consecutiveCollisionTimeHindLeft;

  /**
   * Time for last consecutive collision on hind right leg
   */
  long consecutiveCollisionTimeHindRight;

   /**
   * Time for last consecutive collision on head
   */
  long consecutiveCollisionTimeHead;

   /**
   * Time for last consecutive collision aggregate
   */
  long consecutiveCollisionTimeAggregate;

   /**
   * collision occured on side
   */
  long collisionSide;


  /***/
  bool buttonPressed[BodyPercept::numOfSwitches]; //0 Button not pressed, 1 Button pressed
  unsigned long buttonTime[BodyPercept::numOfSwitches]; // The time as the button was pressed/released
  unsigned long buttonPressedTime[BodyPercept::numOfSwitches]; // How long the last button was pressed (after releasing it)
  unsigned long buttonDuration[BodyPercept::numOfSwitches]; // How long the button is pressed/released
  bool anyBackButtonPressed;  // The same information for any backbutton
  unsigned long anyBackButtonTime;
  unsigned long anyBackButtonPressedTime;
  unsigned long anyBackButtonDuration;
  /***/

  /** Indicates whether something has been measured by the body PSD or not*/
  bool somethingInFrontOfChest;

  /** The distance of objects in front of the body PSD*/
  double distanceToSIFOC;
  
  /** The last time when something has been measured by the body PSD */
  unsigned long int timeWhenSomethingWasInFrontOfChestLast;
};

/**
 * Streaming operator that reads a RobotState from a stream.
 * @param stream The stream from which is read.
 * @param robotState The RobotState object.
 * @return The stream.
 */ 
In& operator>>(In& stream,RobotState& robotState);
 
/**
 * Streaming operator that writes a RobotState to a stream.
 * @param stream The stream to write on.
 * @param robotState The RobotState object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const RobotState& robotState);


#endif //__RobotState_h_

/*
 * Change log :
 * 
 * $Log: RobotState.h,v $
 * Revision 1.6  2004/06/23 11:02:27  goehring
 * Xabsl symbol collision-side added
 *
 * Revision 1.5  2004/06/16 17:07:33  cesarz
 * Moved body PSD calculations
 *
 * Revision 1.4  2004/06/08 18:34:42  juengel
 * Added robot-state.time-since-something-was-in-front-of-chest.
 *
 * Revision 1.3  2004/05/25 12:36:51  tim
 * added body PSD data
 *
 * Revision 1.2  2004/05/24 14:16:06  juengel
 * New button evaluation.
 *
 * Revision 1.1.1.1  2004/05/22 17:25:16  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/02/10 11:12:08  dueffert
 * acceleration added; frameNumber is unsigned long everywhere
 *
 * Revision 1.3  2004/01/24 11:30:16  juengel
 * Added ERS7 switches (head, backFront, backBack).
 *
 * Revision 1.2  2003/11/13 17:39:03  goehring
 * frameNumber added
 *
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.4  2003/09/30 10:51:12  dueffert
 * typos fixed
 *
 * Revision 1.3  2003/09/25 11:23:52  juengel
 * Removed BlobCollection.
 *
 * Revision 1.2  2003/07/06 12:05:31  schumann
 * added foreleg opening angle for ball challenge
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/06/20 15:32:32  dueffert
 * getting from down from wall by rolling added
 *
 * Revision 1.5  2003/06/20 13:27:21  risler
 * added tailLeft tailRight messages
 *
 * Revision 1.4  2003/06/20 10:31:59  goehring
 * Added CollisionStates
 *
 * Revision 1.3  2003/04/01 22:40:44  cesarz
 * added mouth states
 *
 * Revision 1.2  2003/01/30 11:26:47  juengel
 * Added tailPosition to bodyPercept
 *
 * Revision 1.1  2002/09/10 15:26:39  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.4  2002/08/22 14:41:03  risler
 * added some doxygen comments
 *
 * Revision 1.3  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.2  2002/05/15 08:09:10  risler
 * added button down messages
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.7  2002/04/08 17:48:53  risler
 * added pickedUp
 *
 * Revision 1.6  2002/03/28 16:55:58  risler
 * RobotStateDetector receives BodyPercept instead of PerceptCollection
 * added switch duration in RobotStateDetector
 *
 * Revision 1.5  2002/03/28 15:19:20  risler
 * implemented switch messages in RobotStateDetector
 *
 * Revision 1.4  2002/02/23 16:37:15  risler
 * State in RobotState like BodyPercept state
 *
 * Revision 1.3  2001/12/10 17:47:06  risler
 * change log added
 *
 */
