/**
 * @file RobotPose.h
 *
 * The file contains the definition of the class RobotPose.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#ifndef __RobotPose_h_
#define __RobotPose_h_

#include "Tools/Streams/InOut.h"
#include "Tools/Math/Pose2D.h"

/**
 * @class RobotPose
 * A Pose2D with validity.
 */
class RobotPose : public Pose2D
{
  private:
    double validity; /**< The validity of the robot pose. */

  public:
    /**
     * Constructor.
     */
    RobotPose() {validity = 0; frameNumber = 0;}

    const RobotPose& operator=(const RobotPose& other)
    {
      (Pose2D&) *this = (const Pose2D&) other;
      validity = other.validity;
      frameNumber = other.frameNumber;
      return *this;
    }

    /**
     * Cast Contructor.
     */
    RobotPose (const Pose2D& otherPose) {
      (Pose2D&) *this = otherPose;
      validity=0;
    }

    const RobotPose& operator=(const Pose2D& other)
    {
      (Pose2D&) *this = other;
      //don't touch validity and frameNumber here!
      return *this;
    }

    /**
     * The function sets the current pose.
     * @param p The pose.
     */
    void setPose(const Pose2D& p) {
      (Pose2D&) *this = p;
    }

    /**
     * The function sets the validity of the current pose.
     * @param v The validity.
     */
    void setValidity(const double v) {validity = v;}

	/**
     * The function sets the frame number of the current pose.
     */
    void setFrameNumber(unsigned long frameNumber) {this->frameNumber = frameNumber;}

    
	
	/**
     * The function returns the current pose.
     * @return The pose.
     * @attention this function is obsolete and should not be used any more
     * @attention instead this object IS a pose instead of owning one
     */
    const Pose2D getPose() const {
      return (const Pose2D&) *this;
    }

    /**
     * The function returns the validity of the current pose.
     * @return The validity.
     */
    const double& getValidity() const {return validity;}

	/**< The frame number when perceived. */
	unsigned long frameNumber;

  /**< Speed measured by perceived distance to goal. */
  double speedbyDistanceToGoal;

  /**< The distance to the field border */
  double distanceToBorder;

  /**< The angle to the field border */
  double angleToBorder;
};

/**
 * Streaming operator that reads a robot pose from a stream.
 * @param stream The stream from which is read.
 * @param robotPose The robot pose object.
 * @return The stream.
 */ 
In& operator>>(In& stream,RobotPose& robotPose);
 
/**
 * Streaming operator that writes a robot pose to a stream.
 * @param stream The stream to write on.
 * @param robotPose The robot pose object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const RobotPose& robotPose);


#endif //__RobotPose_h_

/*
 * Change log :
 * 
 * $Log: RobotPose.h,v $
 * Revision 1.1.1.1  2004/05/22 17:25:15  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.8  2004/03/27 16:16:37  juengel
 * Added distanceToBorder and angleToBorder.
 *
 * Revision 1.7  2004/01/22 11:42:53  spranger
 * changed speedbydistancetogoal to remove some compiler warnings
 *
 * Revision 1.6  2004/01/22 08:55:43  spranger
 * added speedbydistancetogoal member
 *
 * Revision 1.5  2004/01/20 15:08:26  dueffert
 * missing =operator added
 *
 * Revision 1.4  2004/01/19 14:55:23  dueffert
 * all frameNumbers (and not only some of them) are unsigned long now
 *
 * Revision 1.3  2003/11/18 15:19:24  dueffert
 * serious bugs fixed
 *
 * Revision 1.2  2003/11/13 14:51:25  goehring
 * frameNumber added
 *
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2002/11/27 13:54:07  dueffert
 * doxygen docu added
 *
 * Revision 1.5  2002/11/19 17:38:31  dueffert
 * doxygen bugs corrected
 *
 * Revision 1.4  2002/11/06 21:24:22  loetzsch
 * no message
 *
 * Revision 1.3  2002/11/06 18:28:39  juengel
 * repaired Ball Locator
 *
 * Revision 1.2  2002/09/22 18:40:51  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:26:39  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.5  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.4  2002/06/08 16:24:41  Thomas Rfer
 * Cast constructor improved
 *
 * Revision 1.3  2002/06/08 12:58:19  Thomas Rfer
 * operator= inserted (to avoid MCOOP exception)
 *
 * Revision 1.2  2002/06/08 09:26:31  Thomas Rfer
 * Team ball position, first draft
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.9  2002/02/19 14:28:35  brunn
 * RobotPose ist jetzt auch eine Pose anstatt nur so zu heissen
 *
 * Revision 1.8  2002/02/13 22:43:02  roefer
 * First working versions of DefaultLandmarksPerceptor and MonteCarloSelfLocator
 *
 * Revision 1.7  2002/02/11 00:56:12  loetzsch
 * added two new setPose() functions
 *
 * Revision 1.6  2002/02/05 03:36:12  loetzsch
 * replaced direct member access by
 * inline const VALUE& get...() const   and
 * inline void set...(const Value&) methods.
 *
 * Revision 1.5  2002/01/25 15:40:13  roefer
 * The Oracle
 *
 * Revision 1.4  2002/01/13 12:16:16  AndySHB
 * no message
 *
 * Revision 1.3  2001/12/10 17:47:06  risler
 * change log added
 *
 */
