/**
 * @file PlayerPoseCollection.h
 * 
 * Declaration of class PlayerPoseCollection
 * @author <A href=mailto:kspiess@informatik.uni-bremen.de>Kai Spiess</A>
 */ 

#ifndef __PlayerPoseCollection_h_
#define __PlayerPoseCollection_h_

#include "Tools/Streams/InOut.h"
#include "Tools/Math/Pose2D.h"
#include "Tools/Player.h"


/**
 *  This class contains the informations about the PlayerPose 
 *  of a located robot. This is not for informations about the own robot.
 */
class PlayerPose
{
public:
  /** Constructor */
  PlayerPose(){speed = 0;playerNumber=Player::undefinedPlayerNumber;sigmaMin=-1;sigmaMaj=-1;};
  /** Copy-Constructor */
  PlayerPose(const PlayerPose& other){*this = other;};
  /** Destructor */
  ~PlayerPose(){};

  unsigned long frameNumber; /**< The frame number of the image that was used to create the percept. */
  
  void setFrameNumber(unsigned long frameNumber) {this->frameNumber = frameNumber;}

  /** sets the pose */
  void setPose(const Pose2D& p) {pose = p;}

  /** sets the pose */
  void setPose(const double& x, const double& y, const double& rotation)
  { pose.translation.x = x; pose.translation.y = y; pose.rotation = rotation; }

  /** returns the pose */
  const Pose2D& getPose() const {return pose;}

  /** sets the speed */
  void setSpeed(const double& _speed) {
    speed = _speed;
  }

  /** returns the speed */
  const double& getSpeed() const {return speed;}

  /** sets the validity */
  void setValidity(const double& v) {validity = v;}

  /** returns the validity */
  const double& getValidity() const {return validity;}

  /** sets the playerNumber */
  void setPlayerNumber(const Player::playerNumber n) {playerNumber = n;}

  /** returns the playerRole */
  const Player::playerNumber& getPlayerNumber() const {return playerNumber;}

  /** returns the player number as an Integer */
  int getPlayerNumberAsInt() const {return (int)playerNumber; }

  /** copies another PlayerPose to this one */
  void operator = (const PlayerPose& other);

  /** sets the sigmaMin */
  void setSigmaMin(const double sigma) {sigmaMin = sigma;}

  /** returns the sigmaMin */
  const double& getSigmaMin() const {return sigmaMin;}

  /** sets the sigmaMaj */
  void setSigmaMaj(const double sigma) {sigmaMaj = sigma;}

  /** returns the sigmaMaj */
  const double& getSigmaMaj() const {return sigmaMaj;}

  /** sets the cosAngle */
  void setCosAngle(const double cos) {cosAngle = cos;}

  /** returns the cosAngle */
  const double& getCosAngle() const {return cosAngle;}

  /** sets the sinAngle */
  void setSinAngle(const double sin) {sinAngle = sin;}

  /** returns the sinAngle */
  const double& getSinAngle() const {return sinAngle;}

private:
  /** The position and orientation of the player as a Pose2D */
  Pose2D pose;

  /** The speed of the player */
  double speed;

  /** The validity of the located robot */
  double validity;

  /** Playerrole */
  Player::playerNumber playerNumber;

  /** The variance in Min-direction */
  double sigmaMin;

  /** The variance in Maj-direction */
  double sigmaMaj;

  /** The cosinus of the angle of the variance-ellipse */
  double cosAngle;

  /** The sinus of the angle of the variance-ellipse */
  double sinAngle;

};

/**
 * Streaming operator that reads a PlayerPose from a stream.
 * @param stream The stream from which is read.
 * @param playerPose The PlayerPose object.
 * @return The stream.
 */ 
In& operator>>(In& stream,PlayerPose& playerPose);

/**
 * Streaming operator that writes a PlayerPose to a stream.
 * @param stream The stream to write on.
 * @param playerPose The PlayerPose object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const PlayerPose& playerPose);



/**
 *  This class contains the PlayerPoses of all detected robots.
 */
class PlayerPoseCollection
{
public:
  /** Constructor */
  PlayerPoseCollection();

  /** Copy-Constructor */
  PlayerPoseCollection(PlayerPoseCollection& other);

  /** Destructor */
  ~PlayerPoseCollection();

  /** sets a player pose of the own team at a given index */
  void setOwnPlayerPose(int index, const PlayerPose& pose) {ownPlayerPoses[index] = pose;}

  /** sets a player pose of the own team at a given index */
  void setOwnPlayerPose(int index, const double& x, const double& y, const double& rotation) 
  {ownPlayerPoses[index].setPose(x,y,rotation);}

  /** sets a player pose of the own team at a given index */
  void setOwnPlayerPose(int index, const double& x, const double& y, const double& rotation, const double& validity) 
  {
    ownPlayerPoses[index].setPose(x,y,rotation);
    ownPlayerPoses[index].setValidity(validity);
  }

  /** returns a player pose of the own team for a given index */
  const PlayerPose& getOwnPlayerPose(int index) const {return ownPlayerPoses[index];}

  /** sets a player pose of the opponent team at a given index */
  void setOpponentPlayerPose(int index, const PlayerPose& pose) {opponentPlayerPoses[index] = pose;}

  /** sets a player pose of the opponent team at a given index */
  void setOpponentPlayerPose(int index, const double& x, const double& y, const double& rotation) 
  {opponentPlayerPoses[index].setPose(x,y,rotation);}

  /** sets a player pose of the opponent team at a given index */
  void setOpponentPlayerPose(int index, const double& x, const double& y, const double& rotation, const double& validity) 
  {
    opponentPlayerPoses[index].setPose(x,y,rotation);
    opponentPlayerPoses[index].setValidity(validity);
  }
  
  /** returns a player pose of the opponent team for a given index */
  const PlayerPose& getOpponentPlayerPose(int index) const {return opponentPlayerPoses[index];}

  /** copies another PlayerPoseCollection to this one */
  void operator = (const PlayerPoseCollection& other);

  /** Number of located opponent robots */
  int numberOfOpponentPlayers;

  /** Number of located own robots */
  int numberOfOwnPlayers;

  /** give the In operator acess to protected members */
  friend In& operator>>(In& stream,PlayerPoseCollection& playerPoseCollection);

  /** give the Out operator acess to protected members */
  friend Out& operator<<(Out& stream, const PlayerPoseCollection& playerPoseCollection);

protected:
  /** A list of located opponent robots */
  PlayerPose opponentPlayerPoses[4];

  /** A list of located own robots */
  PlayerPose ownPlayerPoses[3];
};

/**
 * Streaming operator that reads a PlayerPoseCollection from a stream.
 * @param stream The stream from which is read.
 * @param playerPoseCollection The PlayerPoseCollection object.
 * @return The stream.
 */ 
In& operator>>(In& stream,PlayerPoseCollection& playerPoseCollection);
 
/**
 * Streaming operator that writes a PlayerPoseCollection to a stream.
 * @param stream The stream to write on.
 * @param playerPoseCollection The PlayerPoseCollection object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const PlayerPoseCollection& playerPoseCollection);


#endif //__PlayerPoseCollection_h_

/*
 * Change log :
 * 
 * $Log: PlayerPoseCollection.h,v $
 * Revision 1.1.1.1  2004/05/22 17:25:15  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/01/19 14:55:23  dueffert
 * all frameNumbers (and not only some of them) are unsigned long now
 *
 * Revision 1.3  2003/12/06 17:45:33  loetzsch
 * replaced Player::playerRole (goalie, defender, striker1, striker2)
 * by Player::playerNumber (one, two, three, four)
 *
 * Revision 1.2  2003/11/18 16:09:10  goehring
 * frameNumber added
 *
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.7  2003/05/13 14:32:33  mkunz
 * added setOwnPlayerPose-method with validity
 *
 * Revision 1.6  2003/04/19 03:17:59  pruente
 * Merged in changes by Carsten: - Added collision prohibition
 *
 *
 * Revision 1.5  2003/03/16 20:15:12  thiel
 * changed set-methods from const double& to const double
 * operator= completed with missing attributs
 *
 * Revision 1.4  2003/03/12 13:33:29  schmidt
 * Added some result data from the sensor fusion merge process to the playerPose.
 *
 * Revision 1.3  2003/02/27 10:06:59  schmidt
 * Added three variants of a SensorFusionPlayersLocator.
 *
 * Revision 1.2  2002/09/22 18:40:51  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:26:39  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.2  2002/06/10 14:15:34  kspiess
 * new set methode
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.8  2002/02/05 03:36:12  loetzsch
 * replaced direct member access by
 * inline const VALUE& get...() const   and
 * inline void set...(const Value&) methods.
 *
 * Revision 1.7  2002/02/03 14:37:58  juengel
 * Drawing of the world state removed from Berlin2001BehaviorControl.
 * Drawing method for world states added to PaintMethods.
 * Drawing of the world state added to the Processes with BehaviorControl.
 *
 * Revision 1.6  2002/01/15 10:15:30  kspiess
 * Streaming-Operatoren fr PlayerPose hinzugefgt
 *
 * Revision 1.5  2002/01/13 14:08:57  kspiess
 * Pose2D eingebaut
 *
 * Revision 1.4  2001/12/20 23:15:17  kspiess
 * PlayerPoseCollection implementiert
 *
 * Revision 1.3  2001/12/10 17:47:06  risler
 * change log added
 *
 */
