/**
* @file BallModel.cpp
*
* Implementation of class BallModel.
*
* @author <A href=mailto:marc.dassler@web.de>Marc Dassler</A>
*/



#include "LandmarksState.h"

const double LandmarksState::BEACON = 1;
const double LandmarksState::GOAL = 1;
const double LandmarksState::LINE = 0.5;
const double LandmarksState::EDGE = 0.75;

LandmarksState::LandmarksState()
{
  initLandmarks();
  lastSeenBeacon = 0;
  reset();
}

void LandmarksState::initLandmarks()
{

  // BEACON
  // decide which color is on opponent side
  if (getPlayer().getTeamColor() == Player::red)
  {
    landmarks[0] =  LandmarkState(Vector3<double>(xPosFrontFlags, yPosLeftFlags,  flagHeight+ flagColorHeight),Flag::pinkAboveSkyblue,BEACON);
    landmarks[1] =  LandmarkState(Vector3<double>(xPosFrontFlags, yPosRightFlags,  flagHeight+ flagColorHeight),Flag::skyblueAbovePink,BEACON);
    landmarks[2] =  LandmarkState(Vector3<double>(xPosBackFlags, yPosLeftFlags,  flagHeight+ flagColorHeight),Flag::yellowAbovePink,BEACON);
    landmarks[3] =  LandmarkState(Vector3<double>(xPosBackFlags, yPosRightFlags,  flagHeight+ flagColorHeight),Flag::pinkAboveYellow,BEACON);
  }
  else
  {
    landmarks[0] =  LandmarkState(Vector3<double>(xPosFrontFlags, yPosLeftFlags,  flagHeight+ flagColorHeight),Flag::pinkAboveYellow,BEACON);
    landmarks[1] =  LandmarkState(Vector3<double>(xPosFrontFlags, yPosRightFlags,  flagHeight+ flagColorHeight),Flag::yellowAbovePink,BEACON);
    landmarks[2] =  LandmarkState(Vector3<double>(xPosBackFlags, yPosLeftFlags,  flagHeight+ flagColorHeight),Flag::pinkAboveSkyblue,BEACON);
    landmarks[3] =  LandmarkState(Vector3<double>(xPosBackFlags, yPosRightFlags,  flagHeight+ flagColorHeight),Flag::skyblueAbovePink,BEACON);
  }
 
  // OWN PENALTY AREA
  landmarks[4] =  LandmarkState(Vector3<double>(xPosOwnPenaltyArea, yPosLeftPenaltyArea, 0.0),-1, EDGE);
  landmarks[5] =  LandmarkState(Vector3<double>(xPosOwnPenaltyArea, yPosRightPenaltyArea, 0.0),-1, EDGE);

  // OPPONENT PENALTY AREA
  landmarks[6] =  LandmarkState(Vector3<double>(xPosOpponentPenaltyArea, yPosLeftPenaltyArea, 0.0),-1, EDGE);
  landmarks[7] =  LandmarkState(Vector3<double>(xPosOpponentPenaltyArea, yPosRightPenaltyArea, 0.0),-1, EDGE);
  
  // CENTER LINE AND EDGES
  landmarks[8] =  LandmarkState(Vector3<double>(xPosHalfWayLine, 0.0, 0.0),-1, LINE);
  landmarks[9] =  LandmarkState(Vector3<double>(xPosOpponentPenaltyArea, yPosLeftGroundline, 0.0),-1, EDGE);
  landmarks[10]=  LandmarkState(Vector3<double>(xPosOpponentPenaltyArea, yPosRightGroundline, 0.0),-1, EDGE);

  // GOALS
  landmarks[11]=  LandmarkState(Vector3<double>(xPosOpponentGoal, 0.0, 0.0),-1, GOAL);
  landmarks[12]=  LandmarkState(Vector3<double>(xPosOpponentGoal, 0.0, 0.0),-1, GOAL);

}

void LandmarksState::reset()
{
  for (int i=0; i<numOfLandmarks; i++) 
  {
    landmarks[i].visible = false;
    landmarks[i].timeWhenLastSeen = 0;
  }

}

bool LandmarksState::seenAnyBeacon() const
{
	bool ret=false;
	for (int i=0; i<Flag::numberOfFlagTypes; i++) 
    ret = ret | landmarks[i].visible;
	return ret;
}

bool LandmarksState::seenAnyFlag() const
{
  return seenAnyBeacon() | landmarks[11].visible | landmarks[12].visible;
}

int LandmarksState::lastSeenBeaconIndex() const
{
  long theMax = 0;
  int ret=0;
  for (int i=0;i<Flag::numberOfFlagTypes;i++)
  {
    if (landmarks[i].timeWhenLastSeen>theMax)
    {
      theMax = landmarks[i].timeWhenLastSeen;
      ret = i;
    }
  }
  return ret;

}
long LandmarksState::timeOfLastSeenBeacon(int beaconIndex) const
{
  return landmarks[beaconIndex].timeWhenLastSeen;
}

long LandmarksState::timeBetweenSeen2LastBeacons(int beaconIndex) const
{
  //calculate the time difference between the last seen and the others
  long minTime = 1000000;
  for (int i=0;i<Flag::numberOfFlagTypes;i++)
    if (beaconIndex!=i)
      minTime = min(landmarks[beaconIndex].timeWhenLastSeen-landmarks[i].timeWhenLastSeen,minTime);
  return minTime;

}
void LandmarksState::update(const LandmarksPercept& landmarksPercept)
{
  int i;
  for (i=0; i<Flag::numberOfFlagTypes; i++) landmarks[i].visible = false;

  for (i=0; i<landmarksPercept.numberOfFlags; i++) 
    setLandmarkVisibleByBeaconType(landmarksPercept.flags[i].type);
}

void LandmarksState::setLandmarkVisibleByBeaconType(int type)
{
  for (int i=0;i<Flag::numberOfFlagTypes;i++)
    if (landmarks[i].beaconColor == type)
    {
      landmarks[i].visible = true;
      landmarks[i].timeWhenLastSeen = SystemCall::getCurrentSystemTime();
      break;
    }
}

In& operator>>(In& stream,LandmarksState& landmarksState)
{
  for(int i=0; i<landmarksState.numOfLandmarks; i++) 
    stream >> landmarksState.landmarks[i];
  return stream;
}

Out& operator<<(Out& stream, const LandmarksState& landmarksState)
{
  for(int i=0; i<landmarksState.numOfLandmarks; i++) 
    stream << landmarksState.landmarks[i];
  return stream;
}


/*
* Change log :
*
* $Log: LandmarksState.cpp,v $
* Revision 1.5  2004/06/28 09:46:57  dassler
* introduced some more headcontrol symbols
* time-since-last-seen-beacon
* time-between-last-beacons
*
* Revision 1.4  2004/06/17 17:33:28  dassler
* New Line added
*
* Revision 1.3  2004/06/17 15:39:33  dassler
* LandmarkState added
*
*/

