/** 
* @file Logger.cpp
* 
* Implementation of class Logger.
*
* @author Max Risler
*/
 
#include "Logger.h"
#include "Tools/RobotConfiguration.h"
#include "Tools/Debugging/Debugging.h"
#include "Platform/GTAssert.h"
#include "Platform/SystemCall.h"

Logger::Logger() : 
  INIT_DEBUGGING,
  INIT_RECEIVER_SENSORDATA(true),

  mode(QueueFillRequest::immediateReadWrite),
  saveOrSendTime(0),
  fin(0)
{
  debugIn.setSize(400000);
  debugOut.setSize(2000000);
}

Logger::~Logger() 
{
  if (fin)
    delete(fin);
}

int Logger::main() 
{
  WATCH(sendSensorData,idSensorData,bin,theSensorDataBufferReceiver);

  //if somebody attached messages to that file, process them now
  if ((fin->exists())&&(! fin->eof()))
  {
    *fin >> debugIn;
    debugIn.handleAllMessages(*this);
    debugIn.clear();
  }


  INFO(printRobotStatus,idText,text,"Free mem: "<<SystemCall::getFreeMem()<<" bytes"<<endl<<"Remaining capacity: "<<SystemCall::getRemainingPower()<<"%"<<endl << "MAC Address: " << getRobotConfiguration().getMacAddressString());
  
  switch (mode)
  {
  case QueueFillRequest::overwriteOlder:
    debugOut.removeRepetitions();
    break;
  case QueueFillRequest::rejectAll:
    debugOut.clear();
    break;
  case QueueFillRequest::toStickNSeconds:
    if ((SystemCall::getCurrentSystemTime() > saveOrSendTime)&&(saveOrSendTime != 0))
    {
      // save the outgoing queue to the memory stick
      if (!debugOut.isEmpty())
      {
        OutBinaryFile fout("Logfile.log");
        fout << debugOut;
        debugOut.clear();
      }
      saveOrSendTime = 0;
    }
    break;
  case QueueFillRequest::toStickImmediately:
      // save the outgoing queue to the memory stick
      if (!debugOut.isEmpty())
      {
        OutBinaryFile fout("Logfile.log");
        fout << debugOut;
      }
     break;
  }
  
#if defined(_WIN32) || defined(WLAN)
  // send the outgoing queue to RobotControl 
  // (direct in WIN32, via TcpGateway in WLAN)
  if ((mode==QueueFillRequest::immediateReadWrite)||
      ((mode==QueueFillRequest::collectNSeconds)&&(saveOrSendTime < SystemCall::getCurrentSystemTime()))||
      (mode==QueueFillRequest::overwriteOlder))
  {
#ifndef _WIN32
    if (messageWasReceived) // Hallo Hack
#endif
      theDebugSender.send();
  }
#endif

  return 1; // Wait at least 1 millisecond, then recall main()
}

void Logger::init()
{
  fin = new InBinaryFile("requests.dat");
  if ((fin->exists())&&(! fin->eof()))
  {
    *fin >> debugIn;
  }
  debugIn.handleAllMessages(*this);
  debugIn.clear();
  messageWasReceived = false;
}

bool Logger::handleMessage(InMessage& message)
{
  messageWasReceived = true;

  switch(message.getMessageID())
  {
  case idQueueFillRequest:
  	{
    QueueFillRequest qfr;
    message.bin >> qfr;
    mode = qfr.mode;
    saveOrSendTime = 1000*qfr.seconds + SystemCall::getCurrentSystemTime();
    return true;
    }
  default:
    return Process::handleMessage(message);
  }
}

MAKE_PROCESS(Logger);

/*
 * Change log :
 * 
 * $Log: Logger.cpp,v $
 * Revision 1.2  2004/06/16 21:48:22  goehring
 * saveToStickImmediately added, saveToStickAfterNSeconds deletes queue
 *
 * Revision 1.1.1.1  2004/05/22 17:24:57  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1  2003/07/24 13:33:15  risler
 * new process logger
 *
 */
