/**
* @file Platform/Win32Linux/MessageQueueBase.h
* 
* Declaration of class MessageQueueBase for Win32.
*
* @author Martin Ltzsch
*/

#ifndef __MessageQueueBase_h_
#define __MessageQueueBase_h_

#include "Tools/MessageQueue/MessageIDs.h"
#include "Tools/List.h"
#include "Tools/Player.h"

/**
* @class Message
*
* Represents a single message that can be stored in the MessageQueueBase.
*
* @author Martin Ltzsch
*/
class Message
{
public:
  /** Constructor */
  Message();

  /** Destructor */
  ~Message();

  /** returns the size of the message */
  int getSize() const;

  /** returns the time stamp of the message */
  unsigned long getTimeStamp() const;

  /** sets the time stamp of the message */
  void setTimeStamp(unsigned long timeStamp);

  /** returns the message id of the message */
  MessageID getMessageID() const;

  /** sets the message id of the message */
  void setMessageID(MessageID id);

  /** returns the team color of the robot that sent this message. Can be undefined. */
  Player::teamColor getTeamColor() const;

  /** sets the team color of the robot that sent this message. Can be undefined. */
  void setTeamColor(Player::teamColor teamColor);

  /** returns the player number of the robot that sent this message. Can be undefined. */
  Player::playerNumber getPlayerNumber() const;

  /** sets the player number of the robot that sent this message. Can be undefined. */
  void setPlayerNumber(Player::playerNumber playerNumber);

  /** returns whether the message was sent from a physical robot */
  bool getMessageWasSentFromAPhysicalRobot() const;

  /** sets whether the message was sent from a physical robot */
  void setMessageWasSentFromAPhysicalRobot(bool b);

  /** resets the read position to 0 */
  void resetReadPosition();

  /**
  * The function adds a number of bytes to the message.
  * @param p The address the data is located at.
  * @param s The number of bytes to be written.
  */
  virtual void write(const void* p,int s);

  /** 
  * Reads a number of bytes from message.
  * @param p The address the data is written to. Note that p
  *          must point to a memory area that is at least
  *          "s" bytes large.
  * @param s The number of bytes to be read.
  */
  void read(void* p,int s); 

  /**
  * Gives the MessageQueue direct data access for reading to a message.
  * @return the address of the first byte of the message
  */
  const char* getData() const;

  /** Returns if the message was read completely. */
  bool eof() const;

private:
  /** the size of the message */
  int size;

  /** the time stamp of the message */
  unsigned long timeStamp;

  /** the message id of the message */
  MessageID id;

  /** the team color of the robot that sent the message, can be undefined */
  char teamColor;

  /** the player number of the robot that sent the message, can be undefined */
  char playerNumber;

  /** 
  * If true, then the message was sent from a physical robot. 
  * This is always false if messages are set from non-Aperios/OpenR platforms.
  */
  bool messageWasSentFromAPhysicalRobot;

  /** a pointer to the data of the message */
  char *data;

  /** the position where the next bytes are read from the message */
  int readPosition;
};

/**
* @class MessageQueueBase 
*
* The platform dependend base of class MessageQueue.
*
* @author Martin Ltzsch
*/
class MessageQueueBase
{
public:
  /** Constructor */
  MessageQueueBase();

  /** Destructor */
  ~MessageQueueBase();

  /**
  * Sets the size of the queue. Ignored on the Win32 platform.
  * @param size The maximum size of the queue in Bytes. 
  */
  void setSize(int size) {};

  /** Removes all messages from the queue */
  void clear();

  /** Returns the number of the messages in the queue */
  int getNumberOfMessages() const;

  /**
  * The function adds a number of bytes to the last message in the queue.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  */
  virtual void write(const void* p,int size);

  /** 
  * Finishes the last message in the queue. 
  * The number of messages becomes increased and a new message can be started.
  * @param id The type id of the message.
  * @param timeStamp The time stamp of the message
  * @param teamColor the team color of the robot that sent the message
  * @param playerNumber the player number of the robot that sent the message 
  * @param messageWasSentFromAPhysicalRobot If true, then the message was sent 
  *     from a physical robot. Is set to false on all non-Aperios/OpenR platforms. 
  */
  void finishMessage(MessageID id, unsigned long timeStamp,
    Player::teamColor teamColor,Player::playerNumber playerNumber,
    bool messageWasSentFromAPhysicalRobot=false);

  /** Returns if the the currently selected message for reading was read completely. */
  bool eof() const; 

  /** 
  * Reads a number of bytes from the currently selected message for reading 
  * @param p The address the data is written to. Note that p
  *          must point to a memory area that is at least
  *          "size" bytes large.
  * @param size The number of bytes to be read.
  */
  void read(void* p,int size); 

  /** Deletes all older messages from the queue if a newer
    * message of same type is already in the queue.
    * This method should not be called during message handling. */
  void removeRepetitions();

  /**
  * Gives the MessageQueue direct read access to the selected message for reading.
  * @return the address of the first byte of the message
  */
  const char* getData() const;

  /** 
  * Returns the time stamp of a message.
  * @param message the number of the message
  */
  unsigned long getTimeStamp(int message) const;

  /** Returns the message id of the currently selected message for reading */
  MessageID getMessageID() const;

  /** Returns the time stamp of the currently selected message for reading */
  unsigned long getTimeStamp() const;

  /** Returns the message size of the currently selected message for reading */
  int getMessageSize() const;

  /** returns the team color of the robot that sent the currently selected message for reading. */
  Player::teamColor getTeamColor() const;

  /** returns the player number of the robot that sent the currently selected message for reading.  */
  Player::playerNumber getPlayerNumber() const;

  /** returns whether the currently selected message for reading was sent from a physical robot */
  bool getMessageWasSentFromAPhysicalRobot() const;

  /** 
  * Returns the read position of the currently selected message for reading 
  * so that the message can be read again.
  */
  void resetReadPosition();

  /*
  * Sets which message is selected for reading.
  * @param message The number of the message
  */
  void setSelectedMessageForReading(int message);

private:
  /** A List storing all messages */
  List<Message> messages;

  /** the message that is selected for reading */
  List<Message>::Pos selectedMessageForReading;

  /** 
  * Returns a position in the list given by the number of the message
  * @param message the number of the message
  * @return the position of the message
  */
  List<Message>::Pos getPos(int message) const;
};

#endif // __MessageQueueBase_h_

/*
* Change log :
* 
* $Log: MessageQueueBase.h,v $
* Revision 1.1.1.1  2004/05/22 17:24:45  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2003/12/06 23:23:55  loetzsch
* messages in a MessageQueue now contain
* - the team color of the robot which sent the message
* - the player number of the robot which sent the message
* - if the message was sent from a physical robot or not
*
* Revision 1.1  2003/10/07 10:07:00  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:36:26  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2002/12/05 16:13:19  dueffert
* started implementing realtime sending
*
* Revision 1.1  2002/09/10 15:40:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/08/22 14:41:03  risler
* added some doxygen comments
*
* Revision 1.1  2002/08/01 12:52:27  roefer
* RouterCtrl and TcpConnection added
*
* Revision 1.2  2002/07/23 20:43:51  roefer
* selectedMessageForReading is List<Message>::Pos now (is faster)
*
* Revision 1.1  2002/07/23 13:36:39  loetzsch
* - exchanged StaticQueue by MessageQueue with platform dependend
*   base classes
*
*/
