/**
 * @file Platform/Win32/Robot2.h
 *
 * This file declares the class Robot.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#ifndef __Robot_h__
#define __Robot_h__

#include "Controller/Controller.h"
#include "Platform/ProcessFramework.h"
#include "Representations/Perception/Image.h"
#include "Representations/Perception/SensorData.h"
#include "Tools/RobotConfiguration.h"

class OVirtualRobotComm;

/**
 * The class implements a robot as a list of processes.
 */
class Robot : public ProcessList
{
  private:
    OVirtualRobotComm* robotProcess; /**< A pointer to the process that simulates the physical robots. */
    SimObject* obj; /**< A pointer to the associated robot in SimRobot. */
    std::string name; /**< The name of the robot. */
    RobotDesign::Design robotDesign; /**< The design of the robot. */

  public:
    /**
     * Constructor.
     * @param name The name of the robot.
     * @param obj A pointer to the associated robot in SimRobot.
     */
    Robot(const char* name, SimObject* obj, RobotDesign::Design design);

    /**
     * The function updates all sensors and sends motor commands to SimRobot.
     */
    void update();

    /**
     * The function returns the name of the robot.
     * @return The name of the robot.
     */
    const char* getName() const {return name.c_str();}

    /**
     * The function returns the design of the robot.
     * @return The design of the robot.
     */
    RobotDesign::Design getRobotDesign() const {return robotDesign;}

    /**
     * The function returns the SimRobot object representing a robot.
     * @return The SimRobot object pointer that was specified at construction.
     */
    SimObject* getSimRobotObject() const {return obj;}

    /**
     * The function returns a pointer to the process that simulates the physical robots.
     * @return The pointer to the process.
     */
    OVirtualRobotComm* getRobotProcess() const {return robotProcess;}

    /**
     * The function connects a sender and a receiver.
     * @param sender The sender.
     * @param receiver The receiver.
     */
    static void connect(SenderList* sender,ReceiverList* receiver);

    /**
     * The function looks up a sender.
     * @param senderName The Aperios name of the sender. If the process name is missing
     *                   i.e. senderName starts with a dot, the first process with a 
     *                   sender that matches the rest of the name is used.
     * @return A pointer to the sender or 0 if no sender exists with the specified name.
     */
    SenderList* getSender(const char* senderName);

    /**
     * The function looks up a receiver.
     * @param receiverName The Aperios name of the receiver.
     * @return A pointer to the receiver or 0 if no receiver exists with the specified name.
     */
    ReceiverList* getReceiver(const char* receiverName);
};

#endif

/*
 * Changelog:
 * 
 * $Log: Robot2.h,v $
 * Revision 1.1.1.1  2004/05/22 17:24:13  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.5  2004/03/10 23:55:28  roefer
 * ERS7 support for log files
 *
 * Revision 1.4  2004/01/17 19:19:18  roefer
 * Simulator calculates robot pose based on class Kinematics now
 *
 * Revision 1.3  2003/10/21 15:05:51  roefer
 * Added the oracle
 *
 * Revision 1.2  2003/10/20 14:05:54  roefer
 * Views and buttons
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 */
