/**
 * @file Platform/Win32/Robot.h
 *
 * This file declares the class Robot.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#ifndef __ROBOT_H__
#define __ROBOT_H__

#include "Platform/ProcessFramework.h"
#include "Representations/Perception/Image.h"
#include "Representations/Perception/SensorData.h"

class OVirtualRobotComm;

/**
 * The class implements a robot as a list of processes.
 */
class Robot : public ProcessList
{
  private:
    OVirtualRobotComm* robotProcess; /**< A pointer to the process that simulates the physical robots. */
    SIM3DOBJECT obj[5]; /**< A pointer to the associated robot in SimRobot. */
    STRING name;

  public:
    /**
     * Constructor.
     * @param name The name of the robot.
     * @param obj A pointer to the associated robot in SimRobot.
     */
    Robot(const char* name,SIM3DOBJECT obj[5]);

    /**
     * The function updates all sensors and sends motor commands to SimRobot.
     */
    void update();

    /**
     * The function returns the name of the robot.
     * @return The name of the robot.
     */
    const char* getName() const {return name;}

    /**
     * The function returns the SimRobot object representing a robot.
     * @return The SimRobot object pointer that was specified at construction.
     */
    SIM3DOBJECT getSimRobotObject(int i = 0) const {return obj[i];}

    /**
     * The function returns a pointer to the process that simulates the physical robots.
     * @return The pointer to the process.
     */
    OVirtualRobotComm* getRobotProcess() const {return robotProcess;}

    /**
     * Activates or diactivates a certain switch on the robot.
     * @param sensor The switch. Allowed values are: headBack, headFront,
     *               mouth, chin, and back.
     * @param activate Decides whether to activate or deactivate the switch.
     */
    void activateSwitch(SensorData::sensors sensor,bool activate = true);

    /**
     * The function checks whether a certain switch is activated.
     * @param sensor The switch to check.
     * @return Activated or not?
     */
    bool isSwitchActivated(SensorData::sensors sensor) const;

    /**
     * The function connects a sender and a receiver.
     * @param sender The sender.
     * @param receiver The receiver.
     */
    static void connect(SenderList* sender,ReceiverList* receiver);

    /**
     * The function looks up a sender.
     * @param senderName The Aperios name of the sender. If the process name is missing
     *                   i.e. senderName starts with a dot, the first process with a 
     *                   sender that matches the rest of the name is used.
     * @return A pointer to the sender or 0 if no sender exists with the specified name.
     */
    SenderList* getSender(const char* senderName);

    /**
     * The function looks up a receiver.
     * @param receiverName The Aperios name of the receiver.
     * @return A pointer to the receiver or 0 if no receiver exists with the specified name.
     */
    ReceiverList* getReceiver(const char* receiverName);
};

#endif

/*
 * Changelog:
 * 
 * $Log: Robot.h,v $
 * Revision 1.1.1.1  2004/05/22 17:24:13  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:30:28  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2003/04/18 15:21:25  roefer
 * Robot name handling changed
 *
 * Revision 1.2  2002/12/16 14:51:21  dueffert
 * changelog added
 *
 */
