/**
 * @file Platform/Win32/RoboCupCtrl2.cpp
 *
 * This file implements the class RoboCupCtrl.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#include "RoboCupCtrl2.h"
#include "Tools/Streams/InStreams.h"
#include "Representations/Perception/ColorTable64.h"
#include <sys/timeb.h>
#include "Platform/GTAssert.h"
#include "Tools/Location.h"
#include "SimRobotCore/SimObject.h"
#include "SimRobotCore/Surface.h"
#include "Tools/Actorics/Kinematics.h"
#include "Tools/RobotConfiguration.h"

RoboCupCtrl* RoboCupCtrl::controller = 0;
std::string RoboCupCtrl::robotName;

RoboCupCtrl::RoboCupCtrl()
{
  controller = this;
  simTime = false;
  time = 8000 - getCurrentSystemTime();

  adaptColors(RobotDesign::ERS210, colorsERS210);
  colorizeScene(colorsERS210);
  adaptColors(RobotDesign::ERS7, colorsERS7);
  colorizeScene(colorsERS7);

  obj = 0;
  SimObject* group = getObjectReference("GT2004.robots");
  for(currentRobot = 0; currentRobot < group->getNumberOfChildNodes(); ++currentRobot)
  {
    obj = group->getChildNode(currentRobot);
    robotName = obj->getFullName();
    currentDesign = obj->getChildNode(0)->getName() == "body01" ? RobotDesign::ERS7 : RobotDesign::ERS210;
    robots.insert(new Robot(obj->getName().c_str(), obj, currentDesign));
  }

  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    connect(robots[i]);
  
  for(i = robots.getFirst(); i; ++i)
    robots[i].start();

  SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_IDLE);
  lastTime = 0;
}

RoboCupCtrl::~RoboCupCtrl()
{
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    robots[i].announceStop();
  Sleep(1000);
  controller = 0;
}

void RoboCupCtrl::execute()
{
  unsigned t = getCurrentSystemTime();
  if(lastTime + 40 > t)
    Sleep(lastTime + 40 - t);
  lastTime = t;

  for(int j = 0; j < 5; ++j)
  {
    currentRobot = 0;
    for(List<Robot>::Pos i = robots.getFirst(); i; ++i, ++currentRobot)
    {
      colorizeScene(robots[i].getRobotDesign() == RobotDesign::ERS7 ? colorsERS7 : colorsERS210);
      robots[i].update();
    }
    if(simTime)
      time += 8;
  }
  getObjectReference("GT2004.robots")->computeIntersectionSphereRadius();
}

const char* RoboCupCtrl::getRobotName() const
{
  unsigned threadId = GetCurrentThreadId();
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    for(ProcessList::Pos j = robots[i].getFirst(); j; ++j)
      if(robots[i][j].getId() == threadId)
        return robots[i].getName();
  return obj ? obj->getName().c_str() : "AIBO1";
}

int RoboCupCtrl::getRobotIndex() const
{
  unsigned threadId = GetCurrentThreadId();
  int index = 0;
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i, ++index)
    for(ProcessList::Pos j = robots[i].getFirst(); j; ++j)
      if(robots[i][j].getId() == threadId)
        return index;
  ASSERT(currentRobot < ROBOT_MAX);
  return (int) currentRobot; // process is currently in construction
}

int RoboCupCtrl::getProcessIndex() const
{
  unsigned threadId = GetCurrentThreadId();
  int index = 0;
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i, ++index)
    for(ProcessList::Pos j = robots[i].getFirst(); j; ++j, ++index)
      if(robots[i][j].getId() == threadId)
        return index;
  return index + ProcessList::numberOfProcesses; // process is currently in construction
}

RobotDesign::Design RoboCupCtrl::getRobotDesign() const
{
  unsigned threadId = GetCurrentThreadId();
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    for(ProcessList::Pos j = robots[i].getFirst(); j; ++j)
      if(robots[i][j].getId() == threadId)
        return robots[i].getRobotDesign();
  return currentDesign;
}

Pose2D RoboCupCtrl::getPose2D(SimObject* obj)
{
  return Pose2D(obj->getRotation().getZAngle(),
                Vector2<double>(obj->getPosition().v[0], obj->getPosition().v[1]));
}

void RoboCupCtrl::setPose2D(SimObject* obj, const Pose2D& pose, const SensorData* sensorData)
{
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    if(robots[i].getSimRobotObject() == obj)
      break;
  ASSERT(i);
  double angle = 0,
         neck = obj->getPosition().v[2];
  if(sensorData)
  {
    const RobotDimensions& r = getRobotConfiguration().getRobotDimensions();
    RobotVertices rob;
    Kinematics::calcNeckAndLegPositions(*sensorData, rob);
    /*static const SensorData::sensors index[4] =
    {
      SensorData::pawFR,
      SensorData::pawFL,
      SensorData::pawHR,
      SensorData::pawHL
    };
    SensorData& s = const_cast<SensorData&>(*sensorData);
    for(int leg = 0; leg < 4; ++leg)
      s.data[index[leg]] = fabs(rob.footPosition[leg].z + rob.neckHeight) < 1;*/
    angle = rob.bodyTilt;
    neck = (Pose2D(-rob.bodyTilt, Vector2<double>(0, rob.neckHeight)) + Pose2D(Vector2<double>(0, -r.neckToLegsLengthZ)))
            .translation.y;
    neck /= 1000;
  }
  const Robot& robot = robots[i];
  Vector3d position(pose.translation.x, pose.translation.y, neck);
  obj->setPosition(position);
  Matrix3d m;
  m.setRotation(0,0,-obj->getRotation().getZAngle());
  m.rotateY(angle - obj->getRotation().getYAngle());
  m.rotateZ(pose.getAngle());
  obj->rotate(m, obj->getPosition());
}

void RoboCupCtrl::connect(Robot& robot)
{
  const char* gatewaySenderName = "TCPGateway.Sender.",
            * gatewayReceiverName = "TCPGateway.Receiver.";

  // open connect.cfg for actual process layout
  char conncfg[512];
  CString sPathName = AfxGetApp()->m_pszHelpFilePath;
  GetLongPathName(sPathName, conncfg, sizeof(conncfg));
  sPathName = conncfg;
  sPathName = sPathName.Left(sPathName.ReverseFind('.')); // remove .exe
  sPathName = sPathName.Mid(sPathName.ReverseFind('\\') + 1); // remove path
  sPathName = sPathName.Mid(sPathName.Find('r') + 1); // remove Simulator
  if(sPathName == "")
  {
    sPathName = conncfg;
    sPathName = sPathName.Left(sPathName.ReverseFind('\\')); // remove \Simulator.exe
    sPathName = sPathName.Mid(sPathName.ReverseFind('\\') + 1); // remove path
  }
  sprintf(conncfg, "%s/Src/Processes/%s/connect.cfg", File::getGTDir(), (const char*) sPathName);
  InConfigFile stream(conncfg, "ERS-210");
  ASSERT(stream.exists());

  // attach receivers to senders
  char senderName[NAME_LENGTH_MAX],
       receiverName[NAME_LENGTH_MAX];
  while(!stream.eof())
  {
    stream >> senderName >> receiverName;
    SenderList* sender = 0;
    if(strncmp(senderName, gatewaySenderName, strlen(gatewaySenderName)) == 0)
    {
      std::string packageName = &senderName[strlen(gatewaySenderName)];
      packageName = packageName.substr(0, packageName.size() - 2);
      SimObject* obj = robot.getSimRobotObject();
      if(obj)
      {
        std::string netReceiverName = obj->getName() + ".Receiver." +
                                      packageName + ".O";
        char robconcfg[512];
        sprintf(robconcfg, "%s/Bin/connect-for-RobotControl.cfg", File::getGTDir());
        std::string netSenderName = getSenderName(robconcfg, netReceiverName.c_str());
        if(netSenderName.size())
          for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
          {
            SimObject* obj = robots[i].getSimRobotObject();
            if(obj)
            {
              std::string otherRobotName = obj->getName() + ".";
              if(strncmp(netSenderName.c_str(), otherRobotName.c_str(), otherRobotName.size()) == 0)
              {
                std::string otherSenderName = 
                  getSenderName(conncfg, (std::string(gatewayReceiverName) + packageName + ".O").c_str());
                if(otherSenderName.size())
                  sender = robots[i].getSender(otherSenderName.c_str());
                break;
              }
            }
          }
      }
    }
    else
      sender = robot.getSender(senderName);

    if(strncmp(receiverName, gatewayReceiverName, strlen(gatewayReceiverName)) != 0)
      Robot::connect(sender, robot.getReceiver(receiverName));
  }
}

std::string RoboCupCtrl::getSenderName(const char* fileName, const char* searchedReceiverName)
{
  // open connect.cfg for actual process layout
  InConfigFile stream(fileName, "ERS-210");
  if(stream.exists())
  {
    char senderName[NAME_LENGTH_MAX],
         receiverName[NAME_LENGTH_MAX];
    while(!stream.eof())
    {
      stream >> senderName >> receiverName;
      if(strcmp(receiverName, searchedReceiverName) == 0)
        return senderName;
    }
  }
  OutputDebugString(searchedReceiverName);
  OutputDebugString(" not found\n");
  return "";
}

unsigned RoboCupCtrl::getCurrentSystemTime() const
{
  _timeb sysTime;
  _ftime(&sysTime);
  return sysTime.time * 1000 + sysTime.millitm;
}

unsigned RoboCupCtrl::getTime() const
{
  if(simTime)
    return unsigned(time);
  else
    return unsigned(getCurrentSystemTime() + time);
}

void RoboCupCtrl::adaptColors(RobotDesign::Design design, Vector3d* colors)
{
  int c;
  for(c = 1; c < numOfColors; ++c)
    colors[c].v[0] = -1;
    
  ColorTable64 colorTable;
  InBinaryFile stream(getLocation().getModelFilename("coltable.c64", design));
  if(!stream.exists())
    return;
  stream >> colorTable;
  int rSum[numOfColors],
      gSum[numOfColors],
      bSum[numOfColors],
      count[numOfColors];
  for(c = 0; c < numOfColors; ++c)
    rSum[c] = gSum[c] = bSum[c] = count[c] = 0;
  for(int r = 0; r < 255; r += 4)
    for(int g = 0; g < 255; g += 4)
      for(int b = 0; b < 255; b += 4)
      {
        colorClass c = getColorClass(colorTable, r, g, b);
        ASSERT(unsigned(c) < numOfColors);
        rSum[c] += r;
        gSum[c] += g;
        bSum[c] += b;
        ++count[c];
      }
  for(c = 1; c < numOfColors; ++c)
    if(count[c])
    {
      rSum[c] /= count[c];
      gSum[c] /= count[c];
      bSum[c] /= count[c];
      if(getColorClass(colorTable, rSum[c], gSum[c], bSum[c]) != c)
      {
        int rCenter = rSum[c],
            gCenter = gSum[c],
            bCenter = bSum[c],
            minDist = 1000000;
        for(int r = 0; r < 255; r += 4)
          for(int g = 0; g < 255; g += 4)
            for(int b = 0; b < 255; b += 4)
            {
              if(getColorClass(colorTable, r, g, b) == c)
              {
                int d = (r - rCenter) * (r - rCenter) +
                        (g - gCenter) * (g - gCenter) +
                        (b - bCenter) * (b - bCenter);
                if(d < minDist)
                {
                  minDist = d;
                  rSum[c] = r;
                  gSum[c] = g;
                  bSum[c] = b;
                }
              }
            }
        
      }
      colors[c] = Vector3d(rSum[c] / 255.0, gSum[c] / 255.0, bSum[c] / 255.0);
    }
}

void RoboCupCtrl::colorizeScene(const Vector3d* colors)
{
  for(int c = 1; c < numOfColors; ++c)
  {
    Surface* surface = getSurface(ColorClasses::getColorName((colorClass) c));
    if(surface && colors[c].v[0] != -1)
      surface->setColor(colors[c]);
  }
  setBackgroundColor(colors[white] * 0.8);
}

colorClass RoboCupCtrl::getColorClass(const ColorTable& colorTable, int r, int g, int b) const
{
  int y =       (int)( 0.2990 * r + 0.5870 * g + 0.1140 * b),
      v = 127 + (int)(-0.1687 * r - 0.3313 * g + 0.5000 * b),
      u = 127 + (int)( 0.5000 * r - 0.4187 * g - 0.0813 * b);
  if(y < 0) 
    v = 0;
  else if(y > 255)
    v = 255;
  if(u < 0) 
    u = 0; 
  else if(u > 255)
    u = 255;
  if(v < 0)
    v = 0;
  else if(v > 255)
    v = 255;
  return colorTable.getColorClass(y, u, v);
}

/*
 * Change log :
 *
 * $Log: RoboCupCtrl2.cpp,v $
 * Revision 1.5  2004/06/03 15:31:10  roefer
 * Origin of ERS-7 corrected (for oracle)
 *
 * Revision 1.4  2004/05/30 14:32:42  roefer
 * Debugging output removed
 *
 * Revision 1.3  2004/05/27 13:47:59  roefer
 * Adaptation to new location handling
 *
 * Revision 1.2  2004/05/27 10:08:08  thomas
 * added model-specific locations
 *
 * Revision 1.1.1.1  2004/05/22 17:24:13  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.19  2004/04/09 11:35:53  roefer
 * Bremen Byters German Open check-in
 *
 * Revision 1.19  2004/04/02 00:12:19  roefer
 * Crash fixed
 *
 * Revision 1.18  2004/03/27 10:15:12  roefer
 * Outcommented part added
 *
 * Revision 1.17  2004/03/20 09:55:26  roefer
 * Preparation for improved odometry
 *
 * Revision 1.16  2004/03/10 23:55:28  roefer
 * ERS7 support for log files
 *
 * Revision 1.15  2004/02/29 14:54:40  roefer
 * Simulator performs a maximum of 25 simulation steps per second
 *
 * Revision 1.14  2004/01/28 21:55:50  roefer
 * RobotDimensions revised
 *
 * Revision 1.13  2004/01/28 15:36:29  tim
 * adapted to SimRobXP rotation changes
 *
 * Revision 1.12  2004/01/23 00:13:23  roefer
 * ERS-7 simulation, first draft
 *
 * Revision 1.11  2004/01/17 21:34:29  roefer
 * Simulator calculates robot pose based on class Kinematics now
 *
 * Revision 1.10  2004/01/17 20:09:03  roefer
 * Simulator calculates robot pose based on class Kinematics now
 *
 * Revision 1.9  2004/01/17 19:19:18  roefer
 * Simulator calculates robot pose based on class Kinematics now
 *
 * Revision 1.8  2003/12/30 23:56:45  roefer
 * Determination of process layout did not work when SimRobot was started via scene file
 *
 * Revision 1.7  2003/12/29 20:57:12  roefer
 * Surrounded SimRobot weakness
 *
 * Revision 1.6  2003/12/09 21:17:55  roefer
 * Platform-dependent code moved to GUI (release code) project
 *
 * Revision 1.5  2003/12/03 23:59:26  roefer
 * Compatibility with VC2003.NET, GUI works now
 *
 * Revision 1.4  2003/10/26 08:57:15  roefer
 * Symbol SIMROBXP removed
 *
 * Revision 1.3  2003/10/25 11:52:20  roefer
 * Simulator is SimRobXP now
 *
 * Revision 1.2  2003/10/21 22:37:46  roefer
 * Removed some outcommented code
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 */
