/**
 * @file Platform/Win32/RoboCupCtrl.h
 *
 * This file declares the class RoboCupCtrl.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#ifndef __RoboCupCtrl_H__
#define __RoboCupCtrl_H__

#include "RobotControl/SimRobot/Controller.h"
#include "SimRob95/SimRobot/sim3DMoveable.h"
#include "Robot.h"
#include "Representations/Perception/SensorData.h"
#include "Tools/List.h"
#include "Tools/Math/Pose2D.h"


/**
 * The class implements the SimRobot controller for RoboCup.
 */
class RoboCupCtrl : public CONTROLLER
{
  private:
    DECLARE_SYNC;
    static RoboCupCtrl* controller; /**< A pointer to the SimRobot controller. */
    SIM3DOBJECT objDrag; /**< The currently dragged robot. */
    bool drag; /**< Drag or rotate? */
    double angleOffset; /**< Offset between click position and robot orientation. */
    SIM3DOBJECT obj; /**< The current robot constructed. */
  protected:
    static STRING robotName; /**< The name of the robot currently constructed. */
    List<Robot> robots; /**< The list of all robots. */
    VECTOR posOffset; /**< Offset between click position and robot center. */
    bool simTime; /**< Switches between simulation time mode and real time mode. */
    int time; /**< The simulation time. */

    /**
     * The function determines where the user clicked on the field.
     * @param info Information about the click position and the view state.
     * @return The position the user clicked on.
     */
    VECTOR CalcClickPosition(const CLICKINFO& info);

    /**
     * The function returns the current system time.
     * @return The time in milliseconds.
     */
    unsigned getCurrentSystemTime() const;

   /**
     * The function connects all senders of a robot.
     * @param robot The robot.
     */
    void connect(Robot& robot);

   private:
    /**
     * The function searches for a sender that sends to a certain receiver.
     * @param fileName The name of the file to search.
     * @param searchedReceiverName The name of the receiver.
     * @return The name of the sender or "" if it has not been found.
     */
    STRING getSenderName(const char* fileName,const char* searchedReceiverName);

  public:
    /**
     * Constructor.
     */
    RoboCupCtrl();

    /**
     * Destructor.
     */
    ~RoboCupCtrl();

    /**
     * The function is called in each simulation step.
     * @param message Returns a message to the caller that will be displayed
     *                in the status bar.
     * @return Shall the simulation continue?
     */
    BOOLEAN SetActors(STRING& message);

    /**
     * The function is called when the user starts to drag with the mouse in the object view.
     * @param ci Information about the position the user clicked at in the view.
     */
    void OnBeginDrag(const CLICKINFO& ci);

    /**
     * The function is called while the user drags with the mouse in the object view.
     * @param ci Information about the position the user clicked at in the view.
     */
    void OnDrag(const CLICKINFO& ci);

    /**
     * The function is called when the user finishes to drag with the mouse in the object view.
     * @param ci Information about the position the user clicked at in the view.
     */
    void OnEndDrag(const CLICKINFO& ci);

    /**
     * The function returns a pointer to the SimRobot controller.
     * @return The pointer to the SimRobot controller.
     */
    static RoboCupCtrl* getController() {return controller;}

    /**
     * The function returns the name of the robot currently constructed.
     * @return The name of the robot.
     */
    static STRING& getRobotName() {return robotName;}

    /**
     * The function returns the SimRobot object representing the robot associated to the current thread.
     * @return The SimRobot object pointer of the robot.
     */
    SIM3DOBJECT getSimRobotObject() const;

    /**
     * The function returns the index of the robot associated to the current thread.
     * @return The index of the robot. It refers to the sequence in which the robots where defined.
     */
    int getRobotIndex() const;

    /**
     * The function returns the index of the current process.
     * @return The index of the process. It refers to the sequence in which the processes where defined.
     */
    int getProcessIndex() const;

    /**
     * The function determines the 2-D pose of a SimRobot object.
     * @param obj The object of which the pose will be determined.
     * @return The 2-D pose of the specified object.
     */
    Pose2D getPose2D(SIM3DOBJECT obj);

    /**
     * The function sets the 2-D pose of a SimRobot object.
     * @param obj The object of which the pose will be determined.
     * @param pose The desired 2-D pose of the specified object.
     */
    void setPose2D(SIM3DOBJECT obj,const Pose2D& pose);

    /**
     * Activates or diactivates a certain switch on the robot.
     * @param sensor The switch. Allowed values are: headBack, headFront,
     *               mouth, chin, and back.
     * @param activate Decides whether to activate or deactivate the switch.
     */
    void activateSwitch(SensorData::sensors sensor,bool activate = true)
      {robots[robots.getFirst()].activateSwitch(sensor,activate);}

    /**
     * The function checks whether a certain switch is activated.
     * @param sensor The switch to check.
     * @return Activated or not?
     */
    bool isSwitchActivated(SensorData::sensors sensor) const
      {return robots[robots.getFirst()].isSwitchActivated(sensor);}

    /**
     * The function returns the simulation time.
     * @return The pseudo-time in milliseconds.
     */
    unsigned getTime() const;

    /**
     * The function adapts the colors of the simulation to the color table.
     */
    void adaptColors();
};

#endif

/*
 * Change log :
 *
 * $Log: RoboCupCtrl.h,v $
 * Revision 1.1.1.1  2004/05/22 17:24:01  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/10/26 08:57:15  roefer
 * Symbol SIMROBXP removed
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/26 15:30:28  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.2  2003/09/26 11:34:07  juengel
 * no message
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.7  2003/03/23 17:08:18  roefer
 * New player selection in simulation
 *
 * Revision 1.6  2002/12/18 16:22:55  dueffert
 * doxygen docu corrected
 *
 * Revision 1.5  2002/12/07 18:43:55  roefer
 * SimGT2003 log file handling added
 *
 * Revision 1.4  2002/09/22 18:40:55  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.3  2002/09/19 12:05:48  loetzsch
 * MathLib -> Mathlib
 *
 * Revision 1.2  2002/09/18 16:35:12  loetzsch
 * made GT2003 compilable,
 * rechanged message ids
 *
 * Revision 1.1  2002/09/10 15:40:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.4  2002/08/29 14:01:43  dueffert
 * includes in correct case, system includes in <>
 *
 * Revision 1.3  2002/08/04 17:53:18  roefer
 * SimGT2002 connection to physical robots added
 *
 * Revision 1.2  2002/07/09 20:08:26  roefer
 * Extract simulation colors from color table
 *
 * Revision 1.1  2002/06/28 10:26:21  roefer
 * OUTPUT is possible in constructors
 *
 * Revision 1.3  2002/06/01 13:36:48  roefer
 * Inter-robot communication added to RobotControl
 *
 * Revision 1.2  2002/05/16 22:36:11  roefer
 * Team communication and GTMath bugs fixed
 *
 * Revision 1.3  2002/04/25 14:50:35  kallnik
 * changed double/float to double
 * added several #include GTMath
 *
 * PLEASE use double
 *
 * Revision 1.2  2002/04/23 10:38:29  risler
 * renamed headOdometry to headState
 *
 * Revision 1.1  2002/04/20 15:52:20  roefer
 * Project simpified, WATCH and WATCH_PART added
 *
 * Revision 1.9  2002/04/18 10:25:34  roefer
 * Bremen GO
 *
 * Revision 1.8  2002/04/06 09:55:53  roefer
 * Image and SensorData path through DebugQueues changed
 *
 * Revision 1.7  2002/03/24 18:15:01  loetzsch
 * continued change to blocking sensor data receivers
 *
 * Revision 1.6  2002/03/05 13:41:34  roefer
 * Button-API for OVirtualRobotComm
 *
 * Revision 1.5  2002/02/05 18:08:40  roefer
 * PlatformProcess::getRobotIndex() inserted
 *
 * Revision 1.4  2002/01/25 15:40:14  roefer
 * The Oracle
 *
 * Revision 1.3  2002/01/22 14:39:17  roefer
 * Drag and drop for robots and ball
 *
 * Revision 1.2  2002/01/20 23:34:28  loetzsch
 * Sending images and sensor data to processes running in RobotControl now possible
 *
 * Revision 1.1  2002/01/14 22:40:07  loetzsch
 * SimRobot / RobotControl integration started
 *
 *
 * Revision 1.3  2001/12/10 17:47:08  risler
 * change log added
 *
 */
